/**
 * Panadite Academy - Student Portal
 * Modern UI Interactions
 * Version: 2.0
 */

document.addEventListener('DOMContentLoaded', function() {
    // Initialize all UI components
    initSidebar();
    initNavbar();
    initTooltips();
    initPopovers();
    initDarkMode();
    
    // Add ripple effect to buttons
    document.querySelectorAll('.btn').forEach(button => {
        button.addEventListener('click', createRipple);
    });
    
    // Listen for window resize events
    window.addEventListener('resize', handleWindowResize);
    handleWindowResize();
});

/**
 * Sidebar functionality
 */
function initSidebar() {
    const sidebar = document.querySelector('.sidebar');
    const sidebarToggle = document.querySelector('.sidebar-toggle');
    const mobileToggle = document.querySelector('.navbar-mobile-toggle');
    const body = document.body;
    
    // Create backdrop for mobile
    const backdrop = document.createElement('div');
    backdrop.className = 'sidebar-backdrop';
    body.appendChild(backdrop);
    
    // Remove any stored sidebar collapsed preference
    localStorage.removeItem('sidebar-collapsed');
    
    // Remove sidebar-collapsed class if it exists
    body.classList.remove('sidebar-collapsed');
    
    // Initial state - ensure sidebar is hidden on mobile
    const isMobile = window.innerWidth <= 991.98;
    if (isMobile) {
        sidebar.classList.remove('show');
        backdrop.classList.remove('show');
        body.classList.remove('sidebar-open');
    }
    
    // Toggle sidebar only on mobile devices
    if (sidebarToggle) {
        sidebarToggle.addEventListener('click', function() {
            if (window.innerWidth <= 991.98) {
                // Only toggle on mobile
                sidebar.classList.toggle('show');
                backdrop.classList.toggle('show');
            }
        });
    }
    
    // Toggle sidebar on mobile
    if (mobileToggle) {
        // Make sure the sidebar is initially hidden on mobile
        if (window.innerWidth <= 991.98) {
            sidebar.classList.remove('show');
            backdrop.classList.remove('show');
            body.classList.remove('sidebar-open');
        }
        
        mobileToggle.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            sidebar.classList.toggle('show');
            backdrop.classList.toggle('show');
            body.classList.toggle('sidebar-open');
            
            // Toggle icon between menu and close
            const icon = mobileToggle.querySelector('i');
            if (icon) {
                if (sidebar.classList.contains('show')) {
                    icon.className = 'bx bx-x';
                } else {
                    icon.className = 'bx bx-menu';
                }
            }
        });
    }
    
    // Close sidebar when clicking outside on mobile
    backdrop.addEventListener('click', function() {
        sidebar.classList.remove('show');
        backdrop.classList.remove('show');
        body.classList.remove('sidebar-open');
    });
    
    // Highlight active menu item
    const currentPath = window.location.pathname;
    const filename = currentPath.split('/').pop() || 'dashboard.php';
    
    document.querySelectorAll('.sidebar-link').forEach(link => {
        const href = link.getAttribute('href');
        if (href === filename || 
           (filename === 'dashboard.php' && (href === '' || href === '#' || href === 'dashboard.php'))) {
            link.closest('.sidebar-item').classList.add('active');
        }
    });
}

/**
 * Navbar functionality
 */
function initNavbar() {
    // Handle dropdowns
    document.querySelectorAll('[data-toggle="dropdown"]').forEach(toggle => {
        const dropdown = toggle.nextElementSibling;
        
        if (!dropdown) return;
        
        toggle.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            // Close all other dropdowns
            document.querySelectorAll('.dropdown-menu.show').forEach(menu => {
                if (menu !== dropdown) {
                    menu.classList.remove('show');
                }
            });
            
            dropdown.classList.toggle('show');
        });
    });
    
    // Close dropdowns when clicking outside
    document.addEventListener('click', function(e) {
        if (!e.target.closest('.dropdown-menu') && !e.target.closest('[data-toggle="dropdown"]')) {
            document.querySelectorAll('.dropdown-menu.show').forEach(menu => {
                menu.classList.remove('show');
            });
        }
    });
    
    // Search functionality
    const searchInput = document.querySelector('.navbar-search input');
    if (searchInput) {
        searchInput.addEventListener('focus', function() {
            this.parentElement.classList.add('focused');
        });
        
        searchInput.addEventListener('blur', function() {
            this.parentElement.classList.remove('focused');
        });
        
        searchInput.addEventListener('input', debounce(function(e) {
            if (e.target.value.length >= 2) {
                // Implement search functionality here
                console.log('Searching for:', e.target.value);
            }
        }, 300));
    }
}

/**
 * Initialize Bootstrap tooltips if available
 */
function initTooltips() {
    if (typeof bootstrap !== 'undefined' && bootstrap.Tooltip) {
        const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        tooltipTriggerList.map(function(tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    }
}

/**
 * Initialize Bootstrap popovers if available
 */
function initPopovers() {
    if (typeof bootstrap !== 'undefined' && bootstrap.Popover) {
        const popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'));
        popoverTriggerList.map(function(popoverTriggerEl) {
            return new bootstrap.Popover(popoverTriggerEl);
        });
    }
}

/**
 * Dark mode toggle functionality
 */
function initDarkMode() {
    const darkModeToggle = document.getElementById('darkModeToggle');
    const prefersDarkMode = window.matchMedia('(prefers-color-scheme: dark)').matches;
    const storedTheme = localStorage.getItem('theme');
    
    // Set initial theme based on preferences
    if (storedTheme) {
        document.body.classList.toggle('dark-mode', storedTheme === 'dark');
        if (darkModeToggle) {
            darkModeToggle.checked = storedTheme === 'dark';
        }
    } else if (prefersDarkMode) {
        document.body.classList.add('dark-mode');
        if (darkModeToggle) {
            darkModeToggle.checked = true;
        }
    }
    
    // Handle theme toggle click
    if (darkModeToggle) {
        darkModeToggle.addEventListener('change', function() {
            document.body.classList.toggle('dark-mode', this.checked);
            localStorage.setItem('theme', this.checked ? 'dark' : 'light');
        });
    }
}

/**
 * Handle responsive behavior
 */
function handleWindowResize() {
    const isMobile = window.innerWidth < 992;
    const body = document.body;
    const sidebar = document.querySelector('.sidebar');
    const backdrop = document.querySelector('.sidebar-backdrop');
    
    if (isMobile) {
        // On mobile, ensure sidebar is hidden by default
        body.classList.remove('sidebar-collapsed'); // This class should only be for desktop
        
        // Hide sidebar completely on mobile unless explicitly shown
        if (!body.classList.contains('sidebar-open')) {
            sidebar.classList.remove('show');
            if (backdrop) backdrop.classList.remove('show');
        }
        
        // Make sure navbar is full width
        document.querySelector('.navbar').style.left = '0';
        document.querySelector('.navbar').style.width = '100%';
    } else {
        // On desktop/tablet, ensure sidebar is fully expanded
        body.classList.remove('sidebar-collapsed');
        sidebar.classList.remove('show'); // Remove mobile-specific class
        if (backdrop) backdrop.classList.remove('show');
        body.classList.remove('sidebar-open');
        
        // Reset any inline styles that might have been applied
        sidebar.style.left = '';
        sidebar.style.width = '';
        sidebar.style.visibility = '';
        sidebar.style.opacity = '';
    }
}

/**
 * Create a ripple effect on button click
 * @param {Event} e - Click event
 */
function createRipple(e) {
    const button = e.currentTarget;
    
    const circle = document.createElement('span');
    const diameter = Math.max(button.clientWidth, button.clientHeight);
    const radius = diameter / 2;
    
    circle.style.width = circle.style.height = `${diameter}px`;
    circle.style.left = `${e.clientX - button.getBoundingClientRect().left - radius}px`;
    circle.style.top = `${e.clientY - button.getBoundingClientRect().top - radius}px`;
    circle.classList.add('ripple');
    
    const ripple = button.querySelector('.ripple');
    if (ripple) {
        ripple.remove();
    }
    
    button.appendChild(circle);
}

/**
 * Create a debounced function to prevent too many events firing
 * @param {Function} func - The function to debounce
 * @param {number} delay - Delay in milliseconds
 * @returns {Function} - Debounced function
 */
function debounce(func, delay) {
    let timeout;
    return function() {
        const context = this;
        const args = arguments;
        clearTimeout(timeout);
        timeout = setTimeout(() => func.apply(context, args), delay);
    };
}

/**
 * Format date in user-friendly format
 * @param {string} dateString - ISO date string
 * @returns {string} - Formatted date
 */
function formatDate(dateString) {
    if (!dateString) return '';
    
    const options = { year: 'numeric', month: 'short', day: 'numeric' };
    return new Date(dateString).toLocaleDateString(undefined, options);
}

/**
 * Format time in user-friendly format
 * @param {string} timeString - Time string (HH:MM:SS)
 * @returns {string} - Formatted time (hh:mm AM/PM)
 */
function formatTime(timeString) {
    if (!timeString) return '';
    
    const options = { hour: 'numeric', minute: '2-digit', hour12: true };
    return new Date(`2000-01-01T${timeString}`).toLocaleTimeString(undefined, options);
}

/**
 * Format datetime in relative format (e.g., "2 hours ago", "Yesterday")
 * @param {string} dateTimeString - ISO datetime string
 * @returns {string} - Relative time
 */
function timeAgo(dateTimeString) {
    if (!dateTimeString) return '';
    
    const now = new Date();
    const date = new Date(dateTimeString);
    const seconds = Math.floor((now - date) / 1000);
    
    // Less than a minute
    if (seconds < 60) {
        return 'Just now';
    }
    
    // Less than an hour
    const minutes = Math.floor(seconds / 60);
    if (minutes < 60) {
        return `${minutes} minute${minutes > 1 ? 's' : ''} ago`;
    }
    
    // Less than a day
    const hours = Math.floor(minutes / 60);
    if (hours < 24) {
        return `${hours} hour${hours > 1 ? 's' : ''} ago`;
    }
    
    // Less than a week
    const days = Math.floor(hours / 24);
    if (days < 7) {
        if (days === 1) return 'Yesterday';
        return `${days} days ago`;
    }
    
    // Default to formatted date
    return formatDate(dateTimeString);
}

/**
 * Show notification toast
 * @param {string} message - Notification message
 * @param {string} type - Notification type (success, warning, danger, info)
 * @param {number} duration - Duration in milliseconds
 */
function showNotification(message, type = 'info', duration = 5000) {
    // Create toast container if it doesn't exist
    let toastContainer = document.querySelector('.toast-container');
    if (!toastContainer) {
        toastContainer = document.createElement('div');
        toastContainer.className = 'toast-container';
        document.body.appendChild(toastContainer);
    }
    
    // Create toast element
    const toast = document.createElement('div');
    toast.className = `toast toast-${type}`;
    toast.innerHTML = `
        <div class="toast-icon">
            <i class="bx ${type === 'success' ? 'bx-check-circle' : 
                         type === 'warning' ? 'bx-error' : 
                         type === 'danger' ? 'bx-x-circle' : 'bx-info-circle'}"></i>
        </div>
        <div class="toast-content">${message}</div>
        <button type="button" class="toast-close">&times;</button>
    `;
    
    // Add to container
    toastContainer.appendChild(toast);
    
    // Trigger animation
    setTimeout(() => {
        toast.classList.add('show');
    }, 10);
    
    // Handle close button
    const closeButton = toast.querySelector('.toast-close');
    closeButton.addEventListener('click', () => {
        toast.classList.remove('show');
        setTimeout(() => {
            toast.remove();
        }, 300);
    });
    
    // Auto dismiss
    if (duration) {
        setTimeout(() => {
            if (toast.parentNode) {
                toast.classList.remove('show');
                setTimeout(() => {
                    toast.remove();
                }, 300);
            }
        }, duration);
    }
}
