<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
requireRole('student');
require_once '../config/db_connect.php';

// Get current student ID and assignment ID
$student_id = $_SESSION['user_id'];
$assignment_id = intval($_GET['id'] ?? 0);

if (!$assignment_id) {
    header('Location: assignments.php');
    exit;
}

// Get assignment details and submission info - USING CORRECT COLUMN NAMES
$query = "
    SELECT 
        a.assignment_id,
        a.title,
        a.description,
        a.instructions,
        a.due_date,
        a.points as max_points,
        a.created_at,
        c.course_id,
        c.title as course_title,
        c.thumbnail as course_thumbnail,
        u.first_name,
        u.last_name,
        u.email as teacher_email,
        ass.submission_id,
        ass.submission_date,
        ass.submission_text,
        ass.file_url,
        ass.grade,
        ass.feedback,
        ass.submitted_at,
        ass.teacher_feedback_files,
        CASE 
            WHEN ass.grade IS NOT NULL THEN 'graded'
            WHEN ass.submission_id IS NOT NULL THEN 'submitted'
            WHEN a.due_date < NOW() THEN 'overdue'
            ELSE 'pending'
        END as status
    FROM assignments a
    JOIN courses c ON a.course_id = c.course_id
    JOIN users u ON c.teacher_id = u.user_id
    JOIN enrollments e ON c.course_id = e.course_id
    LEFT JOIN assignment_submissions ass ON a.assignment_id = ass.assignment_id AND ass.student_id = ?
    WHERE a.assignment_id = ? AND e.student_id = ?
";

$stmt = $pdo->prepare($query);
$stmt->execute([$student_id, $assignment_id, $student_id]);
$assignment = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$assignment) {
    header('Location: assignments.php');
    exit;
}

// Note: Q&A functionality removed - assignment_questions table doesn't exist
$questions = [];

$page_title = $assignment['title'] . " - Assignment Details";

// Set current page for navigation
$currentPage = 'assignments';

// Include modern header with sidebar and navbar
include_once __DIR__ . '/components/header.php';
?>

<!-- Assignment Details Content -->
<div class="assignment-details-content">
    <style>
        :root {
            --panadite-primary: #2563eb;
            --panadite-secondary: #1d4ed8;
            --panadite-light: #dbeafe;
            --panadite-dark: #1e40af;
            --panadite-accent: #f59e0b;
            --text-primary: #1f2937;
            --text-secondary: #6b7280;
            --border-color: #e5e7eb;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --error-color: #ef4444;
            --bg-light: #f8fafc;
        }

        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: linear-gradient(135deg, var(--bg-light) 0%, #e2e8f0 100%);
            color: var(--text-primary);
            line-height: 1.6;
            min-height: 100vh;
        }

        /* Header */
        .page-header {
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
            box-shadow: 0 4px 20px rgba(37, 99, 235, 0.15);
        }

        .header-content {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem;
        }

        .breadcrumb {
            margin-bottom: 1rem;
            opacity: 0.8;
        }

        .breadcrumb a {
            color: white;
            text-decoration: none;
        }

        .breadcrumb a:hover {
            text-decoration: underline;
        }

        .page-title {
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
            text-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }

        .page-subtitle {
            font-size: 1.1rem;
            opacity: 0.9;
            font-weight: 400;
        }

        /* Main Container */
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem 2rem;
        }

        /* Assignment Details Card */
        .assignment-card {
            background: white;
            border-radius: 20px;
            overflow: hidden;
            box-shadow: 0 6px 25px rgba(0,0,0,0.08);
            margin-bottom: 2rem;
        }

        .assignment-header {
            padding: 2rem;
            background: linear-gradient(135deg, var(--bg-light) 0%, #e2e8f0 50%);
            border-bottom: 1px solid var(--border-color);
        }

        .assignment-meta {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .meta-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 1rem;
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }

        .meta-icon {
            width: 40px;
            height: 40px;
            background: var(--panadite-light);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--panadite-primary);
            font-size: 1.2rem;
        }

        .meta-content h4 {
            font-size: 0.85rem;
            color: var(--text-secondary);
            margin-bottom: 0.25rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .meta-content p {
            font-size: 1rem;
            font-weight: 600;
            color: var(--text-primary);
        }

        /* Status Badge */
        .status-badge {
            padding: 0.75rem 1.25rem;
            border-radius: 25px;
            font-size: 0.9rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .status-pending {
            background: #fef3c7;
            color: #d97706;
        }

        .status-submitted {
            background: #dbeafe;
            color: #2563eb;
        }

        .status-graded {
            background: #d1fae5;
            color: #059669;
        }

        .status-overdue {
            background: #fee2e2;
            color: #dc2626;
        }

        /* Content Sections */
        .content-section {
            padding: 2rem;
            border-bottom: 1px solid var(--border-color);
        }

        .content-section:last-child {
            border-bottom: none;
        }

        .section-title {
            font-size: 1.3rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 1rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .section-icon {
            width: 32px;
            height: 32px;
            background: var(--panadite-primary);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1rem;
        }

        .description-content {
            color: var(--text-secondary);
            font-size: 1rem;
            line-height: 1.7;
            margin-bottom: 1.5rem;
        }

        .instructions-content {
            background: var(--bg-light);
            padding: 1.5rem;
            border-radius: 12px;
            border-left: 4px solid var(--panadite-primary);
            color: var(--text-secondary);
            line-height: 1.7;
        }

        /* Submission Section */
        .submission-info {
            background: var(--bg-light);
            padding: 1.5rem;
            border-radius: 12px;
            margin-bottom: 1rem;
        }

        .submission-item {
            margin-bottom: 1rem;
        }

        .submission-item:last-child {
            margin-bottom: 0;
        }

        .submission-label {
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
            display: block;
        }

        .submission-value {
            color: var(--text-secondary);
            padding: 0.75rem;
            background: white;
            border-radius: 8px;
            border: 1px solid var(--border-color);
        }

        /* Action Buttons */
        .action-buttons {
            display: flex;
            gap: 1rem;
            flex-wrap: wrap;
            margin-top: 2rem;
        }

        .action-btn {
            padding: 1rem 2rem;
            border: none;
            border-radius: 12px;
            font-weight: 600;
            font-size: 1rem;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            position: relative;
            overflow: hidden;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(37, 99, 235, 0.3);
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(37, 99, 235, 0.4);
        }

        .btn-secondary {
            background: linear-gradient(135deg, var(--panadite-accent) 0%, #f97316 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(245, 158, 11, 0.3);
        }

        .btn-secondary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(245, 158, 11, 0.4);
        }

        .btn-outline {
            background: transparent;
            color: var(--panadite-primary);
            border: 2px solid var(--panadite-primary);
        }

        .btn-outline:hover {
            background: var(--panadite-primary);
            color: white;
        }

        /* Teacher Info */
        .teacher-info {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1.5rem;
            background: var(--bg-light);
            border-radius: 12px;
        }

        .teacher-avatar {
            width: 60px;
            height: 60px;
            background: var(--panadite-primary);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.5rem;
            font-weight: 600;
        }

        .teacher-details h4 {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.25rem;
        }

        .teacher-details p {
            color: var(--text-secondary);
            font-size: 0.9rem;
        }

        /* Enhanced Content Styles */
        .overview-card {
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            border: 1px solid var(--border-color);
            margin-bottom: 2rem;
        }

        .overview-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            margin: 0;
        }

        .overview-item {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1.5rem;
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
            border: 1px solid rgba(255,255,255,0.8);
        }

        .overview-icon {
            width: 50px;
            height: 50px;
            background: var(--panadite-light);
            color: var(--panadite-primary);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
        }

        .overview-details h4 {
            font-size: 0.9rem;
            color: var(--text-secondary);
            margin-bottom: 0.25rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .overview-details p {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--text-primary);
        }

        .description-content.enhanced, .instructions-content.enhanced {
            background: white;
            padding: 2rem;
            border-radius: 12px;
            border-left: 4px solid var(--panadite-primary);
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            font-size: 1.1rem;
            line-height: 1.8;
        }

        .empty-state {
            text-align: center;
            padding: 3rem 2rem;
            color: var(--text-secondary);
            font-style: italic;
        }

        .empty-state.large {
            padding: 4rem 2rem;
        }

        .empty-state i {
            font-size: 3rem;
            margin-bottom: 1rem;
            display: block;
            opacity: 0.5;
        }

        .submission-badge {
            padding: 0.5rem 1rem;
            border-radius: 25px;
            font-size: 0.9rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-left: 1rem;
        }

        .submission-badge.submitted {
            background: #dbeafe;
            color: #2563eb;
        }

        .submission-badge.graded {
            background: #d1fae5;
            color: #059669;
        }

        .submission-timeline {
            margin-bottom: 2rem;
        }

        .timeline-item {
            display: flex;
            gap: 1rem;
            margin-bottom: 1.5rem;
            position: relative;
        }

        .timeline-item:not(:last-child)::after {
            content: '';
            position: absolute;
            left: 20px;
            top: 50px;
            bottom: -20px;
            width: 2px;
            background: #e5e7eb;
        }

        .timeline-item.completed::after {
            background: var(--success-color);
        }

        .timeline-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.2rem;
            flex-shrink: 0;
        }

        .timeline-item.completed .timeline-icon {
            background: var(--success-color);
            color: white;
        }

        .timeline-item.pending .timeline-icon {
            background: var(--warning-color);
            color: white;
        }

        .timeline-content h4 {
            margin-bottom: 0.5rem;
            color: var(--text-primary);
        }

        .timeline-content p {
            color: var(--text-secondary);
            margin-bottom: 0.5rem;
        }

        .timeline-content .text-muted {
            font-size: 0.9rem;
        }

        .grade-display {
            margin-top: 0.5rem;
        }

        .grade-score {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--success-color);
        }

        .grade-percentage {
            color: var(--text-secondary);
            margin-left: 0.5rem;
        }

        .submission-content {
            background: #f8fafc;
            border-radius: 12px;
            padding: 2rem;
            margin-top: 1.5rem;
        }

        .content-block {
            margin-bottom: 1.5rem;
        }

        .content-block:last-child {
            margin-bottom: 0;
        }

        .content-block h5 {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin-bottom: 1rem;
            color: var(--text-primary);
            font-weight: 600;
        }

        .text-content {
            background: white;
            padding: 1.5rem;
            border-radius: 8px;
            border-left: 3px solid var(--panadite-primary);
            line-height: 1.7;
        }

        .file-submission {
            background: white;
            border-radius: 8px;
            overflow: hidden;
        }

        .download-btn {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1.5rem;
            text-decoration: none;
            color: var(--text-primary);
            transition: all 0.3s ease;
            border-left: 3px solid var(--panadite-primary);
        }

        .download-btn:hover {
            background: var(--panadite-light);
            color: var(--panadite-dark);
        }

        .download-btn i {
            font-size: 1.5rem;
            color: var(--panadite-primary);
        }

        .download-btn span {
            font-weight: 600;
        }

        .download-btn small {
            display: block;
            color: var(--text-secondary);
            font-size: 0.9rem;
        }

        /* Feedback Section */
        .feedback-section {
            background: linear-gradient(135deg, #fef3e2 0%, #fde68a 100%);
            border: 1px solid #f59e0b;
        }

        .feedback-text {
            margin-bottom: 2rem;
        }

        .feedback-content {
            background: white;
            padding: 2rem;
            border-radius: 12px;
            border-left: 4px solid var(--panadite-accent);
            line-height: 1.8;
            font-size: 1.1rem;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }

        .teacher-files h4 {
            margin-bottom: 1.5rem;
            color: var(--text-primary);
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .files-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 1rem;
        }

        .file-card {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            border: 1px solid rgba(255,255,255,0.8);
            display: flex;
            align-items: center;
            gap: 1rem;
            transition: all 0.3s ease;
        }

        .file-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 20px rgba(0,0,0,0.15);
        }

        .file-card .file-icon {
            width: 50px;
            height: 50px;
            border-radius: 10px;
            background: rgba(0,0,0,0.05);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            flex-shrink: 0;
        }

        .file-info {
            flex: 1;
            min-width: 0;
        }

        .file-info h5 {
            margin: 0 0 0.25rem 0;
            font-size: 1rem;
            font-weight: 600;
            color: var(--text-primary);
            word-break: break-word;
        }

        .file-info p {
            margin: 0;
            font-size: 0.85rem;
            color: var(--text-secondary);
        }

        .download-file-btn {
            padding: 0.5rem 1rem;
            background: var(--panadite-primary);
            color: white;
            text-decoration: none;
            border-radius: 6px;
            font-size: 0.9rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.3s ease;
            white-space: nowrap;
        }

        .download-file-btn:hover {
            background: var(--panadite-dark);
            color: white;
            transform: translateY(-1px);
        }

        .text-success {
            color: var(--success-color) !important;
        }

        .text-warning {
            color: var(--warning-color) !important;
        }

        .text-danger {
            color: var(--error-color) !important;
        }

        /* Mobile Responsive */
        @media (max-width: 768px) {
            .page-title {
                font-size: 2rem;
            }
            
            .assignment-meta {
                grid-template-columns: 1fr;
            }

            .overview-grid {
                grid-template-columns: 1fr;
            }

            .overview-item {
                padding: 1rem;
            }

            .files-grid {
                grid-template-columns: 1fr;
            }

            .file-card {
                padding: 1rem;
            }
            
            .action-buttons {
                flex-direction: column;
            }
            
            .main-container {
                padding: 0 0.5rem 2rem;
            }
            
            .page-header {
                padding: 1.5rem 0;
            }
            
            .content-section {
                padding: 1.5rem;
            }

            .submission-badge {
                margin-left: 0;
                margin-top: 0.5rem;
                display: inline-block;
            }

            .timeline-item {
                gap: 0.75rem;
            }

            .timeline-icon {
                width: 35px;
                height: 35px;
            }
        }    
            .content-section {
                padding: 1.5rem;
            }
        }
    </style>
</head>
<body>
    <?php include_once 'components/sidebar.php'; ?>
    
    <div class="page-header">
        <div class="header-content">
            <div class="breadcrumb">
                <a href="assignments.php"><i class="bx bx-arrow-back"></i> Back to Assignments</a>
            </div>
            <h1 class="page-title">
                <i class="bx bx-task"></i>
                <?= htmlspecialchars($assignment['title']) ?>
            </h1>
            <p class="page-subtitle"><?= htmlspecialchars($assignment['course_title']) ?></p>
        </div>
    </div>

    <div class="container">
        <div class="assignment-card">
            <!-- Assignment Header -->
            <div class="assignment-header">
                <div class="assignment-meta">
                    <div class="meta-item">
                        <div class="meta-icon">
                            <i class="bx bx-calendar"></i>
                        </div>
                        <div class="meta-content">
                            <h4>Due Date</h4>
                            <p><?= date('M d, Y g:i A', strtotime($assignment['due_date'])) ?></p>
                        </div>
                    </div>
                    
                    <div class="meta-item">
                        <div class="meta-icon">
                            <i class="bx bx-award"></i>
                        </div>
                        <div class="meta-content">
                            <h4>Points</h4>
                            <p><?= $assignment['max_points'] ?> Points</p>
                        </div>
                    </div>
                    
                    <div class="meta-item">
                        <div class="meta-icon">
                            <i class="bx bx-time-five"></i>
                        </div>
                        <div class="meta-content">
                            <h4>Status</h4>
                            <div class="status-badge status-<?= $assignment['status'] ?>">
                                <?php
                                $status_icons = [
                                    'pending' => 'bx-time',
                                    'submitted' => 'bx-check-circle',
                                    'graded' => 'bx-star',
                                    'overdue' => 'bx-alarm-exclamation'
                                ];
                                ?>
                                <i class="bx <?= $status_icons[$assignment['status']] ?>"></i>
                                <?= ucfirst($assignment['status']) ?>
                            </div>
                        </div>
                    </div>
                    
                    <?php if($assignment['grade'] !== null): ?>
                    <div class="meta-item">
                        <div class="meta-icon">
                            <i class="bx bx-trophy"></i>
                        </div>
                        <div class="meta-content">
                            <h4>Grade</h4>
                            <p><?= $assignment['grade'] ?>/<?= $assignment['max_points'] ?></p>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Assignment Overview Card -->
            <div class="content-section overview-card">
                <div class="overview-grid">
                    <div class="overview-item">
                        <div class="overview-icon">
                            <i class="bx bx-book-content"></i>
                        </div>
                        <div class="overview-details">
                            <h4>Course</h4>
                            <p><?= htmlspecialchars($assignment['course_title']) ?></p>
                        </div>
                    </div>
                    <div class="overview-item">
                        <div class="overview-icon">
                            <i class="bx bx-user-check"></i>
                        </div>
                        <div class="overview-details">
                            <h4>Instructor</h4>
                            <p><?= htmlspecialchars($assignment['first_name'] . ' ' . $assignment['last_name']) ?></p>
                        </div>
                    </div>
                    <div class="overview-item">
                        <div class="overview-icon">
                            <i class="bx bx-calendar-plus"></i>
                        </div>
                        <div class="overview-details">
                            <h4>Assigned</h4>
                            <p><?= date('M d, Y', strtotime($assignment['created_at'])) ?></p>
                        </div>
                    </div>
                    <div class="overview-item">
                        <div class="overview-icon">
                            <i class="bx bx-time"></i>
                        </div>
                        <div class="overview-details">
                            <h4>Time Left</h4>
                            <p>
                                <?php 
                                $due_time = strtotime($assignment['due_date']);
                                $now = time();
                                if ($due_time > $now) {
                                    $diff = $due_time - $now;
                                    $days = floor($diff / (24 * 60 * 60));
                                    $hours = floor(($diff % (24 * 60 * 60)) / (60 * 60));
                                    if ($days > 0) {
                                        echo "$days days, $hours hours";
                                    } else {
                                        echo "$hours hours";
                                    }
                                } else {
                                    echo "<span class='text-danger'>Overdue</span>";
                                }
                                ?>
                            </p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Assignment Description -->
            <div class="content-section">
                <h3 class="section-title">
                    <div class="section-icon">
                        <i class="bx bx-file-blank"></i>
                    </div>
                    Assignment Description
                </h3>
                <div class="description-content enhanced">
                    <?= $assignment['description'] ? nl2br(htmlspecialchars($assignment['description'])) : '<div class="empty-state"><i class="bx bx-info-circle"></i> No description provided</div>' ?>
                </div>
            </div>

            <!-- Assignment Instructions -->
            <?php if($assignment['instructions']): ?>
            <div class="content-section">
                <h3 class="section-title">
                    <div class="section-icon">
                        <i class="bx bx-list-check"></i>
                    </div>
                    Instructions & Requirements
                </h3>
                <div class="instructions-content enhanced">
                    <?= nl2br(htmlspecialchars($assignment['instructions'])) ?>
                </div>
            </div>
            <?php endif; ?>

            <!-- Submission Status & Details -->
            <div class="content-section">
                <h3 class="section-title">
                    <div class="section-icon">
                        <i class="bx bx-upload"></i>
                    </div>
                    Your Submission
                    <?php if($assignment['submission_id']): ?>
                        <span class="submission-badge <?= $assignment['status'] ?>">
                            <i class="bx <?= $assignment['status'] === 'graded' ? 'bx-check-circle' : ($assignment['status'] === 'submitted' ? 'bx-time-five' : 'bx-info-circle') ?>"></i>
                            <?= ucfirst($assignment['status']) ?>
                        </span>
                    <?php endif; ?>
                </h3>
                
                <?php if($assignment['submission_id']): ?>
                    <div class="submission-details">
                        <div class="submission-timeline">
                            <div class="timeline-item completed">
                                <div class="timeline-icon">
                                    <i class="bx bx-upload"></i>
                                </div>
                                <div class="timeline-content">
                                    <h4>Assignment Submitted</h4>
                                    <p><?= date('M d, Y g:i A', strtotime($assignment['submission_date'] ?? $assignment['submitted_at'])) ?></p>
                                    <small class="text-muted">
                                        <?php
                                        $submitted_time = strtotime($assignment['submission_date'] ?? $assignment['submitted_at']);
                                        $due_time = strtotime($assignment['due_date']);
                                        if ($submitted_time <= $due_time) {
                                            echo "<span class='text-success'><i class='bx bx-check'></i> Submitted on time</span>";
                                        } else {
                                            $late_hours = round(($submitted_time - $due_time) / 3600, 1);
                                            echo "<span class='text-warning'><i class='bx bx-time'></i> Late by $late_hours hours</span>";
                                        }
                                        ?>
                                    </small>
                                </div>
                            </div>
                            
                            <?php if($assignment['grade']): ?>
                            <div class="timeline-item completed">
                                <div class="timeline-icon">
                                    <i class="bx bx-medal"></i>
                                </div>
                                <div class="timeline-content">
                                    <h4>Assignment Graded</h4>
                                    <div class="grade-display">
                                        <span class="grade-score"><?= $assignment['grade'] ?>/<?= $assignment['max_points'] ?></span>
                                        <span class="grade-percentage">(<?= round(($assignment['grade'] / $assignment['max_points']) * 100, 1) ?>%)</span>
                                    </div>
                                </div>
                            </div>
                            <?php else: ?>
                            <div class="timeline-item pending">
                                <div class="timeline-icon">
                                    <i class="bx bx-time"></i>
                                </div>
                                <div class="timeline-content">
                                    <h4>Waiting for Grade</h4>
                                    <p>Your submission is under review</p>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>

                        <!-- Submission Content -->
                        <div class="submission-content">
                            <?php if($assignment['submission_text']): ?>
                            <div class="content-block">
                                <h5><i class="bx bx-edit"></i> Text Submission</h5>
                                <div class="text-content">
                                    <?= nl2br(htmlspecialchars($assignment['submission_text'])) ?>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <?php if($assignment['file_url']): ?>
                            <div class="content-block">
                                <h5><i class="bx bx-paperclip"></i> File Submission</h5>
                                <div class="file-submission">
                                    <a href="<?= htmlspecialchars($assignment['file_url']) ?>" target="_blank" class="download-btn">
                                        <i class="bx bx-download"></i>
                                        <span>Download Submitted File</span>
                                        <small>Click to view or download your submission</small>
                                    </a>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php else: ?>
                    <div class="no-submission">
                        <div class="empty-state large">
                            <i class="bx bx-cloud-upload"></i>
                            <h4>No Submission Yet</h4>
                            <p>You haven't submitted this assignment yet. Click the button below to get started.</p>
                        </div>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Teacher Feedback Section -->
            <?php if($assignment['submission_id'] && ($assignment['feedback'] || $assignment['teacher_feedback_files'])): ?>
            <div class="content-section feedback-section">
                <h3 class="section-title">
                    <div class="section-icon">
                        <i class="bx bx-message-square-detail"></i>
                    </div>
                    Teacher Feedback
                </h3>
                
                <?php if($assignment['feedback']): ?>
                <div class="feedback-text">
                    <div class="feedback-content">
                        <?= nl2br(htmlspecialchars($assignment['feedback'])) ?>
                    </div>
                </div>
                <?php endif; ?>

                <?php if($assignment['teacher_feedback_files']): 
                    $teacher_files = json_decode($assignment['teacher_feedback_files'], true);
                    if($teacher_files && is_array($teacher_files) && count($teacher_files) > 0): ?>
                
                <div class="teacher-files">
                    <h4><i class="bx bx-folder"></i> Additional Resources & Documents</h4>
                    <div class="files-grid">
                        <?php foreach($teacher_files as $file): 
                            $file_ext = strtolower(pathinfo($file['original_name'], PATHINFO_EXTENSION));
                            $file_icon = '';
                            $file_color = '';
                            
                            // Set icons and colors based on file type
                            if(in_array($file_ext, ['pdf'])) {
                                $file_icon = 'bx-file-pdf';
                                $file_color = '#dc2626';
                            } elseif(in_array($file_ext, ['doc', 'docx'])) {
                                $file_icon = 'bx-file-doc';
                                $file_color = '#2563eb';
                            } elseif(in_array($file_ext, ['xls', 'xlsx'])) {
                                $file_icon = 'bx-file-excel';
                                $file_color = '#059669';
                            } elseif(in_array($file_ext, ['ppt', 'pptx'])) {
                                $file_icon = 'bx-file-powerpoint';
                                $file_color = '#d97706';
                            } elseif(in_array($file_ext, ['jpg', 'jpeg', 'png', 'gif'])) {
                                $file_icon = 'bx-image';
                                $file_color = '#7c3aed';
                            } elseif(in_array($file_ext, ['mp4', 'mov', 'avi'])) {
                                $file_icon = 'bx-video';
                                $file_color = '#dc2626';
                            } else {
                                $file_icon = 'bx-file';
                                $file_color = '#6b7280';
                            }
                        ?>
                        <div class="file-card">
                            <div class="file-icon" style="color: <?= $file_color ?>">
                                <i class="bx <?= $file_icon ?>"></i>
                            </div>
                            <div class="file-info">
                                <h5><?= htmlspecialchars($file['original_name']) ?></h5>
                                <p>
                                    <?= isset($file['file_size']) ? number_format($file['file_size'] / 1024, 1) . ' KB' : 'Unknown size' ?>
                                    <?= isset($file['upload_date']) ? ' • ' . date('M d, Y', strtotime($file['upload_date'])) : '' ?>
                                </p>
                            </div>
                            <div class="file-actions">
                                <a href="download_teacher_feedback.php?file=<?= urlencode($file['file_path']) ?>&submission=<?= $assignment['submission_id'] ?>&student=<?= $student_id ?>" 
                                   class="download-file-btn" title="Download <?= htmlspecialchars($file['original_name']) ?>">
                                    <i class="bx bx-download"></i>
                                    Download
                                </a>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <!-- Teacher Information -->
            <div class="content-section">
                <h3 class="section-title">
                    <div class="section-icon">
                        <i class="bx bx-user"></i>
                    </div>
                    Instructor
                </h3>
                <div class="teacher-info">
                    <div class="teacher-avatar">
                        <?= strtoupper(substr($assignment['first_name'], 0, 1) . substr($assignment['last_name'], 0, 1)) ?>
                    </div>
                    <div class="teacher-details">
                        <h4><?= htmlspecialchars($assignment['first_name'] . ' ' . $assignment['last_name']) ?></h4>
                        <p><a href="mailto:<?= htmlspecialchars($assignment['teacher_email']) ?>" style="color: var(--panadite-primary); text-decoration: none;"><?= htmlspecialchars($assignment['teacher_email']) ?></a></p>
                    </div>
                </div>
            </div>

            <!-- Action Buttons -->
            <div class="content-section">
                <div class="action-buttons">
                    <?php if($assignment['status'] === 'pending' || $assignment['status'] === 'overdue'): ?>
                        <a href="submit-assignment.php?id=<?= $assignment['assignment_id'] ?>" class="action-btn btn-primary">
                            <i class="bx bx-upload"></i>
                            <?= $assignment['status'] === 'overdue' ? 'Submit Late' : 'Submit Assignment' ?>
                        </a>
                    <?php elseif($assignment['status'] === 'submitted' && !$assignment['grade']): ?>
                        <a href="submit-assignment.php?id=<?= $assignment['assignment_id'] ?>" class="action-btn btn-secondary">
                            <i class="bx bx-edit"></i>
                            Update Submission
                        </a>
                    <?php endif; ?>
                    
                    <a href="assignments.php" class="action-btn btn-outline">
                        <i class="bx bx-arrow-back"></i>
                        Back to Assignments
                    </a>
                    
                    <a href="mailto:<?= htmlspecialchars($assignment['teacher_email']) ?>?subject=Question about <?= urlencode($assignment['title']) ?>" class="action-btn btn-outline">
                        <i class="bx bx-envelope"></i>
                        Contact Instructor
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Close home-content div -->
        </div>
    </section>
</body>
</html>
