<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';

// Require student role to access this page
requireRole('student');

// Include database connection
require_once '../config/db_connect.php';

// Include AI grading services for submission processing
require_once '../includes/ai_grading_service.php';
require_once '../includes/improved_pdf_processor.php';

// Include course progress tracking service
require_once '../includes/course_progress_service.php';

// Include enhanced submission processor
require_once 'enhanced_submission_processor.php';

// Function wrapper for enhanced submission processing
function processEnhancedSubmission($assignmentId, $studentId, $submissionText, $uploadedFile, $conn) {
    try {
        error_log("🚀 processEnhancedSubmission called with assignmentId=$assignmentId, studentId=$studentId");
        
        if (!class_exists('EnhancedSubmissionProcessor')) {
            throw new Exception('EnhancedSubmissionProcessor class not found');
        }
        
        $processor = new EnhancedSubmissionProcessor($conn, true); // Enable debug mode
        error_log("✅ Enhanced processor instantiated successfully");
        
        $result = $processor->processSubmission($assignmentId, $studentId, $submissionText, $uploadedFile);
        error_log("📊 Enhanced processor result: " . json_encode($result));
        
        return $result;
        
    } catch (Exception $e) {
        error_log("❌ Enhanced processor failed: " . $e->getMessage());
        error_log("❌ Stack trace: " . $e->getTraceAsString());
        
        // Return failure result
        return [
            'success' => false,
            'errors' => [$e->getMessage()],
            'submission_id' => null
        ];
    }
}

// Get student ID
$student_id = $_SESSION['user_id'];

// Get assignment ID
if (!isset($_GET['id']) || empty($_GET['id'])) {
    $_SESSION['error'] = "Assignment ID not provided.";
    header("Location: courses.php");
    exit();
}

$assignment_id = intval($_GET['id']);

// Get assignment details and check if student is enrolled in the course
$assignmentQuery = "
    SELECT a.*, c.title as course_title, c.course_id,
           (SELECT COUNT(*) FROM enrollments WHERE course_id = a.course_id AND student_id = ?) as is_enrolled,
           s.submitted_at, s.file_url as submission_file, s.submission_text, s.grade, s.feedback,
           s.ai_score, s.ai_feedback, s.similarity_score, s.plagiarism_score, 
           s.processing_status, s.ai_confidence
    FROM assignments a
    JOIN courses c ON a.course_id = c.course_id
    LEFT JOIN (
        SELECT assignment_id, student_id, submitted_at, file_url, submission_text, 
               grade, feedback, ai_score, ai_feedback, similarity_score, plagiarism_score, 
               processing_status, ai_confidence,
               ROW_NUMBER() OVER (PARTITION BY assignment_id, student_id ORDER BY submitted_at DESC) as rn
        FROM assignment_submissions 
        WHERE student_id = ?
    ) s ON a.assignment_id = s.assignment_id AND s.rn = 1
    WHERE a.assignment_id = ?
";

$stmt = $conn->prepare($assignmentQuery);
if (!$stmt) {
    $_SESSION['error'] = "Database error. Please try again later.";
    header("Location: courses.php");
    exit();
}

$stmt->bind_param("iii", $student_id, $student_id, $assignment_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    $_SESSION['error'] = "Assignment not found or you don't have permission to access it.";
    header("Location: courses.php");
    exit();
}

$assignment = $result->fetch_assoc();

// Check if student is enrolled in the course
if ($assignment['is_enrolled'] <= 0) {
    $_SESSION['error'] = "You must be enrolled in this course to view its assignments.";
    header("Location: course-detail.php?id=" . $assignment['course_id']);
    exit();
}

// Check if assignment is past due date and no submission has been made
$isPastDue = false;
$isSubmitted = !empty($assignment['submitted_at']);
$isGraded = !empty($assignment['grade']);

if (!empty($assignment['due_date'])) {
    $dueDate = new DateTime($assignment['due_date']);
    $currentDate = new DateTime();
    $isPastDue = $currentDate > $dueDate && !$isSubmitted;
}

// Set page title
$pageTitle = "Assignment: " . $assignment['title'];

// Handle form submission
$submissionSuccess = false;
$submissionError = '';

// Include enhanced submission processor
require_once 'enhanced_submission_processor.php';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_assignment'])) {
    // Only allow submission if not already graded
    if ($isGraded) {
        $submissionError = "This assignment has already been graded. You cannot submit again.";
    } else {
        // Use enhanced submission processor
        error_log("🚀 Attempting enhanced submission processing...");
        error_log("📊 Assignment ID: $assignment_id, Student ID: $student_id");
        error_log("📝 Submission Text Length: " . strlen(trim($_POST['submission_text'] ?? '')));
        error_log("📁 File Upload: " . (isset($_FILES['submission_file']) && $_FILES['submission_file']['error'] === UPLOAD_ERR_OK ? 'YES' : 'NO'));
        
        $result = processEnhancedSubmission(
            $assignment_id, 
            $student_id, 
            trim($_POST['submission_text'] ?? ''), 
            $_FILES['submission_file'] ?? null, 
            $conn
        );
        
        error_log("📊 Enhanced processor result: " . json_encode($result));
        
        if ($result['success']) {
            // Set success message but don't redirect - show modal instead
            $_SESSION['success'] = 'Assignment submitted successfully!';
            $_SESSION['show_ai_modal'] = true;
            $_SESSION['submission_data'] = $result;
            
            // Mark as just submitted to trigger modal
            $justSubmitted = true;
            $submissionResult = $result;
            
            // Refresh to get updated submission data
            $submissionResult = $conn->query("SELECT * FROM assignment_submissions WHERE assignment_id = $assignment_id AND student_id = $student_id ORDER BY submitted_at DESC LIMIT 1");
            if ($submissionResult) {
                $submission = $submissionResult->fetch_assoc();
                $isSubmitted = true;
            }
        } else {
            $submissionError = implode(', ', $result['errors']);
        }
        
        // ⚠️ IMPORTANT: Skip legacy processing if enhanced processor succeeded
        // This prevents duplicate submissions and file_url being NULL
        $skipLegacyProcessing = $result['success'];
        
        // Legacy fallback processing (for compatibility - only if enhanced failed)
        if (!$skipLegacyProcessing) {
            error_log("⚠️ LEGACY PROCESSING ACTIVATED - Enhanced processor failed!");
            error_log("📊 skipLegacyProcessing = " . ($skipLegacyProcessing ? 'true' : 'false'));
            error_log("📊 result['success'] = " . ($result['success'] ? 'true' : 'false'));
            
        $submissionText = trim($_POST['submission_text'] ?? '');
        $uploadedFile = $_FILES['submission_file'] ?? null;
        $submissionFilePath = '';
        
        // Check if file is uploaded
        if (!empty($uploadedFile['name'])) {
            $targetDir = "../uploads/assignments/";
            
            // Create directory if it doesn't exist
            if (!file_exists($targetDir)) {
                mkdir($targetDir, 0755, true);
            }
            
            // Generate unique filename
            $fileExtension = pathinfo($uploadedFile['name'], PATHINFO_EXTENSION);
            $newFileName = 'assignment_' . $assignment_id . '_student_' . $student_id . '_' . time() . '.' . $fileExtension;
            $targetFilePath = $targetDir . $newFileName;
            
            // Check file size (10MB limit)
            if ($uploadedFile['size'] > 10 * 1024 * 1024) {
                $submissionError = "File size must be less than 10MB.";
            } 
            // Check file type (allow common document formats)
            elseif (!in_array(strtolower($fileExtension), ['pdf', 'doc', 'docx', 'ppt', 'pptx', 'xls', 'xlsx', 'txt', 'zip', 'rar', 'png', 'jpg', 'jpeg'])) {
                $submissionError = "Invalid file type. Allowed types: PDF, DOC, DOCX, PPT, PPTX, XLS, XLSX, TXT, ZIP, RAR, PNG, JPG, JPEG.";
            } else {
                if (move_uploaded_file($uploadedFile['tmp_name'], $targetFilePath)) {
                    $submissionFilePath = $newFileName;
                } else {
                    $submissionError = "Failed to upload file. Please try again.";
                }
            }
        }
        
        // If no errors and at least one submission method is used
        if (empty($submissionError) && (!empty($submissionText) || !empty($submissionFilePath))) {
            // Initialize AI grading variables
            $extractedText = '';
            $aiScore = null;
            $aiFeedback = '';
            $similarityScore = null;
            $plagiarismScore = null;
            $processingStatus = 'pending';
            $keywordMatchScore = null;
            $structureScore = null;
            $qualityScore = null;
            $reviewNeeded = 0;
            $aiConfidence = null;
            
            try {
                // Check if AI grading is enabled for this assignment
                $checkAIQuery = "SELECT ai_grading_enabled, memorandum_content FROM assignments WHERE assignment_id = ?";
                $checkStmt = $conn->prepare($checkAIQuery);
                $checkStmt->bind_param("i", $assignment_id);
                $checkStmt->execute();
                $aiResult = $checkStmt->get_result()->fetch_assoc();
                
                $isAIEnabled = $aiResult['ai_grading_enabled'] ?? false;
                $memorandumContent = $aiResult['memorandum_content'] ?? '';
                
                // Prepare full submission text for AI grading
                $fullSubmissionText = $submissionText;
                
                // Extract text from uploaded file if present using robust processor
                if (!empty($submissionFilePath) && file_exists($targetFilePath)) {
                    try {
                        $processor = new ImprovedPDFProcessor(true); // Enable debug logging
                        $fileExtractedText = $processor->extractText($targetFilePath);
                        
                        if (!empty($fileExtractedText)) {
                            $extractedText = $fileExtractedText;
                            $fullSubmissionText .= "\n\n" . $fileExtractedText;
                            error_log("Student submission text extracted successfully: " . strlen($fileExtractedText) . " characters");
                        } else {
                            error_log("No text could be extracted from student submission file: $targetFilePath");
                        }
                    } catch (Exception $e) {
                        error_log("Error extracting text from student submission: " . $e->getMessage());
                        // Continue without file text - don't fail the entire submission
                    }
                }
                
                // Perform AI grading if enabled and memorandum exists
                if ($isAIEnabled && !empty($memorandumContent) && !empty(trim($fullSubmissionText))) {
                    $gradingService = new PHPAIGradingService($conn, true);
                    $gradingResult = $gradingService->gradeAssignment($fullSubmissionText, $memorandumContent, $assignment_id);
                    
                    // Extract grading results
                    $aiScore = $gradingResult['ai_score'] ?? 0;
                    $aiFeedback = $gradingResult['ai_feedback'] ?? 'Automated grading completed.';
                    $similarityScore = $gradingResult['similarity_score'] ?? 0;
                    $plagiarismScore = $gradingResult['plagiarism_score'] ?? 0;
                    $keywordMatchScore = $gradingResult['keyword_match_score'] ?? 0;
                    $structureScore = $gradingResult['structure_score'] ?? 0;
                    $qualityScore = $gradingResult['quality_score'] ?? 0;
                    $reviewNeeded = $gradingResult['review_needed'] ?? 0;
                    $aiConfidence = $gradingResult['ai_confidence'] ?? 0;
                    $processingStatus = $gradingResult['processing_status'] ?? 'completed';
                    
                    // Log successful AI grading
                    error_log("AI Grading completed for assignment $assignment_id, student $student_id: Score $aiScore");
                } else {
                    $processingStatus = $isAIEnabled ? 'no_memorandum' : 'ai_disabled';
                }
                
            } catch (Exception $e) {
                // Log AI grading error but don't fail submission
                error_log("AI Grading Error for assignment $assignment_id: " . $e->getMessage());
                $processingStatus = 'failed';
                $reviewNeeded = 1;
                $aiFeedback = 'Automatic grading failed. Manual review required.';
            }
            
            // ============ DEBUG FILE UPLOAD ISSUE ============
            error_log("🔍 DEBUG FILE UPLOAD:");
            error_log("📁 submissionFilePath: " . ($submissionFilePath ?? 'NULL'));
            error_log("📄 submissionText length: " . strlen($submissionText ?? ''));
            error_log("🆔 assignment_id: " . $assignment_id);
            error_log("👤 student_id: " . $student_id);
            
            // Check if file actually exists
            if (!empty($submissionFilePath)) {
                $fullPath = "../uploads/assignments/" . $submissionFilePath;
                if (file_exists($fullPath)) {
                    error_log("✅ File exists: " . $fullPath . " (size: " . filesize($fullPath) . " bytes)");
                } else {
                    error_log("❌ File NOT found: " . $fullPath);
                }
            }
            
            // Log the exact INSERT parameters
            error_log("📝 INSERT parameters being sent:");
            error_log("1. assignment_id: " . $assignment_id);
            error_log("2. student_id: " . $student_id);  
            error_log("3. submission_text: " . substr($submissionText, 0, 100) . "...");
            error_log("4. file_url: " . ($submissionFilePath ?? 'NULL'));
            error_log("5. extracted_text: " . substr($extractedText, 0, 100) . "...");
            // ================================================
            
            // Insert submission record with AI grading data
            $insertQuery = "
                INSERT INTO assignment_submissions (
                    assignment_id, student_id, submission_text, file_url, 
                    extracted_text, ai_score, ai_feedback, similarity_score, 
                    plagiarism_score, keyword_match_score, structure_score, 
                    quality_score, processing_status, review_needed, ai_confidence, 
                    submitted_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
            ";
            
            $insertStmt = $conn->prepare($insertQuery);
            if (!$insertStmt) {
                $submissionError = "Database error. Please try again.";
            } else {
                $insertStmt->bind_param(
                    "iisssdsdddddsid",
                    $assignment_id, $student_id, $submissionText, $submissionFilePath,
                    $extractedText, $aiScore, $aiFeedback, $similarityScore,
                    $plagiarismScore, $keywordMatchScore, $structureScore,
                    $qualityScore, $processingStatus, $reviewNeeded, $aiConfidence
                );
                
                if ($insertStmt->execute()) {
                    $submissionSuccess = true;
                    
                    // Update overall course progress
                    try {
                        $progressService = new CourseProgressService($conn);
                        $updatedProgress = $progressService->calculateCourseProgress($student_id, $assignment['course_id']);
                        
                        // Log progress update
                        error_log("Course progress updated for student $student_id, course {$assignment['course_id']}: {$updatedProgress['overall_percentage']}%");
                        
                        // Store success message with progress info
                        if (!empty($aiScore)) {
                            $_SESSION['submission_success'] = "Assignment submitted successfully! AI Score: " . round($aiScore, 1) . "%. Course Progress: " . round($updatedProgress['overall_percentage'], 1) . "%";
                        } else {
                            $_SESSION['submission_success'] = "Assignment submitted successfully! Course Progress: " . round($updatedProgress['overall_percentage'], 1) . "%";
                        }
                        
                    } catch (Exception $e) {
                        error_log("Progress update error after assignment submission: " . $e->getMessage());
                        
                        // Store success message without progress info if update fails
                        if (!empty($aiScore)) {
                            $_SESSION['submission_success'] = "Assignment submitted successfully! AI Score: " . round($aiScore, 1) . "%. Your submission has been automatically graded.";
                        } else {
                            $_SESSION['submission_success'] = "Assignment submitted successfully! Your submission will be reviewed by your instructor.";
                        }
                    }
                    
                    // Refresh assignment data to show submission
                    $stmt->execute();
                    $result = $stmt->get_result();
                    $assignment = $result->fetch_assoc();
                    $isSubmitted = true;
                } else {
                    $submissionError = "Failed to save submission. Please try again.";
                    error_log("Database error saving submission: " . $insertStmt->error);
                }
            }
        } elseif (empty($submissionText) && empty($submissionFilePath)) {
            $submissionError = "Please provide either text submission or upload a file.";
        }
        
        } // End of legacy processing if block
    }
}

// Include header
include_once 'components/header.php';

// Include navigation
include_once 'components/navbar.php';
?>

<style>
    /* Modern form styling */
    .file-upload-wrapper:hover {
        border-color: #667eea !important;
        background: rgba(102, 126, 234, 0.1) !important;
        transform: translateY(-2px);
        transition: all 0.3s ease;
    }
    
    #submitBtn:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4) !important;
    }
    
    #watchGradingBtn:hover {
        background: #e3f2fd !important;
        transform: translateY(-1px);
    }
    
    .card:hover {
        box-shadow: 0 10px 30px rgba(0,0,0,0.1) !important;
        transform: translateY(-2px);
        transition: all 0.3s ease;
    }
    
    /* Accordion styling */
    .accordion-button {
        border-radius: 12px !important;
        border: none !important;
        background: linear-gradient(135deg, #f8f9ff 0%, #f0f2ff 100%) !important;
        font-weight: 600;
    }
    
    .accordion-button:not(.collapsed) {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%) !important;
        color: white !important;
    }
    
    .accordion-item {
        border: none !important;
        margin-bottom: 10px;
        border-radius: 12px !important;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }
    
    /* Log styling */
    .log-container::-webkit-scrollbar {
        width: 8px;
    }
    
    .log-container::-webkit-scrollbar-track {
        background: #f1f1f1;
        border-radius: 4px;
    }
    
    .log-container::-webkit-scrollbar-thumb {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        border-radius: 4px;
    }
    
    /* Progress bar animation */
    @keyframes pulseGlow {
        0% { box-shadow: 0 0 5px rgba(102, 126, 234, 0.4); }
        50% { box-shadow: 0 0 20px rgba(102, 126, 234, 0.8), 0 0 30px rgba(102, 126, 234, 0.4); }
        100% { box-shadow: 0 0 5px rgba(102, 126, 234, 0.4); }
    }
    
    .progress-bar-animated {
        animation: pulseGlow 2s ease-in-out infinite;
    }
    
    /* Modal backdrop blur */
    .modal.show {
        backdrop-filter: blur(10px);
    }
    
    /* Score display animations */
    @keyframes bounce {
        0%, 20%, 50%, 80%, 100% { transform: translateY(0); }
        40% { transform: translateY(-10px); }
        60% { transform: translateY(-5px); }
    }
    
    @keyframes scoreCountUp {
        from { transform: scale(0) rotate(-180deg); opacity: 0; }
        to { transform: scale(1) rotate(0deg); opacity: 1; }
    }
    
    @keyframes fadeInUp {
        from { opacity: 0; transform: translateY(20px); }
        to { opacity: 1; transform: translateY(0); }
    }
    
    .score-circle {
        animation: scoreCountUp 1s ease-out;
    }
    
    #gradingResults .card {
        animation: fadeInUp 0.8s ease-out;
    }
    
    /* Hover effects for better interaction */
    .score-circle:hover {
        transform: scale(1.05);
        transition: transform 0.3s ease;
    }
    
    /* Ensure buttons are clickable */
    #submitBtn, #watchGradingBtn {
        position: relative;
        z-index: 10;
        pointer-events: auto;
    }
    
    /* Fix any overlay issues */
    .card-body {
        position: relative;
        z-index: 1;
    }
    
    /* Ensure form elements are properly positioned */
    #assignmentForm {
        position: relative;
        z-index: 1;
    }
</style>

<div class="container-fluid pt-4">
    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?= $_SESSION['success']; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <?= $_SESSION['error']; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['submission_success'])): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class='bx bx-check-circle me-2'></i><?= $_SESSION['submission_success']; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php unset($_SESSION['submission_success']); ?>
    <?php elseif ($submissionSuccess): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class='bx bx-check-circle me-2'></i>Assignment submitted successfully!
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    
    <?php if (!empty($submissionError)): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <?= $submissionError; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    
    <div class="row">
        <div class="col-lg-8">
            <div class="card shadow-sm mb-4">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <div>
                        <h5 class="mb-0"><?= htmlspecialchars($assignment['title']) ?></h5>
                        <p class="text-muted small mb-0">
                            <i class='bx bx-book me-1'></i> 
                            <a href="course-detail.php?id=<?= $assignment['course_id'] ?>" class="text-reset">
                                <?= htmlspecialchars($assignment['course_title']) ?>
                            </a>
                        </p>
                    </div>
                    <?php if ($isSubmitted): ?>
                        <span class="badge bg-success">Submitted</span>
                    <?php elseif ($isPastDue): ?>
                        <span class="badge bg-danger">Overdue</span>
                    <?php else: ?>
                        <span class="badge bg-warning text-dark">Pending</span>
                    <?php endif; ?>
                </div>
                <div class="card-body">
                    <div class="mb-4">
                        <?php if (!empty($assignment['description'])): ?>
                            <h6 class="fw-bold">Instructions:</h6>
                            <div class="p-3 bg-light rounded mb-3">
                                <?= nl2br(htmlspecialchars($assignment['description'])) ?>
                            </div>
                        <?php endif; ?>
                        
                        <div class="row mt-3">
                            <div class="col-md-6">
                                <p><i class='bx bx-time me-1'></i> <strong>Due Date:</strong> 
                                   <?= !empty($assignment['due_date']) ? date('M d, Y h:i A', strtotime($assignment['due_date'])) : 'No deadline' ?>
                                </p>
                            </div>
                            <div class="col-md-6">
                                <p><i class='bx bx-trophy me-1'></i> <strong>Points:</strong> 
                                   <?= !empty($assignment['points']) ? $assignment['points'] : 'Not specified' ?>
                                </p>
                            </div>
                        </div>
                        
                        <?php if (!empty($assignment['file_path'])): ?>
                        <div class="mt-3">
                            <h6 class="fw-bold">Assignment Files:</h6>
                            <div class="d-flex align-items-center p-2 border rounded">
                                <i class='bx bx-file me-3 fs-4'></i>
                                <div>
                                    <span>Assignment Document</span>
                                    <div class="small text-muted">
                                        <?= pathinfo($assignment['file_path'], PATHINFO_EXTENSION) ?> file
                                    </div>
                                </div>
                                <a href="download_assignment.php?id=<?= $assignment_id ?>" class="btn btn-sm btn-outline-primary ms-auto">
                                    <i class='bx bx-download'></i> Download
                                </a>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <?php if ($isGraded): ?>
                    <div class="mt-4">
                        <h6 class="fw-bold">Feedback & Grading:</h6>
                        <div class="card bg-light">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <h5 class="mb-0">Grade: <?= $assignment['grade'] ?> / <?= $assignment['points'] ?></h5>
                                    <span class="badge bg-info">Graded</span>
                                </div>
                                
                                <?php if (!empty($assignment['feedback'])): ?>
                                <div class="mt-3">
                                    <h6>Instructor Feedback:</h6>
                                    <p class="p-3 bg-white rounded border"><?= nl2br(htmlspecialchars($assignment['feedback'])) ?></p>
                                </div>
                                <?php endif; ?>
                                
                                <?php if (!empty($assignment['ai_score']) && $assignment['ai_score'] > 0): ?>
                                <div class="mt-3">
                                    <h6>AI Grading Summary:</h6>
                                    <div class="p-3 bg-white rounded border">
                                        <div class="row">
                                            <div class="col-md-6">
                                                <div class="d-flex justify-content-between align-items-center mb-2">
                                                    <span><strong>AI Score:</strong></span>
                                                    <span class="badge bg-<?= $assignment['ai_score'] >= 70 ? 'success' : ($assignment['ai_score'] >= 50 ? 'warning' : 'danger') ?> fs-6">
                                                        <?= round($assignment['ai_score'], 1) ?>%
                                                    </span>
                                                </div>
                                                <?php if (!empty($assignment['similarity_score'])): ?>
                                                <div class="d-flex justify-content-between align-items-center mb-2">
                                                    <span><small>Content Similarity:</small></span>
                                                    <span class="small"><?= round($assignment['similarity_score'], 1) ?>%</span>
                                                </div>
                                                <?php endif; ?>
                                                <?php if (!empty($assignment['ai_confidence'])): ?>
                                                <div class="d-flex justify-content-between align-items-center mb-2">
                                                    <span><small>AI Confidence:</small></span>
                                                    <span class="small"><?= round($assignment['ai_confidence'], 1) ?>%</span>
                                                </div>
                                                <?php endif; ?>
                                            </div>
                                            <div class="col-md-6">
                                                <?php if (!empty($assignment['plagiarism_score'])): ?>
                                                <div class="d-flex justify-content-between align-items-center mb-2">
                                                    <span><small>Originality:</small></span>
                                                    <span class="small text-<?= $assignment['plagiarism_score'] < 30 ? 'success' : ($assignment['plagiarism_score'] < 60 ? 'warning' : 'danger') ?>">
                                                        <?= round(100 - $assignment['plagiarism_score'], 1) ?>%
                                                    </span>
                                                </div>
                                                <?php endif; ?>
                                                <?php if (!empty($assignment['processing_status'])): ?>
                                                <div class="d-flex justify-content-between align-items-center mb-2">
                                                    <span><small>Status:</small></span>
                                                    <span class="badge badge-sm bg-<?= $assignment['processing_status'] == 'completed' ? 'success' : ($assignment['processing_status'] == 'failed' ? 'danger' : 'warning') ?>">
                                                        <?= ucfirst($assignment['processing_status']) ?>
                                                    </span>
                                                </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                        <?php if (!empty($assignment['ai_feedback'])): ?>
                                        <div class="mt-2 pt-2 border-top">
                                            <small><strong>AI Feedback:</strong></small>
                                            <div class="small text-muted mt-1"><?= nl2br(htmlspecialchars($assignment['ai_feedback'])) ?></div>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($isSubmitted && !$isGraded): ?>
                    <div class="mt-4">
                        <h6 class="fw-bold">Your Submission:</h6>
                        <div class="card bg-light">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <span class="text-muted">Submitted: <?= date('M d, Y h:i A', strtotime($assignment['submitted_at'])) ?></span>
                                    <?php if (!empty($assignment['ai_score']) && $assignment['ai_score'] > 0): ?>
                                        <div>
                                            <span class="badge bg-<?= $assignment['ai_score'] >= 70 ? 'success' : ($assignment['ai_score'] >= 50 ? 'warning' : 'danger') ?>">AI: <?= round($assignment['ai_score'], 1) ?>%</span>
                                            <span class="badge bg-secondary ms-1">Awaiting Review</span>
                                        </div>
                                    <?php else: ?>
                                        <span class="badge bg-secondary">Awaiting Grade</span>
                                    <?php endif; ?>
                                </div>
                                
                                <?php if (!empty($assignment['submission_text'])): ?>
                                <div class="mt-3">
                                    <h6>Your Answer:</h6>
                                    <p class="p-3 bg-white rounded border"><?= nl2br(htmlspecialchars($assignment['submission_text'])) ?></p>
                                </div>
                                <?php endif; ?>
                                
                                <?php if (!empty($assignment['submission_file'])): ?>
                                <div class="mt-3">
                                    <h6>Submitted File:</h6>
                                    <div class="d-flex align-items-center p-2 bg-white rounded border">
                                        <i class='bx bx-file me-3 fs-4'></i>
                                        <div>
                                            <span><?= htmlspecialchars(pathinfo($assignment['submission_file'], PATHINFO_BASENAME)) ?></span>
                                        </div>
                                        <a href="download_submission.php?id=<?= $assignment_id ?>" class="btn btn-sm btn-outline-primary ms-auto">
                                            <i class='bx bx-download'></i> Download
                                        </a>
                                    </div>
                                </div>
                                <?php endif; ?>
                                
                                <?php if (!empty($assignment['ai_score']) && $assignment['ai_score'] > 0): ?>
                                <div class="mt-3">
                                    <h6>AI Grading Results:</h6>
                                    <div class="p-3 bg-white rounded border">
                                        <div class="row">
                                            <div class="col-md-6">
                                                <div class="d-flex justify-content-between align-items-center mb-2">
                                                    <span><strong>AI Score:</strong></span>
                                                    <span class="badge bg-<?= $assignment['ai_score'] >= 70 ? 'success' : ($assignment['ai_score'] >= 50 ? 'warning' : 'danger') ?> fs-6">
                                                        <?= round($assignment['ai_score'], 1) ?>%
                                                    </span>
                                                </div>
                                                <?php if (!empty($assignment['similarity_score'])): ?>
                                                <div class="d-flex justify-content-between align-items-center mb-1">
                                                    <span><small>Content Similarity:</small></span>
                                                    <span class="small"><?= round($assignment['similarity_score'], 1) ?>%</span>
                                                </div>
                                                <?php endif; ?>
                                            </div>
                                            <div class="col-md-6">
                                                <?php if (!empty($assignment['plagiarism_score'])): ?>
                                                <div class="d-flex justify-content-between align-items-center mb-1">
                                                    <span><small>Originality:</small></span>
                                                    <span class="small text-<?= $assignment['plagiarism_score'] < 30 ? 'success' : ($assignment['plagiarism_score'] < 60 ? 'warning' : 'danger') ?>">
                                                        <?= round(100 - $assignment['plagiarism_score'], 1) ?>%
                                                    </span>
                                                </div>
                                                <?php endif; ?>
                                                <?php if (!empty($assignment['ai_confidence'])): ?>
                                                <div class="d-flex justify-content-between align-items-center mb-1">
                                                    <span><small>AI Confidence:</small></span>
                                                    <span class="small"><?= round($assignment['ai_confidence'], 1) ?>%</span>
                                                </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                        <?php if (!empty($assignment['ai_feedback'])): ?>
                                        <div class="mt-2 pt-2 border-top">
                                            <small><strong>AI Feedback:</strong></small>
                                            <div class="small text-muted mt-1"><?= nl2br(htmlspecialchars($assignment['ai_feedback'])) ?></div>
                                        </div>
                                        <?php endif; ?>
                                        <div class="mt-2 pt-2 border-top">
                                            <small class="text-info"><i class='bx bx-info-circle me-1'></i>This is an automated assessment. Your instructor will review and may adjust the final grade.</small>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                    <?php if (!$isSubmitted && !$isPastDue): ?>
                    <div class="mt-4">
                        <div class="card border-0 shadow-sm" style="background: linear-gradient(135deg, #f8f9ff 0%, #f0f2ff 100%); border-radius: 16px;">
                            <div class="card-header border-0 bg-transparent pt-4 pb-3">
                                <h5 class="card-title mb-0 d-flex align-items-center">
                                    <div class="icon-circle me-3" style="width: 48px; height: 48px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border-radius: 12px; display: flex; align-items: center; justify-content: center;">
                                        <i class="fas fa-paper-plane text-white"></i>
                                    </div>
                                    <div>
                                        <span class="h5 mb-0 text-dark">Submit Your Assignment</span>
                                        <p class="text-muted small mb-0 mt-1">Upload your work and get instant AI feedback</p>
                                    </div>
                                </h5>
                            </div>
                            <div class="card-body px-4 pb-4">
                                <form method="POST" enctype="multipart/form-data" id="assignmentForm">
                                    <!-- Text Submission -->
                                    <div class="mb-4">
                                        <label for="submission_text" class="form-label fw-semibold text-dark d-flex align-items-center">
                                            <i class="fas fa-edit me-2 text-primary"></i>
                                            Text Submission
                                        </label>
                                        <textarea class="form-control border-0 shadow-sm" 
                                                  id="submission_text" 
                                                  name="submission_text" 
                                                  rows="6" 
                                                  placeholder="Type your answer here..."
                                                  style="border-radius: 12px; background: rgba(255,255,255,0.8); backdrop-filter: blur(10px);"></textarea>
                                        <div class="form-text d-flex align-items-center mt-2">
                                            <i class="fas fa-info-circle me-1 text-info"></i>
                                            You can submit text, a file, or both together
                                        </div>
                                    </div>
                                    
                                    <!-- File Upload -->
                                    <div class="mb-4">
                                        <label for="submission_file" class="form-label fw-semibold text-dark d-flex align-items-center">
                                            <i class="fas fa-file-upload me-2 text-success"></i>
                                            File Upload
                                        </label>
                                        <div class="file-upload-wrapper" style="border: 2px dashed #e0e6ff; border-radius: 12px; background: rgba(255,255,255,0.5); transition: all 0.3s ease;">
                                            <input class="form-control border-0 shadow-none" 
                                                   type="file" 
                                                   id="submission_file" 
                                                   name="submission_file"
                                                   style="background: transparent;">
                                        </div>
                                        <div class="form-text d-flex align-items-center mt-2">
                                            <i class="fas fa-file-check me-1 text-success"></i>
                                            PDF, DOC, DOCX, PPT, PPTX, XLS, XLSX, TXT, ZIP, RAR, PNG, JPG, JPEG • Max 10MB
                                        </div>
                                    </div>
                                    <!-- Live Status Indicator -->
                                    <div id="liveStatus" class="alert border-0 shadow-sm" 
                                         style="display: none; background: linear-gradient(135deg, #e3f2fd 0%, #e8f5e8 100%); border-radius: 12px;">
                                        <div class="d-flex align-items-center">
                                            <div class="spinner-border spinner-border-sm me-3 text-primary" role="status">
                                                <span class="visually-hidden">Processing...</span>
                                            </div>
                                            <div class="flex-grow-1">
                                                <span id="statusText" class="fw-semibold text-dark">Preparing submission...</span>
                                                <div class="progress mt-2 shadow-sm" style="height: 8px; border-radius: 6px; background: rgba(255,255,255,0.8);">
                                                    <div id="quickProgress" 
                                                         class="progress-bar progress-bar-striped progress-bar-animated" 
                                                         role="progressbar" 
                                                         style="width: 0%; background: linear-gradient(90deg, #36D1DC 0%, #5B86E5 100%);"></div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <!-- Action Buttons -->
                                    <div class="d-flex gap-3 justify-content-end mt-4">
                                        <button type="button" id="watchGradingBtn" 
                                                class="btn btn-outline-info" 
                                                style="display: none; border-radius: 10px; border: 2px solid #e3f2fd;">
                                            <i class="fas fa-eye me-2"></i> 
                                            Watch Live Grading
                                        </button>
                                         <!-- Manual Modal Test Button -->
                                         <button type="button" id="testModal" class="btn btn-info me-2" onclick="startLiveGrading(<?php echo $assignment_id; ?>); return false;">
                                              🔍 Test Modal
                                         </button>
                                          <!-- Simple fallback button for testing -->
                                         <button type="button" id="testSubmit" class="btn btn-success me-2" onclick="handleLiveSubmission(); return false;">
                                             Test Submit
                                         </button>
                                         
                                         <button type="submit" 
                                                 name="submit_assignment" 
                                                 id="submitBtn" 
                                                 class="btn btn-primary shadow-sm px-4 py-3"
                                                 style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border: none; border-radius: 12px; font-weight: 600; transition: all 0.3s ease; position: relative; z-index: 1000; pointer-events: auto;">
                                             <i class="fas fa-paper-plane me-2"></i> 
                                             <span id="btnText">Submit Assignment</span>
                                         </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                    <?php elseif ($isPastDue): ?>
                    <div class="alert alert-danger mt-4" role="alert">
                        <i class='bx bx-time-five me-2'></i> The due date for this assignment has passed. Contact your instructor if you need an extension.
                    </div>
                    <?php endif; ?>
                </div>
                <div class="card-footer bg-white">
                    <a href="course-detail.php?id=<?= $assignment['course_id'] ?>#assignments" class="btn btn-outline-secondary">
                        <i class='bx bx-arrow-back me-1'></i> Back to Assignments
                    </a>
                </div>
            </div>
        </div>
        
        <div class="col-lg-4">
            <!-- Course Progress Widget -->
            <?php 
            // Get current course progress
            try {
                $progressService = new CourseProgressService($conn);
                $courseProgress = $progressService->calculateCourseProgress($student_id, $assignment['course_id']);
            } catch (Exception $e) {
                $courseProgress = null;
                error_log("Progress widget error: " . $e->getMessage());
            }
            ?>
            
            <?php if ($courseProgress): ?>
            <div class="card shadow-sm mb-4">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h6 class="mb-0"><i class='bx bx-chart-line me-1'></i> Course Progress</h6>
                    <span class="badge bg-<?= $courseProgress['overall_percentage'] >= 80 ? 'success' : ($courseProgress['overall_percentage'] >= 50 ? 'warning' : 'primary') ?>">
                        <?= round($courseProgress['overall_percentage'], 1) ?>%
                    </span>
                </div>
                <div class="card-body py-2">
                    <!-- Overall Progress Bar -->
                    <div class="mb-3">
                        <div class="d-flex justify-content-between align-items-center mb-1">
                            <small class="text-muted">Overall Progress</small>
                            <small class="fw-bold"><?= round($courseProgress['overall_percentage'], 1) ?>%</small>
                        </div>
                        <div class="progress" style="height: 8px;">
                            <div class="progress-bar bg-<?= $courseProgress['overall_percentage'] >= 80 ? 'success' : ($courseProgress['overall_percentage'] >= 50 ? 'warning' : 'primary') ?>" 
                                 style="width: <?= $courseProgress['overall_percentage'] ?>%"></div>
                        </div>
                    </div>
                    
                    <!-- Detailed Breakdown -->
                    <div class="row text-center">
                        <div class="col-4">
                            <div class="border-end">
                                <div class="h6 mb-0 text-primary"><?= $courseProgress['videos']['completed'] ?>/<?= $courseProgress['videos']['total'] ?></div>
                                <small class="text-muted">Videos</small>
                            </div>
                        </div>
                        <div class="col-4">
                            <div class="border-end">
                                <div class="h6 mb-0 text-warning"><?= $courseProgress['assignments']['completed'] ?>/<?= $courseProgress['assignments']['total'] ?></div>
                                <small class="text-muted">Assignments</small>
                            </div>
                        </div>
                        <div class="col-4">
                            <div class="h6 mb-0 text-info"><?= $courseProgress['quizzes']['completed'] ?>/<?= $courseProgress['quizzes']['total'] ?></div>
                            <small class="text-muted">Quizzes</small>
                        </div>
                    </div>
                    
                    <!-- Completion Status -->
                    <div class="mt-3 text-center">
                        <?php 
                        $statusConfig = [
                            'completed' => ['badge' => 'success', 'icon' => 'bx-check-circle', 'text' => 'Course Completed!'],
                            'near_completion' => ['badge' => 'warning', 'icon' => 'bx-time-five', 'text' => 'Almost Done'],
                            'in_progress' => ['badge' => 'primary', 'icon' => 'bx-play-circle', 'text' => 'In Progress'],
                            'not_started' => ['badge' => 'secondary', 'icon' => 'bx-pause-circle', 'text' => 'Not Started']
                        ];
                        $status = $statusConfig[$courseProgress['completion_status']] ?? $statusConfig['not_started'];
                        ?>
                        <span class="badge bg-<?= $status['badge'] ?> badge-lg">
                            <i class='bx <?= $status['icon'] ?> me-1'></i><?= $status['text'] ?>
                        </span>
                        
                        <?php if ($courseProgress['overall_percentage'] >= 80): ?>
                        <div class="mt-2">
                            <small class="text-success"><i class='bx bx-award me-1'></i>Certificate Eligible</small>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <?php endif; ?>
            
            <div class="card shadow-sm mb-4">
                <div class="card-header">
                    <h5 class="mb-0">Assignment Status</h5>
                </div>
                <div class="card-body">
                    <ul class="list-group list-group-flush">
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                            Status:
                            <?php if ($isGraded): ?>
                                <span class="badge bg-info">Graded</span>
                            <?php elseif ($isSubmitted): ?>
                                <span class="badge bg-success">Submitted</span>
                            <?php elseif ($isPastDue): ?>
                                <span class="badge bg-danger">Overdue</span>
                            <?php else: ?>
                                <span class="badge bg-warning text-dark">Pending</span>
                            <?php endif; ?>
                        </li>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                            Due Date:
                            <span><?= !empty($assignment['due_date']) ? date('M d, Y', strtotime($assignment['due_date'])) : 'No deadline' ?></span>
                        </li>
                        <?php if ($isSubmitted): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                            Submitted On:
                            <span><?= date('M d, Y', strtotime($assignment['submitted_at'])) ?></span>
                        </li>
                        <?php endif; ?>
                        <?php if ($isGraded): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                            Grade:
                            <span class="fw-bold"><?= $assignment['grade'] ?> / <?= $assignment['points'] ?> points</span>
                        </li>
                        <?php elseif (!empty($assignment['points'])): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                            Possible Points:
                            <span><?= $assignment['points'] ?> points</span>
                        </li>
                        <?php endif; ?>
                    </ul>
                    
                    <?php if (!$isSubmitted && !$isPastDue && !empty($assignment['due_date'])): ?>
                    <div class="mt-3">
                        <div class="alert alert-warning" role="alert">
                            <?php
                            $dueDate = new DateTime($assignment['due_date']);
                            $currentDate = new DateTime();
                            $interval = $currentDate->diff($dueDate);
                            
                            if ($interval->days > 0) {
                                echo "<i class='bx bx-time me-2'></i> Due in " . $interval->format('%a days');
                            } elseif ($interval->h > 0) {
                                echo "<i class='bx bx-time me-2'></i> Due in " . $interval->format('%h hours');
                            } else {
                                echo "<i class='bx bx-time me-2'></i> Due in less than an hour";
                            }
                            ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="card shadow-sm">
                <div class="card-header">
                    <h5 class="mb-0">Tips for Success</h5>
                </div>
                <div class="card-body">
                    <ul class="list-group list-group-flush">
                        <li class="list-group-item">Read all instructions carefully before starting</li>
                        <li class="list-group-item">Check your work before submitting</li>
                        <li class="list-group-item">Submit before the deadline to avoid technical issues</li>
                        <li class="list-group-item">Reach out to your instructor if you have questions</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<?php
// Include footer
include_once 'components/footer.php';
?>

<!-- Live AI Grading Modal -->
<div class="modal fade" id="liveGradingModal" tabindex="-1" aria-labelledby="liveGradingModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl">
        <div class="modal-content" style="height: 85vh; max-height: 85vh;">
            <div class="modal-header bg-gradient" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border-bottom: none; border-radius: 0.375rem 0.375rem 0 0;">
                <h5 class="modal-title d-flex align-items-center" id="liveGradingModalLabel">
                    <div class="spinner-border spinner-border-sm me-3" role="status" id="gradingSpinner">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <i class="fas fa-brain me-2"></i> 
                    AI Grading Analysis
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body p-0" style="height: calc(85vh - 130px); overflow-y: auto;">
                <!-- Fixed Progress Section -->
                <div class="sticky-top bg-white border-bottom p-3" style="z-index: 1020;">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <div class="d-flex justify-content-between align-items-center mb-2">
                                <span class="fw-semibold text-dark">Overall Progress</span>
                                <span class="badge fs-6 text-dark" style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);" id="progressPercent">0%</span>
                            </div>
                            <div class="progress" style="height: 12px; border-radius: 10px; box-shadow: inset 0 1px 3px rgba(0,0,0,0.2);">
                                <div class="progress-bar" role="progressbar" id="gradingProgress" 
                                     style="width: 0%; background: linear-gradient(90deg, #36D1DC 0%, #5B86E5 100%); border-radius: 10px;" 
                                     aria-valuenow="0" aria-valuemin="0" aria-valuemax="100"></div>
                            </div>
                        </div>
                        <div class="col-md-4 text-end">
                            <span class="text-muted small" id="currentStepText">Initializing...</span>
                        </div>
                    </div>
                
                <!-- Scrollable Content Area -->
                <div class="p-4">
                    <!-- Grading Results Section (Initially Hidden) -->
                    <div id="gradingResults" class="mb-4" style="display: none;">
                        <div class="card border-0 shadow-lg" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border-radius: 20px;">
                            <div class="card-body text-center py-4">
                                <div class="row align-items-center">
                                    <div class="col-md-4">
                                        <div class="score-circle mx-auto mb-3" style="width: 120px; height: 120px; border-radius: 50%; background: rgba(255,255,255,0.2); display: flex; align-items: center; justify-content: center; border: 3px solid rgba(255,255,255,0.3);">
                                            <div class="text-center">
                                                <div class="display-4 fw-bold text-white" id="finalScore">0%</div>
                                                <small class="text-white-50">Your Score</small>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-8 text-start">
                                        <h4 class="text-white mb-3">
                                            <i class="fas fa-trophy me-2"></i>
                                            Assignment Graded Successfully!
                                        </h4>
                                        <div class="row text-white">
                                            <div class="col-md-6">
                                                <div class="mb-2">
                                                    <small class="text-white-50">Content Similarity</small>
                                                    <div class="h6 mb-0" id="similarityResult">0%</div>
                                                </div>
                                                <div class="mb-2">
                                                    <small class="text-white-50">Keyword Coverage</small>
                                                    <div class="h6 mb-0" id="keywordResult">0%</div>
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <div class="mb-2">
                                                    <small class="text-white-50">Structure & Quality</small>
                                                    <div class="h6 mb-0" id="qualityResult">0%</div>
                                                </div>
                                                <div class="mb-2">
                                                    <small class="text-white-50">AI Confidence</small>
                                                    <div class="h6 mb-0" id="confidenceResult">0%</div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Feedback Section -->
                                <div class="mt-4">
                                    <div class="card bg-white bg-opacity-20 border-0" style="border-radius: 12px;">
                                        <div class="card-body">
                                            <h6 class="text-white mb-3">
                                                <i class="fas fa-comment-dots me-2"></i>
                                                AI Feedback
                                            </h6>
                                            <div class="text-white-50" id="gradingFeedback" style="font-size: 0.9rem; line-height: 1.6;">
                                                Analyzing your submission...
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- AI Analysis Steps -->
                    <div id="gradingDetails" class="accordion" id="gradingAccordion">
                        <!-- Dynamic content will be added here -->
                    </div>
                    
                    <!-- Processing Log -->
                    <div class="mt-4">
                        <div class="card border-0 shadow-sm">
                            <div class="card-header bg-light border-0 py-3">
                                <h6 class="mb-0 d-flex align-items-center">
                                    <i class="fas fa-terminal me-2 text-primary"></i>
                                    Live Processing Log
                                </h6>
                            </div>
                            <div class="card-body p-0">
                                <div class="log-container" style="height: 180px; overflow-y: auto; font-family: 'Monaco', 'Menlo', 'Ubuntu Mono', monospace; font-size: 0.8rem; line-height: 1.4; background: #f8f9fa;">
                                    <div id="logContent" class="p-3"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer border-0 bg-light">
                <div class="d-flex justify-content-between w-100 align-items-center">
                    <div class="text-muted small">
                        <i class="fas fa-info-circle me-1"></i>
                        <span id="gradingStats">AI Analysis in Progress...</span>
                    </div>
                    <div>
                        <button type="button" class="btn btn-outline-secondary me-2" id="restartGrading" style="display: none;">
                            <i class="fas fa-redo me-1"></i> Restart
                        </button>
                        <button type="button" class="btn btn-primary" data-bs-dismiss="modal" id="closeGradingModal">
                            <i class="fas fa-check me-1"></i> Close
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Font Awesome for icons -->
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
<!-- jQuery -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>

<script>
// Live AI Grading System
let gradingSteps = ['start', 'extract_text', 'check_memorandum', 'ai_analysis', 'generate_grade', 'complete'];
let currentStepIndex = 0;
let gradingModal;

$(document).ready(function() {
    console.log('🚀 jQuery loaded successfully');
    console.log('📊 Form elements found:', {
        form: $('#assignmentForm').length,
        textArea: $('#submission_text').length,
        fileInput: $('#submission_file').length,
        submitBtn: $('#submitBtn').length
    });
    
    gradingModal = new bootstrap.Modal(document.getElementById('liveGradingModal'));
    console.log('🚀 Live AI Grading System Initialized');
    
    // Show system is ready
    showQuickStatus('✅ Live AI Grading Ready!', 'success', 2000);
    
    // DISABLED: Allow normal form submission for file upload fix
    // $('#assignmentForm').on('submit', function(e) {
    //     console.log('📝 Form submission intercepted');
    //     e.preventDefault();
    //     handleLiveSubmission();
    //     return false;
    // });
    
    // DISABLED: Allow normal button submission for file upload fix
    // $(document).on('click', '#submitBtn', function(e) {
    //     console.log('🔄 Submit button clicked - starting live grading');
    //     e.preventDefault();
    //     handleLiveSubmission();
    //     return false;
    // });
    
    // DISABLED: Allow normal button click for file upload fix
    // $('#submitBtn').on('click', function(e) {
    //     console.log('📝 Direct button click handler');
    //     e.preventDefault();
    //     handleLiveSubmission();
    //     return false;
    // });
    
    // Debug button clickability
    $('#submitBtn').on('mouseenter', function() {
        console.log('💭 Button is hoverable - should be clickable');
        $(this).css('cursor', 'pointer');
    });
    
    // Test if button is accessible
    console.log('🔍 Button accessibility check:');
    console.log('Button exists:', $('#submitBtn').length > 0);
    console.log('Button is visible:', $('#submitBtn').is(':visible'));
    console.log('Button computed style:', window.getComputedStyle(document.getElementById('submitBtn')));
    
    // Force enable pointer events on button
    $('#submitBtn').css({
        'pointer-events': 'auto',
        'position': 'relative',
        'z-index': '999'
    });
    
    // Watch grading button
    $('#watchGradingBtn').on('click', function() {
        startLiveGrading(<?php echo $assignment_id ?? 0; ?>);
    });
    
    // Debug session state
    console.log('🔍 Session debug:', {
        hasSubmissionSuccess: <?php echo isset($_SESSION['submission_success']) ? 'true' : 'false'; ?>,
        assignmentId: <?php echo $assignment_id ?? 0; ?>,
        sessionMessage: '<?php echo isset($_SESSION['submission_success']) ? addslashes($_SESSION['submission_success']) : 'none'; ?>'
    });
    
    // Auto-start grading after successful submission
    <?php if (isset($_SESSION['submission_success'])): ?>
    console.log('🚀 SESSION SUCCESS DETECTED - Auto-triggering modal');
    setTimeout(function() {
        console.log('🚀 Executing auto-start grading now');
        startLiveGrading(<?php echo $assignment_id ?? 0; ?>);
    }, 1000);
    <?php unset($_SESSION['submission_success']); ?>
    <?php else: ?>
    console.log('❌ No session success flag - modal will not auto-trigger');
    <?php endif; ?>
    
    // Check if there's a recent submission that needs grading
    <?php if ($latestSubmission && $latestSubmission['processing_status'] === 'pending'): ?>
    // Auto-start live grading for pending submission
    setTimeout(function() {
        startLiveGrading(<?php echo $assignment_id ?? 0; ?>);
    }, 1000);
    <?php endif; ?>
});

// Show quick status updates
function showQuickStatus(message, type = 'info', duration = 3000) {
    let statusDiv = $('#liveStatus');
    let statusText = $('#statusText');
    let quickProgress = $('#quickProgress');
    
    statusText.text(message);
    statusDiv.removeClass('alert-info alert-success alert-warning alert-danger')
             .addClass('alert-' + type)
             .fadeIn();
    
    if (duration > 0) {
        setTimeout(function() {
            statusDiv.fadeOut();
        }, duration);
    }
}

// Update quick progress bar
function updateQuickProgress(percentage) {
    $('#quickProgress').css('width', percentage + '%');
}

// Handle live submission with real-time AI grading
function handleLiveSubmission() {
    console.log('🚀 Starting live assignment submission...');
    
    try {
        // Get form data
        let submissionText = $('#submission_text').val();
        submissionText = submissionText ? submissionText.trim() : '';
        
        let fileInput = $('#submission_file')[0];
        let hasFile = fileInput && fileInput.files && fileInput.files.length > 0;
    
    console.log('📊 Submission data:', {
        textLength: submissionText.length,
        hasFile: hasFile,
        fileName: hasFile ? fileInput.files[0].name : 'none'
    });
    
    // Validate submission
    if (!submissionText && !hasFile) {
        showQuickStatus('⚠️ Please provide text or file!', 'warning', 3000);
        alert('Please provide either text submission or upload a file.');
        return;
    }
    
    // Update button state
    $('#submitBtn').prop('disabled', true);
    $('#btnText').html('<span class="spinner-border spinner-border-sm me-2"></span>Processing...');
    
    // Show immediate feedback
    showQuickStatus('🚀 Starting submission process...', 'info', 0);
    updateQuickProgress(5);
    
    // Show modal immediately with submission status
    gradingModal.show();
    resetGradingUI();
    
    addToLog('📝 Preparing submission...');
    updateProgress(5, 'Validating submission data...');
    
    // Create FormData for submission
    let formData = new FormData();
    formData.append('submission_text', submissionText);
    formData.append('submit_assignment', '1');
    
    if (hasFile) {
        formData.append('submission_file', fileInput.files[0]);
        addToLog('📄 File attached: ' + fileInput.files[0].name + ' (' + (fileInput.files[0].size / 1024 / 1024).toFixed(2) + ' MB)');
    }
    
    addToLog('🚀 Submitting assignment to server...');
    updateProgress(10, 'Submitting assignment...');
    
    // Submit to server using AJAX
    $.ajax({
        url: window.location.href,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function(response, status, xhr) {
            console.log('✅ Submission successful');
            
            // Update quick status
            showQuickStatus('✅ Submitted! Starting AI Analysis...', 'success', 0);
            updateQuickProgress(25);
            
            // Update button
            $('#btnText').html('✅ Submitted - AI Analyzing...');
            $('#watchGradingBtn').show();
            
            addToLog('✅ Assignment submitted successfully!');
            updateProgress(20, 'Assignment submitted! Starting AI analysis...');
            
            // Wait a moment then start live grading - slower for better visibility
            setTimeout(function() {
                startLiveGrading(<?php echo $assignment_id; ?>);
            }, 2500); // Increased to 2.5 seconds
        },
        error: function(xhr, status, error) {
            console.error('❌ Submission failed:', error);
            
            // Reset button state
            $('#submitBtn').prop('disabled', false);
            $('#btnText').text('Submit Assignment');
            
            // Show error status
            showQuickStatus('❌ Submission Failed: ' + error, 'danger', 5000);
            updateQuickProgress(0);
            
            addToLog('❌ Submission failed: ' + error, 'error');
            showError('Failed to submit assignment: ' + error);
        },
        xhr: function() {
            var xhr = new window.XMLHttpRequest();
            
            // Upload progress tracking
            xhr.upload.addEventListener('progress', function(evt) {
                if (evt.lengthComputable) {
                    var percentComplete = evt.loaded / evt.total;
                    var progress = Math.round(percentComplete * 20); // 0-20% for upload
                    var progressPercent = Math.round(percentComplete * 100);
                    
                    // Update both progress bars
                    updateProgress(progress, 'Uploading... ' + progressPercent + '%');
                    updateQuickProgress(progress);
                    showQuickStatus('📄 Uploading file... ' + progressPercent + '%', 'info', 0);
                    
                    addToLog('📄 Upload progress: ' + progressPercent + '%');
                }
            }, false);
            
            return xhr;
        }
    });
    
    } catch(error) {
        console.error('🚨 CRITICAL ERROR in handleLiveSubmission:', error);
        alert('Critical Error: ' + error.message);
        
        // Reset button state
        $('#submitBtn').prop('disabled', false);
        $('#btnText').text('Submit Assignment');
        
        // Show error status
        showQuickStatus('🚨 Critical Error: ' + error.message, 'danger', 10000);
    }
}

function resetGradingUI() {
    console.log('🔄 Resetting grading UI...');
    
    // Reset progress bar
    const progressBar = document.getElementById('gradingProgress');
    if (progressBar) progressBar.style.width = '0%';
    
    // Reset progress percentage
    const progressPercent = document.getElementById('progressPercent');
    if (progressPercent) progressPercent.textContent = '0%';
    
    // Reset current step text
    const currentStepText = document.getElementById('currentStepText');
    if (currentStepText) currentStepText.textContent = 'Initializing...';
    
    // Clear details and logs
    const gradingDetails = document.getElementById('gradingDetails');
    if (gradingDetails) gradingDetails.innerHTML = '';
    
    const logContent = document.getElementById('logContent');
    if (logContent) logContent.innerHTML = '';
    
    // Hide result section
    const gradingResults = document.getElementById('gradingResults');
    if (gradingResults) gradingResults.style.display = 'none';
    
    // Reset button states
    const closeBtn = document.getElementById('closeGradingModal');
    if (closeBtn) closeBtn.style.display = 'none';
    
    const restartBtn = document.getElementById('restartGrading');
    if (restartBtn) restartBtn.style.display = 'none';
    
    // Show spinner
    const spinner = document.getElementById('gradingSpinner');
    if (spinner) spinner.style.display = 'inline-block';
}

function startLiveGrading(assignmentId) {
    currentStepIndex = 0;
    gradingModal.show();
    
    // Reset UI
    document.getElementById('gradingProgress').style.width = '0%';
    document.getElementById('gradingDetails').innerHTML = '';
    document.getElementById('logContent').innerHTML = '';
    document.getElementById('closeGradingModal').style.display = 'none';
    document.getElementById('restartGrading').style.display = 'none';
    
    addToLog('🚀 Starting live AI grading for assignment ' + assignmentId);
    processNextStep(assignmentId);
}

function processNextStep(assignmentId) {
    console.log('🔍 Processing step check:', {
        currentStepIndex: currentStepIndex,
        totalSteps: gradingSteps.length,
        nextStep: gradingSteps[currentStepIndex] || 'NONE'
    });
    
    if (currentStepIndex >= gradingSteps.length) {
        console.log('🎉 All steps completed! Finalizing grading...');
        completeGrading();
        return;
    }
    
    let step = gradingSteps[currentStepIndex];
    console.log('➡️ Starting step:', step, '(' + (currentStepIndex + 1) + '/' + gradingSteps.length + ')');
    addToLog('📋 Processing step: ' + step + ' (' + (currentStepIndex + 1) + '/' + gradingSteps.length + ')');
    
    fetch('ajax/live_ai_grading.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'assignment_id=' + assignmentId + '&step=' + step
    })
    .then(response => response.json())
    .then(data => {
        if (data.error) {
            addToLog('❌ Error: ' + data.error, 'error');
            showError(data.error);
            return;
        }
        
        console.log('📈 Step response:', step, data);
        
        updateProgress(data.progress, data.message);
        addStepDetails(step, data.details);
        addToLog('✅ ' + data.message);
        
        if (data.details) {
            Object.entries(data.details).forEach(([key, value]) => {
                if (typeof value === 'object') {
                    addToLog('   📊 ' + key + ': ' + JSON.stringify(value));
                } else {
                    addToLog('   📊 ' + key + ': ' + value);
                }
            });
        }
        
        // Check if this is the final step
        if (step === 'complete' || data.step === 'complete') {
            console.log('🎉 Grading complete! Finalizing...');
            completeGrading(data);
            return;
        }
        
        // Move to next step
        currentStepIndex++;
        
        // Continue with next step if not complete
        console.log('➡️ Moving to step', currentStepIndex, 'of', gradingSteps.length);
        setTimeout(() => processNextStep(assignmentId), 3000);
    })
    .catch(error => {
        addToLog('❌ Network error: ' + error.message, 'error');
        showError('Network error: ' + error.message);
    });
}

function updateProgress(progress, message) {
    console.log('📈 Updating progress:', progress + '%', message);
    
    // Update main progress bar
    const progressBar = document.getElementById('gradingProgress');
    if (progressBar) {
        progressBar.style.width = progress + '%';
        progressBar.setAttribute('aria-valuenow', progress);
    }
    
    // Update progress percentage
    const progressPercent = document.getElementById('progressPercent');
    if (progressPercent) {
        progressPercent.textContent = Math.round(progress) + '%';
    }
    
    // Update current step text
    const currentStepText = document.getElementById('currentStepText');
    if (currentStepText) {
        currentStepText.textContent = message || 'Processing...';
    }
    
    // Legacy support - try to update old gradingStatus if it exists
    const gradingStatus = document.getElementById('gradingStatus');
    if (gradingStatus) {
        gradingStatus.innerHTML = '<h6 class="text-primary">' + message + '</h6><small class="text-muted">' + progress + '% complete</small>';
    }
}

function addStepDetails(step, details) {
    if (!details) return;
    
    let detailsContainer = document.getElementById('gradingDetails');
    let stepCard = document.createElement('div');
    stepCard.className = 'accordion-item';
    
    let stepTitle = {
        'start': '🎯 Initialization',
        'extract_text': '📄 Text Extraction',
        'check_memorandum': '📚 Memorandum Check',
        'ai_analysis': '🤖 AI Analysis',
        'generate_grade': '📊 Grade Generation',
        'complete': '✅ Completion'
    }[step] || step;
    
    stepCard.innerHTML = `
        <div class="accordion-header">
            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" 
                    data-bs-target="#collapse${step}" aria-expanded="false">
                ${stepTitle}
            </button>
        </div>
        <div id="collapse${step}" class="accordion-collapse collapse" data-bs-parent="#gradingDetails">
            <div class="accordion-body">
                <pre>${JSON.stringify(details, null, 2)}</pre>
            </div>
        </div>
    `;
    
    detailsContainer.appendChild(stepCard);
}

function addToLog(message, type = 'info') {
    console.log('📋 Adding log:', message);
    
    let logContent = document.getElementById('logContent');
    if (!logContent) {
        console.warn('Log content element not found');
        return;
    }
    
    let timestamp = new Date().toLocaleTimeString();
    let colorClass = type === 'error' ? 'text-danger' : 'text-success';
    
    logContent.innerHTML += `<div class="${colorClass}">[${timestamp}] ${message}</div>`;
    logContent.scrollTop = logContent.scrollHeight;
}

function showError(error) {
    console.log('⚠️ Showing error:', error);
    
    // Update progress with error message
    updateProgress(0, '❌ Error: ' + error);
    
    // Update grading stats in footer
    const gradingStats = document.getElementById('gradingStats');
    if (gradingStats) {
        gradingStats.innerHTML = '<i class="fas fa-exclamation-triangle me-1"></i>Error Occurred';
    }
    
    // Try to update legacy gradingStatus if it exists
    const gradingStatus = document.getElementById('gradingStatus');
    if (gradingStatus) {
        gradingStatus.innerHTML = '<h6 class="text-danger">❌ Error occurred</h6><small class="text-muted">' + error + '</small>';
    }
    
    // Show control buttons
    const closeBtn = document.getElementById('closeGradingModal');
    const restartBtn = document.getElementById('restartGrading');
    if (closeBtn) closeBtn.style.display = 'inline-block';
    if (restartBtn) restartBtn.style.display = 'inline-block';
    
    // Hide spinner
    const spinner = document.getElementById('gradingSpinner');
    if (spinner) spinner.style.display = 'none';
}

function completeGrading(gradingData = null) {
    console.log('🎉 Grading completed!', gradingData);
    
    updateProgress(100, '✅ AI grading completed successfully!');
    addToLog('🎉 Grading process completed!');
    
    // Update status in footer
    document.getElementById('gradingStats').innerHTML = '<i class="fas fa-check-circle me-1"></i>Grading Complete!';
    
    // Hide spinner and show close button
    document.getElementById('gradingSpinner').style.display = 'none';
    document.getElementById('closeGradingModal').style.display = 'inline-block';
    
    // Show results section if we have grading data
    if (gradingData && gradingData.details) {
        showGradingResults(gradingData.details);
        
        // Show the results for a few seconds, then refresh the page
        setTimeout(function() {
            addToLog('🔄 Refreshing page to show final results...');
            location.reload();
        }, 5000);
    } else {
        // If no data provided, fetch latest and then refresh
        fetchLatestGradingResults();
        
        // Refresh after a short delay to show the updated grade
        setTimeout(function() {
            addToLog('🔄 Refreshing page to show grading results...');
            location.reload();
        }, 3000);
    }
}

function showGradingResults(details) {
    console.log('📊 Displaying grading results:', details);
    
    // Show the results section
    document.getElementById('gradingResults').style.display = 'block';
    
    // Update score displays
    document.getElementById('finalScore').textContent = Math.round(details.ai_score || 0) + '%';
    document.getElementById('similarityResult').textContent = Math.round(details.similarity_score || 0) + '%';
    document.getElementById('keywordResult').textContent = Math.round(details.keyword_match_score || 0) + '%';
    document.getElementById('qualityResult').textContent = Math.round(details.structure_score || 0) + '%';
    document.getElementById('confidenceResult').textContent = Math.round(details.ai_confidence || 0) + '%';
    
    // Update feedback
    if (details.ai_feedback) {
        document.getElementById('gradingFeedback').innerHTML = details.ai_feedback.replace(/\n/g, '<br>');
    } else {
        document.getElementById('gradingFeedback').textContent = 'No detailed feedback available.';
    }
    
    // Add some celebration animation
    const scoreCircle = document.querySelector('.score-circle');
    if (scoreCircle) {
        scoreCircle.style.animation = 'bounce 1s ease-in-out';
    }
}

function fetchLatestGradingResults() {
    // Fetch the latest grading results from the server
    fetch('ajax/get_latest_grade.php?assignment_id=<?php echo $assignment_id ?? 0; ?>')
        .then(response => response.json())
        .then(data => {
            if (data.success && data.grade_data) {
                showGradingResults(data.grade_data);
            }
        })
        .catch(error => {
            console.log('Could not fetch latest results:', error);
        });
}

// Restart grading functionality
document.getElementById('restartGrading').addEventListener('click', function() {
    startLiveGrading(<?php echo $assignment_id ?? 0; ?>);
});

// Add button to manually trigger live grading for testing
<?php if ($isSubmitted && !$isGraded): ?>
// Add a "Watch AI Grading" button to submissions
document.addEventListener('DOMContentLoaded', function() {
    let submitButton = document.querySelector('.btn-primary');
    if (submitButton && submitButton.textContent.includes('Submit Assignment')) {
        let watchButton = document.createElement('button');
        watchButton.type = 'button';
        watchButton.className = 'btn btn-info ms-2';
        watchButton.innerHTML = '<i class="fas fa-eye"></i> Watch AI Grading';
        watchButton.onclick = () => startLiveGrading(<?php echo $assignment_id ?? 0; ?>);
        submitButton.parentNode.appendChild(watchButton);
    }
});
<?php endif; ?>

// Global access to handleLiveSubmission for testing
window.handleLiveSubmission = handleLiveSubmission;
window.startLiveGrading = startLiveGrading;

// Auto-trigger modal for new submissions
<?php if (isset($justSubmitted) && $justSubmitted): ?>
document.addEventListener('DOMContentLoaded', function() {
    console.log('🎉 New submission detected - triggering AI grading modal');
    
    // Show success message
    <?php if (isset($_SESSION['success'])): ?>
    showSuccess('<?php echo addslashes($_SESSION['success']); unset($_SESSION['success']); ?>');
    <?php endif; ?>
    
    // Automatically start AI grading with modal
    setTimeout(function() {
        console.log('🚀 Starting live AI grading for new submission');
        startLiveGrading(<?php echo $assignment_id; ?>);
    }, 1000);
});
<?php endif; ?>

// Helper function to show success messages
function showSuccess(message) {
    const alertDiv = document.createElement('div');
    alertDiv.className = 'alert alert-success alert-dismissible fade show';
    alertDiv.innerHTML = `
        <strong>Success!</strong> ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    const container = document.querySelector('.container-fluid');
    if (container) {
        container.insertBefore(alertDiv, container.firstChild);
    }
}

</script>
