<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
requireRole('student');
require_once '../config/db_connect.php';

// Get current student ID
$student_id = $_SESSION['user_id'];

// Get regular assignments first
$assignments_query = "
    SELECT 
        a.assignment_id,
        a.title,
        a.description,
        a.due_date,
        a.start_date,
        a.points as max_points,
        a.instructions,
        a.submission_type,
        a.allowed_formats,
        a.max_file_size,
        a.allow_late,
        a.late_penalty,
        a.group_assignment,
        a.max_group_size,
        a.grading_rubric,
        a.is_published,
        a.contributes_to_final_grade,
        c.course_id,
        c.title as course_title,
        c.thumbnail as course_thumbnail,
        u.first_name,
        u.last_name,
        ass.submission_id,
        ass.submission_date,
        ass.grade,
        ass.feedback,
        'regular' as assignment_type,
        NULL as request_id,
        CASE 
            WHEN ass.grade IS NOT NULL THEN 'graded'
            WHEN ass.submission_id IS NOT NULL THEN 'submitted'
            WHEN a.due_date < NOW() THEN 'overdue'
            ELSE 'pending'
        END as status
    FROM assignments a
    JOIN courses c ON a.course_id = c.course_id
    JOIN users u ON c.teacher_id = u.user_id
    JOIN enrollments e ON c.course_id = e.course_id AND e.student_id = ?
    LEFT JOIN assignment_submissions ass ON a.assignment_id = ass.assignment_id AND ass.student_id = ?
    WHERE a.is_published = 1
    ORDER BY a.due_date ASC
";

$stmt = $conn->prepare($assignments_query);
if (!$stmt) {
    die("SQL Error: " . $conn->error);
}

$stmt->bind_param("ii", $student_id, $student_id);
$stmt->execute();
$assignments_result = $stmt->get_result();
$assignments = [];
while ($row = $assignments_result->fetch_assoc()) {
    $assignments[] = $row;
}

// Get SITA accreditation assignments separately
$sita_query = "
    SELECT 
        ar.request_id,
        ar.assignment_instructions,
        ar.assignment_created_date,
        ar.status,
        c.course_id,
        c.title as course_title,
        c.thumbnail as course_thumbnail
    FROM accreditation_requests ar
    JOIN courses c ON ar.course_id = c.course_id
    WHERE ar.student_id = ? AND ar.status = 'assignment_created'
";

$sita_stmt = $conn->prepare($sita_query);
if ($sita_stmt) {
    $sita_stmt->bind_param("i", $student_id);
    $sita_stmt->execute();
    $sita_result = $sita_stmt->get_result();
    
    while ($sita_row = $sita_result->fetch_assoc()) {
        // Convert SITA assignment to assignments format
        $due_date = date('Y-m-d H:i:s', strtotime($sita_row['assignment_created_date'] . ' + 14 days'));
        
        $assignments[] = [
            'assignment_id' => 'sita_' . $sita_row['request_id'],
            'title' => 'SITA Assignment - ' . $sita_row['course_title'],
            'description' => $sita_row['assignment_instructions'],
            'due_date' => $due_date,
            'start_date' => $sita_row['assignment_created_date'],
            'max_points' => 100,
            'instructions' => $sita_row['assignment_instructions'],
            'submission_type' => '["file_upload", "text_entry"]',
            'allowed_formats' => '["pdf", "doc", "docx", "zip", "jpg", "jpeg", "png"]',
            'max_file_size' => 10485760,
            'allow_late' => 1,
            'late_penalty' => 0,
            'group_assignment' => 0,
            'max_group_size' => 1,
            'grading_rubric' => null,
            'is_published' => 1,
            'contributes_to_final_grade' => 1,
            'course_id' => $sita_row['course_id'],
            'course_title' => $sita_row['course_title'],
            'course_thumbnail' => $sita_row['course_thumbnail'],
            'first_name' => 'SITA',
            'last_name' => 'Administrator',
            'submission_id' => null,
            'submission_date' => null,
            'grade' => null,
            'feedback' => null,
            'assignment_type' => 'accreditation',
            'request_id' => $sita_row['request_id'],
            'status' => strtotime($due_date) < time() ? 'overdue' : 'pending'
        ];
    }
}

// Sort all assignments by due date
usort($assignments, function($a, $b) {
    return strtotime($a['due_date']) - strtotime($b['due_date']);
});

$page_title = "Assignments";

// Set current page for navigation
$currentPage = 'assignments';

// Include modern header with sidebar and navbar
include_once __DIR__ . '/components/header.php';
?>

<!-- Assignments Content -->
<div class="assignments-content">
    <style>
        :root {
            --panadite-primary: #2563eb;
            --panadite-secondary: #1d4ed8;
            --panadite-light: #dbeafe;
            --panadite-dark: #1e40af;
            --panadite-accent: #f59e0b;
            --text-primary: #1f2937;
            --text-secondary: #6b7280;
            --border-color: #e5e7eb;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --error-color: #ef4444;
            --bg-light: #f8fafc;
        }

        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: linear-gradient(135deg, var(--bg-light) 0%, #e2e8f0 100%);
            color: var(--text-primary);
            line-height: 1.6;
            min-height: 100vh;
        }

        /* Header */
        .page-header {
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
            box-shadow: 0 4px 20px rgba(37, 99, 235, 0.15);
        }

        .header-content {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem;
            text-align: center;
        }

        .page-title {
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
            text-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }

        .page-subtitle {
            font-size: 1.1rem;
            opacity: 0.9;
            font-weight: 400;
        }

        /* Main Container */
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem 2rem;
        }

        /* Stats Overview */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: white;
            border-radius: 16px;
            padding: 1.5rem;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
            border: 2px solid transparent;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 4px;
            background: linear-gradient(90deg, var(--panadite-primary), var(--panadite-secondary));
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 30px rgba(37, 99, 235, 0.15);
            border-color: var(--panadite-light);
        }

        .stat-icon {
            width: 50px;
            height: 50px;
            background: linear-gradient(135deg, var(--panadite-light) 0%, rgba(37, 99, 235, 0.1) 100%);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            color: var(--panadite-primary);
            margin-bottom: 1rem;
        }

        .stat-number {
            font-size: 2rem;
            font-weight: 700;
            color: var(--panadite-primary);
            margin-bottom: 0.25rem;
        }

        .stat-label {
            font-size: 0.9rem;
            color: var(--text-secondary);
            font-weight: 500;
        }

        /* Quiz Navigation Button */
        .quiz-nav-btn {
            background: linear-gradient(135deg, var(--panadite-primary), var(--panadite-secondary));
            color: white;
            border: none;
            padding: 1rem 2rem;
            border-radius: 12px;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.3s ease;
            margin-bottom: 2rem;
            box-shadow: 0 4px 15px rgba(37, 99, 235, 0.2);
        }

        .quiz-nav-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(37, 99, 235, 0.3);
            color: white;
            text-decoration: none;
        }

        /* Assignment Cards */
        .assignments-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(400px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .assignment-card {
            background: white;
            border-radius: 20px;
            overflow: hidden;
            box-shadow: 0 6px 25px rgba(0,0,0,0.08);
            transition: all 0.3s ease;
            border: 2px solid transparent;
            position: relative;
        }

        .assignment-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 12px 40px rgba(37, 99, 235, 0.15);
            border-color: var(--panadite-light);
        }

        .assignment-header {
            padding: 1.5rem;
            background: linear-gradient(135deg, var(--bg-light) 0%, #e2e8f0 50%);
            border-bottom: 1px solid var(--border-color);
        }

        .assignment-title {
            font-size: 1.3rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
            line-height: 1.3;
        }

        .assignment-course {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            color: var(--text-secondary);
            font-size: 0.9rem;
        }

        .course-badge {
            background: var(--panadite-light);
            color: var(--panadite-primary);
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
        }

        .assignment-content {
            padding: 1.5rem;
        }

        .assignment-description {
            color: var(--text-secondary);
            font-size: 0.95rem;
            margin-bottom: 1rem;
            line-height: 1.5;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }

        .assignment-meta {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
            margin-bottom: 1.5rem;
            padding: 1rem;
            background: var(--bg-light);
            border-radius: 12px;
        }

        .meta-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.9rem;
        }

        .meta-icon {
            color: var(--panadite-primary);
            font-size: 1.1rem;
        }

        /* Status Badges */
        .status-badge {
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            display: inline-flex;
            align-items: center;
            gap: 0.3rem;
            margin-bottom: 1rem;
        }

        .status-pending {
            background: #fef3c7;
            color: #d97706;
        }

        .status-submitted {
            background: #dbeafe;
            color: #2563eb;
        }

        .status-graded {
            background: #d1fae5;
            color: #059669;
        }

        .status-overdue {
            background: #fee2e2;
            color: #dc2626;
        }

        /* Action Buttons */
        .action-buttons {
            display: flex;
            gap: 0.75rem;
            flex-wrap: wrap;
        }

        .action-btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 12px;
            font-weight: 600;
            font-size: 0.9rem;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            position: relative;
            overflow: hidden;
        }

        .action-btn::before {
            content: '';
            position: absolute;
            inset: 0;
            background: linear-gradient(45deg, transparent 30%, rgba(255,255,255,0.3) 50%, transparent 70%);
            transform: translateX(-100%);
            transition: transform 0.5s ease;
        }

        .action-btn:hover::before {
            transform: translateX(100%);
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(37, 99, 235, 0.3);
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(37, 99, 235, 0.4);
        }

        .btn-secondary {
            background: linear-gradient(135deg, var(--panadite-accent) 0%, #f97316 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(245, 158, 11, 0.3);
        }

        .btn-secondary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(245, 158, 11, 0.4);
        }

        .btn-outline {
            background: transparent;
            color: var(--panadite-primary);
            border: 2px solid var(--panadite-primary);
        }

        .btn-outline:hover {
            background: var(--panadite-primary);
            color: white;
        }

        /* Q&A Section */
        .qa-section {
            background: white;
            border-radius: 16px;
            padding: 1.5rem;
            margin-top: 1rem;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
            border: 2px solid var(--border-color);
        }

        .qa-header {
            display: flex;
            align-items: center;
            gap: 1rem;
            margin-bottom: 1rem;
        }

        .qa-icon {
            width: 40px;
            height: 40px;
            background: var(--panadite-primary);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.2rem;
        }

        .qa-title {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--text-primary);
        }

        .question-form {
            display: flex;
            flex-direction: column;
            gap: 1rem;
        }

        .question-input {
            width: 100%;
            padding: 1rem;
            border: 2px solid var(--border-color);
            border-radius: 12px;
            font-size: 0.95rem;
            resize: vertical;
            min-height: 100px;
            transition: all 0.3s ease;
        }

        .question-input:focus {
            outline: none;
            border-color: var(--panadite-primary);
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
            color: var(--text-secondary);
        }

        .empty-icon {
            font-size: 4rem;
            color: var(--border-color);
            margin-bottom: 1rem;
        }

        .empty-title {
            font-size: 1.5rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
            color: var(--text-primary);
        }

        .empty-description {
            font-size: 1rem;
            max-width: 400px;
            margin: 0 auto;
        }

        /* Mobile Responsive */
        @media (max-width: 768px) {
            body {
                margin-left: 0;
            }
            
            .page-title {
                font-size: 2rem;
            }
            
            .assignments-grid {
                grid-template-columns: 1fr;
            }
            
            .stats-grid {
                grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            }
            
            .assignment-meta {
                grid-template-columns: 1fr;
                gap: 0.5rem;
            }
            
            .action-buttons {
                flex-direction: column;
            }
        }

        @media (max-width: 480px) {
            .container {
                padding: 0 0.5rem 2rem;
            }
            
            .page-header {
                padding: 1.5rem 0;
            }
            
            .assignment-header, .assignment-content {
                padding: 1rem;
            }
            
            .qa-section {
                padding: 1rem;
            }
        }
    </style>
</head>
<body>
    <?php include_once 'components/sidebar.php'; ?>
    
    <div class="page-header">
        <div class="header-content">
            <h1 class="page-title">
                <i class="bx bx-task"></i>
                Assignments
            </h1>
            <p class="page-subtitle">Manage your coursework and submit assignments</p>
        </div>
    </div>

    <div class="container">
        <!-- Assignment Stats -->
        <div class="stats-grid">
            <?php
            $assignment_stats = [
                'pending' => 0,
                'submitted' => 0,
                'graded' => 0,
                'overdue' => 0
            ];
            
            foreach($assignments as $assignment) {
                $assignment_stats[$assignment['status']]++;
            }
            ?>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="bx bx-time"></i>
                </div>
                <div class="stat-number"><?= $assignment_stats['pending'] ?></div>
                <div class="stat-label">Pending Assignments</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="bx bx-check-circle"></i>
                </div>
                <div class="stat-number"><?= $assignment_stats['submitted'] ?></div>
                <div class="stat-label">Submitted</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="bx bx-star"></i>
                </div>
                <div class="stat-number"><?= $assignment_stats['graded'] ?></div>
                <div class="stat-label">Graded</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="bx bx-alarm-exclamation"></i>
                </div>
                <div class="stat-number"><?= $assignment_stats['overdue'] ?></div>
                <div class="stat-label">Overdue</div>
            </div>
        </div>

        <!-- Quiz Navigation Button -->
        <a href="all_quizzes.php" class="quiz-nav-btn">
            <i class="bx bx-help-circle"></i>
            View All Quizzes
        </a>

        <!-- Assignment Cards -->
        <?php if (!empty($assignments)): ?>
            <div class="assignments-grid">
                <?php foreach($assignments as $assignment): ?>
                    <?php 
                        // Parse JSON data safely for student display
                        $submission_types = json_decode($assignment['submission_type'] ?? '[]', true) ?: [];
                        $allowed_formats = json_decode($assignment['allowed_formats'] ?? '[]', true) ?: [];
                        $is_overdue = strtotime($assignment['due_date']) < time();
                        $starts_soon = $assignment['start_date'] && strtotime($assignment['start_date']) > time();
                    ?>
                    <div class="assignment-card">
                        <div class="assignment-header">
                            <div class="d-flex justify-content-between align-items-start">
                                <div>
                                    <h3 class="assignment-title"><?= htmlspecialchars($assignment['title']) ?></h3>
                                    <div class="assignment-course">
                                        <i class="bx bx-book"></i>
                                        <span class="course-badge"><?= htmlspecialchars($assignment['course_title']) ?></span>
                                        <span>by <?= htmlspecialchars($assignment['first_name'] . ' ' . $assignment['last_name']) ?></span>
                                    </div>
                                </div>
                                <div class="d-flex flex-column align-items-end gap-1">
                                    <?php if ($assignment['contributes_to_final_grade']): ?>
                                        <span class="badge" style="background: #10b981; color: white; font-size: 0.7rem; padding: 0.25rem 0.5rem;">
                                            <i class="bx bx-certification" style="font-size: 0.8rem;"></i> Final Grade
                                        </span>
                                    <?php else: ?>
                                        <span class="badge" style="background: #6b7280; color: white; font-size: 0.7rem; padding: 0.25rem 0.5rem;">
                                            <i class="bx bx-practice" style="font-size: 0.8rem;"></i> Practice Only
                                        </span>
                                    <?php endif; ?>
                                    <?php if ($assignment['group_assignment']): ?>
                                        <span class="badge" style="background: #3b82f6; color: white; font-size: 0.7rem; padding: 0.25rem 0.5rem;">
                                            <i class="bx bx-group" style="font-size: 0.8rem;"></i> Group Work
                                        </span>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                        
                        <div class="assignment-content">
                            <p class="assignment-description"><?= htmlspecialchars($assignment['description']) ?></p>
                            
                            <!-- Enhanced Assignment Requirements Section -->
                            <div class="assignment-requirements" style="background: #f8fafc; border-radius: 8px; padding: 1rem; margin: 1rem 0; border-left: 4px solid var(--panadite-primary);">
                                <h6 style="color: var(--panadite-primary); margin-bottom: 0.75rem; font-weight: 600;">
                                    <i class="bx bx-list-check"></i> Assignment Requirements
                                </h6>
                                
                                <?php if (!empty($submission_types)): ?>
                                    <div style="margin-bottom: 0.75rem;">
                                        <strong style="color: var(--text-primary); font-size: 0.9rem;">Required Submissions:</strong>
                                        <div style="margin-top: 0.5rem; display: flex; flex-wrap: wrap; gap: 0.5rem;">
                                            <?php foreach($submission_types as $type): ?>
                                                <span class="badge" style="background: #e0f2fe; color: #0277bd; border: 1px solid #81d4fa;">
                                                    <?php 
                                                        $type_icons = [
                                                            'file_upload' => 'bx-upload',
                                                            'video' => 'bx-video',
                                                            'document' => 'bx-file-doc',
                                                            'image' => 'bx-image',
                                                            'excel' => 'bx-spreadsheet',
                                                            'pdf' => 'bx-file-pdf',
                                                            'presentation' => 'bx-slideshow',
                                                            'code' => 'bx-code-alt',
                                                            'text_entry' => 'bx-edit-alt',
                                                            'multiple_files' => 'bx-folder-plus'
                                                        ];
                                                        $icon = $type_icons[$type] ?? 'bx-file';
                                                    ?>
                                                    <i class="bx <?= $icon ?>"></i> <?= ucfirst(str_replace('_', ' ', $type)) ?>
                                                </span>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if (!empty($allowed_formats)): ?>
                                    <div style="margin-bottom: 0.75rem;">
                                        <strong style="color: var(--text-primary); font-size: 0.9rem;">Allowed File Formats:</strong>
                                        <div style="margin-top: 0.25rem; font-size: 0.85rem; color: var(--text-secondary);">
                                            <?= implode(', ', array_map('strtoupper', $allowed_formats)) ?>
                                        </div>
                                    </div>
                                <?php endif; ?>
                                
                                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 0.5rem; font-size: 0.85rem;">
                                    <?php if ($assignment['max_file_size']): ?>
                                        <div>
                                            <strong>Max File Size:</strong> <?= $assignment['max_file_size'] ?>MB
                                        </div>
                                    <?php endif; ?>
                                    
                                    <?php if ($assignment['group_assignment']): ?>
                                        <div>
                                            <strong>Team Size:</strong> Up to <?= $assignment['max_group_size'] ?> members
                                        </div>
                                    <?php endif; ?>
                                    
                                    <?php if (!$assignment['allow_late']): ?>
                                        <div style="color: #dc2626;">
                                            <strong>⚠️ No Late Submissions</strong>
                                        </div>
                                    <?php elseif ($assignment['late_penalty'] > 0): ?>
                                        <div style="color: #f59e0b;">
                                            <strong>Late Penalty:</strong> <?= $assignment['late_penalty'] ?>% per day
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <div class="assignment-meta">
                                <?php if ($assignment['start_date'] && strtotime($assignment['start_date']) > time()): ?>
                                    <div class="meta-item" style="color: var(--panadite-primary);">
                                        <i class="meta-icon bx bx-time-five"></i>
                                        <span><strong>Available:</strong> <?= date('M d, Y g:i A', strtotime($assignment['start_date'])) ?></span>
                                    </div>
                                <?php endif; ?>
                                
                                <div class="meta-item <?= $is_overdue ? 'text-danger' : '' ?>">
                                    <i class="meta-icon bx bx-calendar"></i>
                                    <span><strong>Due:</strong> <?= date('M d, Y g:i A', strtotime($assignment['due_date'])) ?></span>
                                    <?php if ($is_overdue): ?>
                                        <span style="color: #dc2626; font-weight: bold;"> (OVERDUE)</span>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="meta-item">
                                    <i class="meta-icon bx bx-award"></i>
                                    <span><strong><?= $assignment['max_points'] ?> Points</strong>
                                        <?php if ($assignment['contributes_to_final_grade']): ?>
                                            <small style="color: #10b981;">• Counts toward final grade</small>
                                        <?php else: ?>
                                            <small style="color: #6b7280;">• Practice assignment</small>
                                        <?php endif; ?>
                                    </span>
                                </div>
                                
                                <?php if($assignment['submission_date']): ?>
                                    <div class="meta-item">
                                        <i class="meta-icon bx bx-check"></i>
                                        <span>Submitted: <?= date('M d, Y g:i A', strtotime($assignment['submission_date'])) ?></span>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if($assignment['grade'] !== null): ?>
                                    <div class="meta-item">
                                        <i class="meta-icon bx bx-trophy"></i>
                                        <span>Grade: <strong><?= $assignment['grade'] ?>/<?= $assignment['max_points'] ?></strong> 
                                            (<?= round(($assignment['grade'] / $assignment['max_points']) * 100, 1) ?>%)</span>
                                    </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="status-badge status-<?= $assignment['status'] ?>">
                                <?php
                                $status_icons = [
                                    'pending' => 'bx-time',
                                    'submitted' => 'bx-check-circle',
                                    'graded' => 'bx-star',
                                    'overdue' => 'bx-alarm-exclamation'
                                ];
                                ?>
                                <i class="bx <?= $status_icons[$assignment['status']] ?>"></i>
                                <?= ucfirst($assignment['status']) ?>
                            </div>
                            
                            <div class="action-buttons">
                                <?php if ($assignment['assignment_type'] === 'accreditation'): ?>
                                    <!-- SITA Accreditation Assignment Actions -->
                                    <?php if ($assignment['status'] === 'pending' || $assignment['status'] === 'overdue'): ?>
                                        <a href="sita_assignment_submission.php?request_id=<?= $assignment['request_id'] ?>" class="action-btn btn-primary">
                                            <i class="bx bx-award"></i>
                                            Start SITA Assignment
                                        </a>
                                    <?php elseif ($assignment['status'] === 'submitted'): ?>
                                        <div class="action-btn" style="background: #dcfce7; color: #166534; cursor: default; border: 1px solid #bbf7d0;">
                                            <i class="bx bx-check-circle"></i>
                                            Assignment Submitted
                                        </div>
                                    <?php elseif ($assignment['status'] === 'graded'): ?>
                                        <div class="action-btn" style="background: #dbeafe; color: #1e40af; cursor: default; border: 1px solid #93c5fd;">
                                            <i class="bx bx-trophy"></i>
                                            Assessment Passed
                                        </div>
                                    <?php endif; ?>
                                    
                                    <!-- Download Assignment Documents -->
                                    <a href="download_files_simple.php?request_id=<?= $assignment['request_id'] ?>" class="action-btn btn-outline">
                                        <i class="bx bx-download"></i>
                                        Download Files
                                    </a>
                                    
                                    <!-- View Progress -->
                                    <a href="course-detail.php?id=<?= $assignment['course_id'] ?>#certificate-tab" class="action-btn btn-outline">
                                        <i class="bx bx-line-chart"></i>
                                        View Progress
                                    </a>
                                <?php else: ?>
                                    <!-- Regular Assignment Actions -->
                                    <?php if ($starts_soon): ?>
                                        <div class="action-btn" style="background: #fef3c7; color: #92400e; cursor: default; border: 1px solid #fed7aa;">
                                            <i class="bx bx-time"></i>
                                            Available <?= date('M d, g:i A', strtotime($assignment['start_date'])) ?>
                                        </div>
                                    <?php elseif($assignment['status'] === 'pending' || $assignment['status'] === 'overdue'): ?>
                                        <a href="submit-assignment.php?id=<?= $assignment['assignment_id'] ?>" class="action-btn btn-primary">
                                            <i class="bx bx-upload"></i>
                                            <?php if ($assignment['status'] === 'overdue'): ?>
                                                <?php if ($assignment['allow_late']): ?>
                                                    Submit Late <?= $assignment['late_penalty'] > 0 ? '(-'.$assignment['late_penalty'].'% penalty)' : '' ?>
                                                <?php else: ?>
                                                    Submission Closed
                                                <?php endif; ?>
                                            <?php else: ?>
                                                Submit Assignment
                                            <?php endif; ?>
                                        </a>
                                    <?php endif; ?>
                                    
                                    <a href="assignment-details.php?id=<?= $assignment['assignment_id'] ?>" class="action-btn btn-outline">
                                        <i class="bx bx-detail"></i>
                                        View Details
                                    </a>
                                    
                                    <?php if (!empty($assignment['instructions'])): ?>
                                        <a href="#" onclick="showInstructions(<?= $assignment['assignment_id'] ?>)" class="action-btn btn-outline">
                                            <i class="bx bx-info-circle"></i>
                                            Instructions
                                        </a>
                                    <?php endif; ?>
                                    
                                    <?php if($assignment['feedback']): ?>
                                        <a href="assignment-feedback.php?id=<?= $assignment['assignment_id'] ?>" class="action-btn btn-secondary">
                                            <i class="bx bx-message-dots"></i>
                                            View Feedback
                                        </a>
                                    <?php endif; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div class="empty-state">
                <div class="empty-icon">
                    <i class="bx bx-task"></i>
                </div>
                <h3 class="empty-title">No Assignments Available</h3>
                <p class="empty-description">You don't have any assignments at the moment. Check back later for new assignments from your instructors.</p>
            </div>
        <?php endif; ?>
    </div>
    
    <script>
        // Animate cards on load
        document.addEventListener('DOMContentLoaded', function() {
            const cards = document.querySelectorAll('.assignment-card, .stat-card');
            cards.forEach((card, index) => {
                card.style.opacity = '0';
                card.style.transform = 'translateY(20px)';
                
                setTimeout(() => {
                    card.style.transition = 'all 0.6s ease';
                    card.style.opacity = '1';
                    card.style.transform = 'translateY(0)';
                }, index * 100);
            });
        });
        
        // Show assignment instructions modal
        function showInstructions(assignmentId) {
            // Find the assignment data
            const assignments = <?= json_encode($assignments) ?>;
            const assignment = assignments.find(a => a.assignment_id == assignmentId);
            
            if (assignment && assignment.instructions) {
                alert('Assignment Instructions:\n\n' + assignment.instructions);
                // TODO: Replace with a proper modal in the future
            } else {
                alert('No detailed instructions available for this assignment.');
            }
        }
    </script>
</div>

<!-- Close home-content div -->
        </div>
    </section>
</body>
</html>
