<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
requireRole('student');
require_once '../config/db_connect.php';

$page_title = 'Calendar';
$user_id = $_SESSION['user_id'];
$current_date = date('Y-m-d');
$current_month = isset($_GET['month']) ? $_GET['month'] : date('Y-m');

// Function to check if table exists
function tableExists($conn, $tableName) {
    if (!$conn) return false;
    $result = $conn->query("SHOW TABLES LIKE '{$tableName}'");
    return ($result && $result->num_rows > 0);
}

// Get student's courses
$courses = [];
if (isset($conn) && tableExists($conn, 'courses') && tableExists($conn, 'enrollments')) {
    $stmt = $conn->prepare("
        SELECT c.course_id, c.title, c.thumbnail
        FROM courses c
        JOIN enrollments e ON c.course_id = e.course_id
        WHERE e.student_id = ?
    ");
    
    if ($stmt) {
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result) {
            $courses = $result->fetch_all(MYSQLI_ASSOC);
        }
        $stmt->close();
    }
}

// Get events for the month (create sample events if table doesn't exist)
$events = [];
if (isset($conn) && tableExists($conn, 'calendar_events')) {
    $stmt = $conn->prepare("
        SELECT 
            ce.event_id,
            ce.title,
            ce.description,
            ce.start_date,
            ce.end_date,
            ce.event_type,
            ce.course_id,
            c.title as course_name,
            c.thumbnail as course_thumbnail
        FROM calendar_events ce
        LEFT JOIN courses c ON ce.course_id = c.course_id
        WHERE (ce.user_id = ? OR ce.course_id IN (
            SELECT course_id FROM enrollments WHERE student_id = ?
        ))
        AND DATE_FORMAT(ce.start_date, '%Y-%m') = ?
        ORDER BY ce.start_date ASC
    ");
    
    if ($stmt) {
        $stmt->bind_param("iis", $user_id, $user_id, $current_month);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result) {
            $events = $result->fetch_all(MYSQLI_ASSOC);
        }
        $stmt->close();
    }
} else {
    // Create sample events for demonstration
    $events = [
        [
            'event_id' => 1,
            'title' => 'Sample Study Session',
            'description' => 'Review course materials',
            'start_date' => date('Y-m-d H:i:s'),
            'end_date' => date('Y-m-d H:i:s', strtotime('+2 hours')),
            'event_type' => 'study',
            'course_id' => null,
            'course_name' => null,
            'course_thumbnail' => null
        ]
    ];
}

include_once 'components/header.php';
?>

<style>
/* Modern Calendar Styling */
body {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    font-family: 'Inter', 'Segoe UI', Roboto, sans-serif;
    min-height: 100vh;
}

.calendar-wrapper {
    padding: 2rem 0;
    min-height: 100vh;
}

.calendar-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 0 2rem;
}

/* Enhanced Header */
.calendar-header {
    text-align: center;
    margin-bottom: 3rem;
    position: relative;
    padding: 3rem 2rem;
    background: rgba(255, 255, 255, 0.1);
    backdrop-filter: blur(20px);
    border-radius: 30px;
    border: 1px solid rgba(255, 255, 255, 0.2);
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.15);
    overflow: hidden;
}

.calendar-header h1 {
    font-size: 3.5rem;
    font-weight: 800;
    margin-bottom: 1rem;
    color: #ffffff;
    text-shadow: 
        0 2px 4px rgba(0, 0, 0, 0.5),
        0 4px 8px rgba(0, 0, 0, 0.3),
        0 8px 16px rgba(0, 0, 0, 0.2);
}

.calendar-header p {
    color: #ffffff;
    font-size: 1.3rem;
    font-weight: 500;
    text-shadow: 
        0 1px 2px rgba(0, 0, 0, 0.6),
        0 2px 4px rgba(0, 0, 0, 0.4);
    margin: 0;
    opacity: 0.95;
}

/* Main Grid Layout */
.calendar-grid {
    display: grid;
    grid-template-columns: 1fr 350px;
    gap: 2rem;
    margin-bottom: 2rem;
}

/* Calendar Section */
.calendar-main {
    background: white;
    border-radius: 24px;
    padding: 2rem;
    box-shadow: 0 20px 60px rgba(0,0,0,0.1);
    border: 1px solid rgba(255,255,255,0.2);
}

.calendar-controls {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
    padding-bottom: 1rem;
    border-bottom: 2px solid #f1f5f9;
}

.month-navigation {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.nav-btn {
    width: 40px;
    height: 40px;
    border: none;
    border-radius: 12px;
    background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
    color: white;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.3s ease;
}

.nav-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(78,115,223,0.3);
}

.current-month {
    font-size: 1.5rem;
    font-weight: 700;
    color: #2c3e50;
}

.add-event-btn {
    background: linear-gradient(135deg, #1cc88a 0%, #17a673 100%);
    color: white;
    border: none;
    padding: 0.75rem 1.5rem;
    border-radius: 12px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.add-event-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(28,200,138,0.3);
}

/* Calendar Grid */
.calendar-table {
    width: 100%;
    border-collapse: separate;
    border-spacing: 8px;
}

.calendar-table th {
    padding: 1rem;
    text-align: center;
    font-weight: 600;
    color: #64748b;
    font-size: 0.9rem;
    text-transform: uppercase;
    letter-spacing: 1px;
}

.calendar-table td {
    width: 14.28%;
    height: 120px;
    vertical-align: top;
    padding: 0;
    position: relative;
}

.calendar-day {
    width: 100%;
    height: 100%;
    border: 2px solid #f1f5f9;
    border-radius: 12px;
    padding: 0.5rem;
    background: #ffffff;
    cursor: pointer;
    transition: all 0.3s ease;
    position: relative;
}

.calendar-day:hover {
    border-color: #4e73df;
    box-shadow: 0 4px 12px rgba(78,115,223,0.2);
}

.calendar-day.today {
    background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
    color: white;
    border-color: #4e73df;
}

.calendar-day.other-month {
    opacity: 0.3;
    background: #f8fafc;
}

.day-number {
    font-weight: 600;
    margin-bottom: 0.25rem;
}

.event-indicator {
    width: 100%;
    height: 4px;
    background: linear-gradient(90deg, #1cc88a 0%, #4e73df 100%);
    border-radius: 2px;
    margin: 2px 0;
}

/* Sidebar */
.calendar-sidebar {
    display: flex;
    flex-direction: column;
    gap: 1.5rem;
}

.sidebar-section {
    background: white;
    border-radius: 20px;
    padding: 1.5rem;
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    border: 1px solid rgba(255,255,255,0.2);
}

.section-title {
    font-size: 1.2rem;
    font-weight: 700;
    color: #2c3e50;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.today-events {
    max-height: 300px;
    overflow-y: auto;
}

.event-item {
    padding: 1rem;
    border-radius: 12px;
    margin-bottom: 0.75rem;
    border-left: 4px solid #4e73df;
    background: #f8fafc;
    transition: all 0.3s ease;
}

.event-item:hover {
    transform: translateX(5px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
}

.event-title {
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 0.25rem;
}

.event-time {
    color: #64748b;
    font-size: 0.85rem;
}

.course-filter {
    margin-bottom: 1rem;
}

.filter-select {
    width: 100%;
    padding: 0.75rem;
    border: 2px solid #e2e8f0;
    border-radius: 12px;
    font-size: 1rem;
    background: white;
}

.course-list {
    display: flex;
    flex-direction: column;
    gap: 0.75rem;
}

.course-item {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 0.75rem;
    border-radius: 12px;
    background: #f8fafc;
    transition: all 0.3s ease;
    cursor: pointer;
}

.course-item:hover {
    background: #e2e8f0;
    transform: translateX(5px);
}

.course-thumbnail {
    width: 40px;
    height: 40px;
    border-radius: 8px;
    background: linear-gradient(135deg, #4e73df 0%, #1cc88a 100%);
    background-size: cover;
}

.course-name {
    font-weight: 500;
    color: #2c3e50;
}

/* Modal Styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.5);
    backdrop-filter: blur(5px);
}

.modal-content {
    background: white;
    margin: 5% auto;
    padding: 2rem;
    border-radius: 20px;
    width: 90%;
    max-width: 500px;
    box-shadow: 0 20px 60px rgba(0,0,0,0.3);
}

.form-group {
    margin-bottom: 1.5rem;
}

.form-label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 600;
    color: #2c3e50;
}

.form-input {
    width: 100%;
    padding: 0.75rem;
    border: 2px solid #e2e8f0;
    border-radius: 12px;
    font-size: 1rem;
}

.form-input:focus {
    outline: none;
    border-color: #4e73df;
    box-shadow: 0 0 0 3px rgba(78,115,223,0.1);
}

.btn-primary {
    background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
    color: white;
    border: none;
    padding: 0.75rem 2rem;
    border-radius: 12px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-secondary {
    background: #64748b;
    color: white;
    border: none;
    padding: 0.75rem 2rem;
    border-radius: 12px;
    font-weight: 600;
    cursor: pointer;
    margin-left: 0.5rem;
}

/* Mobile Responsive Styles */
@media (max-width: 1024px) {
    .calendar-grid {
        grid-template-columns: 1fr;
        gap: 1.5rem;
    }
    
    .calendar-sidebar {
        order: -1;
    }
}

@media (max-width: 768px) {
    .calendar-container {
        padding: 0 1rem;
    }
    
    .calendar-wrapper {
        padding: 1rem 0;
    }
    
    .calendar-header {
        padding: 2rem 1rem;
        margin-bottom: 2rem;
    }
    
    .calendar-header h1 {
        font-size: 2.5rem;
    }
    
    .calendar-header p {
        font-size: 1.1rem;
    }
    
    .calendar-grid {
        grid-template-columns: 1fr;
        gap: 1.5rem;
    }
    
    .calendar-main {
        padding: 1.5rem;
        border-radius: 20px;
    }
    
    .calendar-controls {
        flex-direction: column;
        gap: 1rem;
        align-items: center;
    }
    
    .month-navigation {
        order: 1;
    }
    
    .add-event-btn {
        order: 2;
        width: 100%;
        justify-content: center;
    }
    
    .current-month {
        font-size: 1.3rem;
    }
    
    .calendar-table {
        border-spacing: 4px;
    }
    
    .calendar-table th {
        padding: 0.5rem;
        font-size: 0.8rem;
    }
    
    .calendar-day {
        height: 80px;
        padding: 0.25rem;
    }
    
    .day-number {
        font-size: 0.9rem;
    }
    
    .sidebar-section {
        padding: 1rem;
        border-radius: 16px;
    }
    
    .section-title {
        font-size: 1.1rem;
    }
    
    .event-item {
        padding: 0.75rem;
    }
    
    .course-item {
        padding: 0.5rem;
    }
    
    .course-thumbnail {
        width: 35px;
        height: 35px;
    }
    
    .modal-content {
        margin: 10% auto;
        padding: 1.5rem;
        width: 95%;
        border-radius: 16px;
    }
}

@media (max-width: 480px) {
    .calendar-header h1 {
        font-size: 2rem;
    }
    
    .calendar-header p {
        font-size: 1rem;
    }
    
    .calendar-main {
        padding: 1rem;
    }
    
    .calendar-table th {
        padding: 0.25rem;
        font-size: 0.7rem;
    }
    
    .calendar-day {
        height: 60px;
        padding: 0.2rem;
    }
    
    .day-number {
        font-size: 0.8rem;
        margin-bottom: 0.1rem;
    }
    
    .event-indicator {
        height: 3px;
        margin: 1px 0;
    }
    
    .nav-btn {
        width: 35px;
        height: 35px;
    }
    
    .current-month {
        font-size: 1.1rem;
    }
    
    .sidebar-section {
        padding: 0.75rem;
    }
    
    .section-title {
        font-size: 1rem;
    }
    
    .event-item {
        padding: 0.5rem;
    }
    
    .event-title {
        font-size: 0.9rem;
    }
    
    .event-time {
        font-size: 0.8rem;
    }
    
    .course-name {
        font-size: 0.9rem;
    }
    
    .modal-content {
        margin: 5% auto;
        padding: 1rem;
        border-radius: 12px;
    }
    
    .form-input {
        padding: 0.6rem;
        font-size: 0.9rem;
    }
    
    .btn-primary,
    .btn-secondary {
        padding: 0.6rem 1.5rem;
        font-size: 0.9rem;
    }
}

/* Horizontal scroll for very small screens */
@media (max-width: 360px) {
    .calendar-table-container {
        overflow-x: auto;
        margin: 0 -1rem;
        padding: 0 1rem;
    }
    
    .calendar-table {
        min-width: 320px;
    }
    
    .calendar-day {
        height: 50px;
    }
    
    .day-number {
        font-size: 0.7rem;
    }
}
</style>

<div class="calendar-wrapper">
    <div class="calendar-container">
        <!-- Enhanced Header -->
        <div class="calendar-header">
            <h1><i class="bx bx-calendar me-3"></i>My Calendar</h1>
            <p>Plan your classes and track important events</p>
        </div>

        <!-- Main Calendar Grid -->
        <div class="calendar-grid">
            <!-- Calendar Main Section -->
            <div class="calendar-main">
                <div class="calendar-controls">
                    <div class="month-navigation">
                        <button class="nav-btn" onclick="changeMonth(-1)">
                            <i class="bx bx-chevron-left"></i>
                        </button>
                        <span class="current-month" id="currentMonth">
                            <?= date('F Y', strtotime($current_month . '-01')) ?>
                        </span>
                        <button class="nav-btn" onclick="changeMonth(1)">
                            <i class="bx bx-chevron-right"></i>
                        </button>
                    </div>
                    <button class="add-event-btn" onclick="showAddEventModal()">
                        <i class="bx bx-plus"></i>Add Event
                    </button>
                </div>

                <!-- Calendar Table -->
                <div class="calendar-table-container">
                    <table class="calendar-table">
                        <thead>
                            <tr>
                                <th>Sun</th><th>Mon</th><th>Tue</th><th>Wed</th><th>Thu</th><th>Fri</th><th>Sat</th>
                            </tr>
                        </thead>
                        <tbody id="calendarBody">
                            <!-- Calendar days will be generated by JavaScript -->
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Sidebar -->
            <div class="calendar-sidebar">
                <!-- Today's Events -->
                <div class="sidebar-section">
                    <h3 class="section-title">
                        <i class="bx bx-time-five"></i>Today's Events
                    </h3>
                    <div class="today-events" id="todayEvents">
                        <?php
                        $today_events = array_filter($events, function($event) {
                            return date('Y-m-d', strtotime($event['start_date'])) === date('Y-m-d');
                        });
                        
                        if (empty($today_events)): ?>
                            <p class="text-muted text-center">No events scheduled for today</p>
                        <?php else: ?>
                            <?php foreach ($today_events as $event): ?>
                                <div class="event-item">
                                    <div class="event-title"><?= htmlspecialchars($event['title']) ?></div>
                                    <div class="event-time">
                                        <?= date('g:i A', strtotime($event['start_date'])) ?>
                                        <?php if ($event['course_name']): ?>
                                            - <?= htmlspecialchars($event['course_name']) ?>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- My Courses -->
                <div class="sidebar-section">
                    <h3 class="section-title">
                        <i class="bx bx-book-open"></i>My Courses
                    </h3>
                    <div class="course-list">
                        <?php foreach ($courses as $course): ?>
                            <div class="course-item" onclick="filterByCourse(<?= $course['course_id'] ?>)">
                                <div class="course-thumbnail" 
                                     style="background-image: url('<?= htmlspecialchars($course['thumbnail'] ?? '../assets/img/courses/default.jpg') ?>')">
                                </div>
                                <div class="course-name"><?= htmlspecialchars($course['title']) ?></div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Event Modal -->
<div id="addEventModal" class="modal">
    <div class="modal-content">
        <h2>Add New Event</h2>
        <form id="eventForm">
            <div class="form-group">
                <label class="form-label">Event Title</label>
                <input type="text" class="form-input" name="title" required>
            </div>
            <div class="form-group">
                <label class="form-label">Description</label>
                <textarea class="form-input" name="description" rows="3"></textarea>
            </div>
            <div class="form-group">
                <label class="form-label">Date & Time</label>
                <input type="datetime-local" class="form-input" name="start_date" required>
            </div>
            <div class="form-group">
                <label class="form-label">Course (Optional)</label>
                <select class="form-input" name="course_id">
                    <option value="">Select Course</option>
                    <?php foreach ($courses as $course): ?>
                        <option value="<?= $course['course_id'] ?>"><?= htmlspecialchars($course['title']) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div style="text-align: right;">
                <button type="button" class="btn-secondary" onclick="closeModal()">Cancel</button>
                <button type="submit" class="btn-primary">Add Event</button>
            </div>
        </form>
    </div>
</div>

<script>
let currentDate = new Date();
let events = <?= json_encode($events) ?>;

function generateCalendar() {
    const year = currentDate.getFullYear();
    const month = currentDate.getMonth();
    
    document.getElementById('currentMonth').textContent = 
        currentDate.toLocaleString('default', { month: 'long', year: 'numeric' });
    
    const firstDay = new Date(year, month, 1);
    const lastDay = new Date(year, month + 1, 0);
    const startDate = new Date(firstDay);
    startDate.setDate(startDate.getDate() - firstDay.getDay());
    
    const calendarBody = document.getElementById('calendarBody');
    calendarBody.innerHTML = '';
    
    for (let week = 0; week < 6; week++) {
        const row = document.createElement('tr');
        
        for (let day = 0; day < 7; day++) {
            const cell = document.createElement('td');
            const cellDate = new Date(startDate);
            cellDate.setDate(startDate.getDate() + (week * 7) + day);
            
            const dayDiv = document.createElement('div');
            dayDiv.className = 'calendar-day';
            
            if (cellDate.getMonth() !== month) {
                dayDiv.classList.add('other-month');
            }
            
            if (cellDate.toDateString() === new Date().toDateString()) {
                dayDiv.classList.add('today');
            }
            
            dayDiv.innerHTML = `
                <div class="day-number">${cellDate.getDate()}</div>
            `;
            
            // Add event indicators
            const dayEvents = events.filter(event => {
                const eventDate = new Date(event.start_date);
                return eventDate.toDateString() === cellDate.toDateString();
            });
            
            dayEvents.forEach(event => {
                const indicator = document.createElement('div');
                indicator.className = 'event-indicator';
                dayDiv.appendChild(indicator);
            });
            
            cell.appendChild(dayDiv);
            row.appendChild(cell);
        }
        
        calendarBody.appendChild(row);
    }
}

function changeMonth(direction) {
    currentDate.setMonth(currentDate.getMonth() + direction);
    generateCalendar();
}

function showAddEventModal() {
    document.getElementById('addEventModal').style.display = 'block';
}

function closeModal() {
    document.getElementById('addEventModal').style.display = 'none';
}

function filterByCourse(courseId) {
    // Implement course filtering logic
    console.log('Filter by course:', courseId);
}

// Initialize calendar
document.addEventListener('DOMContentLoaded', function() {
    generateCalendar();
    
    // Handle event form submission
    document.getElementById('eventForm').addEventListener('submit', function(e) {
        e.preventDefault();
        // Add event submission logic here
        closeModal();
    });
    
    // Close modal when clicking outside
    window.addEventListener('click', function(e) {
        const modal = document.getElementById('addEventModal');
        if (e.target === modal) {
            closeModal();
        }
    });
});
</script>

<?php include_once 'components/footer.php'; ?>
