<?php
/**
 * Student Cart Page
 * Clean shopping cart following the established student layout pattern
 */

// Set current page for navigation
$currentPage = 'cart';

// Authentication - Require student role
require_once '../auth/functions.php';
requireRole('student');

// Initialize session cart if not exists
if (!isset($_SESSION['cart'])) {
    $_SESSION['cart'] = array();
}

// Handle localStorage cart sync to session (for logged-in users)
if (isset($_POST['sync_cart']) && !empty($_POST['cart_data'])) {
    $cartData = json_decode($_POST['cart_data'], true);
    if (is_array($cartData)) {
        // Merge localStorage cart with session cart (avoid duplicates)
        foreach ($cartData as $item) {
            if (isset($item['id'])) {
                $exists = false;
                foreach ($_SESSION['cart'] as $sessionItem) {
                    if ($sessionItem['id'] == $item['id']) {
                        $exists = true;
                        break;
                    }
                }
                if (!$exists) {
                    $_SESSION['cart'][] = $item;
                }
            }
        }
    }
}

// CLEAN UP EXISTING DUPLICATES - Remove duplicate courses from session cart
$uniqueCart = [];
$seenIds = [];
foreach ($_SESSION['cart'] as $item) {
    if (isset($item['id']) && !in_array($item['id'], $seenIds)) {
        $uniqueCart[] = $item;
        $seenIds[] = $item['id'];
    }
}
$_SESSION['cart'] = $uniqueCart;

// FETCH USER'S AVAILABLE DISCOUNTS
$available_discounts = [];
$userId = $_SESSION['user_id'] ?? 0;

if ($userId) {
    require_once '../config/db_connect.php';
    $discountsQuery = "SELECT * FROM student_referral_discounts 
                      WHERE student_id = ? AND is_used = 0 AND expires_at > NOW() 
                      ORDER BY discount_percentage DESC LIMIT 1";
    $discountStmt = $conn->prepare($discountsQuery);
    if ($discountStmt) {
        $discountStmt->bind_param("i", $userId);
        $discountStmt->execute();
        $discountResult = $discountStmt->get_result();
        if ($discountResult->num_rows > 0) {
            $available_discounts = $discountResult->fetch_assoc();
        }
    }
}

// Handle login redirects - sync localStorage first, then check cart and redirect  
if (isset($_GET['new_login']) && $_GET['new_login'] === '1') {
    // First check if we need to sync localStorage items (this will be done via JavaScript)
    // We'll delay the redirect check to allow JavaScript sync to happen
    $needsSyncCheck = true; // Flag for JavaScript to handle sync first
} else {
    $needsSyncCheck = false;
}

if (isset($_GET['check_cart_redirect']) && $_GET['check_cart_redirect'] === '1') {
    $originalRedirect = $_GET['original_redirect'] ?? 'dashboard.php';
    $sessionCartCount = count($_SESSION['cart']);
    
    if ($sessionCartCount === 0) {
        // No items in session cart, redirect to original destination immediately
        header('Location: ' . $originalRedirect);
        exit();
    } else {
        // Cart has items, redirect to checkout if that was the original intent
        if (strpos($originalRedirect, 'checkout.php') !== false) {
            header('Location: ' . $originalRedirect);
            exit();
        }
    }
    // If neither condition met, continue loading cart page normally
}

// Check if user came from guest cart (for syncing)
$cart_redirect = isset($_GET['cart_redirect']) ? true : false;
$from_guest = isset($_GET['from_guest']) ? true : false;
$sync_message = '';

if ($cart_redirect || $from_guest) {
    $sync_message = 'Welcome! Your cart items have been synced to your account.';
}

// Include header with sidebar and navbar
include_once __DIR__ . '/components/header.php';
?>

<!-- Simple Cart Header -->
<div class="simple-cart-header">
    <h1 class="cart-title">
        <i class="bx bx-shopping-bag me-3"></i>
        My Shopping Cart
    </h1>
    <p class="cart-description">Review your selected courses and complete your purchase</p>
</div>

<!-- Cart Content -->
<div class="row">
    <div class="col-lg-8">
        <?php if (!empty($sync_message)): ?>
        <!-- Cart Sync Success Message -->
        <div class="alert alert-success alert-dismissible fade show mb-4" role="alert">
            <div class="d-flex align-items-center">
                <i class="bx bx-check-circle me-3 fs-4"></i>
                <div>
                    <strong>Cart Synced Successfully!</strong><br>
                    <span class="text-muted"><?php echo htmlspecialchars($sync_message); ?></span>
                </div>
            </div>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php endif; ?>



        <!-- Cart Items Card -->
        <div class="card cart-items-card">
            <div class="card-header bg-white border-0 pb-0">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <div>
                        <h5 class="card-title mb-1">
                            <i class="bx bx-list-ul me-2"></i>
                            Cart Items
                        </h5>
                        <p class="card-subtitle text-muted mb-0">Review and manage your selected courses</p>
                    </div>
                    <span class="badge bg-primary rounded-pill" id="itemCount">0 items</span>
                </div>
            </div>
            <div class="card-body">
                <!-- Empty Cart State -->
                <div class="empty-cart-simple text-center py-5" id="emptyCart">
                    <i class="bx bx-shopping-bag empty-cart-icon mb-4"></i>
                    <h4 class="empty-title">Your learning cart awaits!</h4>
                    <p class="empty-description">
                        Start building your skills by adding courses to your cart.<br>
                        Invest in your future with quality education from Panadite Academy.
                    </p>
                    <div class="empty-actions">
                        <a href="courses.php" class="btn btn-primary btn-lg">
                            <i class="bx bx-book me-2"></i>
                            Explore Courses
                        </a>

                    </div>
                </div>

                <!-- Cart Items List -->
                <div id="cartItemsList" class="cart-items-list">
                    <!-- Cart items will be populated by JavaScript -->
                </div>

                <!-- Cart Actions -->
                <div class="cart-actions mt-4" id="cartActions" style="display: none;">
                    <div class="row">
                        <div class="col-md-6">
                            <button class="btn btn-outline-danger" onclick="clearCart()">
                                <i class="bx bx-trash me-2"></i>
                                Clear Cart
                            </button>
                        </div>
                        <div class="col-md-6 text-end">
                            <button class="btn btn-outline-primary" onclick="saveForLater()">
                                <i class="bx bx-bookmark me-2"></i>
                                Save for Later
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="col-lg-4">
        <!-- Order Summary Card -->
        <div class="card summary-card" style="position: sticky; top: 20px; z-index: 100;">
            <div class="card-header modern-header">
                <h5 class="mb-0">
                    <i class="bx bx-calculator me-2"></i>
                    Order Summary
                </h5>
                <small class="text-muted">Review your total before checkout</small>
            </div>
            <div class="card-body">
                <!-- Price Breakdown -->
                <div class="price-breakdown">
                    <div class="price-row">
                        <span class="price-label">
                            <i class="bx bx-tag me-1"></i>
                            Subtotal:
                        </span>
                        <span class="price-value" id="subtotal">R0.00</span>
                    </div>
                    <div class="price-row">
                        <span class="price-label">
                            <i class="bx bx-discount me-1"></i>
                            Discount:
                        </span>
                        <span class="price-value discount-value" id="discount">-R0.00</span>
                    </div>
                    <div class="price-row savings-row" id="savingsRow" style="display: none;">
                        <span class="price-label">
                            <i class="bx bx-trophy me-1"></i>
                            You Save:
                        </span>
                        <span class="price-value savings-value" id="savings">R0.00</span>
                    </div>
                </div>

                <hr class="summary-divider">

                <!-- Total -->
                <div class="total-section">
                    <div class="total-row">
                        <span class="total-label">Total Amount:</span>
                        <span class="total-value" id="total">R0.00</span>
                    </div>
                    <div class="payment-methods">
                        <small class="text-muted">
                            <i class="bx bx-shield-check me-1"></i>
                            Secure payment with PayFast
                        </small>
                    </div>
                </div>



                <!-- Security Badge -->
                <div class="security-badge text-center mb-3">
                    <i class="bx bx-lock-alt text-success me-1"></i>
                    <small class="text-muted">SSL Encrypted Secure Checkout</small>
                </div>

                <!-- Checkout Buttons -->
                <div class="checkout-section">
                    <button class="btn btn-primary btn-checkout w-100 mb-3" onclick="proceedToCheckout()" disabled id="checkoutBtn">
                        <i class="bx bx-credit-card me-2"></i>
                        Proceed to Secure Checkout
                        <span class="checkout-arrow">→</span>
                    </button>
                    <a href="courses.php" class="btn btn-outline-secondary w-100 continue-shopping">
                        <i class="bx bx-arrow-back me-2"></i>
                        Continue Shopping
                    </a>
                </div>


            </div>
        </div>

        <!-- Recently Viewed -->
        <div class="card mt-4">
            <div class="card-header">
                <h6 class="mb-0">
                    <i class="bx bx-history me-2"></i>
                    Recently Viewed
                </h6>
            </div>
            <div class="card-body">
                <div class="recently-viewed-list" id="recentlyViewed">
                    <p class="text-muted text-center">No recently viewed courses</p>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* Simple Cart Styling */
.simple-cart-header {
    margin-top: 2rem;
    margin-bottom: 2rem;
    padding-top: 1rem;
    padding-bottom: 1rem;
    border-bottom: 2px solid #e3e6f0;
}

.cart-title {
    font-size: 2rem;
    font-weight: 600;
    color: #495057;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
}

.cart-title i {
    color: #f59e0b;
}

.cart-description {
    font-size: 1rem;
    color: #6c757d;
    margin: 0;
    font-weight: 400;
}

/* Tips Card */
.tips-card {
    border-left: 4px solid #f59e0b;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
}

.tips-container {
    display: flex;
    flex-direction: column;
    gap: 0.75rem;
}

.tip-item {
    display: flex;
    align-items: flex-start;
    font-size: 0.9rem;
    line-height: 1.4;
}

/* Modern Card Headers */
.modern-header {
    background: linear-gradient(135deg, #f8f9fc 0%, #e9ecef 100%);
    border-bottom: 2px solid #e3e6f0;
}

.bg-gradient-primary {
    background: linear-gradient(135deg, #4e73df 0%, #3b82f6 100%) !important;
}

/* Simple Empty Cart */
.empty-cart-simple {
    padding: 3rem 2rem;
}

.empty-cart-icon {
    font-size: 4rem;
    color: #dee2e6;
    display: block;
}

.empty-title {
    font-size: 1.5rem;
    font-weight: 600;
    color: #495057;
    margin-bottom: 1rem;
}

.empty-description {
    color: #6c757d;
    font-size: 1rem;
    line-height: 1.6;
    margin-bottom: 2rem;
}

.empty-actions {
    display: flex;
    gap: 1rem;
    justify-content: center;
    flex-wrap: wrap;
}

/* Cart Items */
.cart-items-list .cart-item {
    border-bottom: 1px solid #e3e6f0;
    padding: 1.5rem 0;
    transition: all 0.3s ease;
}

.cart-items-list .cart-item:hover {
    background-color: #f8f9fc;
    border-radius: 0.5rem;
    padding: 1.5rem 1rem;
}

.cart-items-list .cart-item:last-child {
    border-bottom: none;
}

.cart-items-list img {
    width: 120px;
    height: 80px;
    object-fit: cover;
    border-radius: 0.5rem;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
}

.item-price {
    font-size: 1.5rem;
    font-weight: 700;
    color: #f59e0b;
    margin-bottom: 0.5rem;
}

.remove-btn {
    color: #dc3545;
    background: none;
    border: none;
    font-size: 0.875rem;
    padding: 0.25rem 0.5rem;
    border-radius: 0.25rem;
    transition: all 0.2s ease;
}

.remove-btn:hover {
    background-color: #dc3545;
    color: white;
}

/* Summary Card */
.summary-card {
    box-shadow: 0 8px 30px rgba(0, 0, 0, 0.12);
    border: none;
    top: 100px;
}

.price-breakdown {
    margin-bottom: 1rem;
}

.price-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 0.75rem;
    font-size: 1rem;
}

.price-label {
    color: #6c757d;
    display: flex;
    align-items: center;
}

.price-value {
    font-weight: 600;
    color: #495057;
}

.discount-value {
    color: #28a745 !important;
}

.savings-value {
    color: #f59e0b !important;
}

.summary-divider {
    border-top: 2px solid #e3e6f0;
    margin: 1.5rem 0;
}

.total-section {
    margin-bottom: 1.5rem;
}

.total-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 0.5rem;
}

.total-label {
    font-size: 1.1rem;
    font-weight: 600;
    color: #495057;
}

.total-value {
    font-size: 1.5rem;
    font-weight: 700;
    color: #f59e0b;
}

/* Promo Code */
.promo-section {
    margin-bottom: 1.5rem;
    padding: 1rem;
    background: #f8f9fc;
    border-radius: 0.5rem;
    border: 1px solid #e3e6f0;
}

.promo-input-group .form-control:focus {
    border-color: #28a745;
    box-shadow: 0 0 0 0.2rem rgba(40, 167, 69, 0.25);
}

.promo-suggestions {
    background: white;
    padding: 0.5rem;
    border-radius: 0.25rem;
    border: 1px solid #dee2e6;
}

/* Security and Guarantee Badges */
.security-badge, .guarantee-section {
    padding: 0.75rem;
    background: #e8f5e8;
    border-radius: 0.5rem;
    border: 1px solid #c3e6cb;
}

.guarantee-badge {
    color: #856404;
    margin-bottom: 0.25rem;
}

/* Checkout Button */
.btn-checkout {
    font-size: 1.1rem;
    font-weight: 600;
    padding: 0.875rem 1.5rem;
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    border: none;
    position: relative;
    overflow: hidden;
    transition: all 0.3s ease;
}

.btn-checkout:hover {
    background: linear-gradient(135deg, #218838 0%, #1e7e34 100%);
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(40, 167, 69, 0.3);
}

.btn-checkout:disabled {
    background: #6c757d;
    cursor: not-allowed;
    transform: none;
    box-shadow: none;
}

.checkout-arrow {
    display: inline-block;
    margin-left: 0.5rem;
    transition: transform 0.2s ease;
}

.btn-checkout:hover .checkout-arrow {
    transform: translateX(3px);
}

.continue-shopping {
    transition: all 0.2s ease;
}

.continue-shopping:hover {
    background-color: #f8f9fc;
    border-color: #4e73df;
    color: #4e73df;
}

/* Discounted Price Styling */
.item-price-container {
    text-align: right;
    min-width: 100px;
}

.original-price {
    font-size: 12px;
    color: #6b7280;
    text-decoration: line-through;
    font-weight: 500;
    margin-bottom: 2px;
}

.discounted-price {
    font-size: 16px;
    color: #dc2626;
    font-weight: 700;
    margin-bottom: 2px;
}

.savings-badge {
    font-size: 10px;
    background: linear-gradient(135deg, #dc2626, #b91c1c);
    color: white;
    padding: 2px 6px;
    border-radius: 4px;
    font-weight: 600;
    display: inline-block;
    animation: savings-pulse 2s ease-in-out infinite;
}

@keyframes savings-pulse {
    0%, 100% { opacity: 1; transform: scale(1); }
    50% { opacity: 0.8; transform: scale(0.95); }
}

.discount-row.discount-active {
    color: #dc2626;
    font-weight: 600;
}

.discount-row.discount-active #discount {
    color: #dc2626;
    font-weight: 700;
}

/* Cart Actions */
.cart-actions {
    background: #f8f9fc;
    padding: 1.5rem;
    border-radius: 0.5rem;
    border: 1px solid #e3e6f0;
}

/* Recently Viewed */
.recently-viewed-list img {
    width: 60px;
    height: 40px;
    object-fit: cover;
    border-radius: 0.25rem;
}

/* Responsive Design */
@media (max-width: 768px) {
    .cart-hero-section {
        padding: 1.5rem 0;
    }
    
    .cart-header-content h1.cart-main-title {
        font-size: 2rem;
    }
    
    .cart-stats {
        flex-direction: column;
        gap: 1rem;
        margin-top: 1rem;
    }
    
    .stat-item {
        display: inline-block;
        margin: 0 0.5rem;
    }
    
    .tips-container {
        font-size: 0.85rem;
    }
    
    .empty-actions {
        flex-direction: column;
        align-items: center;
    }
    
    .cart-items-list img {
        width: 80px;
        height: 55px;
    }
    
    .summary-card {
        position: static !important;
        margin-top: 1rem;
    }
}

@media (max-width: 576px) {
    .cart-hero-section {
        padding: 1rem 0;
    }
    
    .cart-header-content h1.cart-main-title {
        font-size: 1.75rem;
    }
    
    .empty-cart-enhanced {
        padding: 2rem 1rem;
    }
    
    .floating-elements {
        width: 150px;
        height: 150px;
    }
    
    .floating-element {
        font-size: 1.2rem;
    }
}

/* Loading states */
.loading {
    opacity: 0.6;
    pointer-events: none;
}
</style>

<script>
// Cart functionality
let cartItems = [];

// DISCOUNT DATA FROM PHP
const userDiscounts = <?= json_encode($available_discounts) ?>;
console.log('🎯 User discounts loaded:', userDiscounts);

// Calculate discount for a price
function calculateDiscount(originalPrice) {
    if (!userDiscounts || !userDiscounts.discount_percentage || originalPrice <= 0) {
        return { discounted_price: originalPrice, savings: 0, has_discount: false };
    }
    
    const discount_amount = Math.min(
        (originalPrice * userDiscounts.discount_percentage / 100),
        userDiscounts.discount_amount || originalPrice
    );
    const discounted_price = originalPrice - discount_amount;
    
    return {
        discounted_price: Math.max(0, discounted_price),
        savings: discount_amount,
        has_discount: true,
        discount_percentage: userDiscounts.discount_percentage
    };
}

// Initialize cart on page load
document.addEventListener('DOMContentLoaded', function() {
    // Clean up localStorage duplicates first
    cleanupCartDuplicates();
    
    // Check if we need to sync guest cart
    syncGuestCartOnLogin();
    loadCartItems();
    updateCartDisplay();
});

// Clean up duplicate courses in localStorage cart
function cleanupCartDuplicates() {
    const cart = JSON.parse(localStorage.getItem('cart') || '[]');
    const uniqueCart = [];
    const seenIds = [];
    
    cart.forEach(item => {
        if (item.id && !seenIds.includes(item.id)) {
            uniqueCart.push(item);
            seenIds.push(item.id);
        }
    });
    
    if (uniqueCart.length !== cart.length) {
        localStorage.setItem('cart', JSON.stringify(uniqueCart));
        console.log('🧹 Cleaned up cart duplicates. Before:', cart.length, 'After:', uniqueCart.length);
    }
}

// Sync guest cart when user logs in
function syncGuestCartOnLogin() {
    // DEBUG: Check all localStorage keys to understand what's stored
    console.log('🔍 DEBUG: All localStorage keys:', Object.keys(localStorage));
    console.log('🔍 DEBUG: panadite_guest_cart:', localStorage.getItem('panadite_guest_cart'));
    console.log('🔍 DEBUG: guest_cart:', localStorage.getItem('guest_cart'));
    console.log('🔍 DEBUG: cart:', localStorage.getItem('cart'));
    
    // Check multiple possible guest cart keys
    let guestCart = localStorage.getItem('panadite_guest_cart') || localStorage.getItem('guest_cart');
    let guestItems = [];
    
    if (guestCart) {
        try {
            guestItems = JSON.parse(guestCart);
            console.log('✅ Found guest cart with', guestItems.length, 'items:', guestItems);
        } catch (error) {
            console.error('❌ Error parsing guest cart:', error);
            guestItems = [];
        }
    }
    
    // Check existing authenticated cart
    let existingCart = [];
    const authCart = localStorage.getItem('cart');
    if (authCart) {
        try {
            existingCart = JSON.parse(authCart);
            console.log('📦 Found existing authenticated cart with', existingCart.length, 'items:', existingCart);
        } catch (error) {
            console.error('❌ Error parsing existing cart:', error);
            existingCart = [];
        }
    }
    
    // PRESERVE CART LOGIC: Don't clear if user already has items
    let finalCart = [];
    
    if (guestItems && guestItems.length > 0) {
        // Convert guest items to authenticated format
        guestItems.forEach(guestItem => {
            finalCart.push({
                id: guestItem.id,
                title: guestItem.title || 'Course',
                instructor: guestItem.instructor || 'Panadite Academy',
                price: guestItem.price || 0,
                rating: guestItem.rating || '4.5',
                reviews: guestItem.reviews || '0',
                thumbnail: guestItem.thumbnail || '/uploads/thumbnails/default-course.jpg',
                level: guestItem.level || 'Beginner',
                description: guestItem.description || ''
            });
        });
        
        // Clear guest cart after successful sync
        localStorage.removeItem('panadite_guest_cart');
        localStorage.removeItem('guest_cart');
        
        console.log('✅ Guest cart synced successfully:', finalCart.length, 'items');
    } else if (existingCart && existingCart.length > 0) {
        // Keep existing cart if no guest cart to sync
        finalCart = existingCart;
        console.log('📦 Keeping existing authenticated cart:', finalCart.length, 'items');
    } else {
        // Both carts empty
        console.log('📭 No cart items found (guest or authenticated)');
    }
    
    // Save final cart
    localStorage.setItem('cart', JSON.stringify(finalCart));
    console.log('💾 Final cart saved:', finalCart.length, 'items');
    
    // UPDATE CART COUNTER - try multiple methods
    setTimeout(() => {
        if (typeof updateCartCount === 'function') {
            updateCartCount();
            console.log('🔄 Cart counter updated via updateCartCount()');
        }
        
        // Also try direct update
        const cartCountElement = document.getElementById('cartCount');
        if (cartCountElement) {
            cartCountElement.textContent = finalCart.length;
            cartCountElement.style.display = finalCart.length > 0 ? 'flex' : 'none';
            console.log('🔄 Cart counter updated directly:', finalCart.length);
        }
    }, 100);
    
    // AUTO-REDIRECT TO CHECKOUT for new users with cart items
    const urlParams = new URLSearchParams(window.location.search);
    const isNewLogin = urlParams.get('new_login') === '1' || sessionStorage.getItem('just_logged_in') === 'true';
    
    if (isNewLogin && finalCart.length > 0) {
        console.log('🚀 Auto-redirecting new user to checkout with', finalCart.length, 'items');
        sessionStorage.removeItem('just_logged_in'); // Clear flag
        setTimeout(() => {
            window.location.href = 'checkout.php?from_registration=1';
        }, 2000); // Give more time to see the sync
    }
}

// Load cart items from localStorage
function loadCartItems() {
    const storedCart = localStorage.getItem('cart');
    cartItems = storedCart ? JSON.parse(storedCart) : [];
}

// Save cart items to localStorage
function saveCartItems() {
    localStorage.setItem('cart', JSON.stringify(cartItems));
    // Update navbar cart count
    if (typeof updateCartCount === 'function') {
        updateCartCount();
    }
}

// Update cart display
function updateCartDisplay() {
    const cartContainer = document.getElementById('cartItems');
    const emptyCart = document.getElementById('emptyCart');
    const itemCount = document.getElementById('itemCount');
    const checkoutBtn = document.getElementById('checkoutBtn');
    const cartActions = document.getElementById('cartActions');
    const totalItems = document.getElementById('totalItems');
    const totalValue = document.getElementById('totalValue');
    
    if (cartItems.length === 0) {
        emptyCart.style.display = 'block';
        if (cartContainer) cartContainer.style.display = 'none';
        if (cartActions) cartActions.style.display = 'none';
        itemCount.textContent = '0 items';
        if (totalItems) totalItems.textContent = '0';
        if (totalValue) totalValue.textContent = 'R0';
        if (checkoutBtn) checkoutBtn.disabled = true;
    } else {
        emptyCart.style.display = 'none';
        if (cartContainer) cartContainer.style.display = 'block';
        if (cartActions) cartActions.style.display = 'block';
        itemCount.textContent = `${cartItems.length} item${cartItems.length > 1 ? 's' : ''}`;
        if (totalItems) totalItems.textContent = cartItems.length;
        if (checkoutBtn) checkoutBtn.disabled = false;
        
        // Render cart items
        renderCartItems();
    }
    
    updateSummary();
}

// Render cart items
function renderCartItems() {
    const cartItemsList = document.getElementById('cartItemsList');
    cartItemsList.innerHTML = '';
    
    cartItems.forEach((item, index) => {
        const originalPrice = parseFloat(item.price || 0);
        const discountInfo = calculateDiscount(originalPrice);
        
        const cartItemDiv = document.createElement('div');
        cartItemDiv.className = 'cart-item';
        
        // Build price display
        let priceDisplay = '';
        if (discountInfo.has_discount) {
            priceDisplay = `
                <div class="item-price-container">
                    <div class="original-price">R${originalPrice.toFixed(0)}</div>
                    <div class="discounted-price">R${discountInfo.discounted_price.toFixed(0)}</div>
                    <div class="savings-badge">Save R${discountInfo.savings.toFixed(0)}!</div>
                </div>
            `;
        } else {
            priceDisplay = `<div class="item-price">R${originalPrice.toFixed(0)}</div>`;
        }
        
        cartItemDiv.innerHTML = `
            <div class="row align-items-center">
                <div class="col-auto">
                    <img src="${item.thumbnail || '/uploads/thumbnails/default-course.jpg'}" 
                         alt="${item.title}" class="img-fluid">
                </div>
                <div class="col">
                    <h6 class="mb-1">${item.title}</h6>
                    <small class="text-muted">by ${item.instructor || 'Panadite Academy'}</small>

                </div>
                <div class="col-auto">
                    ${priceDisplay}
                    <button class="remove-btn mt-1" onclick="removeFromCart(${index})">
                        <i class="bx bx-trash me-1"></i>Remove
                    </button>
                </div>
            </div>
        `;
        cartItemsList.appendChild(cartItemDiv);
    });
}

// Remove item from cart
function removeFromCart(index) {
    cartItems.splice(index, 1);
    saveCartItems();
    updateCartDisplay();
    showAlert('Item removed from cart', 'success');
}

// Update summary
function updateSummary() {
    let subtotal = 0;
    let totalDiscount = 0;
    let totalSavings = 0;
    
    // Calculate totals with discounts applied
    cartItems.forEach(item => {
        const originalPrice = parseFloat(item.price || 0);
        const discountInfo = calculateDiscount(originalPrice);
        
        subtotal += originalPrice;
        if (discountInfo.has_discount) {
            totalDiscount += discountInfo.savings;
            totalSavings += discountInfo.savings;
        }
    });
    
    const finalTotal = subtotal - totalDiscount;
    
    document.getElementById('subtotal').textContent = `R${subtotal.toFixed(0)}`;
    document.getElementById('discount').textContent = `-R${totalDiscount.toFixed(0)}`;
    document.getElementById('total').textContent = `R${finalTotal.toFixed(0)}`;
    
    // Update header stats with final discounted total
    const totalValue = document.getElementById('totalValue');
    if (totalValue) {
        totalValue.textContent = `R${Math.round(finalTotal)}`;
    }
    
    // Show/hide savings row
    const savingsRow = document.getElementById('savingsRow');
    const savings = document.getElementById('savings');
    if (totalSavings > 0) {
        savingsRow.style.display = 'flex';
        savings.textContent = `R${totalSavings.toFixed(0)}`;
        
        // Make discount row visible and highlighted
        const discountRow = document.querySelector('.discount-row');
        if (discountRow) {
            discountRow.style.display = 'flex';
            discountRow.classList.add('discount-active');
        }
    } else {
        savingsRow.style.display = 'none';
    }
}

// Apply promo code
function applyPromoCode() {
    const promoCode = document.getElementById('promoCode').value.trim();
    
    if (!promoCode) {
        showAlert('Please enter a promo code', 'warning');
        return;
    }
    
    // Simulate promo code validation
    const validCodes = {
        'WELCOME10': 0.10,
        'STUDENT20': 0.20,
        'PANADITE15': 0.15
    };
    
    if (validCodes[promoCode.toUpperCase()]) {
        const discount = validCodes[promoCode.toUpperCase()];
        showAlert(`🎉 Promo code applied! You saved ${(discount * 100)}% on your order!`, 'success');
        
        // Store applied promo code
        localStorage.setItem('appliedPromoCode', promoCode.toUpperCase());
        localStorage.setItem('promoDiscount', discount);
        
        // Update display
        updateSummary();
        
        // Disable promo input
        document.getElementById('promoCode').disabled = true;
        document.querySelector('.promo-input-group button').textContent = 'Applied!';
        document.querySelector('.promo-input-group button').disabled = true;
        document.querySelector('.promo-input-group button').classList.remove('btn-outline-success');
        document.querySelector('.promo-input-group button').classList.add('btn-success');
    } else {
        showAlert('❌ Invalid promo code. Try: WELCOME10, STUDENT20, or PANADITE15', 'danger');
    }
}

// Clear entire cart
function clearCart() {
    if (cartItems.length === 0) {
        showAlert('Your cart is already empty', 'info');
        return;
    }
    
    if (confirm('Are you sure you want to clear all items from your cart?')) {
        cartItems = [];
        saveCartItems();
        updateCartDisplay();
        showAlert('🗑️ Cart cleared successfully', 'success');
        
        // Reset promo code
        resetPromoCode();
    }
}

// Save cart for later
function saveForLater() {
    if (cartItems.length === 0) {
        showAlert('Your cart is empty', 'warning');
        return;
    }
    
    // Save to wishlist in localStorage
    const wishlist = JSON.parse(localStorage.getItem('wishlist') || '[]');
    cartItems.forEach(item => {
        if (!wishlist.find(w => w.id === item.id)) {
            wishlist.push(item);
        }
    });
    localStorage.setItem('wishlist', JSON.stringify(wishlist));
    
    // Clear cart
    cartItems = [];
    saveCartItems();
    updateCartDisplay();
    
    showAlert('💖 Items saved to your wishlist and cart cleared', 'success');
    resetPromoCode();
}

// Reset promo code
function resetPromoCode() {
    localStorage.removeItem('appliedPromoCode');
    localStorage.removeItem('promoDiscount');
    
    document.getElementById('promoCode').disabled = false;
    document.getElementById('promoCode').value = '';
    document.querySelector('.promo-input-group button').textContent = 'Apply';
    document.querySelector('.promo-input-group button').disabled = false;
    document.querySelector('.promo-input-group button').classList.remove('btn-success');
    document.querySelector('.promo-input-group button').classList.add('btn-outline-success');
}

// Proceed to checkout
function proceedToCheckout() {
    if (cartItems.length === 0) {
        showAlert('Your cart is empty', 'warning');
        return;
    }
    
    // Redirect to checkout page or payment gateway
    window.location.href = 'checkout.php';
}

// Show alert
function showAlert(message, type = 'info') {
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert alert-${type} alert-dismissible fade show`;
    alertDiv.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    const container = document.querySelector('.page-header');
    container.parentNode.insertBefore(alertDiv, container.nextSibling);
    
    // Auto remove after 3 seconds
    setTimeout(() => {
        if (alertDiv.parentNode) {
            alertDiv.remove();
        }
    }, 3000);
}

// Add demo items for testing (uncomment to test)
function addDemoItems() {
    const demoItems = [
        {
            id: 1,
            title: 'Complete Web Development Bootcamp',
            instructor: 'John Doe',
            price: 499.99,
            rating: 4.8,
            reviews: 1250,
            thumbnail: '/uploads/thumbnails/web-dev.jpg'
        },
        {
            id: 2,
            title: 'Data Science with Python',
            instructor: 'Jane Smith',
            price: 399.99,
            rating: 4.7,
            reviews: 890,
            thumbnail: '/uploads/thumbnails/data-science.jpg'
        }
    ];
    
    cartItems = demoItems;
    saveCartItems();
    updateCartDisplay();
}

// Uncomment next line to add demo items for testing
// addDemoItems();

// Handle localStorage sync and redirect after login
document.addEventListener('DOMContentLoaded', function() {
    const urlParams = new URLSearchParams(window.location.search);
    const needsSyncCheck = <?php echo $needsSyncCheck ? 'true' : 'false'; ?>;
    
    if (needsSyncCheck && urlParams.get('new_login') === '1') {
        console.log('New login detected, checking for localStorage items to sync');
        
        // Get localStorage cart items
        const localCartItems = JSON.parse(localStorage.getItem('cartItems') || '[]');
        
        if (localCartItems.length > 0) {
            console.log('Found ' + localCartItems.length + ' items in localStorage, syncing to session...');
            
            // Sync localStorage to session via AJAX (faster than form submit)
            const xhr = new XMLHttpRequest();
            xhr.open('POST', '', true);
            xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
            
            xhr.onreadystatechange = function() {
                if (xhr.readyState === 4 && xhr.status === 200) {
                    console.log('Cart sync completed, clearing localStorage');
                    // Clear localStorage after successful sync
                    localStorage.removeItem('cartItems');
                    localStorage.removeItem('appliedPromoCode');
                    localStorage.removeItem('promoDiscount');
                    
                    // Reload page to show synced items and proper redirect logic
                    window.location.reload();
                }
            };
            
            // Send cart data to sync
            const params = 'sync_cart=1&cart_data=' + encodeURIComponent(JSON.stringify(localCartItems));
            xhr.send(params);
            
        } else {
            console.log('No localStorage items, checking session cart for redirect...');
            
            // No localStorage items, check session cart count
            const sessionCartCount = <?php echo count($_SESSION['cart']); ?>;
            
            if (sessionCartCount === 0) {
                console.log('No session items either, redirecting to dashboard');
                window.location.href = 'dashboard.php';
            } else {
                console.log('Session cart has items, staying on cart page');
                // Clear localStorage anyway
                localStorage.removeItem('cartItems');
                localStorage.removeItem('appliedPromoCode');
                localStorage.removeItem('promoDiscount');
            }
        }
    }
    
    // Clear localStorage on other login parameters  
    if (urlParams.get('check_cart_redirect') === '1') {
        console.log('Check cart redirect, clearing localStorage');
        localStorage.removeItem('cartItems');
        localStorage.removeItem('appliedPromoCode');
        localStorage.removeItem('promoDiscount');
    }
});
</script>

<?php include_once __DIR__ . '/components/footer.php'; ?>
