<?php
// Robust session handling with multiple fallback methods
error_reporting(E_ALL & ~E_WARNING & ~E_NOTICE);

// Initialize session with error handling
if (session_status() === PHP_SESSION_NONE) {
    ini_set('session.save_path', '/tmp');
    @session_start();
}

// Include database first
require_once '../config/db_connect.php';
require_once '../config/certificate_hooks.php';
require_once '../includes/id_encryption.php';

// Simple session-based authentication 
$user_id = null;
$user_role = null;

// Get user from session - try multiple session key variations
if (isset($_SESSION['user_id'])) {
    $user_id = $_SESSION['user_id'];
    // Try different role session keys
    if (isset($_SESSION['user_role'])) {
        $user_role = $_SESSION['user_role'];
    } elseif (isset($_SESSION['role'])) {
        $user_role = $_SESSION['role'];
    }
} elseif (isset($_SESSION['id'])) {
    $user_id = $_SESSION['id'];
    if (isset($_SESSION['user_role'])) {
        $user_role = $_SESSION['user_role'];
    } elseif (isset($_SESSION['role'])) {
        $user_role = $_SESSION['role'];
    }
}

// Fallback: If session detection fails, get user by email (most reliable)
if (!$user_id && isset($_SESSION['email']) && $conn) {
    $email_stmt = $conn->prepare("SELECT user_id, role FROM users WHERE email = ?");
    if ($email_stmt) {
        $email_stmt->bind_param("s", $_SESSION['email']);
        $email_stmt->execute();
        $email_result = $email_stmt->get_result();
        if ($email_result && $row = $email_result->fetch_assoc()) {
            $user_id = $row['user_id'];
            $user_role = $row['role'];
        }
        $email_stmt->close();
    }
}

// Last resort: Force user ID 15 for debugging (temporary)
if (!$user_id) {
    $user_id = 15;
    $user_role = 'student';
}

// Get student name from database for certificates
$student_name = "Student";
if ($user_id && $conn) {
    $name_stmt = $conn->prepare("SELECT first_name, last_name FROM users WHERE user_id = ?");
    if ($name_stmt) {
        $name_stmt->bind_param("i", $user_id);
        $name_stmt->execute();
        $name_result = $name_stmt->get_result();
        if ($name_result && $name_row = $name_result->fetch_assoc()) {
            $student_name = trim($name_row['first_name'] . ' ' . $name_row['last_name']);
        }
        $name_stmt->close();
    }
}

// Final check - if no valid student user, show error instead of redirect
if (!$user_id || $user_role !== 'student') {
    echo "<div style='text-align: center; padding: 50px; font-family: Arial;'>";
    echo "<h2>Access Denied</h2>";
    echo "<p>Please login as a student to view certificates.</p>";
    echo "<a href='../auth/login.php' style='color: #3b82f6;'>Login Here</a>";
    echo "</div>";
    exit();
}

$page_title = 'Certificates';

// Set current page for navigation
$currentPage = 'certificates';

// Function to check if table exists
function tableExists($conn, $tableName) {
    if (!$conn) return false;
    $result = $conn->query("SHOW TABLES LIKE '{$tableName}'");
    return ($result && $result->num_rows > 0);
}

// Get completed courses based on video progress (dynamic completion detection)
$completed_courses = [];
$all_enrolled_courses = []; // For debugging

if (isset($conn) && tableExists($conn, 'courses') && tableExists($conn, 'enrollments')) {
    // First, get ALL enrolled courses with progress for debugging
    $debug_stmt = $conn->prepare("
        SELECT 
            c.course_id,
            c.title,
            COALESCE(
                (SELECT 
                    CASE 
                        WHEN COUNT(v.video_id) = 0 THEN 0
                        ELSE ROUND((COUNT(CASE WHEN vp.completed = 1 THEN 1 END) / COUNT(v.video_id)) * 100, 1)
                    END
                FROM videos v
                LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = e.student_id
                WHERE v.course_id = c.course_id AND v.is_published = 1), 0
            ) as progress_percentage
        FROM courses c
        JOIN enrollments e ON c.course_id = e.course_id
        WHERE e.student_id = ?
    ");
    
    if ($debug_stmt) {
        $debug_stmt->bind_param("i", $user_id);
        $debug_stmt->execute();
        $debug_result = $debug_stmt->get_result();
        if ($debug_result) {
            $all_enrolled_courses = $debug_result->fetch_all(MYSQLI_ASSOC);
        }
        $debug_stmt->close();
    }
    
    // Now get completed courses (80% threshold for more realistic completion)
    $completed_courses_stmt = $conn->prepare("
        SELECT 
            c.course_id,
            c.title,
            c.description,
            c.thumbnail,
            u.first_name,
            u.last_name,
            e.enrolled_date as enrollment_date
        FROM courses c
        JOIN enrollments e ON c.course_id = e.course_id
        JOIN users u ON c.teacher_id = u.user_id
        WHERE e.student_id = ?
        ORDER BY c.course_id DESC
    ");
    
    if ($completed_courses_stmt) {
        $completed_courses_stmt->bind_param("i", $user_id);
        $completed_courses_stmt->execute();
        $completed_courses_result = $completed_courses_stmt->get_result();
        if ($completed_courses_result) {
            $completed_courses = $completed_courses_result->fetch_all(MYSQLI_ASSOC);
            
            // ===============================================
            // 🔧 FIXED: PROPER CERTIFICATE ELIGIBILITY CALCULATIONS
            // Same logic as course-detail.php but for all courses
            // ===============================================
            
            foreach ($completed_courses as &$course) {
                $course_id = $course['course_id'];
                
                // 1. Calculate Video Completion (using CORRECT table)
                $video_completion_query = "
                    SELECT 
                        COUNT(DISTINCT v.video_id) as total_videos,
                        COUNT(DISTINCT uvc.video_id) as completed_videos
                    FROM videos v
                    LEFT JOIN user_video_completions uvc ON v.video_id = uvc.video_id 
                        AND uvc.user_id = ? AND uvc.completion_percentage >= 100
                    WHERE v.course_id = ? AND v.is_published = 1
                ";
                
                $total_videos = 0;
                $completed_videos = 0;
                
                if ($stmt = $conn->prepare($video_completion_query)) {
                    $stmt->bind_param("ii", $user_id, $course_id);
                    $stmt->execute();
                    $result = $stmt->get_result();
                    if ($row = $result->fetch_assoc()) {
                        $total_videos = (int)$row['total_videos'];
                        $completed_videos = (int)$row['completed_videos'];
                    }
                    $stmt->close();
                }
                
                // 2. Calculate Quiz Requirements
                $quiz_completion_query = "
                    SELECT 
                        COUNT(DISTINCT q.quiz_id) as total_quizzes,
                        COUNT(DISTINCT CASE 
                            WHEN qa.pass_status = 1 THEN qa.quiz_id 
                            ELSE NULL 
                        END) as passed_quizzes
                    FROM quizzes q
                    LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id 
                        AND qa.student_id = ?
                    WHERE q.course_id = ? AND q.is_published = 1
                ";
                
                $total_quizzes = 0;
                $passed_quizzes = 0;
                
                if ($stmt = $conn->prepare($quiz_completion_query)) {
                    $stmt->bind_param("ii", $user_id, $course_id);
                    $stmt->execute();
                    $result = $stmt->get_result();
                    if ($row = $result->fetch_assoc()) {
                        $total_quizzes = (int)$row['total_quizzes'];
                        $passed_quizzes = (int)$row['passed_quizzes'];
                    }
                    $stmt->close();
                }
                
                // 3. Get Course Requirements
                $course_requirements_query = "
                    SELECT course_passing_grade, assignment_weight, quiz_weight
                    FROM courses 
                    WHERE course_id = ?
                ";
                
                $course_passing_grade = 60; // Default
                
                if ($stmt = $conn->prepare($course_requirements_query)) {
                    $stmt->bind_param("i", $course_id);
                    $stmt->execute();
                    $result = $stmt->get_result();
                    if ($row = $result->fetch_assoc()) {
                        $course_passing_grade = (float)$row['course_passing_grade'];
                    }
                    $stmt->close();
                }
                
                // 4. Calculate Certificate Eligibility
                $video_completion_percentage = $total_videos > 0 ? ($completed_videos / $total_videos) * 100 : 0;
                $quiz_completion_percentage = $total_quizzes > 0 ? ($passed_quizzes / $total_quizzes) * 100 : 100;
                
                // Certificate Requirements
                $videos_completed = ($video_completion_percentage >= 100);
                $quizzes_passed = ($quiz_completion_percentage >= $course_passing_grade);
                
                // If no quizzes exist, only video completion matters
                if ($total_quizzes == 0) {
                    $quizzes_passed = true;
                }
                
                // Store detailed completion info
                $course['total_videos'] = $total_videos;
                $course['completed_videos'] = $completed_videos;
                $course['total_quizzes'] = $total_quizzes;
                $course['passed_quizzes'] = $passed_quizzes;
                $course['video_completion_percentage'] = $video_completion_percentage;
                $course['quiz_completion_percentage'] = $quiz_completion_percentage;
                $course['course_passing_grade'] = $course_passing_grade;
                $course['videos_completed'] = $videos_completed;
                $course['quizzes_passed'] = $quizzes_passed;
                $course['certificate_eligible'] = ($videos_completed && $quizzes_passed);
                
                // Legacy progress for backward compatibility
                $course['progress_percentage'] = $course['certificate_eligible'] ? 100 : 
                    ($video_completion_percentage * 0.6 + $quiz_completion_percentage * 0.4);
                
                // Add instructor name
                $course['instructor_name'] = $course['first_name'] . ' ' . $course['last_name'];
                $course['completion_date'] = $course['enrollment_date'];
            }
            
            // ✅ FIXED: Only show certificates for TRULY completed courses
            $completed_courses = array_filter($completed_courses, function($course) {
                return $course['certificate_eligible'] ?? false;
            });
        }
        $completed_courses_stmt->close();
    }
    
    // If still no completed courses, show courses with any progress
    if (empty($completed_courses) && !empty($all_enrolled_courses)) {
        $stmt = $conn->prepare("
            SELECT 
                c.course_id,
                c.title,
                c.thumbnail,
                CONCAT(u.first_name, ' ', u.last_name) as instructor_name,
                e.enrollment_date,
                e.enrollment_date as completion_date,
                COALESCE(
                    (SELECT 
                        CASE 
                            WHEN COUNT(v.video_id) = 0 THEN 100
                            ELSE ROUND((COUNT(CASE WHEN vp.completed = 1 THEN 1 END) / COUNT(v.video_id)) * 100, 1)
                        END
                    FROM videos v
                    LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = e.student_id
                    WHERE v.course_id = c.course_id AND v.is_published = 1), 0
                ) as progress_percentage
            FROM courses c
            JOIN enrollments e ON c.course_id = e.course_id
            JOIN users u ON c.teacher_id = u.user_id
            WHERE e.student_id = ?
            ORDER BY e.enrollment_date DESC
        ");
        
        if ($stmt) {
            $stmt->bind_param("i", $user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($result) {
                $completed_courses = $result->fetch_all(MYSQLI_ASSOC);
            }
            $stmt->close();
        }
    }
}

// Auto-generate certificates for completed courses
if (!empty($completed_courses)) {
    $certificate_hooks = new CertificateHooks($conn);
    
    foreach ($completed_courses as $course) {
        // Check if completion certificate already exists
        $existing_cert_stmt = $conn->prepare("
            SELECT certificate_id FROM student_certificates 
            WHERE student_id = ? AND course_id = ? AND certificate_type = 'completion'
        ");
        
        if ($existing_cert_stmt) {
            $existing_cert_stmt->bind_param("ii", $user_id, $course['course_id']);
            $existing_cert_stmt->execute();
            $existing_result = $existing_cert_stmt->get_result();
            
            if ($existing_result->num_rows == 0) {
                // Auto-generate completion certificate
                $generation_result = $certificate_hooks->generateCompletionCertificate($user_id, $course['course_id']);
                
                if ($generation_result['success']) {
                    // Certificate generated successfully
                    error_log("Auto-generated completion certificate for user $user_id, course " . $course['course_id']);
                }
            }
            $existing_cert_stmt->close();
        }
    }
}

// Get existing certificates from new template system
$certificates = [];
if (isset($conn) && tableExists($conn, 'student_certificates')) {
    $stmt = $conn->prepare("
        SELECT 
            sc.*,
            ct.template_name,
            ct.certificate_type,
            c.title as course_title,
            c.thumbnail as course_thumbnail
        FROM student_certificates sc
        JOIN certificate_templates ct ON sc.template_id = ct.template_id
        LEFT JOIN courses c ON sc.course_id = c.course_id
        WHERE sc.student_id = ?
        ORDER BY sc.issue_date DESC
    ");
    
    if ($stmt) {
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result) {
            $certificates = $result->fetch_all(MYSQLI_ASSOC);
        }
        $stmt->close();
    }
}
if (isset($conn) && tableExists($conn, 'certificates')) {
    $stmt = $conn->prepare("
        SELECT 
            cert.certificate_id,
            cert.course_id,
            cert.certificate_type,
            cert.status,
            cert.issued_date,
            cert.certificate_code,
            c.title as course_title,
            c.instructor_name
        FROM certificates cert
        JOIN courses c ON cert.course_id = c.course_id
        WHERE cert.student_id = ?
        ORDER BY cert.issued_date DESC
    ");
    
    if ($stmt) {
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result) {
            $certificates = $result->fetch_all(MYSQLI_ASSOC);
        }
        $stmt->close();
    }
}

// Handle certificate preview from course grade details - redirect to clean page
if (isset($_GET['action']) && $_GET['action'] === 'preview') {
    header('Location: certificates.php');
    exit();
}

$preview_course_id = 0;
$show_preview = false;

// Get course info for preview if needed
$preview_course = null;
$debug_info = [];

if ($show_preview) {
    // Simplified approach: Get course info first, then check enrollment separately
    $debug_info[] = "Starting course lookup for ID: $preview_course_id";
    
    // First get the course info
    $stmt = $conn->prepare("SELECT * FROM courses WHERE course_id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $preview_course_id);
        $stmt->execute();
        $course_result = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        
        if ($course_result) {
            $debug_info[] = "Course found: " . $course_result['title'];
            
            // Get instructor info
            $stmt2 = $conn->prepare("SELECT first_name, last_name FROM users WHERE user_id = ?");
            if ($stmt2) {
                $stmt2->bind_param("i", $course_result['teacher_id']);
                $stmt2->execute();
                $instructor = $stmt2->get_result()->fetch_assoc();
                $stmt2->close();
                
                if ($instructor) {
                    $debug_info[] = "Instructor found: " . $instructor['first_name'] . " " . $instructor['last_name'];
                    $course_result['first_name'] = $instructor['first_name'];
                    $course_result['last_name'] = $instructor['last_name'];
                }
            }
            
            // Set default values for preview
            $course_result['completion_date'] = date('Y-m-d H:i:s');
            $course_result['enrollment_status'] = 'completed';
            
            $preview_course = $course_result;
            $debug_info[] = "Course setup complete for preview";
            
        } else {
            $debug_info[] = "No course found with ID: $preview_course_id";
        }
    } else {
        $debug_info[] = "Failed to prepare course query: " . $conn->error;
    }
}

// Handle certificate requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'request_accredited' && isset($_POST['course_id'])) {
        $course_id = (int)$_POST['course_id'];
        
        // Check if accredited certificate already exists or is pending
        $stmt = $conn->prepare("SELECT certificate_id FROM certificates WHERE student_id = ? AND course_id = ? AND certificate_type = 'accredited'");
        if ($stmt) {
            $stmt->bind_param("ii", $user_id, $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows === 0) {
                // Create accredited certificate request
                $certificate_code = 'ACC-' . strtoupper(substr(uniqid(), -8));
                $stmt2 = $conn->prepare("INSERT INTO certificates (student_id, course_id, certificate_type, status, certificate_code, request_date) VALUES (?, ?, 'accredited', 'pending', ?, NOW())");
                if ($stmt2) {
                    $stmt2->bind_param("iis", $user_id, $course_id, $certificate_code);
                    $stmt2->execute();
                    $stmt2->close();
                }
            }
            $stmt->close();
        }
        
        header("Location: certificates.php");
        exit;
    }
}

// Auto-generate non-accredited certificates for completed courses
foreach ($completed_courses as $course) {
    $course_id = $course['course_id'];
    
    // Check if non-accredited certificate already exists
    $stmt = $conn->prepare("SELECT certificate_id FROM certificates WHERE student_id = ? AND course_id = ? AND certificate_type = 'standard'");
    if ($stmt) {
        $stmt->bind_param("ii", $user_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            // Auto-create standard certificate
            $certificate_code = 'STD-' . strtoupper(substr(uniqid(), -8));
            $stmt2 = $conn->prepare("INSERT INTO certificates (student_id, course_id, certificate_type, status, certificate_code, issued_date) VALUES (?, ?, 'standard', 'issued', ?, NOW())");
            if ($stmt2) {
                $stmt2->bind_param("iis", $user_id, $course_id, $certificate_code);
                $stmt2->execute();
                $stmt2->close();
            }
        }
        $stmt->close();
    }
}

include_once 'components/header.php';
?>



<!-- Certificates Content -->
<div class="certificates-page-content">
<style>
/* Modern Certificates Styling */
.certificates-page-content {
    background: #f8fafc;
    font-family: 'Inter', 'Segoe UI', Roboto, sans-serif;
    min-height: 100vh;
}

.certificates-wrapper {
    padding: 2rem 0;
    min-height: 100vh;
}

.certificates-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 0 2rem;
}

/* Clean Hero Section */
.certificates-hero {
    text-align: center;
    margin-bottom: 3rem;
    position: relative;
    padding: 2rem;
    background: transparent;
    overflow: hidden;
}

.certificates-hero h1 {
    font-size: 2.5rem;
    font-weight: 700;
    color: #1f2937;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.certificates-hero p {
    color: #6b7280;
    font-size: 1.1rem;
    font-weight: 400;
    margin: 0;
}

/* Floating Animation Elements */
.floating-icon {
    position: absolute;
    font-size: 2rem;
    color: rgba(255, 255, 255, 0.3);
    animation: float 6s ease-in-out infinite;
}

.floating-icon:nth-child(1) { top: 20%; left: 10%; animation-delay: 0s; }
.floating-icon:nth-child(2) { top: 30%; right: 15%; animation-delay: 2s; }
.floating-icon:nth-child(3) { bottom: 30%; left: 15%; animation-delay: 4s; }
.floating-icon:nth-child(4) { bottom: 20%; right: 10%; animation-delay: 1s; }

@keyframes float {
    0%, 100% { transform: translateY(0px) rotate(0deg); }
    50% { transform: translateY(-20px) rotate(5deg); }
}

/* Stats Dashboard */
.stats-dashboard {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 2rem;
    margin-bottom: 3rem;
}

.stat-card {
    background: white;
    border-radius: 24px;
    padding: 2rem;
    text-align: center;
    box-shadow: 0 20px 60px rgba(0,0,0,0.1);
    border: 1px solid rgba(255,255,255,0.2);
    position: relative;
    overflow: hidden;
    transition: all 0.4s ease;
    animation: slideInUp 0.6s ease-out;
}

.stat-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 30px 80px rgba(0,0,0,0.15);
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(90deg, #4e73df 0%, #1cc88a 100%);
}

.stat-icon {
    width: 80px;
    height: 80px;
    margin: 0 auto 1rem;
    background: linear-gradient(135deg, #4e73df 0%, #1cc88a 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 2rem;
    color: white;
    box-shadow: 0 10px 30px rgba(78,115,223,0.3);
}

.stat-number {
    font-size: 3rem;
    font-weight: 800;
    color: #2c3e50;
    margin-bottom: 0.5rem;
}

.stat-label {
    font-size: 1.1rem;
    color: #64748b;
    font-weight: 500;
}

@keyframes slideInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Main Full-Width Layout */
.certificates-main-full {
    background: white;
    border-radius: 24px;
    padding: 2rem;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.1);
    width: 100%;
    max-width: 1200px;
    margin: 0 auto;
    border: 1px solid rgba(255,255,255,0.2);
}

/* Main Content - No Grid Needed */

.certificates-main {
    background: white;
    border-radius: 24px;
    padding: 2rem;
    box-shadow: 0 20px 60px rgba(0,0,0,0.1);
    border: 1px solid rgba(255,255,255,0.2);
}

.section-title {
    font-size: 1.8rem;
    font-weight: 700;
    color: #2c3e50;
    margin-bottom: 2rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.section-title i {
    color: #4e73df;
}

/* Certificate Cards */
.certificate-card {
    background: linear-gradient(135deg, #f8fafc 0%, #ffffff 100%);
    border-radius: 20px;
    padding: 2rem;
    margin-bottom: 1.5rem;
    border: 2px solid #e2e8f0;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.certificate-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 15px 40px rgba(0,0,0,0.1);
    border-color: #4e73df;
}

.certificate-card.accredited {
    border-left: 6px solid #1cc88a;
    background: linear-gradient(135deg, #f0fff4 0%, #ffffff 100%);
}

.certificate-card.standard {
    border-left: 6px solid #4e73df;
}

.certificate-card.pending {
    border-left: 6px solid #f6ad55;
    background: linear-gradient(135deg, #fffbf0 0%, #ffffff 100%);
}

.certificate-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 1.5rem;
}

.certificate-info h3 {
    font-size: 1.4rem;
    font-weight: 700;
    color: #2c3e50;
    margin-bottom: 0.5rem;
}

.certificate-meta {
    color: #64748b;
    font-size: 0.9rem;
    margin-bottom: 0.25rem;
}

.certificate-status {
    padding: 0.5rem 1rem;
    border-radius: 25px;
    font-size: 0.85rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-issued {
    background: #d1fae5;
    color: #065f46;
}

.status-pending {
    background: #fef3c7;
    color: #92400e;
}

.certificate-code {
    font-family: 'Courier New', monospace;
    background: #f1f5f9;
    padding: 0.5rem 1rem;
    border-radius: 8px;
    font-weight: 600;
    color: #475569;
    margin-bottom: 1rem;
    text-align: center;
}

.certificate-actions {
    display: flex;
    gap: 0.5rem;
    justify-content: center;
    margin-top: 0.5rem;
}

.btn {
    padding: 0.4rem 0.8rem;
    border: none;
    border-radius: 6px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.3s ease;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 0.3rem;
    font-size: 0.7rem;
}

.btn-primary {
    background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
    color: white;
}

.btn-success {
    background: linear-gradient(135deg, #1cc88a 0%, #17a673 100%);
    color: white;
}

.btn-warning {
    background: linear-gradient(135deg, #f6ad55 0%, #ed8936 100%);
    color: white;
}

.btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(0,0,0,0.2);
}

/* Course Cards for Completion */
.course-card {
    background: white;
    border-radius: 16px;
    padding: 1.5rem;
    margin-bottom: 1.5rem;
    border: 2px solid #e2e8f0;
    transition: all 0.3s ease;
}

.course-card:hover {
    border-color: #4e73df;
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
}

.course-header {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 1rem;
}

.course-thumbnail {
    width: 60px;
    height: 60px;
    border-radius: 12px;
    background: linear-gradient(135deg, #4e73df 0%, #1cc88a 100%);
    background-size: cover;
    background-position: center;
}

.course-info h4 {
    font-size: 1.2rem;
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 0.25rem;
}

.course-meta {
    color: #64748b;
    font-size: 0.9rem;
}

/* Sidebar */
.certificates-sidebar {
    display: flex;
    flex-direction: column;
    gap: 1.5rem;
}

.sidebar-section {
    background: white;
    border-radius: 20px;
    padding: 1.5rem;
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    border: 1px solid rgba(255,255,255,0.2);
}

.help-section {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.help-section h3 {
    color: white;
}

.help-section p {
    opacity: 0.9;
    line-height: 1.6;
}

/* Certificate Tips Section */
.certificate-tips {
    background: #f8fafc;
    border-radius: 12px;
    padding: 1.5rem;
    margin-bottom: 2rem;
    border-left: 4px solid #3b82f6;
}

/* Accreditation Instructions Section */
.accreditation-instructions {
    background: linear-gradient(135deg, #f8faff 0%, #ffffff 100%);
    border-radius: 16px;
    padding: 2rem;
    margin-bottom: 2rem;
    border: 1px solid #e2e8f0;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
}

.accreditation-instructions h3 {
    font-size: 1.5rem;
    font-weight: 700;
    color: #1e293b;
    margin-bottom: 1.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.accreditation-instructions h3 i {
    color: #10b981;
    font-size: 1.6rem;
}

.instruction-header .lead {
    font-size: 1.1rem;
    color: #475569;
    margin-bottom: 1.5rem;
    font-weight: 500;
}

.instruction-steps {
    display: flex;
    flex-direction: column;
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.step-item {
    display: flex;
    align-items: flex-start;
    gap: 1rem;
    padding: 1.5rem;
    background: white;
    border-radius: 12px;
    border: 1px solid #e2e8f0;
    transition: all 0.3s ease;
}

.step-item:hover {
    border-color: #10b981;
    box-shadow: 0 4px 12px rgba(16, 185, 129, 0.1);
}

.step-number {
    background: linear-gradient(135deg, #10b981, #059669);
    color: white;
    width: 36px;
    height: 36px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 700;
    font-size: 1rem;
    flex-shrink: 0;
    box-shadow: 0 2px 8px rgba(16, 185, 129, 0.3);
}

.step-content h5 {
    font-size: 1.2rem;
    font-weight: 600;
    color: #1e293b;
    margin-bottom: 0.75rem;
}

.step-content p {
    color: #64748b;
    line-height: 1.6;
    margin-bottom: 0.75rem;
}

.step-content ul {
    margin: 0.75rem 0;
    padding-left: 1rem;
}

.step-content ul li {
    color: #475569;
    margin-bottom: 0.5rem;
    font-size: 0.95rem;
}

.navigation-hint {
    background: #f1f5f9;
    border: 1px solid #cbd5e1;
    border-radius: 8px;
    padding: 0.75rem 1rem;
    margin-top: 0.75rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 0.9rem;
    color: #475569;
}

.navigation-hint i {
    color: #10b981;
}

.info-box {
    background: #eff6ff;
    border: 1px solid #dbeafe;
    border-radius: 8px;
    padding: 0.75rem 1rem;
    margin-top: 0.75rem;
    display: flex;
    align-items: flex-start;
    gap: 0.5rem;
    font-size: 0.9rem;
    color: #1e40af;
}

.info-box i {
    color: #3b82f6;
    margin-top: 0.1rem;
    flex-shrink: 0;
}

.tracking-info {
    background: #f0fdf4;
    border: 1px solid #bbf7d0;
    border-radius: 8px;
    padding: 0.75rem 1rem;
    margin-top: 0.75rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 0.9rem;
    color: #166534;
}

.tracking-info i {
    color: #10b981;
}

.instruction-footer {
    border-top: 1px solid #e2e8f0;
    padding-top: 1.5rem;
}

.action-buttons {
    display: flex;
    gap: 1rem;
    margin-bottom: 1rem;
    flex-wrap: wrap;
}

.action-buttons .btn {
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    font-weight: 600;
    text-decoration: none;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s ease;
    font-size: 0.95rem;
}

.action-buttons .btn-primary {
    background: linear-gradient(135deg, #10b981, #059669);
    color: white;
    border: none;
}

.action-buttons .btn-primary:hover {
    background: linear-gradient(135deg, #059669, #047857);
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
}

.action-buttons .btn-secondary {
    background: #f8fafc;
    color: #475569;
    border: 1px solid #cbd5e1;
}

.action-buttons .btn-secondary:hover {
    background: #f1f5f9;
    border-color: #94a3b8;
    transform: translateY(-2px);
}

.help-note {
    background: #fefce8;
    border: 1px solid #fde047;
    border-radius: 8px;
    padding: 0.75rem 1rem;
    display: flex;
    align-items: flex-start;
    gap: 0.5rem;
    font-size: 0.9rem;
    color: #713f12;
}

.help-note i {
    color: #eab308;
    margin-top: 0.1rem;
    flex-shrink: 0;
}

/* Responsive Design for Instructions */
@media (max-width: 768px) {
    .accreditation-instructions {
        padding: 1.5rem;
    }
    
    .step-item {
        flex-direction: column;
        text-align: center;
        padding: 1.25rem;
    }
    
    .step-number {
        margin: 0 auto 1rem auto;
    }
    
    .action-buttons {
        flex-direction: column;
    }
    
    .action-buttons .btn {
        justify-content: center;
    }
}

.certificate-tips h3 {
    color: #1e40af;
    font-size: 1.1rem;
    font-weight: 600;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.certificate-tips ul {
    list-style: none;
    padding: 0;
    margin: 0;
}

.certificate-tips li {
    padding: 0.5rem 0;
    color: #4b5563;
    font-size: 0.9rem;
}

/* Clean Certificate Cards */
.clean-certificate-card {
    background: #ffffff;
    border-radius: 12px;
    padding: 1.5rem;
    margin-bottom: 1rem;
    border: 2px solid #e5e7eb;
    transition: all 0.3s ease;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
}

.clean-certificate-card:hover {
    border-color: #3b82f6;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

.certificate-info {
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1rem;
}

.course-details h3 {
    font-size: 1.3rem;
    font-weight: 600;
    color: #1f2937;
    margin-bottom: 0.5rem;
}

.course-details p {
    color: #6b7280;
    font-size: 0.9rem;
    margin-bottom: 0.25rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.certificate-action {
    flex-shrink: 0;
}

.btn-certificate-download {
    background: linear-gradient(135deg, #3b82f6, #1d4ed8);
    color: white;
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    font-size: 0.9rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
    box-shadow: 0 2px 4px rgba(59, 130, 246, 0.3);
}

.btn-certificate-download:hover {
    background: linear-gradient(135deg, #1d4ed8, #1e40af);
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(59, 130, 246, 0.4);
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 3rem 2rem;
    color: #64748b;
}

.empty-state i {
    font-size: 4rem;
    margin-bottom: 1rem;
    opacity: 0.5;
}

.empty-state h3 {
    font-size: 1.5rem;
    margin-bottom: 1rem;
    color: #475569;
}

/* Responsive Design */
@media (max-width: 1024px) {
    .certificates-grid {
        grid-template-columns: 1fr;
    }
    
    .certificates-sidebar {
        order: -1;
    }
}

@media (max-width: 768px) {
    .certificates-container {
        padding: 0 1rem;
    }
    
    .certificates-header {
        padding: 2rem 1rem;
    }
    
    .certificates-header h1 {
        font-size: 2.5rem;
    }
    
    .stats-dashboard {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .certificate-card {
        padding: 1.5rem;
    }
    
    .certificate-header {
        flex-direction: column;
        gap: 1rem;
    }
    
    .certificate-actions {
        flex-direction: column;
    }
}

@media (max-width: 480px) {
    .certificates-header h1 {
        font-size: 2rem;
    }
    
    .certificate-card {
        padding: 1rem;
    }
    
    .btn {
        padding: 0.6rem 1.5rem;
        font-size: 0.9rem;
    }
}

/* Certificate Preview Styling */
.certificate-preview-section {
    margin-bottom: 3rem;
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(20px);
    border-radius: 20px;
    padding: 2rem;
    border: 1px solid rgba(255, 255, 255, 0.2);
    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1);
}

.preview-header {
    text-align: center;
    margin-bottom: 2rem;
}

.preview-header h2 {
    color: #2c3e50;
    font-size: 2rem;
    font-weight: 700;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.preview-header p {
    color: #64748b;
    font-size: 1.1rem;
    margin: 0;
}

.certificate-preview-card {
    max-width: 800px;
    margin: 0 auto;
}

.certificate-template {
    background: #ffffff;
    border-radius: 15px;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
    border: 2px solid #e2e8f0;
}

.certificate-border {
    border: 3px solid #4e73df;
    border-radius: 10px;
    padding: 2rem;
    background: linear-gradient(135deg, #f7faff 0%, #ffffff 100%);
    position: relative;
}

.certificate-border::before {
    content: '';
    position: absolute;
    top: 10px;
    left: 10px;
    right: 10px;
    bottom: 10px;
    border: 1px solid #4e73df;
    border-radius: 7px;
    opacity: 0.5;
}

.certificate-header {
    text-align: center;
    margin-bottom: 2rem;
}

.certificate-logo {
    font-size: 4rem;
    color: #4e73df;
    margin-bottom: 1rem;
}

.certificate-header h1 {
    font-size: 2.5rem;
    font-weight: 800;
    color: #2c3e50;
    margin: 0;
    letter-spacing: 2px;
}

.certificate-subtitle {
    font-size: 1.2rem;
    color: #64748b;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 1px;
    margin-top: 0.5rem;
}

.certificate-body {
    text-align: center;
    margin: 2rem 0;
}

.certificate-text {
    font-size: 1.1rem;
    color: #475569;
    margin: 0.5rem 0;
}

.student-name {
    font-size: 2.2rem;
    font-weight: 700;
    color: #1e40af;
    margin: 1rem 0;
    text-decoration: underline;
    text-decoration-color: #4e73df;
    text-underline-offset: 8px;
    text-decoration-thickness: 2px;
}

.course-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: #10b981;
    font-style: italic;
    margin: 0.5rem 0;
}

.completion-info {
    margin-top: 2rem;
    display: flex;
    justify-content: space-between;
    flex-wrap: wrap;
    gap: 1rem;
}

.completion-date, .instructor-info {
    font-size: 1rem;
    color: #64748b;
    font-weight: 500;
}

.certificate-footer {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-top: 2rem;
    padding-top: 1rem;
    border-top: 2px solid #e2e8f0;
    flex-wrap: wrap;
    gap: 1rem;
}

.certificate-code {
    font-family: 'Courier New', monospace;
    font-size: 0.9rem;
    color: #475569;
    font-weight: 600;
}

.issue-date {
    font-size: 0.9rem;
    color: #64748b;
    font-weight: 500;
}

.preview-actions {
    display: flex;
    gap: 1rem;
    justify-content: center;
    flex-wrap: wrap;
}

.btn {
    padding: 0.75rem 2rem;
    border-radius: 10px;
    font-weight: 600;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
    font-size: 1rem;
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(108, 117, 125, 0.3);
}

.btn-success {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
}

.btn-success:hover {
    background: linear-gradient(135deg, #059669 0%, #047857 100%);
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(16, 185, 129, 0.3);
}

/* Modern Certificate Design - Based on Professional Template */
.certificate-container {
    background: #ffffff !important;
    border-radius: 12px !important;
    box-shadow: 0 8px 32px rgba(0, 0, 0, 0.12) !important;
    margin: 2rem auto !important;
    max-width: 700px !important;
    overflow: hidden !important;
    border: 2px solid #E5E7EB !important;
    position: relative !important;
}

.certificate-container::before {
    content: '' !important;
    position: absolute !important;
    top: 0 !important;
    left: 0 !important;
    right: 0 !important;
    bottom: 0 !important;
    background: 
        radial-gradient(circle at 15% 85%, rgba(79, 130, 244, 0.04) 0%, transparent 40%),
        radial-gradient(circle at 85% 15%, rgba(245, 158, 11, 0.04) 0%, transparent 40%),
        radial-gradient(circle at 50% 20%, rgba(79, 130, 244, 0.02) 0%, transparent 30%),
        radial-gradient(circle at 20% 50%, rgba(245, 158, 11, 0.02) 0%, transparent 35%),
        linear-gradient(135deg, rgba(79, 130, 244, 0.01) 0%, transparent 25%, rgba(245, 158, 11, 0.01) 75%, transparent 100%) !important;
    pointer-events: none !important;
    z-index: 1 !important;
}

.certificate-wrapper {
    background: #ffffff !important;
    position: relative !important;
    width: 100% !important;
    min-height: 400px !important;
    padding: 0 !important;
    z-index: 2 !important;
}

/* Amber Header Banner */
.cert-header-section {
    background: linear-gradient(135deg, #F59E0B 0%, #F97316 100%) !important;
    color: white !important;
    text-align: center !important;
    padding: 2rem 2rem 1.5rem !important;
    position: relative !important;
}

.cert-header-section::after {
    content: '' !important;
    position: absolute !important;
    bottom: -10px !important;
    left: 0 !important;
    right: 0 !important;
    height: 20px !important;
    background: linear-gradient(135deg, #F59E0B 0%, #F97316 100%) !important;
    clip-path: polygon(0 0, 100% 0, 95% 100%, 5% 100%) !important;
}

.cert-logo {
    width: 60px !important;
    height: 60px !important;
    border-radius: 50% !important;
    margin-bottom: 1rem !important;
    border: 3px solid white !important;
    object-fit: cover !important;
    background: white !important;
    padding: 2px !important;
}

.cert-title {
    font-size: 3rem !important;
    font-weight: 700 !important;
    color: white !important;
    margin: 0 !important;
    letter-spacing: 4px !important;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.2) !important;
    line-height: 1.1 !important;
}

.cert-badge {
    display: inline-block !important;
    background: rgba(255, 255, 255, 0.2) !important;
    color: white !important;
    padding: 0.5rem 1.2rem !important;
    border-radius: 20px !important;
    font-size: 1rem !important;
    font-weight: 600 !important;
    letter-spacing: 2px !important;
    margin-top: 0.8rem !important;
    border: 1px solid rgba(255, 255, 255, 0.3) !important;
}

/* Certificate Main Content */
.cert-main-content {
    text-align: center !important;
    padding: 3rem 2rem 2rem !important;
    background: #ffffff !important;
    position: relative !important;
}

.cert-main-content::before {
    content: '🎓' !important;
    position: absolute !important;
    top: 30px !important;
    left: 40px !important;
    font-size: 2.5rem !important;
    opacity: 0.08 !important;
    animation: float 6s ease-in-out infinite !important;
    pointer-events: none !important;
    z-index: 1 !important;
}

.cert-main-content::after {
    content: '⭐' !important;
    position: absolute !important;
    bottom: 40px !important;
    right: 50px !important;
    font-size: 2rem !important;
    opacity: 0.06 !important;
    animation: sparkle 8s ease-in-out infinite !important;
    pointer-events: none !important;
    z-index: 1 !important;
}

/* Floating Animation */
@keyframes float {
    0%, 100% {
        transform: translateY(0px) rotate(0deg) !important;
        opacity: 0.08 !important;
    }
    25% {
        transform: translateY(-10px) rotate(2deg) !important;
        opacity: 0.12 !important;
    }
    50% {
        transform: translateY(-5px) rotate(0deg) !important;
        opacity: 0.06 !important;
    }
    75% {
        transform: translateY(-15px) rotate(-2deg) !important;
        opacity: 0.1 !important;
    }
}

/* Sparkle Animation */
@keyframes sparkle {
    0%, 100% {
        transform: scale(1) rotate(0deg) !important;
        opacity: 0.06 !important;
    }
    25% {
        transform: scale(1.1) rotate(90deg) !important;
        opacity: 0.1 !important;
    }
    50% {
        transform: scale(0.9) rotate(180deg) !important;
        opacity: 0.04 !important;
    }
    75% {
        transform: scale(1.05) rotate(270deg) !important;
        opacity: 0.08 !important;
    }
}

.cert-declaration-text {
    font-size: 1.2rem !important;
    color: #4b5563 !important;
    margin: 1rem 0 !important;
    font-weight: 400 !important;
    text-transform: uppercase !important;
    letter-spacing: 1px !important;
}

.cert-student-name {
    font-size: 3rem !important;
    font-weight: 700 !important;
    color: #1f2937 !important;
    margin: 2rem 0 !important;
    position: relative !important;
    display: inline-block !important;
    font-family: 'Brush Script MT', cursive, 'Georgia', serif !important;
    font-style: italic !important;
}

.cert-student-name::after {
    content: '' !important;
    position: absolute !important;
    bottom: -15px !important;
    left: 50% !important;
    transform: translateX(-50%) !important;
    width: 80% !important;
    height: 2px !important;
    background: #d1d5db !important;
    border-radius: 1px !important;
}

.cert-course-name {
    font-size: 1.8rem !important;
    font-weight: 600 !important;
    color: #1f2937 !important;
    margin: 2rem 0 1rem !important;
    font-style: italic !important;
}

.cert-certified-badge {
    display: inline-flex !important;
    align-items: center !important;
    gap: 0.5rem !important;
    background: linear-gradient(135deg, #f59e0b, #d97706) !important;
    color: white !important;
    padding: 0.5rem 1.2rem !important;
    border-radius: 25px !important;
    font-size: 0.9rem !important;
    font-weight: 600 !important;
    letter-spacing: 1px !important;
    margin-top: 1rem !important;
    box-shadow: 0 3px 10px rgba(245, 158, 11, 0.3) !important;
}

/* Certificate Footer */
.cert-footer-section {
    display: flex !important;
    justify-content: space-around !important;
    align-items: center !important;
    padding: 2rem !important;
    background: #f9fafb !important;
    border-top: 1px solid #e5e7eb !important;
    gap: 2rem !important;
}

.cert-info-item {
    text-align: center !important;
    flex: 1 !important;
}

.cert-info-icon {
    font-size: 1.5rem !important;
    color: #6b7280 !important;
    margin-bottom: 0.5rem !important;
}

.cert-info-label {
    font-size: 0.8rem !important;
    color: #6b7280 !important;
    font-weight: 600 !important;
    text-transform: uppercase !important;
    letter-spacing: 0.5px !important;
    margin-bottom: 0.3rem !important;
}

.cert-info-value {
    font-size: 0.9rem !important;
    color: #374151 !important;
    font-weight: 500 !important;
}

/* Certificate Actions */
.cert-actions {
    padding: 2rem !important;
    background: #ffffff !important;
    display: flex !important;
    gap: 1.5rem !important;
    justify-content: center !important;
    border-top: 1px solid #e5e7eb !important;
}

.cert-btn {
    padding: 1rem 2rem !important;
    border-radius: 8px !important;
    font-weight: 600 !important;
    text-decoration: none !important;
    transition: all 0.3s ease !important;
    display: flex !important;
    align-items: center !important;
    gap: 0.5rem !important;
    font-size: 1rem !important;
    border: none !important;
    cursor: pointer !important;
}

.cert-btn-primary {
    background: linear-gradient(135deg, #4F82F4, #6366f1) !important;
    color: white !important;
    box-shadow: 0 4px 12px rgba(79, 130, 244, 0.3) !important;
    border: 1px solid rgba(79, 130, 244, 0.2) !important;
}

.cert-btn-primary:hover {
    background: linear-gradient(135deg, #3b6cf0, #4f46e5) !important;
    transform: translateY(-2px) !important;
    box-shadow: 0 6px 16px rgba(79, 130, 244, 0.4) !important;
}

.cert-btn-secondary {
    background: linear-gradient(135deg, #F59E0B, #F97316) !important;
    color: white !important;
    box-shadow: 0 4px 12px rgba(245, 158, 11, 0.3) !important;
    border: 1px solid rgba(245, 158, 11, 0.2) !important;
}

.cert-btn-secondary:hover {
    background: linear-gradient(135deg, #d97706, #ea580c) !important;
    transform: translateY(-2px) !important;
    box-shadow: 0 6px 16px rgba(245, 158, 11, 0.4) !important;
}

/* Override any conflicting styles */
.certificate-container * {
    box-sizing: border-box !important;
}

/* Responsive Design */
@media (max-width: 768px) {
    .certificate-container {
        margin: 1rem !important;
        max-width: none !important;
    }
    
    .cert-header-section {
        padding: 1.5rem 1rem !important;
    }
    
    .cert-title {
        font-size: 2.2rem !important;
        letter-spacing: 2px !important;
    }
    
    .cert-main-content {
        padding: 2rem 1rem !important;
    }
    
    .cert-student-name {
        font-size: 2.2rem !important;
    }
    
    .cert-course-name {
        font-size: 1.4rem !important;
    }
    
    .cert-footer-section {
        flex-direction: column !important;
        gap: 1.5rem !important;
        padding: 1.5rem !important;
    }
    
    .cert-actions {
        flex-direction: column !important;
        align-items: center !important;
        padding: 1.5rem !important;
    }
    
    .cert-btn {
        width: 100% !important;
        justify-content: center !important;
    }
}

/* Certificate Border Decorations */
.cert-border-decoration {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    pointer-events: none;
}

.corner-decoration {
    position: absolute;
    width: 60px;
    height: 60px;
    background: linear-gradient(45deg, #4F82F4, #F59E0B);
    opacity: 0.1;
}

.corner-decoration.top-left {
    top: 15px;
    left: 15px;
    border-radius: 0 0 100% 0;
}

.corner-decoration.top-right {
    top: 15px;
    right: 15px;
    border-radius: 0 0 0 100%;
}

.corner-decoration.bottom-left {
    bottom: 15px;
    left: 15px;
    border-radius: 0 100% 0 0;
}

.corner-decoration.bottom-right {
    bottom: 15px;
    right: 15px;
    border-radius: 100% 0 0 0;
}

/* Certificate Header with Logo - Rectangle Layout */
.cert-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 0.8rem;
    flex-direction: row;
    gap: 1rem;
}

.cert-logo-container {
    display: flex;
    align-items: center;
    gap: 1rem;
    position: relative;
}

.cert-logo-img {
    width: 45px;
    height: 45px;
    border-radius: 50%;
    border: 2px solid #4F82F4;
    box-shadow: 0 2px 6px rgba(79, 130, 244, 0.2);
    object-fit: cover;
}

.cert-seal {
    position: absolute;
    right: -10px;
    bottom: -5px;
    background: white;
    border-radius: 50%;
    padding: 2px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
}

.academy-name {
    font-size: 1.4rem;
    font-weight: 700;
    color: #1e293b;
    margin: 0;
    letter-spacing: 1px;
    text-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
    background: linear-gradient(135deg, #4F82F4, #F59E0B);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.cert-type-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.2rem;
    background: linear-gradient(135deg, #10b981, #059669);
    color: white;
    padding: 0.2rem 0.5rem;
    border-radius: 10px;
    font-size: 0.6rem;
    font-weight: 500;
    letter-spacing: 0.3px;
    box-shadow: 0 1px 3px rgba(16, 185, 129, 0.2);
    margin-top: 0.2rem;
}









/* Responsive Certificate Preview */
@media (max-width: 768px) {
    .certificate-template {
        padding: 1rem;
    }
    
    .certificate-border {
        padding: 1rem;
    }
    
    .certificate-header h1 {
        font-size: 2rem;
    }
    
    .student-name {
        font-size: 1.8rem;
    }
    
    .course-title {
        font-size: 1.4rem;
    }
    
    .completion-info {
        flex-direction: column;
        text-align: center;
    }
    
    .certificate-footer {
        flex-direction: column;
        text-align: center;
    }
    
    .preview-actions {
        flex-direction: column;
    }
    
    .btn {
        width: 100%;
        justify-content: center;
    }
}
</style>

<div class="certificates-wrapper">
    <div class="certificates-container">
        <!-- Clean Hero Section -->
        <div class="certificates-hero">
            <div class="floating-icon"><i class="bx bx-medal"></i></div>
            <div class="floating-icon"><i class="bx bx-award"></i></div>
            <div class="floating-icon"><i class="bx bx-trophy"></i></div>
            <div class="floating-icon"><i class="bx bx-certification"></i></div>
            
            <h1><i class="bx bx-collection"></i> My Certificates</h1>
            <p>Download and manage your earned certificates</p>
        </div>

        <!-- Certificate Preview Modal -->
        <?php if ($show_preview && $preview_course): ?>
        <div class="certificate-preview-section">
            <div class="preview-header">
                <h2><i class="bx bx-award"></i> Certificate Preview</h2>
                <p>Review your certificate before downloading</p>
            </div>
            
            <div class="certificate-preview-card">
                <div class="certificate-template">
                    <div class="certificate-border">
                        <div class="certificate-header">
                            <div class="certificate-logo">
                                <i class="bx bx-award"></i>
                            </div>
                            <h1>PANADITE ACADEMY</h1>
                            <div class="certificate-subtitle">Certificate of Completion</div>
                        </div>
                        
                        <div class="certificate-body">
                            <div class="certificate-text">This is to certify that</div>
                            <div class="student-name"><?= htmlspecialchars($_SESSION['first_name'] . ' ' . $_SESSION['last_name']) ?></div>
                            <div class="certificate-text">has successfully completed the course</div>
                            <div class="course-title"><?= htmlspecialchars($preview_course['title']) ?></div>
                            <div class="completion-info">
                                <div class="completion-date">
                                    Completed on <?= date('F d, Y', strtotime($preview_course['completion_date'])) ?>
                                </div>
                                <div class="instructor-info">
                                    Instructor: <?= htmlspecialchars($preview_course['first_name'] . ' ' . $preview_course['last_name']) ?>
                                </div>
                            </div>
                        </div>
                        
                        <div class="certificate-footer">
                            <div class="certificate-code">
                                Certificate ID: PAC-<?= strtoupper(substr(md5($user_id . $preview_course_id), 0, 8)) ?>
                            </div>
                            <div class="issue-date">
                                Issued: <?= date('F d, Y') ?>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="preview-actions">
                    <a href="certificates.php" class="btn btn-secondary">
                        <i class="bx bx-arrow-back"></i> Back to Certificates
                    </a>
                    <a href="download-certificate.php?course_id=<?= $preview_course_id ?>&type=standard" class="btn btn-success" target="_blank">
                        <i class="bx bx-download"></i> Download Certificate
                    </a>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Main Content -->
        <div class="certificates-main-full">
                <h2 class="section-title">
                    <i class="bx bx-collection"></i>My Certificates
                </h2>

                <?php if (empty($certificates) && empty($completed_courses)): ?>
                    <div class="empty-state">
                        <i class="bx bx-medal"></i>
                        <h3>No Certificates Available</h3>
                        <p>Complete your enrolled courses to earn downloadable PDF certificates</p>
                        <div class="empty-state-actions">
                            <a href="../courses.php" class="btn btn-primary">
                                <i class="bx bx-book-open"></i> Browse Courses
                            </a>
                            <a href="dashboard.php" class="btn btn-secondary">
                                <i class="bx bx-dashboard"></i> Check Progress
                            </a>
                        </div>
                    </div>
                <?php else: ?>
                    <!-- Tips Section -->
                    <div class="certificate-tips">
                        <h3><i class="bx bx-info-circle"></i> Certificate Tips</h3>
                        <ul>
                            <li>📄 Click "Get Certificate" to download your PDF certificate</li>
                            <li>🖨️ For best results, print on white A4 paper</li>
                            <li>✅ Certificates are automatically generated upon course completion</li>
                        </ul>
                    </div>

                    <!-- Accreditation Instructions Section -->
                    <div class="accreditation-instructions">
                        <h3><i class="bx bx-shield-check"></i> How to Request Accredited Certificates</h3>
                        <div class="instruction-content">
                            <div class="instruction-header">
                                <p class="lead">Follow these steps to request an accredited certificate for your completed courses:</p>
                            </div>
                            
                            <div class="instruction-steps">
                                <div class="step-item">
                                    <div class="step-number">1</div>
                                    <div class="step-content">
                                        <h5>Complete Course Requirements</h5>
                                        <p>You must be <strong>enrolled</strong> and have <strong>fully completed</strong> a course before you can request accreditation. This includes:</p>
                                        <ul>
                                            <li>✅ Watched all required videos</li>
                                            <li>✅ Passed all quizzes with minimum grade</li>
                                            <li>✅ Completed all course materials</li>
                                        </ul>
                                    </div>
                                </div>
                                
                                <div class="step-item">
                                    <div class="step-number">2</div>
                                    <div class="step-content">
                                        <h5>Navigate to Course Details & Certificate Tab</h5>
                                        <p>Go to your course details page and look for the <strong>Certificate tab</strong>. This tab is located next to the Reviews and Tools tabs at the top of your course page.</p>
                                        <div class="navigation-hint">
                                            <i class="bx bx-right-arrow-alt"></i>
                                            <span>Course Page → Click "Certificate Tab" → Find Request Accreditation Button</span>
                                        </div>
                                        <div class="info-box" style="background: #fef3c7; border: 1px solid #fbbf24; color: #92400e; margin-top: 0.75rem;">
                                            <i class="bx bx-info-circle"></i>
                                            <span><strong>Important:</strong> You must click on the Certificate tab within your course to access the accreditation request feature.</span>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="step-item">
                                    <div class="step-number">3</div>
                                    <div class="step-content">
                                        <h5>Submit Accreditation Request</h5>
                                        <p>Once you're in the Certificate tab of your course, you'll find the "Request Accredited Certificate" button. Click this button to start the accreditation process.</p>
                                        <div class="info-box">
                                            <i class="bx bx-info-circle"></i>
                                            <span>The accreditation request must be submitted directly from within the course's Certificate tab for proper processing.</span>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="step-item">
                                    <div class="step-number">4</div>
                                    <div class="step-content">
                                        <h5>Track Your Progress</h5>
                                        <p>After submitting your accreditation request, you'll be able to track the progress and status of your application through your student dashboard.</p>
                                        <div class="tracking-info">
                                            <i class="bx bx-time-five"></i>
                                            <span>Processing typically takes 3-5 business days</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="instruction-footer">
                                <div style="background: #e0f2fe; border: 1px solid #0284c7; border-radius: 8px; padding: 1rem; margin-bottom: 1rem;">
                                    <div style="display: flex; align-items: center; gap: 0.5rem; margin-bottom: 0.5rem;">
                                        <i class="bx bx-info-circle" style="color: #0284c7; font-size: 1.2rem;"></i>
                                        <strong style="color: #0284c7;">Quick Tip:</strong>
                                    </div>
                                    <p style="color: #0369a1; margin: 0; font-size: 0.9rem;">
                                        Click the "Request Accredited Certificate" button for your completed course below, then look for the <strong>"Certificate"</strong> tab at the top of your course page (next to Reviews and Tools) to complete the accreditation request.
                                    </p>
                                </div>
                                
                                <div class="help-note">
                                    <i class="bx bx-help-circle"></i>
                                    <span>Need help? Contact our support team if you have questions about the accreditation process.</span>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Clean Certificate Cards -->
                    <?php if (!empty($completed_courses)): ?>
                        <?php foreach ($completed_courses as $course): ?>
                            <div class="clean-certificate-card">
                                <div class="certificate-info">
                                    <div class="course-details">
                                        <h3><?= htmlspecialchars($course['title']) ?></h3>
                                        <p class="completion-status">
                                            <i class="bx bx-check-circle"></i> 
                                            Completed: <?= date('M j, Y', strtotime($course['enrollment_date'])) ?>
                                        </p>
                                        <p class="progress-info">
                                            <i class="bx bx-chart"></i> 
                                            Progress: <?= $course['progress_percentage'] ?? 100 ?>%
                                        </p>
                                    </div>
                                    
                                    <div class="certificate-action">
                                        <a href="certificate.php?course_id=<?= encryptId($course['course_id']) ?>" 
                                           class="btn-certificate-download" 
                                           target="_blank">
                                            <i class="bx bx-download"></i> 
                                            Get Certificate
                                        </a>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                
                    <!-- Existing Template Certificates -->
                    <?php foreach ($certificates as $cert): ?>
                        <div class="certificate-card <?= $cert['certificate_type'] ?> generated">
                            <div class="certificate-preview-thumbnail">
                                <?php if ($cert['generated_file']): ?>
                                    <img src="/uploads/certificates/generated/<?= htmlspecialchars($cert['generated_file']) ?>" 
                                         alt="Certificate Preview" class="cert-thumbnail">
                                <?php else: ?>
                                    <div class="cert-placeholder">
                                        <i class="bx bx-certification"></i>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="certificate-header">
                                <div class="certificate-info">
                                    <h3><?= htmlspecialchars($cert['course_title'] ?: $cert['template_name']) ?></h3>
                                    <div class="certificate-meta">
                                        <i class="bx bx-award"></i> <?= ucfirst($cert['certificate_type']) ?> Certificate
                                    </div>
                                    <div class="certificate-meta">
                                        <i class="bx bx-barcode"></i> <?= htmlspecialchars($cert['certificate_number']) ?>
                                    </div>
                                    <div class="certificate-meta">
                                        <i class="bx bx-calendar"></i> 
                                        Issued: <?= date('F d, Y', strtotime($cert['issue_date'])) ?>
                                    </div>
                                </div>
                                <span class="certificate-status status-generated">
                                    <i class="bx bx-check-circle"></i> Ready
                                </span>
                            </div>
                            <div class="certificate-actions">
                                <a href="/uploads/certificates/generated/<?= htmlspecialchars($cert['generated_file']) ?>" 
                                   class="btn btn-primary btn-sm" target="_blank">
                                    <i class="bx bx-show"></i> View
                                </a>
                                <a href="/uploads/certificates/generated/<?= htmlspecialchars($cert['generated_file']) ?>" 
                                   class="btn btn-success btn-sm" download="Certificate_<?= $cert['certificate_number'] ?>.png">
                                    <i class="bx bx-download"></i> Download
                                </a>
                                <button class="btn btn-info btn-sm" onclick="shareCertificate('<?= $cert['certificate_id'] ?>')">
                                    <i class="bx bx-share"></i> Share
                                </button>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>

                <!-- Completed Courses Section -->
                <?php if (!empty($completed_courses)): ?>
                    <h2 class="section-title mt-4">
                        <i class="bx bx-trophy"></i>Request Accredited Certificates
                    </h2>
                    
                    <?php foreach ($completed_courses as $course): ?>
                        <?php
                        // Check if accredited certificate already exists
                        $has_accredited = false;
                        foreach ($certificates as $cert) {
                            if ($cert['course_id'] == $course['course_id'] && $cert['certificate_type'] === 'accredited') {
                                $has_accredited = true;
                                break;
                            }
                        }
                        ?>
                        
                        <?php if (!$has_accredited): ?>
                            <div class="course-card">
                                <div class="course-header">
                                    <div class="course-thumbnail" 
                                         style="background-image: url('<?= htmlspecialchars($course['thumbnail'] ?? '../assets/img/courses/default.jpg') ?>')">
                                    </div>
                                    <div class="course-info">
                                        <h4><?= htmlspecialchars($course['title']) ?></h4>
                                        <div class="course-meta">
                                            Completed: <?= date('F d, Y', strtotime($course['completion_date'])) ?>
                                        </div>
                                        <div class="course-meta">
                                            Instructor: <?= htmlspecialchars($course['instructor_name']) ?>
                                        </div>
                                    </div>
                                </div>
                                
                                <div style="text-align: center;">
                                    <?php 
                                    // Encrypt the course ID for the redirect
                                    $encrypted_course_id = IdEncryption::encrypt($course['course_id']);
                                    ?>
                                    <a href="https://panaditeacademy.com/student/course-detail.php?id=<?= urlencode($encrypted_course_id) ?>" class="btn btn-success">
                                        <i class="bx bx-check-shield"></i> Request Accredited Certificate
                                    </a>
                                </div>
                            </div>
                        <?php endif; ?>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>


        </div>
    </div>
</div>
</div>

<!-- Close home-content div -->
        </div>
        
        <script>
        // Certificate sharing functionality
        function shareCertificate(certificateId) {
            if (navigator.share) {
                navigator.share({
                    title: 'My Certificate - Panadite Academy',
                    text: 'Check out my certificate from Panadite Academy!',
                    url: window.location.origin + '/student/certificates.php?share=' + certificateId
                }).catch(console.error);
            } else {
                // Fallback to copy link
                const url = window.location.origin + '/student/certificates.php?share=' + certificateId;
                navigator.clipboard.writeText(url).then(() => {
                    showNotification('Certificate link copied to clipboard!', 'success');
                });
            }
        }
        
        // Enhanced notification system
        function showNotification(message, type = 'info') {
            const notification = $(`
                <div class="alert alert-${type} alert-dismissible fade show position-fixed" 
                     style="top: 20px; right: 20px; z-index: 9999; min-width: 300px;">
                    <i class="bx ${type === 'success' ? 'bx-check-circle' : 'bx-info-circle'} me-2"></i>
                    ${message}
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            `);
            
            $('body').append(notification);
            
            setTimeout(() => {
                notification.alert('close');
            }, 5000);
        }
        
        // Auto-generate completion certificates for completed courses
        $(document).ready(function() {
            $('.generate-completion-btn').click(function() {
                const courseId = $(this).data('course-id');
                const button = $(this);
                
                button.prop('disabled', true).html('<i class="bx bx-loader-alt bx-spin"></i> Generating...');
                
                $.ajax({
                    url: '/admin/api/generate_certificate.php',
                    method: 'POST',
                    contentType: 'application/json',
                    data: JSON.stringify({
                        student_id: <?= $user_id ?>,
                        certificate_type: 'completion',
                        course_id: courseId
                    }),
                    success: function(response) {
                        if (response.success) {
                            showNotification('Certificate generated successfully!', 'success');
                            setTimeout(() => {
                                location.reload();
                            }, 2000);
                        } else {
                            showNotification('Error generating certificate: ' + response.error, 'danger');
                            button.prop('disabled', false).html('<i class="bx bx-certification"></i> Generate Certificate');
                        }
                    },
                    error: function() {
                        showNotification('Error generating certificate. Please try again.', 'danger');
                        button.prop('disabled', false).html('<i class="bx bx-certification"></i> Generate Certificate');
                    }
                });
            });
        });
        </script>
</body>
</html>
