<?php
/**
 * Student Checkout Page
 * Panadite Academy
 * 
 * Professional checkout process with course review and payment integration
 */

// Authentication and session
require_once '../auth/functions.php';
requireRole('student');

// FETCH USER'S AVAILABLE DISCOUNTS FOR CHECKOUT
$available_discounts = [];
$userId = $_SESSION['user_id'] ?? 0;

if ($userId) {
    require_once '../config/db_connect.php';
    $discountsQuery = "SELECT * FROM student_referral_discounts 
                      WHERE student_id = ? AND is_used = 0 AND expires_at > NOW() 
                      ORDER BY discount_percentage DESC LIMIT 1";
    $discountStmt = $conn->prepare($discountsQuery);
    if ($discountStmt) {
        $discountStmt->bind_param("i", $userId);
        $discountStmt->execute();
        $discountResult = $discountStmt->get_result();
        if ($discountResult->num_rows > 0) {
            $available_discounts = $discountResult->fetch_assoc();
        }
    }
}

// Handle guest cart transfer if redirected from guest cart
if (isset($_GET['from_guest_cart']) && $_GET['from_guest_cart'] === '1') {
    // JavaScript will handle the cart transfer on page load
    $transfer_guest_cart = true;
} else {
    $transfer_guest_cart = false;
}

// Include layout components
include_once 'components/header.php';
?>

<!-- Simple Checkout Header -->
<div class="simple-checkout-header">
    <h1 class="checkout-title">
        <i class="bx bx-credit-card me-3"></i>
        Checkout
    </h1>
    <p class="checkout-description">Review your order and complete your purchase</p>
</div>

<!-- Checkout Content -->
<div class="container-fluid px-4">
    <div class="row">
        <!-- Left Column - Order Review -->
        <div class="col-lg-8">
            <!-- Order Summary Card -->
            <div class="card checkout-items-card">
                <div class="card-header bg-white border-0 pb-0">
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <div>
                            <h5 class="card-title mb-1">
                                <i class="bx bx-list-ul me-2"></i>
                                Order Review
                            </h5>
                            <p class="card-subtitle text-muted mb-0">Review your selected courses</p>
                        </div>
                        <span class="badge bg-primary rounded-pill" id="checkoutItemCount">0 items</span>
                    </div>
                </div>
                <div class="card-body">
                    <!-- Empty Checkout State -->
                    <div class="empty-checkout-simple text-center py-5" id="emptyCheckout">
                        <i class="bx bx-shopping-bag empty-checkout-icon mb-4"></i>
                        <h4 class="empty-title">No items in checkout!</h4>
                        <p class="empty-description">
                            Add some courses to your cart first.<br>
                            Then proceed to checkout to complete your purchase.
                        </p>
                        <div class="empty-actions">
                            <a href="course-catalog.php" class="btn btn-primary btn-lg">
                                <i class="bx bx-book me-2"></i>
                                Browse Courses
                            </a>
                            <a href="cart.php" class="btn btn-outline-primary btn-lg ms-2">
                                <i class="bx bx-cart me-2"></i>
                                View Cart
                            </a>
                        </div>
                    </div>

                    <!-- Checkout Items Container -->
                    <div id="checkoutItems" style="display: none;">
                        <!-- Items will be populated by JavaScript -->
                    </div>
                </div>
            </div>

            <!-- Payment Method Card -->
            <div class="card payment-method-card mt-4">
                <div class="card-header bg-white border-0">
                    <h5 class="card-title mb-0">
                        <i class="bx bx-credit-card me-2"></i>
                        Payment Method
                    </h5>
                </div>
                <div class="card-body">
                    <div class="payment-options">
                        <div class="payment-option selected" data-method="payfast">
                            <div class="payment-option-header">
                                <div class="payment-radio">
                                    <input type="radio" name="payment_method" id="payfast" value="payfast" checked>
                                    <label for="payfast" class="payment-label">
                                        <img src="../assets/images/payfast-logo.png" alt="PayFast" class="payment-logo" onerror="this.style.display='none'">
                                        <span class="payment-name">PayFast</span>
                                    </label>
                                </div>
                                <div class="payment-security">
                                    <i class="bx bx-shield-check text-success"></i>
                                    <span class="security-text">Secure Payment</span>
                                </div>
                            </div>
                            <div class="payment-description">
                                <p class="text-muted mb-0">
                                    <i class="bx bx-info-circle me-1"></i>
                                    Pay securely with credit card, debit card, or bank transfer via PayFast
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Right Column - Order Summary -->
        <div class="col-lg-4">
            <div class="card order-summary-card" style="position: sticky; top: 20px; z-index: 100;">
                <div class="card-header bg-white border-0">
                    <h5 class="card-title mb-0">
                        <i class="bx bx-receipt me-2"></i>
                        Order Summary
                    </h5>
                    <small class="text-muted">Review your total before checkout</small>
                </div>
                <div class="card-body">
                    <!-- Summary Items -->
                    <div id="summaryItems">
                        <!-- Will be populated by JavaScript -->
                    </div>

                    <!-- Price Breakdown -->
                    <div class="price-breakdown">
                        <div class="price-row">
                            <span>Subtotal:</span>
                            <span id="checkoutSubtotal">R0.00</span>
                        </div>
                        <div class="price-row savings-row" id="checkoutSavingsRow" style="display: none;">
                            <span class="text-success">
                                <i class="bx bx-tag me-1"></i>
                                Discount:
                            </span>
                            <span class="text-success" id="checkoutSavings">-R0.00</span>
                        </div>
                        <div class="price-row total-row">
                            <span class="fw-bold">Total:</span>
                            <span class="fw-bold text-primary fs-5" id="checkoutTotal">R0.00</span>
                        </div>
                    </div>



                    <!-- Security Badges -->
                    <div class="security-badges">
                        <div class="security-badge">
                            <i class="bx bx-shield-check text-success"></i>
                            SSL Encrypted
                        </div>

                    </div>

                    <!-- Checkout Button -->
                    <button class="btn btn-primary btn-lg w-100 checkout-btn" id="proceedCheckoutBtn" onclick="proceedToPayment()" disabled>
                        <i class="bx bx-credit-card me-2"></i>
                        Complete Purchase
                        <span class="checkout-total-badge" id="checkoutTotalBadge">R0.00</span>
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* Simple Checkout Styling */
.simple-checkout-header {
    margin-top: 2rem;
    margin-bottom: 2rem;
    padding-top: 1rem;
    padding-bottom: 1rem;
    border-bottom: 2px solid #e3e6f0;
}

.checkout-title {
    font-size: 2rem;
    font-weight: 600;
    color: #495057;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
}

.checkout-title i {
    color: #f59e0b;
}

.checkout-description {
    font-size: 1rem;
    color: #6c757d;
    margin: 0;
    font-weight: 400;
}

/* Checkout Cards */
.checkout-items-card, .payment-method-card, .order-summary-card {
    border: 1px solid #e3e6f0;
    border-radius: 12px;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
    transition: all 0.3s ease;
}

.checkout-items-card:hover, .payment-method-card:hover {
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
}

/* Empty Checkout State */
.empty-checkout-simple {
    padding: 3rem 2rem;
}

.empty-checkout-icon {
    font-size: 4rem;
    color: #dee2e6;
    display: block;
}

.empty-title {
    font-size: 1.5rem;
    font-weight: 600;
    color: #495057;
    margin-bottom: 1rem;
}

.empty-description {
    color: #6c757d;
    font-size: 1rem;
    line-height: 1.6;
    margin-bottom: 2rem;
}

.empty-actions {
    display: flex;
    gap: 1rem;
    justify-content: center;
    flex-wrap: wrap;
}

/* Checkout Items */
.checkout-item {
    display: flex;
    align-items: center;
    padding: 1rem;
    border: 1px solid #e9ecef;
    border-radius: 8px;
    margin-bottom: 1rem;
    transition: all 0.3s ease;
}

.checkout-item:hover {
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
}

.checkout-item-image {
    width: 80px;
    height: 60px;
    object-fit: cover;
    border-radius: 6px;
    margin-right: 1rem;
}

.checkout-item-details {
    flex: 1;
}

.checkout-item-title {
    font-size: 1rem;
    font-weight: 600;
    color: #495057;
    margin-bottom: 0.25rem;
}

.checkout-item-instructor {
    font-size: 0.875rem;
    color: #6c757d;
    margin-bottom: 0.25rem;
}

.checkout-item-rating {
    font-size: 0.875rem;
    color: #ffc107;
}

.checkout-item-price {
    font-size: 1.25rem;
    font-weight: 600;
    color: #f59e0b;
    text-align: right;
}

/* Payment Method */
.payment-options {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.payment-option {
    border: 2px solid #e9ecef;
    border-radius: 8px;
    padding: 1rem;
    cursor: pointer;
    transition: all 0.3s ease;
}

.payment-option.selected {
    border-color: #3b82f6;
    background-color: #f8faff;
}

.payment-option-header {
    display: flex;
    justify-content: between;
    align-items: center;
    margin-bottom: 0.5rem;
}

.payment-radio {
    display: flex;
    align-items: center;
    flex: 1;
}

.payment-label {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    cursor: pointer;
    margin-bottom: 0;
}

.payment-logo {
    height: 24px;
    width: auto;
}

.payment-name {
    font-weight: 600;
    color: #495057;
}

.payment-security {
    display: flex;
    align-items: center;
    gap: 0.25rem;
    font-size: 0.875rem;
    color: #10b981;
}

.payment-description {
    margin-top: 0.5rem;
    font-size: 0.875rem;
}

/* Order Summary */
.order-summary-card {
    position: sticky;
    top: 2rem;
}

.price-breakdown {
    border-top: 1px solid #e9ecef;
    padding-top: 1rem;
    margin-top: 1rem;
}

.price-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 0.75rem;
}

.total-row {
    border-top: 1px solid #e9ecef;
    padding-top: 0.75rem;
    margin-top: 0.75rem;
}

/* Promo Code */
.promo-section {
    margin: 1rem 0;
    padding: 1rem;
    background-color: #f8f9fa;
    border-radius: 8px;
}

.promo-input-group .btn {
    border-left: none;
}

/* Security Badges */
.security-badges {
    display: flex;
    justify-content: space-between;
    margin: 1rem 0;
    padding: 0.75rem;
    background-color: #f0f9ff;
    border-radius: 6px;
}

.security-badge {
    display: flex;
    align-items: center;
    gap: 0.25rem;
    font-size: 0.875rem;
    color: #10b981;
    font-weight: 500;
}

/* Checkout Button */
.checkout-btn {
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    border: none;
    border-radius: 8px;
    padding: 1rem;
    font-size: 1.1rem;
    font-weight: 600;
    position: relative;
    overflow: hidden;
    transition: all 0.3s ease;
}

.checkout-btn:hover:not(:disabled) {
    background: linear-gradient(135deg, #2563eb 0%, #1e40af 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(59, 130, 246, 0.4);
}

.checkout-btn:disabled {
    background: #6c757d;
    cursor: not-allowed;
}

.checkout-total-badge {
    position: absolute;
    right: 1rem;
    top: 50%;
    transform: translateY(-50%);
    background: #4e73df;
    color: white;
    padding: 0.5rem 1rem;
    border-radius: 6px;
    font-weight: 700;
    font-size: 1.1rem;
}

/* Checkout Discount Price Styling */
.checkout-price-container {
    text-align: right;
    min-width: 100px;
}

.original-price-small {
    font-size: 12px;
    color: #6b7280;
    text-decoration: line-through;
    font-weight: 500;
    margin-bottom: 2px;
}

.discounted-price-large {
    font-size: 18px;
    color: #dc2626;
    font-weight: 700;
    margin-bottom: 2px;
}

.savings-indicator {
    font-size: 10px;
    background: linear-gradient(135deg, #dc2626, #b91c1c);
    color: white;
    padding: 2px 6px;
    border-radius: 4px;
    font-weight: 600;
    display: inline-block;
    animation: checkout-savings-pulse 2s ease-in-out infinite;
}

@keyframes checkout-savings-pulse {
    0%, 100% { opacity: 1; transform: scale(1); }
    50% { opacity: 0.85; transform: scale(0.98); }
}

.discount-highlight {
    color: #dc2626 !important;
    font-weight: 600 !important;
}

#referralDiscountInfo {
    display: none;
    background: #f0f9ff;
    border: 1px solid #0ea5e9;
    border-radius: 6px;
    padding: 8px 12px;
    margin-bottom: 12px;
    font-size: 13px;
    color: #0369a1;
}

/* Summary Items */
.summary-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.5rem 0;
    border-bottom: 1px solid #f1f3f4;
}

.summary-item-title {
    font-size: 0.9rem;
    color: #495057;
    font-weight: 500;
}

.summary-item-price {
    font-size: 0.9rem;
    color: #f59e0b;
    font-weight: 600;
}

/* Responsive Design */
@media (max-width: 768px) {
    .simple-checkout-header {
        margin-top: 1rem;
        padding-top: 0.5rem;
    }
    
    .checkout-title {
        font-size: 1.5rem;
    }
    
    .checkout-item {
        flex-direction: column;
        text-align: center;
    }
    
    .checkout-item-image {
        margin-right: 0;
        margin-bottom: 1rem;
    }
    
    .order-summary-card {
        position: static;
        margin-top: 2rem;
    }
    
    .empty-actions {
        flex-direction: column;
        align-items: center;
    }
}
</style>

<script>
let checkoutItems = [];
let transferGuestCart = <?= json_encode($transfer_guest_cart) ?>;

// DISCOUNT DATA FROM PHP
const userDiscounts = <?= json_encode($available_discounts) ?>;
console.log('🎯 CHECKOUT: User discounts loaded:', userDiscounts);

// Calculate discount for a price
function calculateDiscount(originalPrice) {
    if (!userDiscounts || !userDiscounts.discount_percentage || originalPrice <= 0) {
        return { discounted_price: originalPrice, savings: 0, has_discount: false };
    }
    
    const discount_amount = Math.min(
        (originalPrice * userDiscounts.discount_percentage / 100),
        userDiscounts.discount_amount || originalPrice
    );
    const discounted_price = originalPrice - discount_amount;
    
    return {
        discounted_price: Math.max(0, discounted_price),
        savings: discount_amount,
        has_discount: true,
        discount_percentage: userDiscounts.discount_percentage
    };
}

// Initialize checkout on page load
document.addEventListener('DOMContentLoaded', function() {
    console.log('🚀 CHECKOUT: Page loaded, transfer guest cart?', transferGuestCart);
    
    // Handle guest cart transfer if needed
    if (transferGuestCart) {
        handleGuestCartTransfer();
    }
    
    loadCheckoutItems();
    updateCheckoutDisplay();
    
    <?php if ($transfer_guest_cart): ?>
    // Transfer guest cart items if redirected from guest cart
    transferGuestCartItems();
    <?php endif; ?>
});

// Sync guest cart when user logs in
function syncGuestCartOnLogin() {
    // DEBUG: Check what carts exist
    console.log('🔍 CHECKOUT DEBUG: Checking cart states...');
    console.log('🔍 panadite_guest_cart:', localStorage.getItem('panadite_guest_cart'));
    console.log('🔍 cart:', localStorage.getItem('cart'));
    
    // Check if we have guest cart items to sync
    const guestCart = localStorage.getItem('panadite_guest_cart');
    if (guestCart) {
        try {
            const guestItems = JSON.parse(guestCart);
            if (guestItems && guestItems.length > 0) {
                console.log('✅ CHECKOUT: Found guest cart, syncing', guestItems.length, 'items');
                
                // Convert guest items to authenticated format
                let currentCart = [];
                guestItems.forEach(guestItem => {
                    currentCart.push({
                        id: guestItem.id,
                        title: guestItem.title || 'Course',
                        instructor: guestItem.instructor || 'Panadite Academy',
                        price: guestItem.price || 0,
                        rating: guestItem.rating || '4.5',
                        reviews: guestItem.reviews || '0',
                        thumbnail: guestItem.thumbnail || '/uploads/thumbnails/default-course.jpg',
                        level: guestItem.level || 'Beginner',
                        description: guestItem.description || ''
                    });
                });
                
                // Save synced cart
                localStorage.setItem('cart', JSON.stringify(currentCart));
                localStorage.removeItem('panadite_guest_cart');
                
                console.log('✅ CHECKOUT: Guest cart synced successfully:', currentCart.length, 'items');
            }
        } catch (error) {
            console.error('❌ CHECKOUT: Error syncing guest cart:', error);
        }
    } else {
        // NO GUEST CART - check if authenticated cart exists (from previous sync)
        const existingCart = localStorage.getItem('cart');
        if (existingCart) {
            try {
                const cartItems = JSON.parse(existingCart);
                console.log('📦 CHECKOUT: No guest cart, but found existing authenticated cart with', cartItems.length, 'items');
                console.log('📦 CHECKOUT: Keeping existing cart (likely from cart page sync)');
                // DON'T clear existing cart - it might be from cart page sync
            } catch (error) {
                console.log('🔒 CHECKOUT: Invalid existing cart, clearing');
                localStorage.removeItem('cart');
            }
        } else {
            console.log('📭 CHECKOUT: No carts found (fresh user session)');
        }
    }
    
    // UPDATE CART COUNTER
    if (typeof updateCartCount === 'function') {
        updateCartCount();
    }
}

function loadCheckoutItems() {
    checkoutItems = JSON.parse(localStorage.getItem('cart') || '[]');
}

function updateCheckoutDisplay() {
    const checkoutContainer = document.getElementById('checkoutItems');
    const emptyCheckout = document.getElementById('emptyCheckout');
    const itemCount = document.getElementById('checkoutItemCount');
    const checkoutBtn = document.getElementById('proceedCheckoutBtn');

    if (checkoutItems.length === 0) {
        emptyCheckout.style.display = 'block';
        checkoutContainer.style.display = 'none';
        itemCount.textContent = '0 items';
        checkoutBtn.disabled = true;
    } else {
        emptyCheckout.style.display = 'none';
        checkoutContainer.style.display = 'block';
        itemCount.textContent = `${checkoutItems.length} item${checkoutItems.length > 1 ? 's' : ''}`;
        checkoutBtn.disabled = false;
        
        renderCheckoutItems();
        renderSummaryItems();
    }
    
    updateCheckoutSummary();
}

function renderCheckoutItems() {
    const container = document.getElementById('checkoutItems');
    container.innerHTML = '';
    
    checkoutItems.forEach((item, index) => {
        const originalPrice = parseFloat(item.price || 0);
        const discountInfo = calculateDiscount(originalPrice);
        
        // Build price display
        let priceDisplay = '';
        if (discountInfo.has_discount) {
            priceDisplay = `
                <div class="checkout-price-container">
                    <div class="original-price-small">R${originalPrice.toFixed(0)}</div>
                    <div class="discounted-price-large">R${discountInfo.discounted_price.toFixed(0)}</div>
                    <div class="savings-indicator">Save R${discountInfo.savings.toFixed(0)}!</div>
                </div>
            `;
        } else {
            priceDisplay = `<div class="checkout-item-price">R${originalPrice.toFixed(0)}</div>`;
        }
        
        const itemDiv = document.createElement('div');
        itemDiv.className = 'checkout-item';
        itemDiv.innerHTML = `
            <img src="${item.thumbnail || '/uploads/thumbnails/default-course.jpg'}" 
                 alt="${item.title}" class="checkout-item-image">
            <div class="checkout-item-details">
                <div class="checkout-item-title">${item.title}</div>
                <div class="checkout-item-instructor">by ${item.instructor || 'Panadite Academy'}</div>

            </div>
            ${priceDisplay}
        `;
        container.appendChild(itemDiv);
    });
}

function renderSummaryItems() {
    const container = document.getElementById('summaryItems');
    container.innerHTML = '';
    
    checkoutItems.forEach(item => {
        const summaryDiv = document.createElement('div');
        summaryDiv.className = 'summary-item';
        summaryDiv.innerHTML = `
            <span class="summary-item-title">${item.title}</span>
            <span class="summary-item-price">R${parseFloat(item.price || 0).toFixed(2)}</span>
        `;
        container.appendChild(summaryDiv);
    });
}

function updateCheckoutSummary() {
    let subtotal = 0;
    let referralDiscount = 0;
    let finalTotal = 0;
    
    // Calculate totals with referral discounts
    checkoutItems.forEach(item => {
        const originalPrice = parseFloat(item.price || 0);
        const discountInfo = calculateDiscount(originalPrice);
        
        subtotal += originalPrice;
        if (discountInfo.has_discount) {
            referralDiscount += discountInfo.savings;
        }
        finalTotal += discountInfo.discounted_price;
    });
    
    // Add promo code discount on top of referral discount (if applied)
    const discountPercent = parseFloat(localStorage.getItem('checkoutPromoDiscount') || 0);
    const promoDiscount = finalTotal * discountPercent;
    const total = finalTotal - promoDiscount;
    const totalDiscount = referralDiscount + promoDiscount;
    
    document.getElementById('checkoutSubtotal').textContent = `R${subtotal.toFixed(0)}`;
    document.getElementById('checkoutSavings').textContent = `-R${totalDiscount.toFixed(0)}`;
    document.getElementById('checkoutTotal').textContent = `R${total.toFixed(0)}`;
    
    // UPDATE COMPLETE PURCHASE BUTTON PRICE
    const completeButton = document.getElementById('proceedCheckoutBtn');
    if (completeButton) {
        completeButton.innerHTML = `<i class="bx bx-credit-card me-2"></i>Complete Purchase R${total.toFixed(0)}`;
    }
    
    // Show discount row if there's any discount
    const discountRow = document.querySelector('.summary-discount-row');
    if (totalDiscount > 0) {
        discountRow.style.display = 'flex';
        discountRow.classList.add('discount-highlight');
    } else {
        discountRow.style.display = 'none';
    }
    
    // Show referral discount info if applicable
    if (referralDiscount > 0) {
        const referralInfo = document.getElementById('referralDiscountInfo');
        if (referralInfo) {
            referralInfo.style.display = 'block';
            referralInfo.innerHTML = `<i class="bx bx-gift text-success me-2"></i>Referral Discount Applied: -R${referralDiscount.toFixed(0)}`;
        }
    }
}

function applyCheckoutPromoCode() {
    const promoCode = document.getElementById('checkoutPromoCode').value.trim();
    
    if (!promoCode) {
        alert('Please enter a promo code');
        return;
    }
    
    // Simulate promo code validation
    const validCodes = {
        'WELCOME10': 0.10,
        'STUDENT20': 0.20,
        'PANADITE15': 0.15
    };
    
    if (validCodes[promoCode.toUpperCase()]) {
        const discount = validCodes[promoCode.toUpperCase()];
        alert(`🎉 Promo code applied! You saved ${(discount * 100)}% on your order!`);
        
        // Store applied promo code
        localStorage.setItem('appliedCheckoutPromoCode', promoCode.toUpperCase());
        localStorage.setItem('checkoutPromoDiscount', discount);
        
        // Update display
        updateCheckoutSummary();
        
        // Disable promo input
        document.getElementById('checkoutPromoCode').disabled = true;
        document.querySelector('.promo-input-group button').textContent = 'Applied!';
        document.querySelector('.promo-input-group button').disabled = true;
        document.querySelector('.promo-input-group button').classList.remove('btn-outline-success');
        document.querySelector('.promo-input-group button').classList.add('btn-success');
    } else {
        alert('❌ Invalid promo code. Try: WELCOME10, STUDENT20, or PANADITE15');
    }
}

function proceedToPayment() {
    if (checkoutItems.length === 0) {
        alert('Your cart is empty!');
        return;
    }
    
    // Show loading state
    const checkoutBtn = document.getElementById('proceedCheckoutBtn');
    const originalHTML = checkoutBtn.innerHTML;
    checkoutBtn.innerHTML = '<i class="bx bx-loader-alt bx-spin me-2"></i>Processing...';
    checkoutBtn.disabled = true;
    
    // Create form to submit cart data to PayFast processing
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = 'process-checkout.php';
    form.style.display = 'none';
    
    // Add cart items as JSON
    const cartInput = document.createElement('input');
    cartInput.type = 'hidden';
    cartInput.name = 'cart_items';
    cartInput.value = JSON.stringify(checkoutItems);
    form.appendChild(cartInput);
    
    // Add promo code if applied
    const appliedPromoCode = localStorage.getItem('appliedCheckoutPromoCode');
    if (appliedPromoCode) {
        const promoInput = document.createElement('input');
        promoInput.type = 'hidden';
        promoInput.name = 'promo_code';
        promoInput.value = appliedPromoCode;
        form.appendChild(promoInput);
    }
    
    // Submit form to PayFast processing
    document.body.appendChild(form);
    form.submit();
}
</script>

<?php include_once 'components/footer.php'; ?>
