<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';
require_once '../includes/id_encryption.php';

// Require student role to access this page
requireRole('student');

// Include database connection
require_once '../config/db_connect.php';

// Get student ID
$student_id = $_SESSION['user_id'];

// Decrypt and validate course ID
$course_id = getSecureId('id');
if (!$course_id) {
    $_SESSION['error'] = "Invalid or corrupted course ID";
    header('Location: courses.php');
    exit();
}

// Check if student is enrolled in this course
$enrollQuery = "SELECT * FROM enrollments WHERE student_id = ? AND course_id = ?";
$enrollStmt = $conn->prepare($enrollQuery);
$enrollStmt->bind_param("ii", $student_id, $course_id);
$enrollStmt->execute();
$enrollResult = $enrollStmt->get_result();

$is_enrolled = ($enrollResult->num_rows > 0);

// Get course details
$courseQuery = "
    SELECT c.*, cat.category_name, 
           u.first_name, u.last_name,
           (SELECT COUNT(*) FROM enrollments WHERE course_id = c.course_id) as enrolled_students
    FROM courses c
    JOIN categories cat ON c.category_id = cat.category_id
    JOIN users u ON c.teacher_id = u.user_id
    WHERE c.course_id = ?
";

$courseStmt = $conn->prepare($courseQuery);
$courseStmt->bind_param("i", $course_id);
$courseStmt->execute();
$courseResult = $courseStmt->get_result();

if ($courseResult->num_rows == 0) {
    $_SESSION['error'] = "Course not found";
    header('Location: courses.php');
    exit();
}

$course = $courseResult->fetch_assoc();
$page_title = $course['title'] . " - Assignments";

// Include header
include_once 'components/header.php';
?>

<!-- Course Header -->
<div class="container-fluid pt-4">
    <div class="row">
        <div class="col-12">
            <div class="course-banner">
                <div class="course-banner-content">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h2 class="course-title"><?= htmlspecialchars($course['title']) ?></h2>
                            <p class="course-instructor">
                                <i class='bx bx-user'></i> Instructor: <?= htmlspecialchars($course['first_name'] . ' ' . $course['last_name']) ?>
                            </p>
                            <div class="mt-2">
                                <span class="course-category"><?= htmlspecialchars($course['category_name'] ?? 'General') ?></span>
                                <?php if (!empty($course['level'])): ?>
                                <span class="course-category"><?= htmlspecialchars(ucfirst($course['level'])) ?> Level</span>
                                <?php endif; ?>
                                <?php if (!empty($course['enrolled_students'])): ?>
                                <span class="course-category"><i class='bx bx-group'></i> <?= $course['enrolled_students'] ?> students</span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php if ($course && $is_enrolled) { ?>
<!-- Course Navigation Menu -->
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="course-nav">
                <ul class="nav nav-pills nav-fill">
                    <li class="nav-item">
                        <a href="course-detail.php?id=<?php echo $course_id; ?>" class="nav-link">
                            <i class='bx bx-book-open'></i> Course Content
                        </a>
                    </li>
                    <li class="nav-item">
                        <a href="course-materials.php?id=<?php echo $course_id; ?>" class="nav-link">
                            <i class='bx bx-file'></i> Materials
                        </a>
                    </li>
                    <li class="nav-item">
                        <a href="course-assignments.php?id=<?php echo $course_id; ?>" class="nav-link active">
                            <i class='bx bx-task'></i> Assignments
                        </a>
                    </li>
                    <li class="nav-item">
                        <a href="course-quizzes.php?id=<?php echo $course_id; ?>" class="nav-link">
                            <i class='bx bx-question-mark'></i> Quizzes
                        </a>
                    </li>
                    <li class="nav-item">
                        <a href="course-discussions.php?id=<?php echo $course_id; ?>" class="nav-link">
                            <i class='bx bx-chat'></i> Discussions
                        </a>
                    </li>
                </ul>
            </div>
        </div>
    </div>
</div>

<!-- Assignments Content -->
<div class="container-fluid mb-5">
    <div class="row">
        <div class="col-12">
            <div class="content-card">
                <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <h4 class="mb-0">Course Assignments</h4>
                    </div>
                </div>
                <div class="card-body">
                    <div class="filter-buttons mb-4">
                        <button type="button" class="filter-button active assignment-filter" data-filter="all">All Assignments</button>
                        <button type="button" class="filter-button assignment-filter" data-filter="pending">Pending</button>
                        <button type="button" class="filter-button assignment-filter" data-filter="submitted">Submitted</button>
                        <button type="button" class="filter-button assignment-filter" data-filter="graded">Graded</button>
                    </div>
                    
                    <?php
                    // Check if assignments table exists
                    $tableCheckQuery = "SHOW TABLES LIKE 'assignments'";
                    $tableCheckResult = $conn->query($tableCheckQuery);
                    $assignmentsTableExists = $tableCheckResult->num_rows > 0;
                    
                    // Check if submissions table exists
                    $submissionsTableCheckQuery = "SHOW TABLES LIKE 'submissions'";
                    $submissionsTableCheckResult = $conn->query($submissionsTableCheckQuery);
                    $submissionsTableExists = $submissionsTableCheckResult->num_rows > 0;
                    
                    if ($assignmentsTableExists) {
                        // Check if status column exists in assignments table
                        $statusColumnQuery = "SHOW COLUMNS FROM assignments LIKE 'status'";
                        $statusColumnResult = $conn->query($statusColumnQuery);
                        $statusColumnExists = $statusColumnResult->num_rows > 0;
                        
                        // Get course assignments with submission status
                        if ($submissionsTableExists) {
                            if ($statusColumnExists) {
                                $assignmentsQuery = "
                                    SELECT a.*, 
                                           COALESCE(s.status, 'not_submitted') as submission_status,
                                           s.submission_id, s.grade, s.feedback, s.submitted_at
                                    FROM assignments a
                                    LEFT JOIN submissions s ON a.assignment_id = s.assignment_id AND s.student_id = ?
                                    WHERE a.course_id = ? AND a.status = 'active'
                                    ORDER BY a.due_date ASC
                                ";
                            } else {
                                // If status column doesn't exist, don't include it in WHERE clause
                                $assignmentsQuery = "
                                    SELECT a.*, 
                                           COALESCE(s.status, 'not_submitted') as submission_status,
                                           s.submission_id, s.grade, s.feedback, s.submitted_at
                                    FROM assignments a
                                    LEFT JOIN submissions s ON a.assignment_id = s.assignment_id AND s.student_id = ?
                                    WHERE a.course_id = ?
                                    ORDER BY a.due_date ASC
                                ";
                            }
                        } else {
                            // If submissions table doesn't exist, just query assignments
                            if ($statusColumnExists) {
                                $assignmentsQuery = "
                                    SELECT a.*, 
                                           'not_submitted' as submission_status,
                                           NULL as submission_id, NULL as grade, NULL as feedback, NULL as submitted_at
                                    FROM assignments a
                                    WHERE a.course_id = ? AND a.status = 'active'
                                    ORDER BY a.due_date ASC
                                ";
                            } else {
                                // If status column doesn't exist, don't include it in WHERE clause
                                $assignmentsQuery = "
                                    SELECT a.*, 
                                           'not_submitted' as submission_status,
                                           NULL as submission_id, NULL as grade, NULL as feedback, NULL as submitted_at
                                    FROM assignments a
                                    WHERE a.course_id = ?
                                    ORDER BY a.due_date ASC
                                ";
                            }
                        }
                        
                        $assignmentStmt = $conn->prepare($assignmentsQuery);
                    if ($assignmentStmt) {
                        if ($submissionsTableExists) {
                            $assignmentStmt->bind_param("ii", $student_id, $course_id);
                        } else {
                            $assignmentStmt->bind_param("i", $course_id);
                        }
                        $assignmentStmt->execute();
                        $assignmentsResult = $assignmentStmt->get_result();
                        
                        if ($assignmentsResult->num_rows > 0) {
                            // Display assignments in cards
                            echo '<div class="row g-3">';
                            
                            while ($assignment = $assignmentsResult->fetch_assoc()) {
                                // Determine status and badge color
                                $status = $assignment['submission_status'];
                                $statusBadgeClass = 'bg-warning';
                                $statusText = 'Pending';
                                $filterCategory = 'pending';
                                
                                if ($status === 'submitted') {
                                    $statusBadgeClass = 'bg-info';
                                    $statusText = 'Submitted';
                                    $filterCategory = 'submitted';
                                } else if ($status === 'graded') {
                                    $statusBadgeClass = 'bg-success';
                                    $statusText = 'Graded';
                                    $filterCategory = 'graded';
                                } else if ($status === 'not_submitted') {
                                    // Check if past due date
                                    $dueDate = new DateTime($assignment['due_date']);
                                    $now = new DateTime();
                                    
                                    if ($now > $dueDate) {
                                        $statusBadgeClass = 'bg-danger';
                                        $statusText = 'Overdue';
                                    }
                                }
                                
                                // Format dates
                                $dueDateObj = new DateTime($assignment['due_date']);
                                $formattedDueDate = $dueDateObj->format('M d, Y - h:i A');
                                
                                $publishDateObj = new DateTime($assignment['created_at']);
                                $formattedPublishDate = $publishDateObj->format('M d, Y');
                                
                                $submittedDate = !empty($assignment['submitted_at']) ? 
                                    (new DateTime($assignment['submitted_at']))->format('M d, Y - h:i A') : '';
                                
                                // Calculate time remaining
                                $timeRemaining = '';
                                $now = new DateTime();
                                $interval = $now->diff($dueDateObj);
                                
                                if ($dueDateObj < $now) {
                                    $timeRemaining = '<span class="text-danger">Overdue</span>';
                                } else if ($interval->days > 0) {
                                    $timeRemaining = $interval->format('%d days, %h hours remaining');
                                } else if ($interval->h > 0) {
                                    $timeRemaining = $interval->format('%h hours, %i minutes remaining');
                                } else {
                                    $timeRemaining = $interval->format('%i minutes remaining');
                                }
                                
                                // Display the assignment card
                                echo "<div class='col-lg-6 col-xl-4 assignment-item' data-filter='{$filterCategory}'>";
                                echo "<div class='card h-100 shadow-sm'>";
                                echo "<div class='card-header d-flex justify-content-between align-items-center'>";
                                echo "<h5 class='mb-0 fs-6'>".htmlspecialchars($assignment['title'])."</h5>";
                                echo "<span class='badge {$statusBadgeClass}'>{$statusText}</span>";
                                echo "</div>";
                                
                                echo "<div class='card-body'>";
                                
                                // Points and due date
                                echo "<div class='d-flex justify-content-between mb-3'>";
                                echo "<div><small class='text-muted'>Points: <strong>{$assignment['points']}</strong></small></div>";
                                echo "<div><small class='text-muted'>Due: <strong>{$formattedDueDate}</strong></small></div>";
                                echo "</div>";
                                
                                // Description
                                $description = $assignment['description'];
                                $truncatedDesc = (strlen($description) > 100) ? substr($description, 0, 100) . '...' : $description;
                                echo "<p class='card-text'>".nl2br(htmlspecialchars($truncatedDesc))."</p>";
                                
                                // Time remaining
                                echo "<p class='small text-muted mb-0'><i class='bx bx-time me-1'></i> {$timeRemaining}</p>";
                                
                                // Assignment file download if available
                                if (!empty($assignment['file_path'])) {
                                    $fileExt = strtoupper(pathinfo($assignment['file_path'], PATHINFO_EXTENSION));
                                    echo "<div class='mt-2'><small class='text-muted'><i class='bx bx-file me-1'></i> {$fileExt} file attached</small></div>";
                                }
                                
                                echo "</div>";
                                
                                echo "<div class='card-footer d-flex justify-content-between bg-white'>";
                                
                                // Download button for assignment file (if available)
                                if (!empty($assignment['file_path'])) {
                                    echo "<a href='download_assignment.php?id={$assignment['assignment_id']}' class='btn btn-outline-secondary btn-sm'><i class='bx bx-download me-1'></i> Download</a>";
                                } else {
                                    echo "<span></span>"; // Empty span for flexbox alignment
                                }
                                
                                echo "<div>";
                                
                                if ($status === 'not_submitted') {
                                    echo "<a href='assignment_view.php?id={$assignment['assignment_id']}' class='btn btn-primary btn-sm'>Submit Assignment</a>";
                                } else if ($status === 'submitted') {
                                    echo "<a href='assignment_view.php?id={$assignment['assignment_id']}' class='btn btn-info btn-sm'>View Submission</a>";
                                } else { // graded
                                    echo "<a href='assignment_view.php?id={$assignment['assignment_id']}' class='btn btn-success btn-sm'>View Feedback</a>";
                                }
                                
                                echo "</div>"; // Close action buttons div
                                echo "</div>"; // Close card footer
                                echo "</div>";
                                echo "</div>";
                            }
                            
                            echo '</div>';
                        } else {
                            echo '<div class="empty-state">';
                            echo '<i class="bx bx-task"></i>';
                            echo '<h5 class="mt-3">No assignments yet</h5>';
                            echo '<p class="text-muted">No assignments have been added to this course yet.</p>';
                            echo '</div>';
                        }
                    }
                    ?>
                    
                    <script>
                    // Filter assignments by status
                    document.addEventListener('DOMContentLoaded', function() {
                        const filterButtons = document.querySelectorAll('.assignment-filter');
                        const assignmentItems = document.querySelectorAll('.assignment-item');
                        
                        filterButtons.forEach(button => {
                            button.addEventListener('click', function() {
                                const filterValue = this.getAttribute('data-filter');
                                
                                // Toggle active class
                                filterButtons.forEach(btn => btn.classList.remove('active'));
                                this.classList.add('active');
                                
                                // Filter items
                                assignmentItems.forEach(item => {
                                    if (filterValue === 'all' || item.getAttribute('data-filter') === filterValue) {
                                        item.style.display = '';
                                    } else {
                                        item.style.display = 'none';
                                    }
                                });
                            });
                        });
                    });
                    </script>
                </div>
            </div>
        </div>
    </div>
</div>
<?php 
    } // Close if ($assignmentsTableExists)
    else {
        // Content for non-enrolled users
        ?>
        <div class="container-fluid mb-5">
            <div class="row">
                <div class="col-12">
                    <div class="card shadow-sm">
                        <div class="card-body p-4">
                            <div class="alert alert-warning">
                                <h5><i class="bx bx-lock me-2"></i> Course assignments are only available to enrolled students</h5>
                                <p>To access this content, please enroll in this course.</p>
                                <a href="course-enrollment.php?id=<?= $course_id ?>" class="btn btn-primary mt-2">Enroll Now</a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
<?php
    } // Close else
} // Close if ($course && $is_enrolled)

// Include footer
include_once 'components/footer.php';
?>
