<?php
/**
 * Student Course Catalog - Browse All Available Courses
 * Panadite Academy
 * 
 * Clean, modern course browsing interface with dynamic search
 */

// Authentication and session
require_once '../auth/functions.php';
require_once '../includes/id_encryption.php';
requireRole('student');

// Initialize variables
$availableCourses = [];
$userId = $_SESSION['user_id'] ?? 0;

// Database operations with comprehensive error handling
try {
    require_once '../config/db_connect.php';
    
    if (isset($conn) && $userId) {
        // Fetch all available courses with comprehensive data
        $coursesQuery = "
            SELECT c.course_id as id, c.title, c.description, c.thumbnail, c.price,
                   u.first_name, u.last_name, 'General' as category,
                   (SELECT COUNT(*) FROM enrollments WHERE course_id = c.course_id) as total_students,
                   COALESCE(4.5, 4.5) as rating,
                   COALESCE(c.duration, (SELECT COUNT(*) FROM videos v WHERE v.course_id = c.course_id AND v.is_published = 1)) as duration, 
                   'Beginner' as level,
                   EXISTS(SELECT 1 FROM enrollments WHERE student_id = ? AND course_id = c.course_id) as is_enrolled
            FROM courses c
            JOIN users u ON c.teacher_id = u.user_id
            WHERE c.is_published = 1
            ORDER BY c.created_at DESC
        ";
        
        $stmt = $conn->prepare($coursesQuery);
        if (!$stmt) {
            error_log("Courses SQL prepare failed: " . $conn->error);
            throw new Exception("Courses database query preparation failed");
        }
        
        $stmt->bind_param("i", $userId);
        if (!$stmt->execute()) {
            error_log("Courses SQL execute failed: " . $stmt->error);
            throw new Exception("Courses database query execution failed");
        }
        
        $result = $stmt->get_result();
        
        // FETCH USER'S AVAILABLE DISCOUNTS
        $available_discounts = [];
        $discountsQuery = "SELECT * FROM student_referral_discounts 
                          WHERE student_id = ? AND is_used = 0 AND expires_at > NOW() 
                          ORDER BY discount_percentage DESC LIMIT 1";
        $discountStmt = $conn->prepare($discountsQuery);
        if ($discountStmt) {
            $discountStmt->bind_param("i", $userId);
            $discountStmt->execute();
            $discountResult = $discountStmt->get_result();
            if ($discountResult->num_rows > 0) {
                $available_discounts = $discountResult->fetch_assoc();
            }
        }
        
        while ($row = $result->fetch_assoc()) {
            $original_price = $row['price'];
            $discounted_price = $original_price;
            $has_discount = false;
            
            // APPLY DISCOUNT IF AVAILABLE
            if (!empty($available_discounts) && $original_price > 0) {
                $discount_amount = min(
                    ($original_price * $available_discounts['discount_percentage'] / 100),
                    $available_discounts['discount_amount']
                );
                $discounted_price = $original_price - $discount_amount;
                $has_discount = true;
            }
            
            $availableCourses[] = [
                'id' => $row['id'],
                'title' => $row['title'],
                'description' => $row['description'],
                'thumbnail' => $row['thumbnail'] ?: 'default-course.jpg',
                'price' => $original_price,
                'discounted_price' => $discounted_price,
                'has_discount' => $has_discount,
                'discount_percentage' => $has_discount ? $available_discounts['discount_percentage'] : 0,
                'savings' => $has_discount ? ($original_price - $discounted_price) : 0,
                'instructor' => trim($row['first_name'] . ' ' . $row['last_name']),
                'category' => $row['category'],
                'students' => $row['total_students'],
                'rating' => floatval($row['rating']),
                'duration' => $row['duration'],
                'level' => $row['level'],
                'is_enrolled' => (bool)$row['is_enrolled']
            ];
        }
        
        // 🎲 RANDOM SHUFFLE: Display courses in random order each page load
        if (!empty($availableCourses)) {
            shuffle($availableCourses);
        }
        
    } else {
        error_log("Database connection failed or user not authenticated");
        throw new Exception("Database connection failed");
    }
    
} catch (Exception $e) {
    error_log("Course catalog error: " . $e->getMessage());
    $availableCourses = [];
}

// Set current page for navigation
$currentPage = 'course-catalog';
include_once 'components/header.php';

?>

<!-- Clean Course Catalog Page Content -->
<div class="page-content">
    
    <!-- Clean Page Header with Floating Icons -->
    <div class="catalog-header">
        <div class="header-content">
            <h1><i class='bx bx-compass'></i> Browse Courses</h1>
            <p class="subtitle">Discover amazing courses and expand your knowledge</p>
            
            <!-- Floating Icons Animation -->
            <div class="floating-icons">
                <div class="floating-icon icon-1"><i class='bx bx-book-open'></i></div>
                <div class="floating-icon icon-2"><i class='bx bx-bulb'></i></div>
                <div class="floating-icon icon-3"><i class='bx bx-code-alt'></i></div>
                <div class="floating-icon icon-4"><i class='bx bx-brain'></i></div>
                <div class="floating-icon icon-5"><i class='bx bx-award'></i></div>
                <div class="floating-icon icon-6"><i class='bx bx-graduation'></i></div>
                <div class="floating-icon icon-7"><i class='bx bx-pencil'></i></div>
                <div class="floating-icon icon-8"><i class='bx bx-desktop'></i></div>
            </div>
        </div>
    </div>

    <!-- Dynamic Search Section -->
    <div class="search-section">
        <div class="search-container">
            <div class="search-box">
                <i class='bx bx-search'></i>
                <input type="text" id="courseSearch" placeholder="Search by course name, instructor, or topic..." autocomplete="off">
                <button type="button" id="clearSearch" class="clear-btn" style="display: none;">
                    <i class='bx bx-x'></i>
                </button>
            </div>
        </div>
        
        <!-- Search Results Info -->
        <div class="search-info">
            <span id="resultsCount"><?= count($availableCourses) ?> courses available</span>
            <div class="filter-options">
                <select id="levelFilter" class="filter-select">
                    <option value="">All Levels</option>
                    <option value="Beginner">Beginner</option>
                    <option value="Intermediate">Intermediate</option>
                    <option value="Advanced">Advanced</option>
                </select>
                <select id="priceFilter" class="filter-select">
                    <option value="">All Prices</option>
                    <option value="free">Free Courses</option>
                    <option value="paid">Paid Courses</option>
                </select>
            </div>
        </div>
    </div>

    <!-- Courses Grid -->
    <div class="courses-container">
        <?php if (empty($availableCourses)): ?>
            <div class="empty-state">
                <div class="empty-illustration">
                    <i class='bx bx-book-open'></i>
                </div>
                <h3>No Courses Available</h3>
                <p>Check back later for new courses and learning opportunities</p>
            </div>
        <?php else: ?>
            <div class="courses-grid" id="coursesGrid">
                <?php foreach ($availableCourses as $index => $course): ?>
                    <div class="course-card" 
                         data-course-id="<?= IdEncryption::encrypt($course['id']) ?>"
                         data-title="<?= strtolower($course['title']) ?>"
                         data-instructor="<?= strtolower($course['instructor']) ?>"
                         data-description="<?= strtolower($course['description']) ?>"
                         data-level="<?= strtolower($course['level']) ?>"
                         data-price="<?= $course['price'] <= 0 ? 'free' : 'paid' ?>"
                         style="animation-delay: <?= $index * 0.1 ?>s">
                        
                        <!-- Course Image -->
                        <div class="course-image-container">
                            <img src="<?= htmlspecialchars($course['thumbnail']) ?>" 
                                 alt="<?= htmlspecialchars($course['title']) ?>" 
                                 class="course-image"
                                 onerror="this.src='./assets/img/courses/default.jpg'">
                            
                            <!-- Price Badge -->
                            <?php if ($course['price'] <= 0): ?>
                                <div class="course-badge free-badge">Free</div>
                            <?php else: ?>
                                <?php if ($course['has_discount']): ?>
                                    <div class="course-badge price-badge discount-active">
                                        <span class="discount-percent">-<?= $course['discount_percentage'] ?>%</span>
                                        <span class="original-price">R<?= number_format($course['price'], 0) ?></span>
                                        <span class="discounted-price">R<?= number_format($course['discounted_price'], 0) ?></span>
                                    </div>
                                <?php else: ?>
                                    <div class="course-badge price-badge">R<?= number_format($course['price'], 0) ?></div>
                                <?php endif; ?>
                            <?php endif; ?>
                            
                            <!-- Enrollment Status -->
                            <?php if ($course['is_enrolled']): ?>
                                <div class="enrollment-badge">
                                    <i class='bx bx-check-circle'></i> Enrolled
                                </div>
                            <?php endif; ?>
                        </div>

                        <!-- Course Content -->
                        <div class="course-content">
                            <!-- Course Title -->
                            <h3 class="course-title" title="<?= htmlspecialchars($course['title']) ?>">
                                <?= htmlspecialchars($course['title']) ?>
                            </h3>
                            
                            <!-- Course Description -->
                            <p class="course-description">
                                <?= htmlspecialchars(substr($course['description'], 0, 60)) ?>...
                            </p>
                            
                            <!-- Instructor Info -->
                            <div class="instructor-info">
                                <i class='bx bx-user'></i>
                                <span><?= htmlspecialchars($course['instructor']) ?></span>
                            </div>
                            
                            <!-- Course Meta -->
                            <div class="course-meta">
                                <?php if ($course['duration'] && $course['duration'] > 0): ?>
                                <div class="meta-item">
                                    <i class='bx bx-time'></i>
                                    <span><?= $course['duration'] ?> hours</span>
                                </div>
                                <?php endif; ?>
                                <div class="meta-item">
                                    <i class='bx bx-signal-2'></i>
                                    <span><?= $course['level'] ?></span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Course Actions -->
                        <div class="course-actions">
                            <?php if ($course['is_enrolled']): ?>
                                <button class="action-btn continue-btn" onclick="continueCourse('<?= IdEncryption::encrypt($course['id']) ?>')" title="Continue Learning">
                                    <i class='bx bx-play-circle'></i>
                                </button>
                            <?php else: ?>
                                <button class="action-btn cart-btn" onclick="addToCart('<?= IdEncryption::encrypt($course['id']) ?>', this)" title="Add to Cart">
                                    <i class='bx bx-cart-add'></i>
                                </button>
                                <?php if ($course['price'] <= 0): ?>
                                    <button class="action-btn enroll-btn" onclick="enrollInCourse('<?= IdEncryption::encrypt($course['id']) ?>')" title="Enroll Now - FREE">
                                        <i class='bx bx-plus'></i>
                                    </button>
                                <?php else: ?>
                                    <button class="action-btn enroll-btn" 
                                            onclick="enrollInCourse('<?= IdEncryption::encrypt($course['id']) ?>', <?= $course['has_discount'] ? $course['discounted_price'] : $course['price'] ?>)" 
                                            title="<?= $course['has_discount'] ? 'Enroll Now - R' . number_format($course['discounted_price'], 0) . ' (Save R' . number_format($course['savings'], 0) . '!)' : 'Enroll - R' . number_format($course['price'], 0) ?>">
                                        <i class='bx bx-credit-card'></i>
                                    </button>
                                <?php endif; ?>
                            <?php endif; ?>
                            
                            <a href="<?= generateSecureUrl('course-details.php', $course['id']) ?>" class="action-btn view-btn" title="View Course Details">
                                <i class='bx bx-show'></i>
                            </a>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Modern Course Catalog CSS -->
<style>
/* Clean Page Header */
.catalog-header {
    text-align: center;
    padding: 2rem 0 1rem 0;
    margin-bottom: 2rem;
    position: relative;
    overflow: hidden;
}

.header-content h1 {
    font-size: 2.5rem;
    font-weight: 700;
    color: #1f2937;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.75rem;
}

.header-content h1 i {
    font-size: 2.2rem;
    color: #3b82f6;
}

.subtitle {
    font-size: 1.1rem;
    color: #6b7280;
    font-weight: 400;
    margin: 0;
}

/* Floating Icons Animation */
.floating-icons {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    pointer-events: none;
    z-index: 1;
}

.floating-icon {
    position: absolute;
    width: 20px;
    height: 20px;
    border-radius: 50%;
    background: rgba(59, 130, 246, 0.1);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 12px;
    color: #3b82f6;
    opacity: 0.7;
    animation: float 4s ease-in-out infinite;
}

.floating-icon i {
    font-size: 10px;
}

/* Individual Icon Positions and Delays */
.icon-1 {
    top: 20%;
    left: 10%;
    animation-delay: 0s;
    animation-duration: 6s;
}

.icon-2 {
    top: 30%;
    right: 15%;
    animation-delay: -1s;
    animation-duration: 5s;
}

.icon-3 {
    top: 60%;
    left: 20%;
    animation-delay: -2s;
    animation-duration: 7s;
}

.icon-4 {
    top: 15%;
    right: 30%;
    animation-delay: -3s;
    animation-duration: 4s;
}

.icon-5 {
    top: 70%;
    right: 10%;
    animation-delay: -4s;
    animation-duration: 6s;
}

.icon-6 {
    top: 25%;
    left: 70%;
    animation-delay: -5s;
    animation-duration: 5s;
}

.icon-7 {
    top: 80%;
    left: 40%;
    animation-delay: -6s;
    animation-duration: 7s;
}

.icon-8 {
    top: 40%;
    right: 40%;
    animation-delay: -7s;
    animation-duration: 4s;
}

/* Float Animation Keyframes */
@keyframes float {
    0%, 100% {
        transform: translateY(0px) translateX(0px) rotate(0deg);
        opacity: 0.4;
    }
    25% {
        transform: translateY(-10px) translateX(5px) rotate(90deg);
        opacity: 0.7;
    }
    50% {
        transform: translateY(-5px) translateX(-8px) rotate(180deg);
        opacity: 0.5;
    }
    75% {
        transform: translateY(-15px) translateX(3px) rotate(270deg);
        opacity: 0.8;
    }
}

/* Hover Effect on Header Content */
.header-content:hover .floating-icon {
    animation-play-state: paused;
    opacity: 0.9;
    transform: scale(1.2);
    transition: all 0.3s ease;
}

/* Dynamic Search Section */
.search-section {
    background: #ffffff;
    padding: 2rem;
    margin-bottom: 2rem;
    border-radius: 15px;
    border: 1px solid #e5e7eb;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
}

.search-container {
    max-width: 600px;
    margin: 0 auto 1.5rem auto;
}

.search-box {
    position: relative;
    background: #f8fafc;
    border-radius: 50px;
    border: 2px solid #e2e8f0;
    transition: all 0.3s ease;
    overflow: hidden;
}

.search-box:focus-within {
    border-color: #3b82f6;
    background: #ffffff;
    box-shadow: 0 0 0 4px rgba(59, 130, 246, 0.1);
}

.search-box i {
    position: absolute;
    left: 20px;
    top: 50%;
    transform: translateY(-50%);
    color: #6b7280;
    font-size: 1.2rem;
    z-index: 2;
}

.search-box input {
    width: 100%;
    padding: 18px 60px 18px 50px;
    border: none;
    background: transparent;
    font-size: 1.1rem;
    color: #1f2937;
    outline: none;
}

.search-box input::placeholder {
    color: #9ca3af;
    font-weight: 400;
}

.clear-btn {
    position: absolute;
    right: 15px;
    top: 50%;
    transform: translateY(-50%);
    background: #ef4444;
    color: white;
    border: none;
    border-radius: 50%;
    width: 30px;
    height: 30px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.2s ease;
    z-index: 2;
}

.clear-btn:hover {
    background: #dc2626;
    transform: translateY(-50%) scale(1.1);
}

/* Search Info */
.search-info {
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1rem;
}

#resultsCount {
    color: #6b7280;
    font-weight: 500;
    font-size: 0.95rem;
}

.filter-options {
    display: flex;
    gap: 1rem;
}

.filter-select {
    padding: 8px 16px;
    border: 1px solid #d1d5db;
    border-radius: 8px;
    background: white;
    color: #374151;
    font-size: 0.9rem;
    cursor: pointer;
    transition: all 0.2s ease;
}

.filter-select:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

/* Courses Grid */
.courses-container {
    /* Dynamic Courses Grid - Force 5 Cards Per Row */
    max-width: 100% !important;
    width: 100% !important;
}

.courses-container .courses-grid,
.courses-grid,
#coursesGrid {
    display: grid !important;
    grid-template-columns: repeat(5, 1fr) !important;
    grid-template-rows: auto !important;
    gap: 0.8rem !important;
    padding: 1rem 0 !important;
    max-width: 100% !important;
    width: 100% !important;
    margin: 0 !important;
}

/* Force mobile layout with higher specificity */
body .courses-container .courses-grid,
body .courses-grid,
body #coursesGrid {
    display: grid !important;
}

/* Responsive Grid - Adjusted Breakpoints */
@media (max-width: 1200px) {
    .courses-grid {
        grid-template-columns: repeat(4, 1fr) !important;
        gap: 1rem !important;
    }
}

@media (max-width: 900px) {
    .courses-grid {
        grid-template-columns: repeat(3, 1fr) !important;
        gap: 1rem !important;
    }
}

/* Mobile: FORCE 2 cards per row with maximum specificity */
@media screen and (max-width: 768px) {
    body .courses-container .courses-grid,
    body .courses-grid,
    body #coursesGrid,
    .courses-container .courses-grid,
    .courses-grid,
    #coursesGrid {
        display: grid !important;
        grid-template-columns: repeat(2, 1fr) !important;
        grid-template-rows: auto !important;
        gap: 0.75rem !important;
        padding: 1rem 0.5rem !important;
        width: 100% !important;
        max-width: 100% !important;
        margin: 0 auto !important;
        box-sizing: border-box !important;
    }
    
    body .course-card,
    .course-card {
        height: 280px !important;
        max-width: 100% !important;
        width: 100% !important;
        min-width: 0 !important;
        flex: none !important;
        display: flex !important;
        flex-direction: column !important;
        box-sizing: border-box !important;
    }
    
    body .course-content,
    .course-content {
        padding: 1rem 0.75rem !important;
        flex: 1 !important;
        display: flex !important;
        flex-direction: column !important;
    }
    
    body .course-title,
    .course-title {
        font-size: 0.9rem !important;
        line-height: 1.3 !important;
        margin-bottom: 0.5rem !important;
    }
    
    body .course-description,
    .course-description {
        font-size: 0.8rem !important;
        line-height: 1.4 !important;
        margin-bottom: 0.75rem !important;
    }
    
    body .course-meta,
    .course-meta {
        gap: 0.5rem !important;
        margin-top: auto !important;
    }
    
    body .meta-item,
    .meta-item {
        font-size: 0.75rem !important;
    }
}

/* Very Small Screens: FORCE 2 cards but smaller with maximum specificity */
@media screen and (max-width: 480px) {
    body .courses-container .courses-grid,
    body .courses-grid,
    body #coursesGrid,
    .courses-container .courses-grid,
    .courses-grid,
    #coursesGrid {
        display: grid !important;
        grid-template-columns: repeat(2, 1fr) !important;
        grid-template-rows: auto !important;
        gap: 0.5rem !important;
        padding: 1rem 0.25rem !important;
        width: 100% !important;
        max-width: 100% !important;
        margin: 0 auto !important;
        box-sizing: border-box !important;
    }
    
    body .course-card,
    .course-card {
        height: 260px !important;
        max-width: 100% !important;
        width: 100% !important;
        min-width: 0 !important;
        flex: none !important;
        display: flex !important;
        flex-direction: column !important;
        box-sizing: border-box !important;
    }
    
    body .course-image-container,
    .course-image-container {
        height: 100px !important;
        width: 100% !important;
        flex-shrink: 0 !important;
    }
    
    body .course-content,
    .course-content {
        padding: 0.75rem 0.5rem !important;
        flex: 1 !important;
        display: flex !important;
        flex-direction: column !important;
    }
    
    body .course-title,
    .course-title {
        font-size: 0.85rem !important;
        line-height: 1.2 !important;
        margin-bottom: 0.4rem !important;
        overflow: hidden !important;
        text-overflow: ellipsis !important;
        display: -webkit-box !important;
        -webkit-line-clamp: 2 !important;
        -webkit-box-orient: vertical !important;
    }
    
    body .course-description,
    .course-description {
        font-size: 0.75rem !important;
        margin-bottom: 0.5rem !important;
        line-height: 1.3 !important;
        overflow: hidden !important;
        text-overflow: ellipsis !important;
        display: -webkit-box !important;
        -webkit-line-clamp: 2 !important;
        -webkit-box-orient: vertical !important;
    }
    
    body .instructor-info,
    .instructor-info {
        font-size: 0.75rem !important;
        margin-bottom: 0.5rem !important;
    }
    
    body .course-meta,
    .course-meta {
        gap: 0.25rem !important;
        margin-top: auto !important;
        flex-wrap: wrap !important;
    }
    
    body .meta-item,
    .meta-item {
        font-size: 0.7rem !important;
        white-space: nowrap !important;
    }
    
    body .course-badge,
    .course-badge {
        padding: 0.2rem 0.4rem !important;
        font-size: 0.7rem !important;
        top: 0.5rem !important;
        right: 0.5rem !important;
    }
}

/* FINAL MOBILE OVERRIDE - MAXIMUM PRIORITY */
@media screen and (max-width: 768px) {
    html body .page-content .courses-container .courses-grid,
    html body .page-content .courses-grid,
    html body .page-content #coursesGrid,
    html body .courses-container .courses-grid,
    html body .courses-grid,
    html body #coursesGrid {
        display: grid !important;
        grid-template-columns: 1fr 1fr !important;
        grid-gap: 0.75rem !important;
        gap: 0.75rem !important;
        padding: 1rem 0.5rem !important;
        width: 100% !important;
        max-width: 100% !important;
        margin: 0 !important;
        box-sizing: border-box !important;
        justify-content: space-between !important;
    }
    
    html body .course-card {
        width: 100% !important;
        max-width: 100% !important;
        min-width: 0 !important;
        height: 320px !important;
        flex: none !important;
        display: flex !important;
        flex-direction: column !important;
        box-sizing: border-box !important;
        margin: 0 !important;
    }
    
    /* Course Image Mobile Fixes */
    html body .course-image-container,
    body .course-image-container {
        width: 100% !important;
        height: 120px !important;
        overflow: hidden !important;
        position: relative !important;
        flex-shrink: 0 !important;
        border-radius: 10px 10px 0 0 !important;
    }
    
    html body .course-image,
    body .course-image {
        width: 100% !important;
        height: 100% !important;
        object-fit: cover !important;
        display: block !important;
    }
    
    /* Course Content Mobile */
    html body .course-content,
    body .course-content {
        flex: 1 !important;
        padding: 0.75rem !important;
        display: flex !important;
        flex-direction: column !important;
        justify-content: space-between !important;
    }
    
    /* Course Actions Mobile - FORCE horizontal flex row */
    html body .course-actions,
    body .course-actions {
        display: flex !important;
        flex-direction: row !important;
        justify-content: center !important;
        align-items: center !important;
        padding: 0.5rem 0.75rem !important;
        border-top: 1px solid #f1f5f9 !important;
        background: #fafafa !important;
        margin-top: auto !important;
        gap: 0.4rem !important;
        flex-wrap: nowrap !important;
        width: 100% !important;
        box-sizing: border-box !important;
    }
    
    html body .action-btn,
    body .action-btn {
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        width: 30px !important;
        height: 30px !important;
        border-radius: 6px !important;
        border: none !important;
        cursor: pointer !important;
        font-size: 13px !important;
        transition: all 0.2s ease !important;
        flex-shrink: 0 !important;
        flex-grow: 1 !important;
        max-width: 40px !important;
    }
    
    /* Specific button styles */
    html body .cart-btn,
    body .cart-btn {
        background: #3b82f6 !important;
        color: white !important;
    }
    
    html body .enroll-btn,
    body .enroll-btn {
        background: #10b981 !important;
        color: white !important;
    }
    
    html body .continue-btn,
    body .continue-btn {
        background: #8b5cf6 !important;
        color: white !important;
    }
    
    html body .view-btn,
    body .view-btn {
        background: #6b7280 !important;
        color: white !important;
        text-decoration: none !important;
    }
    
    /* Course badges mobile */
    html body .course-badge,
    body .course-badge {
        position: absolute !important;
        top: 0.5rem !important;
        right: 0.5rem !important;
        z-index: 2 !important;
        font-size: 0.75rem !important;
        padding: 0.25rem 0.5rem !important;
        border-radius: 4px !important;
        font-weight: 600 !important;
    }
    
    html body .enrollment-badge,
    body .enrollment-badge {
        position: absolute !important;
        top: 0.5rem !important;
        left: 0.5rem !important;
        background: rgba(16, 185, 129, 0.9) !important;
        color: white !important;
        padding: 0.25rem 0.5rem !important;
        border-radius: 4px !important;
        font-size: 0.7rem !important;
        z-index: 2 !important;
    }
}

@media screen and (max-width: 480px) {
    html body .page-content .courses-container .courses-grid,
    html body .page-content .courses-grid,
    html body .page-content #coursesGrid,
    html body .courses-container .courses-grid,
    html body .courses-grid,
    html body #coursesGrid {
        display: grid !important;
        grid-template-columns: 1fr 1fr !important;
        grid-gap: 0.5rem !important;
        gap: 0.5rem !important;
        padding: 1rem 0.25rem !important;
        width: 100% !important;
        max-width: 100% !important;
        margin: 0 !important;
        box-sizing: border-box !important;
        justify-content: space-between !important;
    }
    
    html body .course-card {
        width: 100% !important;
        max-width: 100% !important;
        min-width: 0 !important;
        height: 300px !important;
        flex: none !important;
        display: flex !important;
        flex-direction: column !important;
        box-sizing: border-box !important;
        margin: 0 !important;
    }
    
    /* Small Mobile Course Image Fixes */
    html body .course-image-container,
    body .course-image-container {
        width: 100% !important;
        height: 100px !important;
        overflow: hidden !important;
        position: relative !important;
        flex-shrink: 0 !important;
        border-radius: 10px 10px 0 0 !important;
    }
    
    html body .course-image,
    body .course-image {
        width: 100% !important;
        height: 100% !important;
        object-fit: cover !important;
        display: block !important;
    }
    
    /* Small Mobile Course Content */
    html body .course-content,
    body .course-content {
        flex: 1 !important;
        padding: 0.5rem !important;
        display: flex !important;
        flex-direction: column !important;
        justify-content: space-between !important;
    }
    
    /* Small Mobile Course Actions - FORCE horizontal flex row */
    html body .course-actions,
    body .course-actions {
        display: flex !important;
        flex-direction: row !important;
        justify-content: center !important;
        align-items: center !important;
        padding: 0.4rem 0.5rem !important;
        border-top: 1px solid #f1f5f9 !important;
        background: #fafafa !important;
        margin-top: auto !important;
        gap: 0.3rem !important;
        flex-wrap: nowrap !important;
        width: 100% !important;
        box-sizing: border-box !important;
    }
    
    html body .action-btn,
    body .action-btn {
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        width: 26px !important;
        height: 26px !important;
        border-radius: 5px !important;
        border: none !important;
        cursor: pointer !important;
        font-size: 11px !important;
        transition: all 0.2s ease !important;
        flex-shrink: 0 !important;
        flex-grow: 1 !important;
        max-width: 35px !important;
    }
    
    /* Small Mobile Button Colors */
    html body .cart-btn,
    body .cart-btn {
        background: #3b82f6 !important;
        color: white !important;
    }
    
    html body .enroll-btn,
    body .enroll-btn {
        background: #10b981 !important;
        color: white !important;
    }
    
    html body .continue-btn,
    body .continue-btn {
        background: #8b5cf6 !important;
        color: white !important;
    }
    
    html body .view-btn,
    body .view-btn {
        background: #6b7280 !important;
        color: white !important;
        text-decoration: none !important;
    }
    
    /* Small Mobile Badges */
    html body .course-badge,
    body .course-badge {
        position: absolute !important;
        top: 0.4rem !important;
        right: 0.4rem !important;
        z-index: 2 !important;
        font-size: 0.7rem !important;
        padding: 0.2rem 0.4rem !important;
        border-radius: 3px !important;
        font-weight: 600 !important;
    }
    
    html body .enrollment-badge,
    body .enrollment-badge {
        position: absolute !important;
        top: 0.4rem !important;
        left: 0.4rem !important;
        background: rgba(16, 185, 129, 0.9) !important;
        color: white !important;
        padding: 0.2rem 0.4rem !important;
        border-radius: 3px !important;
        font-size: 0.65rem !important;
        z-index: 2 !important;
    }
}

.course-card {
    background: #ffffff;
    border-radius: 10px;
    border: 1px solid #e5e7eb;
    overflow: hidden;
    transition: all 0.3s ease;
    animation: slideUp 0.6s ease forwards;
    box-shadow: 0 2px 6px rgba(0, 0, 0, 0.05);
    display: flex;
    flex-direction: column;
    height: 260px;
    max-width: 100%;
}

.course-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.12);
    border-color: #cbd5e1;
}

/* Course Image */
.course-image-container {
    position: relative;
    width: 100%;
    height: 120px;
    overflow: hidden;
}

.course-image {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.3s ease;
}

.course-card:hover .course-image {
    transform: scale(1.05);
}

/* Course badges */
.course-badge {
    position: absolute;
    top: 0.75rem;
    right: 0.75rem;
    padding: 0.25rem 0.5rem;
    border-radius: 0.375rem;
    font-size: 0.75rem;
    font-weight: 600;
    z-index: 2;
}

.free-badge {
    background: #10b981;
    color: white;
}

.price-badge {
    background: rgba(59, 130, 246, 0.9);
    color: white;
}

/* Enhanced Discount Badge */
.price-badge.discount-active {
    background: linear-gradient(135deg, #ef4444, #dc2626);
    padding: 8px 12px;
    min-width: 70px;
    text-align: center;
    animation: discountPulse 3s infinite;
}

.discount-percent {
    display: block;
    font-size: 10px;
    font-weight: bold;
    color: #fef3c7;
    margin-bottom: 1px;
    line-height: 1;
}

.original-price {
    display: block;
    font-size: 9px;
    text-decoration: line-through;
    opacity: 0.8;
    line-height: 1;
    margin-bottom: 1px;
}

.discounted-price {
    display: block;
    font-size: 11px;
    font-weight: bold;
    color: white;
    line-height: 1;
}

@keyframes discountPulse {
    0%, 100% { 
        transform: scale(1);
        box-shadow: 0 2px 4px rgba(239, 68, 68, 0.3);
    }
    50% { 
        transform: scale(1.05);
        box-shadow: 0 4px 8px rgba(239, 68, 68, 0.5);
    }
}

.enrollment-badge {
    position: absolute;
    top: 15px;
    left: 15px;
    background: rgba(34, 197, 94, 0.95);
    color: white;
    padding: 6px 10px;
    border-radius: 15px;
    font-size: 11px;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 4px;
    backdrop-filter: blur(10px);
    border: 1px solid rgba(255, 255, 255, 0.2);
}

/* Course Content */
.course-content {
    padding: 12px;
    flex: 1;
    display: flex;
    flex-direction: column;
    gap: 6px;
}

.course-title {
    font-size: 0.95rem;
    font-weight: 600;
    color: #1f2937;
    line-height: 1.3;
    margin: 0;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    cursor: pointer;
    height: 1.2em;
}

.course-title:hover {
    color: #3b82f6;
}

.course-description {
    font-size: 0.75rem;
    color: #6b7280;
    line-height: 1.3;
    margin: 0;
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
    height: 2.6em;
}

.instructor-info {
    display: flex;
    align-items: center;
    gap: 6px;
    font-size: 0.8rem;
    color: #374151;
    font-weight: 500;
}

.instructor-info i {
    font-size: 1rem;
    color: #6b7280;
}

.course-meta {
    display: flex;
    gap: 4px;
    font-size: 0.75rem;
    color: #6b7280;
}

.meta-item i {
    font-size: 1rem;
}

.course-rating {
    display: flex;
    align-items: center;
    gap: 8px;
}

.stars {
    display: flex;
    gap: 2px;
}

.stars i {
    font-size: 1rem;
    color: #fbbf24;
}

.rating-text {
    font-size: 0.85rem;
    color: #6b7280;
    font-weight: 500;
}

/* Prominent Discount Badge */
.discount-badge {
    position: absolute;
    top: 10px;
    right: 10px;
    background: linear-gradient(135deg, #ef4444, #dc2626);
    color: white;
    padding: 8px 12px;
    border-radius: 8px;
    font-size: 0.75rem;
    font-weight: bold;
    text-align: center;
    box-shadow: 0 2px 8px rgba(239, 68, 68, 0.3);
    z-index: 10;
    animation: pulseDiscount 2s infinite;
}

.discount-percent {
    display: block;
    font-size: 0.9rem;
    line-height: 1;
    margin-bottom: 2px;
}

.discount-text {
    display: block;
    font-size: 0.6rem;
    opacity: 0.9;
    line-height: 1;
}

@keyframes pulseDiscount {
    0%, 100% { 
        transform: scale(1);
        box-shadow: 0 2px 8px rgba(239, 68, 68, 0.3);
    }
    50% { 
        transform: scale(1.05);
        box-shadow: 0 4px 12px rgba(239, 68, 68, 0.5);
    }
}

/* Clean Price Display Inside Button */
.price-display {
    display: flex;
    flex-direction: column;
    align-items: center;
    line-height: 1.2;
}

.original-price {
    font-size: 0.7rem;
    color: rgba(255, 255, 255, 0.7);
    text-decoration: line-through;
}

.discounted-price {
    font-size: 0.85rem;
    color: white;
    font-weight: bold;
}

/* Course Actions */
.course-actions {
    padding: 8px;
    border-top: 1px solid #f3f4f6;
    background: #f9fafb;
    display: flex;
    gap: 6px;
    justify-content: center;
}

.action-btn {
    background: #3b82f6;
    color: white;
    border: none;
    border-radius: 6px;
    padding: 6px;
    cursor: pointer;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    width: 32px;
    height: 32px;
    font-size: 14px;
}

.action-btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(5, 150, 105, 0.3);
}

    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
}

.view-btn {
    background: #64748b;
    color: white;
    border-color: #64748b;
}

.view-btn:hover {
    background: #475569;
    border-color: #475569;
}

.cart-btn {
    background: #fbbf24;
    color: white;
    border-color: #fbbf24;
}

.cart-btn:hover {
    background: #f59e0b;
    border-color: #f59e0b;
}

.enroll-btn {
    background: #3b82f6;
    color: white;
    border-color: #3b82f6;
}

.enroll-btn:hover {
    background: #2563eb;
    border-color: #2563eb;
}

.continue-btn {
    background: #3b82f6;
    color: white;
    border-color: #3b82f6;
}

.continue-btn:hover {
    background: #2563eb;
    border-color: #2563eb;
}



/* Empty State */
.empty-state {
    text-align: center;
    padding: 4rem 2rem;
}

.empty-illustration i {
    font-size: 5rem;
    color: #e5e7eb;
    margin-bottom: 1.5rem;
}

.empty-state h3 {
    font-size: 1.5rem;
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.5rem;
}

.empty-state p {
    color: #6b7280;
    font-size: 1rem;
}

/* Animations */
@keyframes slideUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Responsive Design for 5-Card Layout */
@media (max-width: 1400px) {
    .courses-grid {
        grid-template-columns: repeat(4, 1fr);
        gap: 1rem;
    }
}

@media (max-width: 1200px) {
    .courses-grid {
        grid-template-columns: repeat(3, 1fr);
        gap: 1rem;
    }
}

@media (max-width: 900px) {
    .courses-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 1rem;
    }
    
    .course-image-container {
        height: 140px;
    }
}

@media (max-width: 768px) {
    .catalog-header {
        padding: 1rem 0;
        margin-bottom: 1.5rem;
    }
    
    .header-content h1 {
        font-size: 2rem;
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .search-section {
        padding: 1.5rem;
        margin: 0 1rem 1.5rem 1rem;
    }
    
    .search-box input {
        padding: 16px 50px 16px 45px;
        font-size: 1rem;
    }
    
    .search-info {
        flex-direction: column;
        align-items: stretch;
        gap: 1rem;
    }
    
    .filter-options {
        justify-content: space-between;
    }
    
    .courses-grid {
        grid-template-columns: 1fr;
        gap: 1.5rem;
        padding: 0 1rem;
    }
    
    .course-actions {
        flex-direction: column;
        gap: 8px;
    }
    
    .course-meta {
        gap: 12px;
    }
    
    .course-image-container {
        height: 180px;
    }
}

@media (max-width: 480px) {
    .search-box input {
        padding: 14px 45px 14px 40px;
        font-size: 0.95rem;
    }
    
    .courses-grid {
        padding: 0 0.5rem;
    }
    
    .course-content {
        padding: 16px;
    }
    
    .course-actions {
        padding: 0 16px 16px 16px;
    }
}

/* Desktop 5-Card Layout Optimization */
@media (min-width: 1600px) {
    .courses-grid {
        grid-template-columns: repeat(5, 1fr);
        gap: 1.5rem;
    }
    
    .course-content {
        padding: 18px;
    }
    
    .course-image-container {
        height: 180px;
    }
}
</style>

<!-- Dynamic Search and Filter JavaScript -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('courseSearch');
    const clearBtn = document.getElementById('clearSearch');
    const levelFilter = document.getElementById('levelFilter');
    const priceFilter = document.getElementById('priceFilter');
    const coursesGrid = document.getElementById('coursesGrid');
    const resultsCount = document.getElementById('resultsCount');
    const allCourses = document.querySelectorAll('.course-card');
    
    let searchTimeout;
    
    // Search functionality
    function performSearch() {
        const searchTerm = searchInput.value.toLowerCase().trim();
        const levelValue = levelFilter.value;
        const priceValue = priceFilter.value;
        
        let visibleCount = 0;
        
        allCourses.forEach(course => {
            const title = course.dataset.title || '';
            const instructor = course.dataset.instructor || '';
            const description = course.dataset.description || '';
            const level = course.dataset.level || '';
            const price = course.dataset.price || '';
            
            // Text search match
            const searchMatch = !searchTerm || 
                title.includes(searchTerm) || 
                instructor.includes(searchTerm) || 
                description.includes(searchTerm);
            
            // Level filter match
            const levelMatch = !levelValue || level === levelValue;
            
            // Price filter match
            const priceMatch = !priceValue || price === priceValue;
            
            // Show/hide course
            if (searchMatch && levelMatch && priceMatch) {
                course.style.display = 'flex';
                visibleCount++;
            } else {
                course.style.display = 'none';
            }
        });
        
        // Update results count
        resultsCount.textContent = `${visibleCount} course${visibleCount !== 1 ? 's' : ''} available`;
        
        // Show/hide clear button
        if (searchTerm) {
            clearBtn.style.display = 'flex';
        } else {
            clearBtn.style.display = 'none';
        }
        
        // Show no results message if needed
        showNoResultsMessage(visibleCount === 0);
    }
    
    // Show/hide no results message
    function showNoResultsMessage(show) {
        let noResultsMsg = document.getElementById('noResultsMessage');
        
        if (show && !noResultsMsg) {
            noResultsMsg = document.createElement('div');
            noResultsMsg.id = 'noResultsMessage';
            noResultsMsg.className = 'empty-state';
            noResultsMsg.innerHTML = `
                <div class="empty-illustration">
                    <i class='bx bx-search-alt-2'></i>
                </div>
                <h3>No Courses Found</h3>
                <p>Try adjusting your search criteria or browse all available courses</p>
            `;
            coursesGrid.appendChild(noResultsMsg);
        } else if (!show && noResultsMsg) {
            noResultsMsg.remove();
        }
    }
    
    // Real-time search with debounce
    searchInput.addEventListener('input', function() {
        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(performSearch, 300);
    });
    
    // Enter key search
    searchInput.addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            e.preventDefault();
            clearTimeout(searchTimeout);
            performSearch();
        }
    });
    
    // Clear search
    clearBtn.addEventListener('click', function() {
        searchInput.value = '';
        clearBtn.style.display = 'none';
        performSearch();
        searchInput.focus();
    });
    
    // Filter changes
    levelFilter.addEventListener('change', performSearch);
    priceFilter.addEventListener('change', performSearch);
    
    // Focus search on page load
    searchInput.focus();
});

// Course action functions
function enrollInCourse(courseId, price = 0) {
    const button = event.target;
    const originalHTML = button.innerHTML;
    
    // Show loading state
    button.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i> Processing...';
    button.disabled = true;
    
    if (price > 0) {
        // Redirect to payment for paid courses
        window.location.href = `process-payment.php?course_id=${courseId}&amount=${price}`;
    } else {
        // Direct enrollment for free courses
        fetch('enroll-course.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `course_id=${courseId}`
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                button.innerHTML = '<i class="bx bx-check-circle"></i> Enrolled!';
                button.className = 'btn-primary enrolled-btn';
                button.onclick = () => continueCourse(courseId);
                
                // Show success notification
                showNotification('Successfully enrolled in course!', 'success');
                
                setTimeout(() => {
                    button.innerHTML = '<i class="bx bx-play-circle"></i> Continue Learning';
                }, 2000);
            } else {
                throw new Error(data.message || 'Enrollment failed');
            }
        })
        .catch(error => {
            console.error('Enrollment error:', error);
            button.innerHTML = originalHTML;
            button.disabled = false;
            showNotification('Failed to enroll in course', 'error');
        });
    }
}

function continueCourse(courseId) {
    // courseId is now encrypted, so we can pass it directly to the URL
    window.location.href = `course-detail.php?id=${courseId}`;
}

function previewCourse(courseId) {
    window.location.href = `course-detail.php?id=${courseId}&preview=1`;
}

function addToCart(courseId, buttonElement) {
    console.log('addToCart called with courseId:', courseId, 'buttonElement:', buttonElement);
    
    // Get the cart button - either passed as parameter or find by course ID
    let cartButton = buttonElement;
    
    if (!cartButton) {
        // Try to find the button by course ID in the card
        cartButton = document.querySelector(`[data-course-id="${courseId}"] .cart-btn`);
        console.log('Searched for cart button with selector:', `[data-course-id="${courseId}"] .cart-btn`);
    }
    
    if (!cartButton) {
        console.error('Cart button not found for course:', courseId);
        console.log('Available course cards with data-course-id:');
        document.querySelectorAll('[data-course-id]').forEach(card => {
            console.log('- Card ID:', card.getAttribute('data-course-id'));
        });
        showNotification('❌ Unable to add course to cart - button not found', 'error');
        return;
    }
    
    console.log('Cart button found:', cartButton);
    
    // Check if course is already in cart
    const existingCart = JSON.parse(localStorage.getItem('cart') || '[]');
    if (existingCart.find(item => item.id == courseId)) {
        showNotification('📚 Course is already in your cart!', 'warning');
        return;
    }
    
    // Visual feedback - show loading state
    const originalHTML = cartButton.innerHTML;
    const originalBg = cartButton.style.background;
    const originalColor = cartButton.style.color;
    
    cartButton.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i>';
    cartButton.disabled = true;
    cartButton.style.background = '#6c757d';
    
    // Make API call to validate and get course data
    fetch('api/add-to-cart.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        },
        body: JSON.stringify({
            course_id: courseId
        })
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.json();
    })
    .then(data => {
        if (data.success && data.course_data) {
            // Add course to localStorage cart
            const cart = JSON.parse(localStorage.getItem('cart') || '[]');
            cart.push(data.course_data);
            localStorage.setItem('cart', JSON.stringify(cart));
            
            // Success visual feedback
            cartButton.innerHTML = '<i class="bx bx-check"></i>';
            cartButton.style.background = '#10b981';
            cartButton.style.color = 'white';
            
            // Show success notification
            showNotification(`🎉 "${data.course_data.title}" added to cart successfully!`, 'success');
            
            // Update cart count in navbar
            updateNavbarCartCount();
            
            // Change button to "In Cart" state after delay
            setTimeout(() => {
                cartButton.innerHTML = '<i class="bx bx-check"></i>';
                cartButton.style.background = '#10b981';
                cartButton.style.color = 'white';
                cartButton.disabled = true;
                cartButton.classList.add('in-cart');
                cartButton.title = 'Added to Cart';
            }, 1500);
            
        } else {
            throw new Error(data.message || 'Failed to add course to cart');
        }
    })
    .catch(error => {
        console.error('Add to cart error:', error);
        
        // Error visual feedback
        cartButton.innerHTML = '<i class="bx bx-error-circle"></i>';
        cartButton.style.background = '#ef4444';
        cartButton.style.color = 'white';
        
        // Show error notification
        const errorMessage = error.message.includes('Authentication') 
            ? 'Please log in to add courses to cart' 
            : error.message || 'Failed to add course to cart';
        showNotification(`❌ ${errorMessage}`, 'error');
        
        // Reset button after delay
        setTimeout(() => {
            cartButton.innerHTML = originalHTML;
            cartButton.style.background = originalBg;
            cartButton.style.color = originalColor;
            cartButton.disabled = false;
        }, 3000);
    });
}

// Update navbar cart count
function updateNavbarCartCount() {
    const cart = JSON.parse(localStorage.getItem('cart') || '[]');
    const cartCount = cart.length;
    
    // Update cart badge in navbar
    const cartBadge = document.querySelector('.cart-count');
    if (cartBadge) {
        cartBadge.textContent = cartCount;
        cartBadge.style.display = cartCount > 0 ? 'inline-flex' : 'none';
    }
    
    // Update cart icon badge (alternative selector)
    const cartIconBadge = document.querySelector('#cart-count');
    if (cartIconBadge) {
        cartIconBadge.textContent = cartCount;
        cartIconBadge.style.display = cartCount > 0 ? 'inline-flex' : 'none';
    }
    
    console.log('Cart count updated:', cartCount);
}

// Notification system
function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="bx ${type === 'success' ? 'bx-check-circle' : type === 'error' ? 'bx-error-circle' : 'bx-info-circle'}"></i>
            <span>${message}</span>
        </div>
    `;
    
    // Add notification styles
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#3b82f6'};
        color: white;
        padding: 12px 20px;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        z-index: 10000;
        animation: slideInRight 0.3s ease;
    `;
    
    // Add to page
    document.body.appendChild(notification);
    
    // Remove after 3 seconds
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 3000);
}

// Add notification animations
const notificationStyles = document.createElement('style');
notificationStyles.textContent = `
    @keyframes slideInRight {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    @keyframes slideOutRight {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
    .notification-content {
        display: flex;
        align-items: center;
        gap: 8px;
        font-weight: 500;
    }
`;
document.head.appendChild(notificationStyles);

// ===============================
// CART FUNCTIONALITY
// ===============================

function addToCart(courseId) {
    // Get the cart button for this specific course
    const cartButton = document.querySelector(`[data-course-id="${courseId}"] .cart-btn`);
    
    if (!cartButton) {
        console.error('Cart button not found for course:', courseId);
        return;
    }
    
    // Check if course is already in cart
    const existingCart = JSON.parse(localStorage.getItem('cart') || '[]');
    if (existingCart.find(item => item.id == courseId)) {
        showNotification('📚 Course is already in your cart!', 'warning');
        return;
    }
    
    // Visual feedback - show loading state
    const originalHTML = cartButton.innerHTML;
    const originalBg = cartButton.style.background;
    const originalColor = cartButton.style.color;
    
    cartButton.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i>';
    cartButton.disabled = true;
    cartButton.style.background = '#6c757d';
    
    // Make API call to validate and get course data
    fetch('api/add-to-cart.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        },
        body: JSON.stringify({
            course_id: courseId
        })
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.json();
    })
    .then(data => {
        if (data.success && data.course_data) {
            // Add course to localStorage cart
            const cart = JSON.parse(localStorage.getItem('cart') || '[]');
            cart.push(data.course_data);
            localStorage.setItem('cart', JSON.stringify(cart));
            
            // Success visual feedback
            cartButton.innerHTML = '<i class="bx bx-check"></i>';
            cartButton.style.background = '#10b981';
            cartButton.style.color = 'white';
            
            // Show success notification
            showNotification(`🎉 "${data.course_data.title}" added to cart successfully!`, 'success');
            
            // Update cart count in navbar
            updateNavbarCartCount();
            
            // Change button to "In Cart" state after delay
            setTimeout(() => {
                cartButton.innerHTML = '<i class="bx bx-check"></i>';
                cartButton.style.background = '#10b981';
                cartButton.style.color = 'white';
                cartButton.disabled = true;
                cartButton.classList.add('in-cart');
                cartButton.title = 'Added to Cart';
            }, 1500);
            
        } else {
            throw new Error(data.message || 'Failed to add course to cart');
        }
    })
    .catch(error => {
        console.error('Add to cart error:', error);
        
        // Error visual feedback
        cartButton.innerHTML = '<i class="bx bx-error-circle"></i>';
        cartButton.style.background = '#ef4444';
        cartButton.style.color = 'white';
        
        // Show error notification
        const errorMessage = error.message.includes('Authentication') 
            ? 'Please log in to add courses to cart' 
            : error.message || 'Failed to add course to cart';
        showNotification(`❌ ${errorMessage}`, 'error');
        
        // Reset button after delay
        setTimeout(() => {
            cartButton.innerHTML = originalHTML;
            cartButton.style.background = originalBg;
            cartButton.style.color = originalColor;
            cartButton.disabled = false;
        }, 3000);
    });
}

// Update navbar cart count
function updateNavbarCartCount() {
    const cart = JSON.parse(localStorage.getItem('cart') || '[]');
    const cartCount = cart.length;
    
    // Update cart badge in navbar
    const cartBadge = document.querySelector('.cart-count');
    if (cartBadge) {
        cartBadge.textContent = cartCount;
        cartBadge.style.display = cartCount > 0 ? 'inline-flex' : 'none';
    }
    
    // Update cart icon badge (alternative selector)
    const cartIconBadge = document.querySelector('#cart-count');
    if (cartIconBadge) {
        cartIconBadge.textContent = cartCount;
        cartIconBadge.style.display = cartCount > 0 ? 'inline-flex' : 'none';
    }
    
    console.log('Cart count updated:', cartCount);
}

// Initialize cart on page load
function initializeCart() {
    // Create notification container if it doesn't exist
    if (!document.querySelector('.cart-notification')) {
        const notificationContainer = document.createElement('div');
        notificationContainer.className = 'cart-notification';
        notificationContainer.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 9999;
            min-width: 300px;
            max-width: 400px;
        `;
        document.body.appendChild(notificationContainer);
    }
    
    // Update cart count on page load
    updateNavbarCartCount();
    
    // Mark courses already in cart
    const cart = JSON.parse(localStorage.getItem('cart') || '[]');
    cart.forEach(cartItem => {
        const cartButton = document.querySelector(`[data-course-id="${cartItem.id}"] .cart-btn`);
        if (cartButton) {
            cartButton.innerHTML = '<i class="bx bx-check"></i>';
            cartButton.style.background = '#10b981';
            cartButton.style.color = 'white';
            cartButton.disabled = true;
            cartButton.classList.add('in-cart');
            cartButton.title = 'Added to Cart';
        }
    });
}

// Enhanced notification system
function showNotification(message, type = 'info', duration = 4000) {
    const container = document.querySelector('.cart-notification') || createNotificationContainer();
    
    const alert = document.createElement('div');
    alert.style.opacity = '0';
    
    const alertTypes = {
        'success': 'alert-success',
        'error': 'alert-danger', 
        'warning': 'alert-warning',
        'info': 'alert-info'
    };
    
    alert.className = `alert ${alertTypes[type] || 'alert-info'} alert-dismissible`;
    alert.style.cssText = `
        margin-bottom: 10px;
        border-radius: 10px;
        padding: 12px 16px;
        font-weight: 500;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        animation: slideInRight 0.3s ease-out;
    `;
    
    alert.innerHTML = `
        <div class="d-flex align-items-center">
            <div class="flex-grow-1">${message}</div>
            <button type="button" class="btn-close btn-close-white" aria-label="Close"></button>
        </div>
    `;
    
    container.appendChild(alert);
    
    // Animate in
    setTimeout(() => {
        alert.style.opacity = '1';
    }, 10);
    
    // Auto dismiss
    const timeoutId = setTimeout(() => {
        dismissNotification(alert);
    }, duration);
    
    // Manual dismiss
    const closeBtn = alert.querySelector('.btn-close');
    closeBtn.addEventListener('click', () => {
        clearTimeout(timeoutId);
        dismissNotification(alert);
    });
}

function createNotificationContainer() {
    const container = document.createElement('div');
    container.className = 'cart-notification';
    container.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        z-index: 9999;
        min-width: 300px;
        max-width: 400px;
    `;
    document.body.appendChild(container);
    return container;
}

function dismissNotification(alert) {
    alert.style.animation = 'slideOutRight 0.3s ease-in';
    setTimeout(() => {
        if (alert.parentNode) {
            alert.parentNode.removeChild(alert);
        }
    }, 300);
}

// Enroll in Course function - FIXED MISSING FUNCTION
function enrollInCourse(encryptedCourseId, price = null) {
    console.log('💳 Enrolling in course:', encryptedCourseId, 'Price:', price);
    
    if (price === null || price <= 0) {
        // Free course - handle direct enrollment
        console.log('🆓 Free course enrollment');
        // For free courses, you might want to add direct enrollment logic here
        showNotification('🎉 Enrolling in free course!', 'success');
        // Could redirect to course details or handle enrollment directly
        window.location.href = `course-details.php?id=${encryptedCourseId}`;
        return;
    }
    
    // Paid course - redirect to payment page
    console.log('💰 Paid course - redirecting to payment');
    showNotification('🔄 Redirecting to payment...', 'info');
    
    // Redirect to payment page with encrypted ID
    window.location.href = `course-payment.php?id=${encryptedCourseId}`;
}

// Continue Course function - for enrolled courses
function continueCourse(encryptedCourseId) {
    console.log('▶️ Continuing course:', encryptedCourseId);
    showNotification('🔄 Loading course...', 'info');
    
    // Redirect to course details page
    window.location.href = `course-details.php?id=${encryptedCourseId}`;
}

// Initialize cart when page loads
document.addEventListener('DOMContentLoaded', function() {
    initializeCart();
});

</script>


