// 🎯 CLEAN JAVASCRIPT - NO PHP SYNTAX AT ALL
// Note: All window variables (currentLessonId, currentVideoId, etc.) are set by course-detail.php

console.log('🔧 Course Detail Main JS loaded successfully');
console.log('🐛 DEBUG: Window variables available:', {
    currentLessonId: typeof window.currentLessonId !== 'undefined' ? window.currentLessonId : 'not set',
    currentVideoId: typeof window.currentVideoId !== 'undefined' ? window.currentVideoId : 'not set',
    currentCourseId: typeof window.currentCourseId !== 'undefined' ? window.currentCourseId : 'not set',
    currentStudentId: typeof window.currentStudentId !== 'undefined' ? window.currentStudentId : 'not set'
});

/**
                completedBadge.style.display = 'none';
            }
            
            // Reset progress display
            if (progressPercentage) {
                progressPercentage.textContent = Math.round(progressPercent) + '%';
            }
            if (progressBar) {
                progressBar.style.width = progressPercent + '%';
            }
            if (progressText) {
                progressText.textContent = progressPercent > 0 ? `Progress: ${Math.round(progressPercent)}% complete` : 'Not started';
            }
            
            break;
    }
    
    console.log('✅ Accordion badges updated successfully');
}

/**
 * 🎯 AJAX VIDEO LOADING - NO PAGE RELOAD
 * Loads a new lesson video without refreshing the entire page
 * @param {number} lessonId - The lesson ID to load
 */
function loadLesson(lessonId) {
    console.log(`🔄 AJAX LOAD: Loading lesson ${lessonId} without page reload`);
    console.log('🐛 DEBUG: loadLesson function called with lessonId:', lessonId);
    console.log('🐛 DEBUG: Event object:', event);
    
    // CRITICAL: Prevent any default behavior or page reload
    try {
        if (event) {
            console.log('🛑 Preventing default event behavior...');
            event.preventDefault();
            event.stopPropagation();
        }
    } catch (e) {
        console.warn('⚠️ Could not prevent event:', e);
    }
    
    // Get encrypted course ID from URL
    const encryptedCourseId = window.location.search.match(/[?&]id=([^&]*)/)?.[1] || '';
    if (!encryptedCourseId) {
        console.error('No encrypted course ID available');
        showVideoNotification('Error: Unable to load lesson', 'error');
        return false;
    }
    
    // Show loading notification
    showVideoNotification('Loading lesson...', 'info', 2000);
    
    // Make AJAX request to get lesson data
    fetch('ajax/get_lesson_data.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            lesson_id: lessonId,
            course_id: encryptedCourseId
        })
    })
    .then(response => {
        console.log('📡 AJAX Response status:', response.status);
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.json();
    })
    .then(data => {
        console.log('📦 AJAX Response data:', data);
        
        if (data.success) {
            // Update the video player
            updateVideoPlayer(data.lesson);
            
            // ✅ UPDATE CURRENT LESSON TITLE IN COURSE PROFILE
            updateCurrentLessonTitle(data.lesson.title);
            
            // Update window variables for other scripts
            window.currentLessonId = lessonId;
            window.currentVideoId = lessonId;
            
            // Update URL without page reload
            const newUrl = `course-detail.php?id=${encryptedCourseId}&lesson=${lessonId}`;
            window.history.pushState({lessonId: lessonId}, '', newUrl);
            
            // Use new unified badge system
            setLessonWatching(lessonId);
            
            console.log('✅ Updated accordion badges for lesson:', lessonId);
            
            // Reinitialize YouTube progress tracking if needed
            if (typeof window.initializeYouTubeProgress === 'function') {
                setTimeout(() => {
                    window.initializeYouTubeProgress();
                }, 1000);
            }
            
            showVideoNotification('Lesson loaded successfully!', 'success', 3000);
            console.log('✅ AJAX lesson load completed successfully');
            
        } else {
            throw new Error(data.message || 'Failed to load lesson data');
        }
    })
    .catch(error => {
        console.error('❌ AJAX lesson load failed:', error);
        
        // COMPLETELY DISABLE ALL REDIRECTS - NO MATTER WHAT
        console.log('🛑 REDIRECT DISABLED - No page reload allowed!');
        showVideoNotification('Lesson navigation temporarily disabled to prevent reloads', 'warning', 4000);
        
        // Override any potential redirect attempts
        const originalLocation = window.location.href;
        setTimeout(() => {
            if (window.location.href !== originalLocation) {
                console.log('🚫 REDIRECT DETECTED AND BLOCKED');
                window.location.href = originalLocation;
            }
        }, 100);
        
        // Don't redirect - just stay on current lesson
        console.log('✅ Staying on current page - no redirect executed');
    });
    
    // Always return false to prevent any default behavior
    return false;
}

/**
 * 🏷️ UPDATE CURRENT LESSON TITLE
 * Updates the main lesson title in the banner when lessons change
 * @param {string} lessonTitle - The title of the current lesson
 */
function updateCurrentLessonTitle(lessonTitle) {
    console.log('🏷️ Updating main lesson title to:', lessonTitle);
    
    const mainLessonTitle = document.getElementById('main-lesson-title');
    if (mainLessonTitle) {
        // Smooth transition effect
        mainLessonTitle.style.transition = 'opacity 0.3s ease-in-out';
        mainLessonTitle.style.opacity = '0.7';
        
        setTimeout(() => {
            mainLessonTitle.textContent = lessonTitle;
            mainLessonTitle.style.opacity = '1';
            console.log('✅ Main lesson title updated successfully');
        }, 150);
    } else {
        console.warn('⚠️ Main lesson title element #main-lesson-title not found');
    }
}

/**
 * 🎯 UPDATE VIDEO PLAYER
 * Updates the video player content with new lesson data
 * @param {Object} lesson - The lesson data from AJAX response
 */
function updateVideoPlayer(lesson) {
    console.log('🚀🚀🚀 CACHE-BUST-18:14-DEC16 🚀🚀🚀 Updating video player with lesson:', lesson);
    
    // Find the specific video player element (not the outer container)
    const videoPlayer = document.getElementById('videoPlayer');
    if (!videoPlayer) {
        console.error('❌ Video player element #videoPlayer not found');
        console.log('🔍 Available elements:', document.querySelectorAll('.video-container, .video-player'));
        return;
    }
    
    console.log('✅ Found video player element:', videoPlayer);
    
    if (!lesson.video_url) {
        console.error('❌ No video URL provided');
        return;
    }
    
    console.log('🎥 Video type:', lesson.video_type);
    console.log('🎥 Video URL:', lesson.video_url);
    console.log('🔍 DEBUG: Checking video type conditions...');
    console.log('🔍 DEBUG: lesson.video_type === "youtube"?', lesson.video_type === 'youtube');
    console.log('🔍 DEBUG: lesson.video_type === "local"?', lesson.video_type === 'local');
    console.log('🔍 DEBUG: lesson.video_type === "file"?', lesson.video_type === 'file');
    console.log('🔍 DEBUG: Combined file condition?', (lesson.video_type === 'local' || lesson.video_type === 'file'));
    
    if (lesson.video_type === 'youtube') {
        // YouTube video - create iframe with API parameters
        let videoUrl = lesson.video_url;
        
        // Add YouTube API parameters if not already present
        if (videoUrl.includes('youtube.com/embed')) {
            const separator = videoUrl.includes('?') ? '&' : '?';
            videoUrl += `${separator}autoplay=0&rel=0&showinfo=0&controls=1&enablejsapi=1&origin=${window.location.origin}`;
        }
        
        videoPlayer.innerHTML = `
            <iframe id="youtube-player" 
                    src="${videoUrl}" 
                    width="100%" 
                    height="400" 
                    frameborder="0" 
                    allowfullscreen 
                    allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture">
            </iframe>
        `;
        console.log('✅ YouTube iframe created with URL:', videoUrl);
        
    } else if (lesson.video_type === 'local' || lesson.video_type === 'file') {
        // Local/File video - create HTML5 video player WITHOUT NATIVE CONTROLS
        videoPlayer.innerHTML = `
            <video id="lessonVideo" 
                   preload="metadata"
                   width="100%" 
                   height="400"
                   style="width: 100%; height: 100%; background: #000;">
                <source src="${lesson.video_url}" type="video/mp4">
                <source src="${lesson.video_url}" type="video/webm">
                <source src="${lesson.video_url}" type="video/ogg">
                Your browser does not support the video tag.
            </video>
        `;
        console.log('✅ Local video player created WITHOUT native controls, URL:', lesson.video_url);
        console.log('🎬 Custom video controls will be used instead of native HTML5 controls');
        
        // 🎯 CRITICAL: Reinitialize VideoProgressTracker for dynamically loaded file videos
        setTimeout(() => {
            console.log('🎬 FILE VIDEO: Reinitializing VideoProgressTracker for dynamic video...');
            
            // Destroy existing tracker if present
            if (window.fileVideoProgressTracker) {
                console.log('🔄 FILE VIDEO: Destroying existing tracker...');
                if (typeof window.fileVideoProgressTracker.destroy === 'function') {
                    window.fileVideoProgressTracker.destroy();
                }
            }
            
            // Initialize new tracker for the switched video
            if (window.VideoProgressTracker && lesson.lesson_id && window.currentCourseId && window.currentStudentId) {
                try {
                    window.fileVideoProgressTracker = new VideoProgressTracker(
                        lesson.lesson_id,        // Use lesson ID from the loaded lesson
                        window.currentCourseId,  
                        window.currentStudentId,
                        {
                            debug: true,
                            updateFrequency: 2000,
                            completionThreshold: 90,
                            
                            onProgressUpdate: function(progressData) {
                                console.log('📊 DYNAMIC FILE VIDEO Progress:', progressData);
                                
                                // 🛡️ DON'T OVERWRITE COMPLETED BADGE - only set watching if not completed
                                if (progressData.progress < 95) { // Only set watching badge if not near completion
                                    setBadge(lesson.lesson_id, 'watching');
                                }
                            },
                            
                            onCompletion: function(completionData) {
                                console.log('🎉 DYNAMIC FILE VIDEO Completed:', completionData);
                                
                                // 💾 SAVE COMPLETION TO DATABASE via AJAX
                                console.log('💾 PREPARING DATABASE SAVE:', completionData);
                                const formData = new FormData();
                                formData.append('video_id', completionData.lessonId);
                                formData.append('course_id', completionData.courseId);
                                formData.append('auto_complete', '1');
                                
                                console.log('💾 SENDING AJAX REQUEST to mark_lesson_completed.php');
                                
                                fetch('mark_lesson_completed.php', {
                                    method: 'POST',
                                    body: formData
                                })
                                .then(response => response.json())
                                .then(data => {
                                    console.log('💾 DATABASE SAVE Result:', data);
                                    if (data.success) {
                                        console.log('✅ Video completion saved to database!');
                                        
                                        // Update debug panel if it exists
                                        const debugDatabase = document.querySelector(`#debug-database-${completionData.lessonId}`);
                                        if (debugDatabase) {
                                            debugDatabase.textContent = 'Database: ✅ Saved successfully';
                                            debugDatabase.className = 'text-success';
                                        }
                                    } else {
                                        console.error('❌ Database save failed:', data.message);
                                        
                                        // Update debug panel with error
                                        const debugDatabase = document.querySelector(`#debug-database-${completionData.lessonId}`);
                                        if (debugDatabase) {
                                            debugDatabase.textContent = `Database: ❌ Save failed - ${data.message}`;
                                            debugDatabase.className = 'text-danger';
                                        }
                                    }
                                })
                                .catch(error => {
                                    console.error('❌ AJAX Error saving completion:', error);
                                    
                                    // Update debug panel with error
                                    const debugDatabase = document.querySelector(`#debug-database-${completionData.lessonId}`);
                                    if (debugDatabase) {
                                        debugDatabase.textContent = `Database: ❌ AJAX Error - ${error.message}`;
                                        debugDatabase.className = 'text-danger';
                                    }
                                });
                                
                                // Update UI badge
                                if (window.setBadge) {
                                    setBadge(lesson.lesson_id, 'completed');
                                }
                                
                                // Show success toast
                                if (window.Toastify) {
                                    Toastify({
                                        text: `🎉 "${lesson.title}" completed! Great job!`,
                                        duration: 5000,
                                        style: { background: "linear-gradient(to right, #00b09b, #96c93d)" }
                                    }).showToast();
                                }
                            },
                            
                            onError: function(error) {
                                console.error('❌ DYNAMIC FILE VIDEO Error:', error);
                            }
                        }
                    );
                    
                    console.log('✅ DYNAMIC FILE VIDEO: VideoProgressTracker reinitialized successfully!');
                } catch (error) {
                    console.error('❌ DYNAMIC FILE VIDEO: Failed to reinitialize VideoProgressTracker:', error);
                }
            } else {
                console.error('❌ DYNAMIC FILE VIDEO: Missing data for VideoProgressTracker:', {
                    VideoProgressTracker: !!window.VideoProgressTracker,
                    lessonId: lesson.lesson_id,
                    courseId: window.currentCourseId,
                    studentId: window.currentStudentId,
                    allWindowVars: {
                        currentCourseId: window.currentCourseId,
                        currentStudentId: window.currentStudentId,
                        currentLessonId: window.currentLessonId,
                        currentVideoId: window.currentVideoId
                    }
                });
                
                // 🔧 TRY TO EXTRACT VALUES FROM PHP OR DOM
                if (!window.currentCourseId || !window.currentStudentId) {
                    console.log('🔧 TRYING TO EXTRACT missing values from page...');
                    
                    // Try to get course ID from URL or page data
                    const urlParams = new URLSearchParams(window.location.search);
                    const courseIdFromUrl = urlParams.get('id');
                    
                    // Try to extract from page elements or PHP data
                    const courseIdElement = document.querySelector('[data-course-id]');
                    const studentIdElement = document.querySelector('[data-student-id]');
                    
                    console.log('🔧 EXTRACTION ATTEMPT:', {
                        courseIdFromUrl: courseIdFromUrl,
                        courseIdElement: courseIdElement?.dataset?.courseId,
                        studentIdElement: studentIdElement?.dataset?.studentId
                    });
                }
            }
            
            // 🎮 CRITICAL: Reinitialize video control functions for new video element
            if (typeof window.initializeVideoControls === 'function') {
                console.log('🎮 DYNAMIC VIDEO: Reinitializing video controls...');
                window.initializeVideoControls();
                console.log('✅ DYNAMIC VIDEO: Video controls reinitialized successfully!');
            } else {
                console.error('❌ DYNAMIC VIDEO: initializeVideoControls function not found');
            }
        }, 500); // Wait for video element to be fully rendered
        
    } else {
        // Fallback - assume it's a YouTube video if type is not specified
        console.warn('⚠️ Video type not specified, assuming YouTube');
        videoPlayer.innerHTML = `
            <iframe id="youtube-player" 
                    src="${lesson.video_url}" 
                    width="100%" 
                    height="400" 
                    frameborder="0" 
                    allowfullscreen 
                    allow="autoplay; encrypted-media">
            </iframe>
        `;
        console.log('✅ Fallback YouTube iframe created');
    }
    
    console.log('✅ Video player updated successfully');
}

// Video notification system
function showVideoNotification(message, type = 'info', duration = 4000) {
    // Remove existing notifications
    const existingNotifications = document.querySelectorAll('.video-notification');
    existingNotifications.forEach(notif => notif.remove());
    
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `video-notification alert alert-${type === 'success' ? 'success' : type === 'error' ? 'danger' : 'info'} alert-dismissible fade show`;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        z-index: 9999;
        min-width: 300px;
        max-width: 500px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        border-radius: 8px;
        animation: slideInRight 0.3s ease-out;
    `;
    
    notification.innerHTML = `
        <div class="d-flex align-items-center">
            <div class="me-2">
                ${type === 'success' ? '✅' : type === 'error' ? '❌' : 'ℹ️'}
            </div>
            <div class="flex-grow-1">${message}</div>
            <button type="button" class="btn-close ms-2" data-bs-dismiss="alert"></button>
        </div>
    `;
    
    document.body.appendChild(notification);
    
    // Auto-remove after duration
    if (duration > 0) {
        setTimeout(() => {
            if (notification && notification.parentNode) {
                notification.classList.remove('show');
                setTimeout(() => notification.remove(), 150);
            }
        }, duration);
    }
    
    return notification;
}

/**
 * 🎯 UNIFIED BADGE SYSTEM - Simple & Clean
 * Creates and manages badges dynamically
 */
function setBadge(lessonId, badgeType) {
    console.log(`🎯 UNIFIED: Setting badge for lesson ${lessonId} to "${badgeType}"`);
    
    const badgeContainer = document.getElementById(`lesson-badges-${lessonId}`);
    if (!badgeContainer) {
        console.log(`❌ UNIFIED: Badge container not found for lesson ${lessonId}`);
        return;
    }
    
    // Clear existing watching and completed badges
    const existingBadges = badgeContainer.querySelectorAll('.badge-watching-accordion, .badge-completed-accordion');
    existingBadges.forEach(badge => badge.remove());
    
    // Create new badge based on type
    switch(badgeType) {
        case 'watching':
            const watchingBadge = document.createElement('span');
            watchingBadge.className = 'badge badge-watching-accordion me-1';
            watchingBadge.style.fontSize = '0.6rem';
            watchingBadge.title = 'Currently Watching';
            watchingBadge.id = `watching-badge-${lessonId}`;
            watchingBadge.innerHTML = '<i class="bx bx-play"></i> Watching';
            badgeContainer.appendChild(watchingBadge);
            badgeContainer.style.display = 'flex';
            console.log(`✅ UNIFIED: Created and showing WATCHING badge for lesson ${lessonId}`);
            break;
            
        case 'completed':
            const completedBadge = document.createElement('span');
            completedBadge.className = 'badge badge-completed-accordion me-1';
            completedBadge.style.fontSize = '0.6rem';
            completedBadge.title = 'Video Completed';
            completedBadge.id = `completed-badge-${lessonId}`;
            completedBadge.innerHTML = '<i class="bx bx-check"></i> Completed';
            badgeContainer.appendChild(completedBadge);
            badgeContainer.style.display = 'flex';
            console.log(`✅ UNIFIED: Created and showing COMPLETED badge for lesson ${lessonId}`);
            break;
            
        case 'none':
            // Hide the badge container if no badges
            const remainingBadges = badgeContainer.querySelectorAll('.badge');
            if (remainingBadges.length === 0) {
                badgeContainer.style.display = 'none';
            }
            console.log(`✅ UNIFIED: Removed all badges for lesson ${lessonId}`);
            break;
            
        default:
            console.log(`⚠️ UNIFIED: Unknown badge type "${badgeType}" for lesson ${lessonId}`);
    }
}

/**
 * 🧹 CLEAR ALL BADGES - Simple Version
 * Clears all watching and completed badges
 */
function clearAllBadges() {
    console.log('🧹 UNIFIED: Clearing all badges...');
    
    // Clear all watching badges
    document.querySelectorAll('[id*="watching-badge-"]').forEach(badge => {
        badge.style.display = 'none';
    });
    
    // Clear all completed badges (optional - usually we preserve these)
    // document.querySelectorAll('[id*="completed-badge-"]').forEach(badge => {
    //     badge.style.display = 'none';
    // });
    
    console.log('✅ UNIFIED: All badges cleared');
}

/**
 * 🎯 SET LESSON AS WATCHING
 * Shows watching badge for current lesson, hides all others
 * Preserves completed badges for already completed lessons
 */
function setLessonWatching(lessonId) {
    console.log(`🎯 UNIFIED: Setting lesson ${lessonId} as watching`);
    
    // Clear all watching badges from other lessons
    document.querySelectorAll('[id*="watching-badge-"]').forEach(badge => {
        badge.remove();
    });
    
    // Check if this lesson is already completed
    if (isLessonCompleted(lessonId)) {
        console.log(`🎯 UNIFIED: Lesson ${lessonId} is already completed - keeping completed badge and adding watching`);
        
        // For completed lessons: Keep completed badge and add watching badge
        const badgeContainer = document.getElementById(`lesson-badges-${lessonId}`);
        if (badgeContainer) {
            // Remove any existing watching badge for this lesson
            const existingWatching = badgeContainer.querySelector('.badge-watching-accordion');
            if (existingWatching) existingWatching.remove();
            
            // Create and add watching badge alongside completed badge
            const watchingBadge = document.createElement('span');
            watchingBadge.className = 'badge badge-watching-accordion me-1';
            watchingBadge.style.fontSize = '0.6rem';
            watchingBadge.title = 'Currently Rewatching';
            watchingBadge.id = `watching-badge-${lessonId}`;
            watchingBadge.innerHTML = '<i class="bx bx-play"></i> Watching';
            badgeContainer.appendChild(watchingBadge);
            badgeContainer.style.display = 'flex';
            
            console.log(`✅ UNIFIED: Added watching badge alongside completed badge for lesson ${lessonId}`);
        }
    } else {
        // Show watching badge for incomplete lessons only
        setBadge(lessonId, 'watching');
    }
}

/**
 * 🔍 CHECK IF LESSON IS COMPLETED
 * Reads completion status from PHP debug data or database
 */
function isLessonCompleted(lessonId) {
    // Check for PHP debug comment with completion data
    const debugComments = document.querySelectorAll('*');
    for (let element of debugComments) {
        const htmlContent = element.innerHTML || '';
        if (htmlContent.includes(`DEBUG LESSON ${lessonId}:`)) {
            const match = htmlContent.match(new RegExp(`DEBUG LESSON ${lessonId}: ({.*?})`));
            if (match) {
                try {
                    const debugData = JSON.parse(match[1]);
                    const isCompleted = debugData.completed_bool === true || debugData.completed_from_query === 1;
                    console.log(`🔍 UNIFIED: Lesson ${lessonId} completion from PHP debug:`, isCompleted, debugData);
                    return isCompleted;
                } catch (e) {
                    console.log(`⚠️ UNIFIED: Could not parse debug data for lesson ${lessonId}`);
                }
            }
        }
    }
    
    // Fallback: check if a completed badge was initially rendered
    const completedBadge = document.querySelector(`#completed-badge-${lessonId}`);
    if (completedBadge && completedBadge.style.display !== 'none') {
        console.log(`🔍 UNIFIED: Lesson ${lessonId} has visible completed badge`);
        return true;
    }
    
    console.log(`🔍 UNIFIED: Lesson ${lessonId} is not completed`);
    return false;
}

/**
 * 🏆 SET LESSON AS COMPLETED  
 * Shows completed badge for lesson, removes watching badge
 */
function setLessonCompleted(lessonId) {
    console.log(`🏆 UNIFIED: Setting lesson ${lessonId} as completed`);
    setBadge(lessonId, 'completed');
}

/**
 * 🔧 DEBUG: Check all lessons completion status
 * Call this in console to see which lessons are marked as completed
 */
function debugAllLessonsCompletion() {
    console.log('🔧 UNIFIED DEBUG: Checking completion status for all lessons...');
    
    const badgeContainers = document.querySelectorAll('[id^="lesson-badges-"]');
    const completionData = [];
    
    badgeContainers.forEach(container => {
        const lessonId = container.id.replace('lesson-badges-', '');
        const isCompleted = isLessonCompleted(lessonId);
        completionData.push({
            lessonId: parseInt(lessonId),
            isCompleted: isCompleted,
            container: container
        });
    });
    
    console.log('🔧 COMPLETION DATA:', completionData);
    
    // Show completed lessons specifically
    const completedLessons = completionData.filter(lesson => lesson.isCompleted);
    console.log('🏆 COMPLETED LESSONS:', completedLessons.map(l => l.lessonId));
    
    const incompleteLessons = completionData.filter(lesson => !lesson.isCompleted);
    console.log('⏳ INCOMPLETE LESSONS:', incompleteLessons.map(l => l.lessonId));
    
    return completionData;
}

// Make debug function globally available
window.debugAllLessonsCompletion = debugAllLessonsCompletion;

/**
 * 🚀 INITIALIZE ALL BADGES ON PAGE LOAD
 * Shows completed badges for all completed lessons automatically
 */
function initializeBadgesOnLoad() {
    console.log('🚀 UNIFIED: Initializing badges for all lessons...');
    
    const badgeContainers = document.querySelectorAll('[id^="lesson-badges-"]');
    let completedCount = 0;
    let incompleteCount = 0;
    
    badgeContainers.forEach(container => {
        const lessonId = container.id.replace('lesson-badges-', '');
        
        if (isLessonCompleted(lessonId)) {
            setBadge(lessonId, 'completed');
            completedCount++;
            console.log(`🏆 UNIFIED: Auto-initialized completed badge for lesson ${lessonId}`);
        } else {
            // Don't show watching badge on load, only when user clicks
            incompleteCount++;
            console.log(`⏳ UNIFIED: Lesson ${lessonId} is incomplete - no badge on load`);
        }
    });
    
    console.log(`✅ UNIFIED: Badge initialization complete - ${completedCount} completed, ${incompleteCount} incomplete`);
}

// Run badge initialization when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    console.log('📄 UNIFIED: DOM loaded, initializing badges...');
    // Small delay to ensure PHP debug comments are rendered
    setTimeout(initializeBadgesOnLoad, 100);
});

// Also run if DOM is already loaded
if (document.readyState === 'loading') {
    // DOM still loading, use DOMContentLoaded
} else {
    // DOM already loaded
    setTimeout(initializeBadgesOnLoad, 100);
}

/**
 * 🎯 TAB SWITCHING FUNCTIONALITY
 * Handles switching between Course Description, Reviews, Notes, Certificate, and Learning Tools tabs
 */
function switchTab(tabName) {
    console.log('🔄 Switching to tab:', tabName);
    
    // Remove active class from all tab buttons
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    
    // Add active class to clicked tab button
    const activeTabBtn = document.querySelector(`[data-tab="${tabName}"]`);
    if (activeTabBtn) {
        activeTabBtn.classList.add('active');
    }
    
    // Hide all tab panes
    document.querySelectorAll('.tab-pane').forEach(pane => {
        pane.classList.remove('active');
    });
    
    // Show the selected tab pane
    const activeTabPane = document.getElementById(tabName);
    if (activeTabPane) {
        activeTabPane.classList.add('active');
        console.log(`✅ Successfully switched to ${tabName} tab`);
    } else {
        console.error(`❌ Could not find tab pane with id: ${tabName}`);
    }
}

/**
 * 🎯 ACCORDION FUNCTIONALITY
 * Enhanced accordion toggle functionality for Bootstrap accordions
 */
function toggleAccordion(accordionId) {
    console.log('🔄 Toggling accordion:', accordionId);
    
    const accordion = document.getElementById(accordionId);
    if (accordion) {
        // Use Bootstrap's collapse functionality if available
        if (typeof bootstrap !== 'undefined' && bootstrap.Collapse) {
            const bsCollapse = new bootstrap.Collapse(accordion, {
                toggle: true
            });
        } else {
            // Fallback manual toggle
            accordion.classList.toggle('show');
        }
        console.log(`✅ Toggled accordion: ${accordionId}`);
    } else {
        console.error(`❌ Could not find accordion with id: ${accordionId}`);
    }
}

/**
 * 🎯 BUTTON CLICK HANDLERS
 * Generic button click handler for various interactive elements
 */
function handleButtonClick(action, data) {
    console.log('🖱️ Button clicked:', { action, data });
    
    switch(action) {
        case 'tab':
            switchTab(data.tabName);
            break;
        case 'accordion':
            toggleAccordion(data.accordionId);
            break;
        case 'loadLesson':
            loadLesson(data.lessonId);
            break;
        default:
            console.log('🔄 Unhandled button action:', action);
    }
}

// 🎯 HELPER FUNCTION FOR NESTED ACCORDIONS (Based on Working Emergency Fix)
function setupNestedAccordions(nestedHeaders) {
    console.log(`📌 Setting up ${nestedHeaders.length} nested accordion headers...`);
    
    // Remove any existing listeners to avoid duplicates (like emergency fix)
    nestedHeaders.forEach((header, index) => {
        // Clone and replace to remove old listeners
        const newHeader = header.cloneNode(true);
        header.parentNode.replaceChild(newHeader, header);
    });
    
    // Get the refreshed headers after cloning
    const refreshedHeaders = document.querySelectorAll('.nested-accordion-header');
    
    refreshedHeaders.forEach((header, index) => {
        console.log(`📌 Setting up header ${index + 1}:`, header);
        
        header.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            console.log('🎯 NESTED ACCORDION CLICKED:', this);
            
            // Get target from both Bootstrap versions (like emergency fix)
            const target = this.getAttribute('data-bs-target') || this.getAttribute('data-target');
            console.log('🔍 Target:', target);
            
            if (target) {
                const targetElement = document.querySelector(target);
                console.log('🔍 Element:', targetElement);
                
                if (targetElement) {
                    // Use the exact same logic as the working emergency fix
                    const isVisible = targetElement.style.display !== 'none' && targetElement.classList.contains('show');
                    
                    if (isVisible) {
                        targetElement.style.display = 'none';
                        targetElement.classList.remove('show');
                        this.classList.add('collapsed');
                    } else {
                        targetElement.style.display = 'block';
                        targetElement.classList.add('show');
                        this.classList.remove('collapsed');
                    }
                    
                    console.log('✅ Toggled:', target, 'Visible:', !isVisible);
                } else {
                    console.error('❌ Target element not found for:', target);
                }
            } else {
                console.error('❌ No target attribute found on:', this);
            }
        });
    });
    
    console.log('✅ All nested accordion headers set up successfully');
}

// 🚀 MAIN INITIALIZATION FUNCTION
document.addEventListener('DOMContentLoaded', function() {
    console.log('🚀 Course Detail Main Script Loaded');
    
    // 🎯 IMMEDIATE NESTED ACCORDION SETUP (Before anything else)
    try {
        console.log('🔧 IMMEDIATE: Setting up nested accordions...');
        const nestedHeaders = document.querySelectorAll('.nested-accordion-header');
        console.log(`🔍 IMMEDIATE: Found ${nestedHeaders.length} nested accordion headers`);
        
        if (nestedHeaders.length === 0) {
            console.warn('⚠️ No nested accordion headers found immediately - trying with timeout');
            setTimeout(() => {
                const delayedHeaders = document.querySelectorAll('.nested-accordion-header');
                console.log(`🔍 DELAYED: Found ${delayedHeaders.length} nested accordion headers`);
                setupNestedAccordions(delayedHeaders);
            }, 1000);
        } else {
            setupNestedAccordions(nestedHeaders);
        }
    } catch (error) {
        console.error('❌ Error setting up nested accordions:', error);
    }
    
    // 🎯 GLOBAL VARIABLES
    let currentVideoId = null;
    let currentLessonId = null;
    let currentTab = 'description';
    let progressCheckInterval = null;
    
    // 🎯 TAB SWITCHING
    console.log('🔧 Setting up tab switching...');
    document.querySelectorAll('.tab-btn').forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            const tabName = this.getAttribute('data-tab');
            if (tabName) {
                switchTab(tabName);
            }
        });
    });
    console.log('✅ Tab switching initialized');
    
    // 🎯 INITIALIZE ACCORDION FUNCTIONALITY (Bootstrap 4 Compatible)
    console.log('🔧 Setting up accordion functionality...');
    document.querySelectorAll('.accordion-button').forEach(button => {
        button.addEventListener('click', function(e) {
            // Bootstrap 4 uses data-target, Bootstrap 5 uses data-bs-target
            const target = this.getAttribute('data-bs-target') || this.getAttribute('data-target');
            if (target) {
                const accordionId = target.replace('#', '');
                console.log('🔄 Accordion button clicked for:', accordionId);
                
                // Try to trigger Bootstrap 4 collapse
                const targetElement = document.getElementById(accordionId);
                if (targetElement && typeof $ !== 'undefined' && $.fn.collapse) {
                    console.log('✨ Triggering Bootstrap 4 collapse for:', accordionId);
                    $(targetElement).collapse('toggle');
                } else if (targetElement) {
                    console.log('🔄 Manual toggle for:', accordionId);
                    // Manual fallback toggle
                    if (targetElement.classList.contains('show')) {
                        targetElement.classList.remove('show');
                        targetElement.style.display = 'none';
                    } else {
                        targetElement.classList.add('show');
                        targetElement.style.display = 'block';
                    }
                }
            }
        });
    });
    console.log('✅ Accordion functionality initialized');
    
    // 🎯 INITIALIZE BOOTSTRAP COMPONENTS (if available)
    if (typeof bootstrap !== 'undefined') {
        console.log('✅ Bootstrap detected - enhanced functionality available');
        
        // Initialize Bootstrap accordions
        document.querySelectorAll('.accordion').forEach(accordion => {
            // Bootstrap will handle this automatically
        });
        
        // Initialize Bootstrap tabs (fallback)
        document.querySelectorAll('[data-bs-toggle="tab"]').forEach(tab => {
            tab.addEventListener('click', function(e) {
                e.preventDefault();
                const tabName = this.getAttribute('data-tab');
                if (tabName) {
                    switchTab(tabName);
                }
            });
        });
    } else {
        console.log('⚠️ Bootstrap not detected - using fallback functionality');
    }
    
    // Test accordion badge update on page load if we have current lesson
    if (typeof window.currentLessonId !== 'undefined' && window.currentLessonId) {
        console.log('🧪 Testing accordion badge for current lesson:', window.currentLessonId);
        // Set a small delay to ensure DOM is fully rendered
        setTimeout(() => {
            // Old badge call removed - will use unified system
        }, 500);
    }
    
    console.log('🎉 All interactive functionality initialized successfully!');
});

console.log('✅ Course Detail Main JS loaded completely - no PHP syntax!');
