<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';
require_once '../includes/id_encryption.php';

// Require student role to access this page
requireRole('student');

// Include database connection
require_once '../config/db_connect.php';

// Get student ID
$student_id = $_SESSION['user_id'];

// Decrypt and validate course ID
$course_id = getSecureId('id');
if (!$course_id) {
    $_SESSION['error'] = "Invalid or corrupted course ID";
    header('Location: courses.php');
    exit();
}

// Check if student is enrolled in this course
$enrollQuery = "SELECT * FROM enrollments WHERE student_id = ? AND course_id = ?";
$enrollStmt = $conn->prepare($enrollQuery);
$enrollStmt->bind_param("ii", $student_id, $course_id);
$enrollStmt->execute();
$enrollResult = $enrollStmt->get_result();

$is_enrolled = ($enrollResult->num_rows > 0);

// Get course details
$courseQuery = "
    SELECT c.*, cat.category_name, 
           u.first_name, u.last_name,
           (SELECT COUNT(*) FROM enrollments WHERE course_id = c.course_id) as enrolled_students
    FROM courses c
    JOIN categories cat ON c.category_id = cat.category_id
    JOIN users u ON c.teacher_id = u.user_id
    WHERE c.course_id = ?
";

$courseStmt = $conn->prepare($courseQuery);
$courseStmt->bind_param("i", $course_id);
$courseStmt->execute();
$courseResult = $courseStmt->get_result();

if ($courseResult->num_rows == 0) {
    $_SESSION['error'] = "Course not found";
    header('Location: courses.php');
    exit();
}

$course = $courseResult->fetch_assoc();
$page_title = $course['title'] . " - Discussions";

// Include header
include_once 'components/header.php';
?>

<!-- Course Header -->
<div class="container-fluid pt-4">
    <div class="row">
        <div class="col-12">
            <div class="course-banner">
                <div class="course-banner-content">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h2 class="course-title"><?= htmlspecialchars($course['title']) ?></h2>
                            <p class="course-instructor">
                                <i class='bx bx-user'></i> Instructor: <?= htmlspecialchars($course['first_name'] . ' ' . $course['last_name']) ?>
                            </p>
                            <div class="mt-2">
                                <span class="course-category"><?= htmlspecialchars($course['category_name'] ?? 'General') ?></span>
                                <?php if (!empty($course['level'])): ?>
                                <span class="course-category"><?= htmlspecialchars(ucfirst($course['level'])) ?> Level</span>
                                <?php endif; ?>
                                <?php if (!empty($course['enrolled_students'])): ?>
                                <span class="course-category"><i class='bx bx-group'></i> <?= $course['enrolled_students'] ?> students</span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php if ($course && $is_enrolled) { ?>
<!-- Course Navigation Menu -->
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="course-nav">
                <ul class="nav nav-pills nav-fill">
                    <li class="nav-item">
                        <a href="course-detail.php?id=<?php echo $course_id; ?>" class="nav-link">
                            <i class='bx bx-book-open'></i> Course Content
                        </a>
                    </li>
                    <li class="nav-item">
                        <a href="course-materials.php?id=<?php echo $course_id; ?>" class="nav-link">
                            <i class='bx bx-file'></i> Materials
                        </a>
                    </li>
                    <li class="nav-item">
                        <a href="course-assignments.php?id=<?php echo $course_id; ?>" class="nav-link">
                            <i class='bx bx-task'></i> Assignments
                        </a>
                    </li>
                    <li class="nav-item">
                        <a href="course-quizzes.php?id=<?php echo $course_id; ?>" class="nav-link">
                            <i class='bx bx-question-mark'></i> Quizzes
                        </a>
                    </li>
                    <li class="nav-item">
                        <a href="course-discussions.php?id=<?php echo $course_id; ?>" class="nav-link active">
                            <i class='bx bx-chat'></i> Discussions
                        </a>
                    </li>
                </ul>
            </div>
        </div>
    </div>
</div>

<!-- Discussions Content -->
<div class="container-fluid mb-5">
    <div class="row">
        <div class="col-12">
            <div class="content-card">
                <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <h4 class="mb-0">Course Discussions</h4>
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#newDiscussionModal">
                            <i class="bx bx-plus-circle"></i> New Discussion
                        </button>
                    </div>
                </div>
                <div class="card-body">
                    
                    <?php
                    // Check if discussions table exists
                    $discussionsTableQuery = "SHOW TABLES LIKE 'discussions'";
                    $discussionsTableResult = $conn->query($discussionsTableQuery);
                    $discussionsTableExists = $discussionsTableResult->num_rows > 0;
                    
                    // Check if discussion_replies table exists
                    $repliesTableQuery = "SHOW TABLES LIKE 'discussion_replies'";
                    $repliesTableResult = $conn->query($repliesTableQuery);
                    $repliesTableExists = $repliesTableResult->num_rows > 0;
                    
                    if ($discussionsTableExists) {
                        // Check if status column exists in discussions table
                        $statusColumnQuery = "SHOW COLUMNS FROM discussions LIKE 'status'";
                        $statusColumnResult = $conn->query($statusColumnQuery);
                        $statusColumnExists = $statusColumnResult->num_rows > 0;
                        
                        // Check if is_pinned column exists in discussions table
                        $pinnedColumnQuery = "SHOW COLUMNS FROM discussions LIKE 'is_pinned'";
                        $pinnedColumnResult = $conn->query($pinnedColumnQuery);
                        $pinnedColumnExists = $pinnedColumnResult->num_rows > 0;
                        
                        // Check if created_at column exists in discussions table
                        $createdAtColumnQuery = "SHOW COLUMNS FROM discussions LIKE 'created_at'";
                        $createdAtColumnResult = $conn->query($createdAtColumnQuery);
                        $createdAtColumnExists = $createdAtColumnResult->num_rows > 0;
                        
                        // Build WHERE clause
                        $whereClause = "WHERE d.course_id = ?";
                        if ($statusColumnExists) {
                            $whereClause .= " AND d.status = 'active'";
                        }
                        
                        // Build ORDER BY clause
                        $orderByClause = "ORDER BY ";
                        if ($pinnedColumnExists) {
                            $orderByClause .= "d.is_pinned DESC";
                            if ($createdAtColumnExists) {
                                $orderByClause .= ", d.created_at DESC";
                            } else {
                                $orderByClause .= ", d.discussion_id DESC"; // Fallback
                            }
                        } else if ($createdAtColumnExists) {
                            $orderByClause .= "d.created_at DESC";
                        } else {
                            $orderByClause .= "d.discussion_id DESC"; // Default fallback
                        }
                        
                        // Get course discussions
                        if ($repliesTableExists) {
                            $discussionsQuery = "
                                SELECT d.*, 
                                       u.first_name, u.last_name, u.profile_image,
                                       (SELECT COUNT(*) FROM discussion_replies WHERE discussion_id = d.discussion_id) as reply_count,
                                       (SELECT MAX(created_at) FROM discussion_replies WHERE discussion_id = d.discussion_id) as last_reply
                                FROM discussions d
                                JOIN users u ON d.user_id = u.user_id
                                {$whereClause}
                                {$orderByClause}
                            ";
                        } else {
                            // If discussion_replies table doesn't exist, just query discussions without reply info
                            $discussionsQuery = "
                                SELECT d.*, 
                                       u.first_name, u.last_name, u.profile_image,
                                       0 as reply_count,
                                       NULL as last_reply
                                FROM discussions d
                                JOIN users u ON d.user_id = u.user_id
                                {$whereClause}
                                {$orderByClause}
                            ";
                        }
                        
                        $discussionStmt = $conn->prepare($discussionsQuery);
                    if ($discussionStmt) {
                        $discussionStmt->bind_param("i", $course_id);
                        $discussionStmt->execute();
                        $discussionsResult = $discussionStmt->get_result();
                        
                        if ($discussionsResult->num_rows > 0) {
                            echo '<div class="list-group">';
                            
                            while ($discussion = $discussionsResult->fetch_assoc()) {
                                // Format dates
                                $createdAt = new DateTime($discussion['created_at']);
                                $formattedCreatedAt = $createdAt->format('M d, Y - h:i A');
                                
                                $lastReply = !empty($discussion['last_reply']) ? 
                                    (new DateTime($discussion['last_reply']))->format('M d, Y - h:i A') : null;
                                
                                // Profile image or default
                                $profileImage = !empty($discussion['profile_image']) ? 
                                    htmlspecialchars($discussion['profile_image']) : 
                                    '../assets/img/default-avatar.png';
                                
                                // Determine if pinned
                                $isPinned = $discussion['is_pinned'] == 1;
                                
                                echo '<a href="discussion-view.php?id=' . $discussion['discussion_id'] . '" class="list-group-item list-group-item-action">';
                                echo '<div class="d-flex w-100 justify-content-between align-items-center">';
                                
                                // User image and title
                                echo '<div class="d-flex align-items-center">';
                                echo '<img src="' . $profileImage . '" class="rounded-circle me-3" width="40" height="40" alt="User">';
                                
                                echo '<div>';
                                echo '<h5 class="mb-1">';
                                
                                // Pinned icon
                                if ($isPinned) {
                                    echo '<i class="bx bx-pin me-2 text-primary"></i>';
                                }
                                
                                echo htmlspecialchars($discussion['title']) . '</h5>';
                                
                                echo '<p class="mb-1 text-muted">By ' . htmlspecialchars($discussion['first_name'] . ' ' . $discussion['last_name']) . ' • ' . $formattedCreatedAt . '</p>';
                                echo '</div>';
                                echo '</div>';
                                
                                // Reply count and date
                                echo '<div class="text-end">';
                                echo '<span class="badge bg-primary rounded-pill">' . $discussion['reply_count'] . ' replies</span>';
                                
                                if ($lastReply) {
                                    echo '<p class="small text-muted mb-0">Last reply: ' . $lastReply . '</p>';
                                }
                                
                                echo '</div>';
                                echo '</div>';
                                
                                // Short preview of content
                                $content = $discussion['content'];
                                $preview = (strlen($content) > 150) ? substr($content, 0, 150) . '...' : $content;
                                echo '<p class="mt-2 mb-0">' . htmlspecialchars($preview) . '</p>';
                                
                                echo '</a>';
                            }
                            
                            echo '</div>';
                        } else {
                            echo '<div class="empty-state">';
                            echo '<i class="bx bx-chat"></i>';
                            echo '<h5 class="mt-3">No discussions yet</h5>';
                            echo '<p class="text-muted">No discussions have been created for this course yet. Be the first to start a discussion!</p>';
                            echo '</div>';
                        }
                    }
                    ?>
                </div>
            </div>
        </div>
        
        <!-- Discussion Categories/Tags -->
        <div class="col-12">
            <div class="card shadow-sm border-0">
                <div class="card-header bg-light">
                    <h5 class="mb-0">Discussion Categories</h5>
                </div>
                <div class="card-body">
                    <div class="d-flex flex-wrap gap-2">
                        <a href="course-discussions.php?id=<?= $course_id ?>&tag=general" class="btn btn-sm btn-outline-primary">General</a>
                        <a href="course-discussions.php?id=<?= $course_id ?>&tag=questions" class="btn btn-sm btn-outline-primary">Questions</a>
                        <a href="course-discussions.php?id=<?= $course_id ?>&tag=resources" class="btn btn-sm btn-outline-primary">Resources</a>
                        <a href="course-discussions.php?id=<?= $course_id ?>&tag=announcements" class="btn btn-sm btn-outline-primary">Announcements</a>
                        <a href="course-discussions.php?id=<?= $course_id ?>&tag=assignments" class="btn btn-sm btn-outline-primary">Assignments</a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<?php 
    } // Close if ($discussionsTableExists)
    else {
        // Content for non-enrolled users
        ?>
        <div class="container-fluid mb-5">
            <div class="row">
                <div class="col-12">
                    <div class="card shadow-sm">
                        <div class="card-body p-4">
                            <div class="alert alert-warning">
                                <h5><i class="bx bx-lock me-2"></i> Course discussions are only available to enrolled students</h5>
                                <p>To access this content, please enroll in this course.</p>
                                <a href="course-enrollment.php?id=<?= $course_id ?>" class="btn btn-primary mt-2">Enroll Now</a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
<?php
    } // Close else
} // Close if ($course && $is_enrolled)

// Include footer
include_once 'components/footer.php';
?>
