<?php
/**
 * Student Dashboard - Clean Modern Design
 * Professional admin-style interface with clean functionality
 */

// Set current page for navigation
$currentPage = 'dashboard';

// Authentication - Require student role
session_start();
include '../db_connection.php';
include '../includes/id_encryption.php';  // Include project's encryption system
require_once '../auth/functions.php';
requireRole('student');

// Initialize default values
$enrolledCourses = 0;
$completedCourses = 0;
$assignmentsDue = 0;
$certificatesEarned = 0;
$totalLearningHours = 0;
$upcomingClasses = 0;
$overallProgress = 0;

$userId = $_SESSION['user_id'] ?? 0;
$userName = $_SESSION['first_name'] ?? 'Student';
$userLastName = $_SESSION['last_name'] ?? '';
$fullName = trim($userName . ' ' . $userLastName);

// Database queries for dashboard statistics
try {
    require_once '../config/db_connect.php';
    
    if (isset($conn) && $userId) {
        // Get enrolled courses count (with error checking)
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM enrollments WHERE student_id = ?");
        if ($stmt) {
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $enrolledCourses = $row['count'];
            }
            $stmt->close();
        }
        
        // Get completed courses count using FIXED CERTIFICATE ELIGIBILITY LOGIC
        // Same logic as certificate system: 100% videos + quiz requirements met
        $completedCourses = 0;
        if ($enrolledCourses > 0) {
            // Get all enrolled courses and check certificate eligibility for each
            $enrolledCoursesQuery = $conn->prepare("
                SELECT e.course_id 
                FROM enrollments e
                WHERE e.student_id = ?
            ");
            
            if ($enrolledCoursesQuery) {
                $enrolledCoursesQuery->bind_param("i", $userId);
                $enrolledCoursesQuery->execute();
                $coursesResult = $enrolledCoursesQuery->get_result();
                
                while ($courseRow = $coursesResult->fetch_assoc()) {
                    $courseId = $courseRow['course_id'];
                    
                    // 1. Check video completion (FIXED - uses user_video_completions)
                    $videoQuery = $conn->prepare("
                        SELECT 
                            COUNT(DISTINCT v.video_id) as total_videos,
                            COUNT(DISTINCT uvc.video_id) as completed_videos
                        FROM videos v
                        LEFT JOIN user_video_completions uvc ON v.video_id = uvc.video_id 
                            AND uvc.user_id = ? AND uvc.completion_percentage >= 100
                        WHERE v.course_id = ? AND v.is_published = 1
                    ");
                    
                    $total_videos = 0;
                    $completed_videos = 0;
                    
                    if ($videoQuery) {
                        $videoQuery->bind_param("ii", $userId, $courseId);
                        $videoQuery->execute();
                        $videoResult = $videoQuery->get_result();
                        if ($videoRow = $videoResult->fetch_assoc()) {
                            $total_videos = (int)$videoRow['total_videos'];
                            $completed_videos = (int)$videoRow['completed_videos'];
                        }
                        $videoQuery->close();
                    }
                    
                    // 2. Check quiz requirements
                    $quizQuery = $conn->prepare("
                        SELECT 
                            COUNT(DISTINCT q.quiz_id) as total_quizzes,
                            COUNT(DISTINCT CASE WHEN qa.pass_status = 1 THEN qa.quiz_id END) as passed_quizzes,
                            c.course_passing_grade
                        FROM quizzes q
                        LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id AND qa.student_id = ?
                        LEFT JOIN courses c ON q.course_id = c.course_id
                        WHERE q.course_id = ? AND q.is_published = 1
                        GROUP BY c.course_passing_grade
                    ");
                    
                    $total_quizzes = 0;
                    $passed_quizzes = 0;
                    $course_passing_grade = 60;
                    
                    if ($quizQuery) {
                        $quizQuery->bind_param("ii", $userId, $courseId);
                        $quizQuery->execute();
                        $quizResult = $quizQuery->get_result();
                        if ($quizRow = $quizResult->fetch_assoc()) {
                            $total_quizzes = (int)$quizRow['total_quizzes'];
                            $passed_quizzes = (int)$quizRow['passed_quizzes'];
                            $course_passing_grade = (float)$quizRow['course_passing_grade'] ?: 60;
                        }
                        $quizQuery->close();
                    }
                    
                    // 3. Apply SAME certificate eligibility logic
                    $video_completion_percentage = $total_videos > 0 ? ($completed_videos / $total_videos) * 100 : 0;
                    $videos_completed = ($video_completion_percentage >= 100);
                    
                    if ($total_quizzes == 0) {
                        $quizzes_passed = true; // No quizzes = automatic pass
                    } else {
                        $quiz_pass_rate = ($passed_quizzes / $total_quizzes) * 100;
                        $quizzes_passed = ($quiz_pass_rate >= $course_passing_grade);
                    }
                    
                    // Course is "completed" if certificate eligible
                    if ($videos_completed && $quizzes_passed) {
                        $completedCourses++;
                    }
                }
                
                $enrolledCoursesQuery->close();
            }
        }
        
        // Calculate Activities (Quizzes) Statistics - Total, Completed, and Incomplete
        $totalQuizActivities = 0;
        $completedQuizActivities = 0;
        $incompleteQuizActivities = 0;
        
        if ($enrolledCourses > 0) {
            // Get total quiz activities across all enrolled courses
            $totalActivitiesQuery = $conn->prepare("
                SELECT COUNT(DISTINCT q.quiz_id) as total_activities
                FROM quizzes q
                JOIN enrollments e ON q.course_id = e.course_id
                WHERE e.student_id = ? AND q.is_published = 1
            ");
            
            if ($totalActivitiesQuery) {
                $totalActivitiesQuery->bind_param("i", $userId);
                $totalActivitiesQuery->execute();
                $result = $totalActivitiesQuery->get_result();
                if ($row = $result->fetch_assoc()) {
                    $totalQuizActivities = (int)$row['total_activities'];
                }
                $totalActivitiesQuery->close();
            }
            
            // Get completed quiz activities (passed quizzes)
            $completedActivitiesQuery = $conn->prepare("
                SELECT COUNT(DISTINCT qa.quiz_id) as completed_activities
                FROM quiz_attempts qa
                JOIN quizzes q ON qa.quiz_id = q.quiz_id
                JOIN enrollments e ON q.course_id = e.course_id
                WHERE e.student_id = ? AND qa.student_id = ? AND qa.pass_status = 1 AND q.is_published = 1
            ");
            
            if ($completedActivitiesQuery) {
                $completedActivitiesQuery->bind_param("ii", $userId, $userId);
                $completedActivitiesQuery->execute();
                $result = $completedActivitiesQuery->get_result();
                if ($row = $result->fetch_assoc()) {
                    $completedQuizActivities = (int)$row['completed_activities'];
                }
                $completedActivitiesQuery->close();
            }
            
            // Calculate incomplete activities
            $incompleteQuizActivities = $totalQuizActivities - $completedQuizActivities;
        }
        
        // Get certificates available count using SAME CERTIFICATE ELIGIBILITY LOGIC
        // Shows courses that are eligible for certificates (same as completed courses)
        $certificatesEarned = $completedCourses; // Use same logic for consistency
        
        // Calculate total learning hours from enrolled courses (with error checking)
        $stmt = $conn->prepare("SELECT SUM(COALESCE(c.duration, 0)) as hours FROM courses c 
                              JOIN enrollments e ON c.course_id = e.course_id 
                              WHERE e.student_id = ?");
        if ($stmt) {
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $totalLearningHours = $row['hours'] ?? 0;
                // Debug: Check if hours are being calculated correctly
                error_log("Dashboard hours debug: User $userId has total hours: " . $totalLearningHours);
            }
            $stmt->close();
        }
        
        // Get upcoming classes (with error checking)
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM schedules s 
                              JOIN enrollments e ON s.course_id = e.course_id 
                              WHERE e.student_id = ? AND s.schedule_date >= CURDATE()");
        if ($stmt) {
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $upcomingClasses = $row['count'];
            }
            $stmt->close();
        }
        
        // Calculate real overall progress using FIXED logic (same as certificates)
        // Combines video completion + quiz performance across ALL enrolled courses
        $overallProgress = 0;
        if ($enrolledCourses > 0) {
            // Get all enrolled courses and calculate combined progress
            $enrolledCoursesQuery = $conn->prepare("
                SELECT e.course_id 
                FROM enrollments e
                WHERE e.student_id = ?
            ");
            
            if ($enrolledCoursesQuery) {
                $enrolledCoursesQuery->bind_param("i", $userId);
                $enrolledCoursesQuery->execute();
                $coursesResult = $enrolledCoursesQuery->get_result();
                
                $totalVideoProgress = 0;
                $totalQuizProgress = 0;
                $courseCount = 0;
                
                while ($courseRow = $coursesResult->fetch_assoc()) {
                    $courseId = $courseRow['course_id'];
                    $courseCount++;
                    
                    // 1. Calculate video completion for this course (FIXED - uses user_video_completions)
                    $videoQuery = $conn->prepare("
                        SELECT 
                            COUNT(DISTINCT v.video_id) as total_videos,
                            COUNT(DISTINCT uvc.video_id) as completed_videos
                        FROM videos v
                        LEFT JOIN user_video_completions uvc ON v.video_id = uvc.video_id 
                            AND uvc.user_id = ? AND uvc.completion_percentage >= 100
                        WHERE v.course_id = ? AND v.is_published = 1
                    ");
                    
                    if ($videoQuery) {
                        $videoQuery->bind_param("ii", $userId, $courseId);
                        $videoQuery->execute();
                        $videoResult = $videoQuery->get_result();
                        if ($videoRow = $videoResult->fetch_assoc()) {
                            $courseVideoProgress = $videoRow['total_videos'] > 0 
                                ? ($videoRow['completed_videos'] / $videoRow['total_videos']) * 100 
                                : 100; // If no videos, consider 100%
                            $totalVideoProgress += $courseVideoProgress;
                        }
                        $videoQuery->close();
                    }
                    
                    // 2. Calculate quiz performance for this course
                    $quizQuery = $conn->prepare("
                        SELECT 
                            COUNT(DISTINCT q.quiz_id) as total_quizzes,
                            COUNT(DISTINCT CASE WHEN qa.pass_status = 1 THEN qa.quiz_id END) as passed_quizzes,
                            c.course_passing_grade
                        FROM quizzes q
                        LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id AND qa.student_id = ?
                        LEFT JOIN courses c ON q.course_id = c.course_id
                        WHERE q.course_id = ? AND q.is_published = 1
                        GROUP BY c.course_passing_grade
                    ");
                    
                    if ($quizQuery) {
                        $quizQuery->bind_param("ii", $userId, $courseId);
                        $quizQuery->execute();
                        $quizResult = $quizQuery->get_result();
                        if ($quizRow = $quizResult->fetch_assoc()) {
                            if ($quizRow['total_quizzes'] > 0) {
                                $quizPassRate = ($quizRow['passed_quizzes'] / $quizRow['total_quizzes']) * 100;
                                $courseQuizProgress = min($quizPassRate, 100); // Cap at 100%
                            } else {
                                $courseQuizProgress = 100; // No quizzes = 100%
                            }
                            $totalQuizProgress += $courseQuizProgress;
                        } else {
                            $totalQuizProgress += 100; // No quizzes found = 100%
                        }
                        $quizQuery->close();
                    }
                }
                
                // Calculate weighted overall progress: 60% videos + 40% quizzes
                if ($courseCount > 0) {
                    $avgVideoProgress = $totalVideoProgress / $courseCount;
                    $avgQuizProgress = $totalQuizProgress / $courseCount;
                    $overallProgress = round(($avgVideoProgress * 0.6) + ($avgQuizProgress * 0.4));
                }
                
                $enrolledCoursesQuery->close();
            }
        }
        
        // Fetch enrolled courses with FIXED progress calculation for Continue Learning section
        $enrolledCoursesData = [];
        if ($enrolledCourses > 0) {
            // Get basic course info first
            $coursesQuery = $conn->prepare("
                SELECT c.course_id as id, c.title, c.description, c.thumbnail, 
                       u.first_name, u.last_name, e.enrolled_date
                FROM enrollments e
                JOIN courses c ON e.course_id = c.course_id
                JOIN users u ON c.teacher_id = u.user_id
                WHERE e.student_id = ?
                ORDER BY e.enrolled_date DESC
                LIMIT 4
            ");
            
            if ($coursesQuery) {
                $coursesQuery->bind_param("i", $userId);
                if ($coursesQuery->execute()) {
                    $result = $coursesQuery->get_result();
                    while ($row = $result->fetch_assoc()) {
                        $courseId = $row['id'];
                        
                        // Calculate REAL progress using FIXED logic (same as certificates)
                        
                        // 1. Video completion (FIXED - uses user_video_completions)
                        $videoQuery = $conn->prepare("
                            SELECT 
                                COUNT(DISTINCT v.video_id) as total_videos,
                                COUNT(DISTINCT uvc.video_id) as completed_videos
                            FROM videos v
                            LEFT JOIN user_video_completions uvc ON v.video_id = uvc.video_id 
                                AND uvc.user_id = ? AND uvc.completion_percentage >= 100
                            WHERE v.course_id = ? AND v.is_published = 1
                        ");
                        
                        $total_videos = 0;
                        $completed_videos = 0;
                        
                        if ($videoQuery) {
                            $videoQuery->bind_param("ii", $userId, $courseId);
                            $videoQuery->execute();
                            $videoResult = $videoQuery->get_result();
                            if ($videoRow = $videoResult->fetch_assoc()) {
                                $total_videos = (int)$videoRow['total_videos'];
                                $completed_videos = (int)$videoRow['completed_videos'];
                            }
                            $videoQuery->close();
                        }
                        
                        // 2. Quiz performance 
                        $quizQuery = $conn->prepare("
                            SELECT 
                                COUNT(DISTINCT q.quiz_id) as total_quizzes,
                                COUNT(DISTINCT CASE WHEN qa.pass_status = 1 THEN qa.quiz_id END) as passed_quizzes
                            FROM quizzes q
                            LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id AND qa.student_id = ?
                            WHERE q.course_id = ? AND q.is_published = 1
                        ");
                        
                        $total_quizzes = 0;
                        $passed_quizzes = 0;
                        
                        if ($quizQuery) {
                            $quizQuery->bind_param("ii", $userId, $courseId);
                            $quizQuery->execute();
                            $quizResult = $quizQuery->get_result();
                            if ($quizRow = $quizResult->fetch_assoc()) {
                                $total_quizzes = (int)$quizRow['total_quizzes'];
                                $passed_quizzes = (int)$quizRow['passed_quizzes'];
                            }
                            $quizQuery->close();
                        }
                        
                        // 🚨 CRITICAL PROFESSIONAL FIX: Use SAME logic as other fixed files
                        // PREVENT courses with NO CONTENT from showing 100% completion!
                        
                        $has_any_content = ($total_videos > 0 || $total_quizzes > 0);
                        
                        if (!$has_any_content) {
                            // 🚫 NO CONTENT = 0% progress (NOT 100%!)
                            $overall_course_progress = 0;
                            error_log("DASHBOARD.PHP CONTINUE LEARNING: Course ID $courseId has NO CONTENT - Progress set to 0%");
                        } else {
                            // Course has content - calculate normally
                            $video_progress = $total_videos > 0 ? ($completed_videos / $total_videos) * 100 : 0;
                            $quiz_progress = $total_quizzes > 0 ? ($passed_quizzes / $total_quizzes) * 100 : 0;
                            
                            // Only weight if both exist, otherwise use the available one
                            if ($total_videos > 0 && $total_quizzes > 0) {
                                $overall_course_progress = round(($video_progress * 0.6) + ($quiz_progress * 0.4));
                            } elseif ($total_videos > 0) {
                                $overall_course_progress = round($video_progress);
                            } elseif ($total_quizzes > 0) {
                                $overall_course_progress = round($quiz_progress);
                            } else {
                                $overall_course_progress = 0;
                            }
                        }
                        
                        $enrolledCoursesData[] = [
                            'id' => $row['id'],
                            'title' => $row['title'],
                            'description' => $row['description'],
                            'instructor' => $row['first_name'] . ' ' . $row['last_name'],
                            'thumbnail' => !empty($row['thumbnail']) ? $row['thumbnail'] : './assets/img/courses/default.jpg',
                            'progress' => $overall_course_progress,
                            'status' => $overall_course_progress >= 100 ? 'completed' : 'in_progress'
                        ];
                    }
                }
                $coursesQuery->close();
            }
        }
    }
} catch (Exception $e) {
    // Log error but don't display to user
    error_log("Dashboard data error: " . $e->getMessage());
}

// Include modern header with sidebar and navbar
include_once __DIR__ . '/components/header.php';
?>

<!-- Clean Modern Dashboard -->
<div class="dashboard-content">
    <!-- Overall Progress Circle -->
    <div class="progress-section">
        <div class="progress-card">
            <div class="progress-circle-container">
                <div class="circular-progress" data-percentage="<?= $overallProgress ?>">
                    <div class="progress-ring">
                        <svg class="progress-svg" viewBox="0 0 120 120">
                            <circle class="progress-bg" cx="60" cy="60" r="54"></circle>
                            <circle class="progress-bar" cx="60" cy="60" r="54" data-progress="<?= $overallProgress ?>"></circle>
                        </svg>
                        <div class="progress-text">
                            <span class="progress-percentage"><?= $overallProgress ?>%</span>
                            <span class="progress-label">Overall Progress</span>
                        </div>
                    </div>
                </div>
            </div>
            <div class="progress-details">
                <h3>Learning Journey</h3>
                <p>Keep up the great work! You're making excellent progress on your learning path.</p>

                <!-- Browse Courses Button -->
                <div class="banner-action-section">
                    <a href="https://panaditeacademy.com/student/course-catalog.php" class="browse-courses-btn">
                        <i class='bx bx-search-alt'></i>
                        <span>Browse Courses</span>
                        <i class='bx bx-right-arrow-alt'></i>
                    </a>
                </div>

                
                <!-- Enrolled Courses with Grades -->
                <div class="enrolled-courses-section">
                    <div class="enrolled-courses-card">
                        <div class="cycling-header">
                            <h3><i class='bx bx-book-open'></i> My Enrolled Courses</h3>
                            <div class="courses-badge">
                                <span class="badge-text">Track your progress</span>
                            </div>
                        </div>
                        
                        <!-- Learning Stats Mini Cards -->
                        <div class="learning-stats-mini">
                            <div class="mini-stat-card completed">
                                <div class="mini-stat-icon">
                                    <i class='bx bx-check-circle'></i>
                                </div>
                                <div class="mini-stat-content">
                                    <span class="mini-stat-number"><?= $completedCourses ?></span>
                                    <span class="mini-stat-label">Completed</span>
                                </div>
                            </div>
                            <div class="mini-stat-card progress">
                                <div class="mini-stat-icon">
                                    <i class='bx bx-time-five'></i>
                                </div>
                                <div class="mini-stat-content">
                                    <span class="mini-stat-number"><?= $enrolledCourses - $completedCourses ?></span>
                                    <span class="mini-stat-label">In Progress</span>
                                </div>
                            </div>
                            <div class="mini-stat-card hours">
                                <div class="mini-stat-icon">
                                    <i class='bx bx-time'></i>
                                </div>
                                <div class="mini-stat-content">
                                    <span class="mini-stat-number"><?= $totalLearningHours ?></span>
                                    <span class="mini-stat-label">Hours</span>
                                </div>
                            </div>
                        </div>
                        
                        <div class="enrolled-courses-grid" id="enrolledCoursesGrid">
                            <div class="course-loading">
                                <div class="loading-spinner"></div>
                                <p>Loading your courses...</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modern Statistics Grid -->
    <div class="modern-stats-grid">
        <!-- Enrolled Courses -->
        <div class="modern-stat-card enrolled">
            <div class="stat-header">
                <div class="stat-icon bg-primary">
                    <i class='bx bx-book-open'></i>
                </div>
                <div class="stat-trend">
                    <i class='bx bx-trending-up'></i>
                </div>
            </div>
            <div class="stat-body">
                <div class="stat-number"><?= $enrolledCourses ?></div>
                <div class="stat-label">Enrolled Courses</div>
                <div class="stat-description">Your active learning journey</div>
            </div>
            <div class="stat-footer">
                <a href="courses.php" class="stat-link">
                    <span>View All</span>
                    <i class='bx bx-right-arrow-alt'></i>
                </a>
            </div>
        </div>

        <!-- Completed Courses -->
        <div class="modern-stat-card completed">
            <div class="stat-header">
                <div class="stat-icon bg-success">
                    <i class='bx bx-check-circle'></i>
                </div>
                <div class="stat-badge">
                    <?= $overallProgress ?>%
                </div>
            </div>
            <div class="stat-body">
                <div class="stat-number"><?= $completedCourses ?></div>
                <div class="stat-label">Completed Courses</div>
                <div class="stat-description"><?= $overallProgress > 0 ? 'Great progress!' : 'Start your journey!' ?></div>
            </div>
            <div class="stat-progress">
                <div class="progress-container">
                    <div class="progress-bar">
                        <div class="progress-fill" style="width: <?= $overallProgress ?>%"></div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Activities (Quizzes) -->
        <div class="modern-stat-card activities">
            <div class="stat-header">
                <div class="stat-icon bg-warning">
                    <i class='bx bx-brain'></i>
                </div>
                <?php if ($incompleteQuizActivities > 0): ?>
                <div class="stat-alert">
                    <i class='bx bx-timer'></i>
                </div>
                <?php endif; ?>
                
                <!-- Small Clean Activity Indicators -->
                <div class="activity-indicators">
                    <div class="indicator completed" title="Completed Activities">
                        <span class="indicator-number"><?= $completedQuizActivities ?></span>
                        <i class='bx bx-trending-up'></i>
                    </div>
                    <div class="indicator incomplete" title="Remaining Activities">
                        <span class="indicator-number"><?= $incompleteQuizActivities ?></span>
                        <i class='bx bx-trending-down'></i>
                    </div>
                </div>
            </div>
            <div class="stat-body">
                <div class="stat-number"><?= $totalQuizActivities ?></div>
                <div class="stat-label">Learning Activities</div>
                <div class="stat-description">
                    <?php if ($totalQuizActivities == 0): ?>
                        No activities yet
                    <?php elseif ($completedQuizActivities == $totalQuizActivities): ?>
                        All activities completed! 🎉
                    <?php elseif ($completedQuizActivities > 0): ?>
                        <?= $completedQuizActivities ?>/<?= $totalQuizActivities ?> completed
                    <?php else: ?>
                        Ready to start activities
                    <?php endif; ?>
                </div>
            </div>
            <div class="stat-footer">
                <a href="quizzes.php" class="stat-link">
                    <span>View Activities</span>
                    <i class='bx bx-right-arrow-alt'></i>
                </a>
            </div>
        </div>

        <!-- Certificates Earned -->
        <div class="modern-stat-card certificates">
            <div class="stat-header">
                <div class="stat-icon bg-info">
                    <i class='bx bx-award'></i>
                </div>
                <?php if ($certificatesEarned > 0): ?>
                <div class="stat-celebration">
                    <i class='bx bx-trophy'></i>
                </div>
                <?php endif; ?>
            </div>
            <div class="stat-body">
                <div class="stat-number"><?= $certificatesEarned ?></div>
                <div class="stat-label">Certificates Earned</div>
                <div class="stat-description"><?= $certificatesEarned > 0 ? 'Well deserved!' : 'Coming soon...' ?></div>
            </div>
            <div class="stat-footer">
                <a href="certificates.php" class="stat-link">
                    <span>View All</span>
                    <i class='bx bx-right-arrow-alt'></i>
                </a>
            </div>
        </div>
    </div>

    <!-- Main Content Grid -->
    <div class="dashboard-grid">
        <!-- Continue Learning Card -->
        <div class="modern-card">
            <div class="card-header">
                <div class="header-content">
                    <h3><i class='bx bx-play-circle'></i> Continue Learning</h3>
                    <span class="course-count"><?= $enrolledCourses ?> courses</span>
                </div>
                <a href="courses.php" class="view-all-btn">
                    <span>View All</span>
                    <i class='bx bx-right-arrow-alt'></i>
                </a>
            </div>
            <div class="card-content">
                <?php if ($enrolledCourses > 0 && !empty($enrolledCoursesData)): ?>
                <div class="continue-learning-grid">
                    <?php foreach ($enrolledCoursesData as $course): ?>
                     <div class="continue-course-block" onclick="window.location.href='course-detail.php?id=<?= encryptId($course['id']) ?>'">
                        <div class="block-thumbnail">
                            <?php if (!empty($course['thumbnail'])): ?>
                                <img src="<?= htmlspecialchars($course['thumbnail']) ?>" alt="<?= htmlspecialchars($course['title']) ?>" onerror="this.parentElement.innerHTML='<i class=\'bx bx-play-circle\'></i>'">
                            <?php else: ?>
                                <i class='bx bx-play-circle'></i>
                            <?php endif; ?>
                        </div>
                        <div class="block-content">
                            <h4 class="block-title"><?= htmlspecialchars($course['title']) ?></h4>
                            <div class="block-progress">
                                <span class="progress-label"><?= $course['progress'] ?>% Progress</span>
                                <div class="mini-progress-bar">
                                    <div class="mini-progress-fill" style="width: <?= $course['progress'] ?>%"></div>
                                </div>
                            </div>
                            <div class="block-instructor">
                                <i class='bx bx-user'></i>
                                <?= htmlspecialchars($course['instructor']) ?>
                            </div>
                        </div>
                        <div class="block-action">
                            <i class='bx bx-play'></i>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php elseif ($enrolledCourses > 0): ?>
                <div class="modern-empty-state">
                    <div class="empty-illustration">
                        <i class='bx bx-book-reader'></i>
                    </div>
                    <div class="empty-content">
                        <h4>Loading Courses...</h4>
                        <p>Your enrolled courses are being loaded. Please wait.</p>
                    </div>
                </div>
                <?php else: ?>
                <div class="modern-empty-state">
                    <div class="empty-illustration">
                        <i class='bx bx-book-add'></i>
                    </div>
                    <div class="empty-content">
                        <h4>Start Your Learning Journey</h4>
                        <p>Discover amazing courses and begin building new skills today!</p>
                        <a href="courses.php" class="modern-btn">
                            <i class='bx bx-search'></i>
                            <span>Browse Courses</span>
                        </a>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Recent Activity Card -->
        <div class="modern-card">
            <div class="card-header">
                <div class="header-content">
                    <h3><i class='bx bx-pulse'></i> Recent Activity</h3>
                    <span class="activity-indicator">
                        <span class="indicator-dot"></span>
                        <span>Live</span>
                    </span>
                </div>
            </div>
            <div class="card-content">
                <?php
                // Get recent learning activities based on real data
                $recentActivities = [];
                
                // Get recent video completions
                $recentVideoQuery = $conn->prepare("
                    SELECT uvc.completed_at, v.title as video_title, c.title as course_title, c.course_id,
                           'video_completion' as activity_type
                    FROM user_video_completions uvc
                    JOIN videos v ON uvc.video_id = v.video_id
                    JOIN courses c ON v.course_id = c.course_id
                    WHERE uvc.user_id = ? AND uvc.completion_percentage >= 100
                    ORDER BY uvc.completed_at DESC
                    LIMIT 3
                ");
                
                if ($recentVideoQuery) {
                    $recentVideoQuery->bind_param("i", $userId);
                    $recentVideoQuery->execute();
                    $videoResult = $recentVideoQuery->get_result();
                    while ($row = $videoResult->fetch_assoc()) {
                        $recentActivities[] = $row;
                    }
                    $recentVideoQuery->close();
                }
                
                // Get recent quiz attempts
                $recentQuizQuery = $conn->prepare("
                    SELECT qa.completed_at, q.title as quiz_title, c.title as course_title, c.course_id,
                           qa.pass_status, qa.score,
                           'quiz_attempt' as activity_type
                    FROM quiz_attempts qa
                    JOIN quizzes q ON qa.quiz_id = q.quiz_id
                    JOIN courses c ON q.course_id = c.course_id
                    WHERE qa.student_id = ?
                    ORDER BY qa.completed_at DESC
                    LIMIT 3
                ");
                
                if ($recentQuizQuery) {
                    $recentQuizQuery->bind_param("i", $userId);
                    $recentQuizQuery->execute();
                    $quizResult = $recentQuizQuery->get_result();
                    while ($row = $quizResult->fetch_assoc()) {
                        $recentActivities[] = $row;
                    }
                    $recentQuizQuery->close();
                }
                
                // Sort all activities by date
                usort($recentActivities, function($a, $b) {
                    return strtotime($b['completed_at']) - strtotime($a['completed_at']);
                });
                
                // Take top 5 most recent activities
                $recentActivities = array_slice($recentActivities, 0, 5);
                ?>
                
                <?php if (!empty($recentActivities)): ?>
                <div class="activity-list">
                    <?php foreach ($recentActivities as $activity): ?>
                    <div class="activity-item">
                        <div class="activity-icon">
                            <?php if ($activity['activity_type'] == 'video_completion'): ?>
                                <i class='bx bx-play-circle' style="color: #1cc88a;"></i>
                            <?php elseif ($activity['activity_type'] == 'quiz_attempt'): ?>
                                <?php if ($activity['pass_status'] == 1): ?>
                                    <i class='bx bx-check-circle' style="color: #1cc88a;"></i>
                                <?php else: ?>
                                    <i class='bx bx-x-circle' style="color: #e74a3b;"></i>
                                <?php endif; ?>
                            <?php endif; ?>
                        </div>
                        <div class="activity-content">
                            <div class="activity-title">
                                <?php if ($activity['activity_type'] == 'video_completion'): ?>
                                    Completed: <?= htmlspecialchars($activity['video_title']) ?>
                                <?php elseif ($activity['activity_type'] == 'quiz_attempt'): ?>
                                    <?php if ($activity['pass_status'] == 1): ?>
                                        Passed Quiz: <?= htmlspecialchars($activity['quiz_title']) ?>
                                    <?php else: ?>
                                        Attempted Quiz: <?= htmlspecialchars($activity['quiz_title']) ?>
                                    <?php endif; ?>
                                <?php endif; ?>
                            </div>
                            <div class="activity-meta">
                                <span class="course-name"><?= htmlspecialchars($activity['course_title']) ?></span>
                                <span class="activity-time"><?= date('M j, g:i A', strtotime($activity['completed_at'])) ?></span>
                            </div>
                            <?php if ($activity['activity_type'] == 'quiz_attempt' && isset($activity['score'])): ?>
                            <div class="activity-score">
                                Score: <?= round($activity['score']) ?>%
                            </div>
                            <?php endif; ?>
                        </div>
                        <div class="activity-status">
                            <?php if ($activity['activity_type'] == 'video_completion'): ?>
                                <span class="status-badge completed">✓ Completed</span>
                            <?php elseif ($activity['activity_type'] == 'quiz_attempt'): ?>
                                <?php if ($activity['pass_status'] == 1): ?>
                                    <span class="status-badge passed">✓ Passed</span>
                                <?php else: ?>
                                    <span class="status-badge attempted">⚠ Need Retry</span>
                                <?php endif; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php else: ?>
                <div class="modern-empty-state">
                    <div class="empty-illustration">
                        <i class='bx bx-trending-up'></i>
                    </div>
                    <div class="empty-content">
                        <h4>No Recent Activity</h4>
                        <p>Complete videos or take quizzes to see your learning activity here!</p>
                        <a href="courses.php" class="modern-btn">
                            <i class='bx bx-play'></i>
                            <span>Start Learning</span>
                        </a>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Featured Courses Section -->
    <div class="featured-courses-section">
        <div class="featured-courses-card">
            <div class="cycling-header">
                <h3><i class='bx bx-sparkles'></i> Featured Courses</h3>
                <div class="cycling-indicator">
                    <span class="indicator-dot active"></span>
                    <span class="indicator-dot"></span>
                    <span class="indicator-dot"></span>
                </div>
            </div>
            <div class="cycling-courses" id="cyclingCourses">
                <!-- Loading placeholder -->
                <div class="course-loading">
                    <div class="loading-spinner"></div>
                    <p>Loading courses...</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Modern Quick Actions -->
    <div class="modern-card full-width">
        <div class="card-header">
            <div class="header-content">
                <h3><i class='bx bx-zap'></i> Quick Actions</h3>
                <span class="subtitle">Everything you need at your fingertips</span>
            </div>
        </div>
        <div class="card-content">
            <div class="modern-actions-grid">
                <a href="courses.php" class="modern-action-card courses">
                    <div class="action-icon-wrapper">
                        <i class='bx bx-book-open'></i>
                    </div>
                    <div class="action-content">
                        <h4>Browse Courses</h4>
                        <p>Discover new learning opportunities</p>
                        <span class="action-arrow"><i class='bx bx-right-arrow-alt'></i></span>
                    </div>
                </a>
                
                <a href="courses.php" class="modern-action-card activities">
                    <div class="action-icon-wrapper">
                        <i class='bx bx-brain'></i>
                        <?php if ($incompleteQuizActivities > 0): ?>
                        <span class="action-badge"><?= $incompleteQuizActivities ?></span>
                        <?php endif; ?>
                    </div>
                    <div class="action-content">
                        <h4>Learning Activities</h4>
                        <p>Practice with <?= $totalQuizActivities ?> quiz activities</p>
                        <span class="action-arrow"><i class='bx bx-right-arrow-alt'></i></span>
                    </div>
                </a>
                
                <a href="certificates.php" class="modern-action-card certificates">
                    <div class="action-icon-wrapper">
                        <i class='bx bx-award'></i>
                        <?php if ($certificatesEarned > 0): ?>
                        <span class="action-badge"><?= $certificatesEarned ?></span>
                        <?php endif; ?>
                    </div>
                    <div class="action-content">
                        <h4>Certificates</h4>
                        <p>View your earned certificates</p>
                        <span class="action-arrow"><i class='bx bx-right-arrow-alt'></i></span>
                    </div>
                </a>
            </div>
        </div>
    </div>


</div>

<!-- Clean Modern CSS Styling -->
<style>
/* Dashboard Content */
.dashboard-content {
    background: whitesmoke;
    min-height: 100vh;
    padding: 20px;
    max-width: 100%;
    box-sizing: border-box;
}

/* Welcome Banner */
.welcome-banner {
    background: whitesmoke;
    border-radius: 1rem;
    padding: 2rem;
    margin-bottom: 2rem;
    color: #2c3e50;
    box-shadow: 0 0.25rem 1rem rgba(0, 0, 0, 0.1);
    color: #4e73df;
    animation: float 8s ease-in-out infinite;
    transition: opacity 0.3s ease;
}

.floating-icon:nth-child(1) {
    top: 20%;
    left: 85%;
}

.floating-icon:nth-child(2) {
    top: 60%;
    left: 90%;
}

.floating-icon:nth-child(3) {
    top: 15%;
    left: 10%;
}

.floating-icon:nth-child(4) {
    top: 70%;
    left: 15%;
}

.floating-icon:nth-child(5) {
    top: 40%;
    left: 5%;
}

.floating-icon:nth-child(6) {
    top: 30%;
    left: 95%;
}

@keyframes float {
    0%, 100% {
        transform: translateY(0px) translateX(0px);
        opacity: 0.1;
    }
    25% {
        transform: translateY(-10px) translateX(5px);
        opacity: 0.15;
    }
    50% {
        transform: translateY(-5px) translateX(-3px);
        opacity: 0.12;
    }
    75% {
        transform: translateY(-15px) translateX(7px);
        opacity: 0.18;
    }
}

.welcome-banner:hover .floating-icon {
    opacity: 0.2;
}

/* Recent Activity List Styling */
.activity-list {
    display: flex;
    flex-direction: column;
    gap: 0.75rem;
}

.activity-item {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 0.75rem;
    background: #f8f9fc;
    border-radius: 0.5rem;
    border-left: 3px solid #4e73df;
    transition: all 0.3s ease;
}

.activity-item:hover {
    background: #e3f2fd;
    transform: translateX(2px);
}

.activity-icon {
    flex-shrink: 0;
    width: 2rem;
    height: 2rem;
    display: flex;
    align-items: center;
    justify-content: center;
    background: white;
    border-radius: 50%;
    font-size: 1.2rem;
}

.activity-content {
    flex: 1;
    min-width: 0;
}

.activity-title {
    font-weight: 600;
    font-size: 0.875rem;
    color: #2c3e50;
    margin-bottom: 0.25rem;
    display: -webkit-box;
    -webkit-line-clamp: 1;
    -webkit-box-orient: vertical;
    overflow: hidden;
}

.activity-meta {
    display: flex;
    justify-content: space-between;
    align-items: center;
    font-size: 0.75rem;
    color: #5a5c69;
    margin-bottom: 0.25rem;
}

.course-name {
    font-weight: 500;
    color: #4e73df;
}

.activity-time {
    color: #858796;
}

.activity-score {
    font-size: 0.75rem;
    color: #1cc88a;
    font-weight: 600;
}

.activity-status {
    flex-shrink: 0;
}

.status-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.25rem;
    padding: 0.25rem 0.5rem;
    border-radius: 0.375rem;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
}

.status-badge.completed {
    background: #d4edda;
    color: #1cc88a;
    border: 1px solid #1cc88a;
}

.status-badge.passed {
    background: #d4edda;
    color: #1cc88a;
    border: 1px solid #1cc88a;
}

.status-badge.attempted {
    background: #f8d7da;
    color: #e74a3b;
    border: 1px solid #e74a3b;
}

/* Small Clean Activity Indicators Styling */
.activity-indicators {
    position: absolute;
    top: 0.5rem;
    right: 0.5rem;
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
    z-index: 3;
}

.indicator {
    display: flex;
    align-items: center;
    gap: 0.2rem;
    padding: 0.15rem 0.4rem;
    border-radius: 0.75rem;
    font-size: 0.65rem;
    font-weight: 600;
    backdrop-filter: blur(10px);
    border: 1px solid rgba(255, 255, 255, 0.2);
    transition: all 0.3s ease;
    min-width: 2rem;
    justify-content: center;
}

.indicator.completed {
    background: linear-gradient(135deg, rgba(34, 197, 94, 0.15), rgba(34, 197, 94, 0.25));
    color: #16a34a;
    border-color: rgba(34, 197, 94, 0.3);
}

.indicator.incomplete {
    background: linear-gradient(135deg, rgba(239, 68, 68, 0.15), rgba(239, 68, 68, 0.25));
    color: #dc2626;
    border-color: rgba(239, 68, 68, 0.3);
}

.indicator:hover {
    transform: scale(1.05);
    box-shadow: 0 0.25rem 0.5rem rgba(0, 0, 0, 0.1);
}

.indicator-number {
    font-size: 0.7rem;
    font-weight: 700;
    line-height: 1;
}

.indicator i {
    font-size: 0.8rem;
    opacity: 0.8;
}

/* Ensure stats card header has relative positioning for indicators */
.modern-stat-card .stat-header {
    position: relative;
}

/* Ensure banner content is above floating icons */
.banner-content {
    position: relative;
    z-index: 2;
    display: flex;
    flex-direction: column;
    align-items: center;
    text-align: center;
    gap: 1.5rem;
}

.welcome-text h1 {
    font-size: 2.5rem;
    font-weight: 700;
    margin: 0 0 0.5rem 0;
    color: #2c3e50;
}

.welcome-text p {
    font-size: 1.1rem;
    margin: 0 0 1.5rem 0;
    color: #5a5c69;
}

.welcome-actions {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
    justify-content: center;
}

.welcome-actions .btn {
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    border-radius: 0.5rem;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s ease;
    border: 2px solid transparent;
}

.welcome-actions .btn-primary {
    background: #4e73df;
    color: white;
    border-color: #4e73df;
}

.welcome-actions .btn-primary:hover {
    background: #2e59d9;
    border-color: #2e59d9;
    transform: translateY(-2px);
}

.welcome-actions .btn-outline-primary {
    background: transparent;
    color: #4e73df;
    border-color: #4e73df;
}

.welcome-actions .btn-outline-primary:hover {
    background: #4e73df;
    color: white;
}

/* Featured Courses Section */
.featured-courses-section {
    margin-bottom: 2rem;
}

.featured-courses-card {
    background: white;
    border-radius: 1rem;
    padding: 2rem;
    box-shadow: 0 0.25rem 1rem rgba(0, 0, 0, 0.1);
    border: 1px solid #e3e6f0;
}

.cycling-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
}

.cycling-header h3 {
    margin: 0;
    font-size: 1.2rem;
    font-weight: 700;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #1a2332;
}

.cycling-header h3 i {
    color: #3366ff;
    font-size: 1.3rem;
}

.cycling-indicator {
    display: flex;
    gap: 0.5rem;
}

.indicator-dot {
    width: 8px;
    height: 8px;
    border-radius: 50%;
    background: rgba(44, 62, 80, 0.3);
    transition: all 0.3s ease;
}

.indicator-dot.active {
    background: #4e73df;
    transform: scale(1.2);
}

/* Cycling Courses Grid */
.cycling-courses {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 0.5rem;
    transition: all 0.5s ease;
    max-width: 800px;
    margin: 0 auto;
}

/* Small Horizontal Course Cards */
.mini-course-card {
    background: rgba(255, 255, 255, 0.95);
    border-radius: 0.5rem;
    padding: 0.5rem;
    color: #333;
    transition: all 0.3s ease;
    cursor: pointer;
    transform: translateY(20px);
    opacity: 0;
    animation: slideInUp 0.5s ease forwards;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    height: 70px;
    min-width: 180px;
}

.mini-course-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    background: white;
}

.course-thumbnail {
    width: 50px;
    height: 50px;
    border-radius: 0.375rem;
    object-fit: cover;
    background: #f8f9fc;
    flex-shrink: 0;
}

.course-info {
    flex: 1;
    display: flex;
    flex-direction: column;
    justify-content: center;
    min-width: 0;
}

.course-info h4 {
    font-size: 0.7rem;
    font-weight: 600;
    margin: 0 0 0.2rem 0;
    color: #2c3e50;
    line-height: 1.1;
    display: -webkit-box;
    -webkit-line-clamp: 1;
    -webkit-box-orient: vertical;
    overflow: hidden;
    white-space: nowrap;
    text-overflow: ellipsis;
}

.course-meta {
    display: flex;
    justify-content: space-between;
    align-items: center;
    font-size: 0.6rem;
    margin-bottom: 0.1rem;
    gap: 0.25rem;
}

.course-level {
    background: #4e73df;
    color: white;
    padding: 0.05rem 0.3rem;
    border-radius: 0.2rem;
    font-weight: 500;
    font-size: 0.55rem;
    white-space: nowrap;
}

.course-rating {
    color: #f6c23e;
    font-weight: 600;
    font-size: 0.6rem;
}

.course-price {
    font-size: 0.65rem;
    font-weight: 700;
    color: #1cc88a;
    white-space: nowrap;
}

/* Loading Animation */
.course-loading {
    grid-column: 1 / -1;
    text-align: center;
    padding: 2rem 1rem;
    color: #5a5c69;
}

.loading-spinner {
    width: 30px;
    height: 30px;
    border: 3px solid rgba(78, 115, 223, 0.2);
    border-top: 3px solid #4e73df;
    border-radius: 50%;
    margin: 0 auto 0.5rem;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

@keyframes slideInUp {
    to {
        transform: translateY(0);
        opacity: 1;
    }
}

/* Card Animation Delays */
.mini-course-card:nth-child(1) { animation-delay: 0.1s; }
.mini-course-card:nth-child(2) { animation-delay: 0.2s; }
.mini-course-card:nth-child(3) { animation-delay: 0.3s; }
.mini-course-card:nth-child(4) { animation-delay: 0.4s; }
.mini-course-card:nth-child(5) { animation-delay: 0.5s; }
.mini-course-card:nth-child(6) { animation-delay: 0.6s; }

/* Enrolled Courses Section - Clean Panadite Branding */
.enrolled-courses-section {
    margin-bottom: 2rem;
}

.enrolled-courses-card {
    background: white;
    border-radius: 0.75rem;
    padding: 1.5rem 1rem;
    border: 1px solid #e8ecf4;
    transition: all 0.3s ease;
    max-width: 100%;
    box-sizing: border-box;
    overflow: hidden;
}

.enrolled-courses-card:hover {
    border-color: #3366ff;
    transform: translateY(-1px);
}

.courses-badge {
    background: #f0f4ff;
    color: #3366ff;
    padding: 0.4rem 1rem;
    border-radius: 2rem;
    font-size: 0.75rem;
    font-weight: 600;
    border: 1px solid #e0ebff;
}

.enrolled-courses-grid {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 0.5rem;
    transition: all 0.5s ease;
    max-width: 100%;
    margin: 0;
    padding: 0;
    width: 100%;
    box-sizing: border-box;
}

/* Small Enrolled Course Cards - Clean Panadite Style */
.enrolled-mini-card {
    background: #fafbff;
    border-radius: 0.6rem;
    padding: 0.75rem;
    transition: all 0.3s ease;
    opacity: 1;
    transform: translateY(20px);
    opacity: 0;
    animation: slideInUp 0.5s ease forwards;
    display: flex;
    align-items: center;
    gap: 0.75rem;
    min-height: 85px;
    height: auto;
    width: 100%;
    max-width: 100%;
    border: 1px solid #e8ecf4;
    box-sizing: border-box;
}

.enrolled-mini-card:hover {
    transform: translateY(-2px);
    background: #f0f4ff;
    border-color: #3366ff;
    box-shadow: 0 4px 12px rgba(51, 102, 255, 0.08);
}

.enrolled-course-thumbnail {
    width: 56px;
    height: 56px;
    border-radius: 0.5rem;
    object-fit: cover;
    background: #f0f4ff;
    flex-shrink: 0;
    display: flex;
    align-items: center;
    justify-content: center;
    border: 1px solid #e0ebff;
}

.enrolled-course-thumbnail img {
    width: 100%;
    height: 100%;
    border-radius: 0.5rem;
    object-fit: cover;
}

.enrolled-course-thumbnail i {
    font-size: 1.8rem;
    color: #3366ff;
}

.enrolled-course-info {
    flex: 1;
    display: flex;
    flex-direction: column;
    justify-content: center;
    min-width: 0;
    padding: 0.1rem 0;
}

.enrolled-course-info h4 {
    font-size: 0.8rem;
    font-weight: 700;
    margin: 0 0 0.4rem 0;
    color: #1a2332;
    line-height: 1.3;
    overflow: hidden;
    white-space: nowrap;
    text-overflow: ellipsis;
    max-width: 100%;
}

.enrolled-course-meta {
    display: flex;
    justify-content: flex-start;
    align-items: center;
    font-size: 0.65rem;
    margin-bottom: 0.15rem;
    gap: 0.4rem;
    flex-wrap: wrap;
}

.enrolled-course-progress {
    background: #e8f2ff;
    color: #3366ff;
    padding: 0.15rem 0.5rem;
    border-radius: 1rem;
    font-weight: 600;
    font-size: 0.6rem;
    white-space: nowrap;
    border: 1px solid #d4e6ff;
}

.enrolled-course-grade {
    background: #e8f5e8;
    color: #22c55e;
    padding: 0.15rem 0.5rem;
    border-radius: 1rem;
    font-weight: 600;
    font-size: 0.6rem;
    white-space: nowrap;
    border: 1px solid #bbf7d0;
}

.enrolled-course-instructor {
    font-size: 0.7rem;
    color: #6b7280;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    display: flex;
    align-items: center;
    gap: 0.25rem;
}

.enrolled-course-instructor::before {
    content: "👨‍🏫";
    font-size: 0.7rem;
}

/* Learning Stats Mini Cards */
.learning-stats-mini {
    display: flex;
    gap: 0.5rem;
    margin: 1rem 0 1.5rem 0;
    justify-content: flex-start;
    flex-wrap: wrap;
    width: 100%;
    clear: both;
}

.mini-stat-card {
    background: #fafbff;
    border: 1px solid #e8ecf4;
    border-radius: 0.4rem;
    padding: 0.5rem 0.75rem;
    display: flex;
    align-items: center;
    gap: 0.4rem;
    flex: 1;
    max-width: 140px;
    min-width: 100px;
    height: 56px;
    transition: all 0.3s ease;
    cursor: pointer;
    position: relative;
}

.mini-stat-card:hover {
    transform: translateY(-1px);
    box-shadow: 0 2px 8px rgba(51, 102, 255, 0.06);
}

.mini-stat-card.completed {
    border-color: #bbf7d0;
    background: #f0fdf4;
}

.mini-stat-card.completed:hover {
    border-color: #22c55e;
    background: #ecfdf5;
}

.mini-stat-card.progress {
    border-color: #dbeafe;
    background: #eff6ff;
}

.mini-stat-card.progress:hover {
    border-color: #3366ff;
    background: #dbeafe;
}

.mini-stat-card.hours {
    border-color: #fef3c7;
    background: #fffbeb;
}

.mini-stat-card.hours:hover {
    border-color: #f59e0b;
    background: #fef3c7;
}

.mini-stat-icon {
    width: 28px;
    height: 28px;
    border-radius: 0.3rem;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 0.9rem;
    flex-shrink: 0;
}

.mini-stat-card.completed .mini-stat-icon {
    background: #dcfce7;
    color: #22c55e;
}

.mini-stat-card.progress .mini-stat-icon {
    background: #dbeafe;
    color: #3366ff;
}

.mini-stat-card.hours .mini-stat-icon {
    background: #fef3c7;
    color: #f59e0b;
}

.mini-stat-content {
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: flex-start;
    min-width: 0;
    flex: 1;
    height: 100%;
}

.mini-stat-number {
    font-size: 1.1rem;
    font-weight: 700;
    color: #1a2332;
    line-height: 1;
    margin-bottom: 0.1rem;
}

.mini-stat-label {
    font-size: 0.65rem;
    font-weight: 600;
    color: #6b7280;
    text-transform: uppercase;
    letter-spacing: 0.02em;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    max-width: 100%;
}

/* Responsive Mini Stats */
@media (max-width: 768px) {
    .learning-stats-mini {
        justify-content: space-between;
        gap: 0.4rem;
    }
    
    .mini-stat-card {
        min-width: 90px;
        max-width: 120px;
        padding: 0.4rem 0.6rem;
    }
    
    .mini-stat-icon {
        width: 24px;
        height: 24px;
        font-size: 0.8rem;
    }
    
    .mini-stat-number {
        font-size: 1rem;
    }
    
    .mini-stat-label {
        font-size: 0.6rem;
    }
}

/* Continue Learning - Small Clean Blocks Design */
.continue-learning-grid {
    display: flex;
    flex-direction: column;
    gap: 0.75rem;
    max-width: 100%;
}

.continue-course-block {
    background: white;
    border-radius: 0.6rem;
    border: 1px solid #e8ecf0;
    padding: 0.8rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
    cursor: pointer;
    transition: all 0.25s ease;
    box-shadow: 0 1px 3px rgba(0,0,0,0.05);
}

.continue-course-block:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(78,115,223,0.15);
    border-color: #4e73df;
}

.block-thumbnail {
    width: 45px;
    height: 45px;
    border-radius: 0.4rem;
    overflow: hidden;
    flex-shrink: 0;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
}

.block-thumbnail img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.block-thumbnail i {
    color: white;
    font-size: 1.2rem;
}

.block-content {
    flex: 1;
    min-width: 0;
}

.block-title {
    font-size: 0.85rem;
    font-weight: 600;
    color: #2d3748;
    margin: 0 0 0.3rem 0;
    line-height: 1.2;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.block-progress {
    margin-bottom: 0.25rem;
}

.progress-label {
    font-size: 0.7rem;
    font-weight: 500;
    color: #4e73df;
    display: block;
    margin-bottom: 0.15rem;
}

.mini-progress-bar {
    width: 100%;
    height: 3px;
    background: #f1f5f9;
    border-radius: 1.5px;
    overflow: hidden;
}

.mini-progress-fill {
    height: 100%;
    background: linear-gradient(90deg, #4e73df 0%, #36b9cc 100%);
    border-radius: 1.5px;
    transition: width 0.3s ease;
}

.block-instructor {
    font-size: 0.7rem;
    color: #718096;
    display: flex;
    align-items: center;
    gap: 0.2rem;
}

.block-instructor i {
    font-size: 0.65rem;
}

.block-action {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    background: linear-gradient(135deg, #4e73df 0%, #36b9cc 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 0.9rem;
    flex-shrink: 0;
    transition: all 0.25s ease;
}

.continue-course-block:hover .block-action {
    transform: scale(1.08);
}
    }
}

@media (max-width: 480px) {
    .learning-stats-mini {
        flex-direction: column;
        gap: 0.5rem;
        align-items: stretch;
    }
    
    .mini-stat-card {
        max-width: none;
        min-width: auto;
        flex: none;
    }
}

/* Responsive Featured Courses */
@media (max-width: 1200px) {
    .cycling-courses {
        grid-template-columns: repeat(3, 1fr);
        max-width: 650px;
    }
}

@media (max-width: 992px) {
    .featured-courses-card {
        padding: 1.5rem;
    }
    
    .cycling-courses {
        grid-template-columns: repeat(2, 1fr);
        max-width: 450px;
        margin: 0 auto;
    }
}

@media (max-width: 768px) {
    .welcome-banner {
        padding: 1.5rem;
    }
    
    .cycling-courses {
        grid-template-columns: repeat(2, 1fr);
        gap: 0.4rem;
        max-width: 400px;
    }
    
    .mini-course-card {
        padding: 0.4rem;
        height: 60px;
        min-width: 160px;
    }
    
    .course-thumbnail {
        width: 40px;
        height: 40px;
    }
    
    .course-info h4 {
        font-size: 0.65rem;
    }
    
    .course-meta {
        font-size: 0.55rem;
    }
    
    .course-price {
        font-size: 0.6rem;
    }
    
    .course-level {
        font-size: 0.5rem;
        padding: 0.03rem 0.25rem;
    }
}

@media (max-width: 576px) {
    .cycling-courses {
        grid-template-columns: 1fr;
        max-width: 220px;
    }
    
    .mini-course-card {
        min-width: 200px;
        height: 55px;
    }
    
    .course-thumbnail {
        width: 35px;
        height: 35px;
    }
    
    .cycling-header h3 {
        font-size: 1rem;
    }
}

/* Overall Progress Section */
.progress-section {
    margin-bottom: 2rem;
}

.progress-card {
    background: white;
    border-radius: 1rem;
    padding: 2rem;
    box-shadow: 0 0.25rem 1rem rgba(0, 0, 0, 0.1);
    border: 1px solid #e3e6f0;
    display: grid;
    grid-template-columns: auto 1fr;
    gap: 2rem;
    align-items: center;
}

.progress-circle-container {
    display: flex;
    justify-content: center;
    align-items: center;
}

.circular-progress {
    position: relative;
    width: 140px;
    height: 140px;
}

.progress-ring {
    position: relative;
    width: 100%;
    height: 100%;
}

.progress-svg {
    width: 100%;
    height: 100%;
    transform: rotate(-90deg);
}

.progress-bg {
    fill: none;
    stroke: #f1f3f4;
    stroke-width: 8;
}

.progress-bar {
    fill: none;
    stroke: #4e73df;
    stroke-width: 8;
    stroke-linecap: round;
    stroke-dasharray: 339.29; /* 2 * π * 54 */
    stroke-dashoffset: 339.29;
    transition: stroke-dashoffset 2s ease-in-out;
}

.progress-text {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    text-align: center;
}

.progress-percentage {
    display: block;
    font-size: 1.8rem;
    font-weight: 700;
    color: #2c3e50;
    line-height: 1;
}

.progress-label {
    display: block;
    font-size: 0.8rem;
    color: #5a5c69;
    margin-top: 0.25rem;
    font-weight: 500;
}

.progress-details h3 {
    margin: 0 0 0.5rem 0;
    color: #2c3e50;
    font-size: 1.5rem;
    font-weight: 600;
}

.progress-details p {
    margin: 0 0 1.5rem 0;
    color: #5a5c69;
    line-height: 1.5;
}

.progress-stats {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 1rem;
}

.progress-stat {
    text-align: center;
    padding: 1rem;
    background: #f8f9fc;
    border-radius: 0.5rem;
    border: 1px solid #e3e6f0;
}

.progress-stat .stat-number {
    display: block;
    font-size: 1.5rem;
    font-weight: 700;
    color: #4e73df;
    line-height: 1;
}

.progress-stat .stat-label {
    display: block;
    font-size: 0.8rem;
    color: #5a5c69;
    margin-top: 0.25rem;
    font-weight: 500;
}

/* Responsive Progress Card */
@media (max-width: 992px) {
    .progress-card {
        grid-template-columns: 1fr;
        text-align: center;
        gap: 1.5rem;
    }
    
    .progress-stats {
        grid-template-columns: repeat(3, 1fr);
        gap: 0.75rem;
    }
}

@media (max-width: 768px) {
    .progress-card {
        padding: 1.5rem;
    }
    
    .circular-progress {
        width: 120px;
        height: 120px;
    }
    
    .progress-percentage {
        font-size: 1.5rem;
    }
    
    .progress-stats {
        grid-template-columns: 1fr;
        gap: 0.5rem;
    }
    
    .progress-stat {
        padding: 0.75rem;
    }
}

/* Modern Card System */
.modern-card {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 0.25rem 1rem rgba(0, 0, 0, 0.08);
    border: 1px solid #e3e6f0;
    overflow: hidden;
    transition: all 0.3s ease;
    margin-bottom: 2rem;
}

.modern-card:hover {
    box-shadow: 0 0.5rem 2rem rgba(0, 0, 0, 0.12);
    transform: translateY(-2px);
}

.modern-card.full-width {
    grid-column: 1 / -1;
}

/* Card Headers */
.card-header {
    padding: 1.5rem 2rem 1rem 2rem;
    border-bottom: 1px solid #f1f3f4;
    background: linear-gradient(135deg, #f8f9fc 0%, #ffffff 100%);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.header-content h3 {
    margin: 0;
    color: #2c3e50;
    font-size: 1.2rem;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.header-content h3 i {
    color: #4e73df;
    font-size: 1.3rem;
}

.course-count {
    background: #e3f2fd;
    color: #1976d2;
    padding: 0.25rem 0.75rem;
    border-radius: 1rem;
    font-size: 0.8rem;
    font-weight: 600;
    margin-left: 1rem;
}

.subtitle {
    color: #5a5c69;
    font-size: 0.9rem;
    margin-top: 0.25rem;
    display: block;
}

.view-all-btn {
    color: #4e73df;
    text-decoration: none;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    border-radius: 0.5rem;
    transition: all 0.3s ease;
    font-size: 0.9rem;
}

.view-all-btn:hover {
    background: #4e73df;
    color: white;
    transform: translateX(3px);
}

.activity-indicator {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #1cc88a;
    font-size: 0.8rem;
    font-weight: 600;
}

.indicator-dot {
    width: 8px;
    height: 8px;
    background: #1cc88a;
    border-radius: 50%;
    animation: pulse 2s infinite;
}

@keyframes pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.4; }
}

/* Card Content */
.card-content {
    padding: 2rem;
}

/* Modern Empty States */
.modern-empty-state {
    text-align: center;
    padding: 3rem 2rem;
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 1.5rem;
}

.empty-illustration {
    width: 80px;
    height: 80px;
    background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-bottom: 1rem;
    position: relative;
    overflow: hidden;
}

.empty-illustration::before {
    content: '';
    position: absolute;
    top: -50%;
    left: -50%;
    width: 200%;
    height: 200%;
    background: linear-gradient(45deg, transparent, rgba(255,255,255,0.3), transparent);
    animation: shine 3s infinite;
}

@keyframes shine {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.empty-illustration i {
    font-size: 2.5rem;
    color: white;
    z-index: 1;
    position: relative;
}

.empty-content h4 {
    margin: 0 0 0.5rem 0;
    color: #2c3e50;
    font-size: 1.3rem;
    font-weight: 600;
}

.empty-content p {
    margin: 0 0 1.5rem 0;
    color: #5a5c69;
    line-height: 1.6;
    max-width: 300px;
}

.modern-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    background: #4e73df;
    color: white;
    padding: 0.75rem 1.5rem;
    border-radius: 0.5rem;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    box-shadow: 0 4px 15px rgba(78, 115, 223, 0.3);
}

.modern-btn:hover {
    background: #2e59d9;
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(78, 115, 223, 0.4);
    color: white;
}

/* Activity Preview */
.activity-preview {
    display: flex;
    flex-direction: column;
    gap: 1rem;
    margin-top: 1.5rem;
}

.preview-item {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 0.75rem;
    background: #f8f9fc;
    border-radius: 0.5rem;
    border-left: 3px solid #4e73df;
}

.preview-item i {
    color: #4e73df;
    font-size: 1.1rem;
}

.preview-item span {
    color: #5a5c69;
    font-size: 0.9rem;
}

/* Modern Actions Grid */
.modern-actions-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
}

.modern-action-card {
    background: linear-gradient(135deg, #f8f9fc 0%, #ffffff 100%);
    border: 2px solid #e3e6f0;
    border-radius: 1rem;
    padding: 1.5rem;
    text-decoration: none;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 1rem;
    position: relative;
    overflow: hidden;
}

.modern-action-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 4px;
    height: 100%;
    background: #4e73df;
    transform: scaleY(0);
    transition: transform 0.3s ease;
}

.modern-action-card:hover::before {
    transform: scaleY(1);
}

.modern-action-card:hover {
    border-color: #4e73df;
    box-shadow: 0 8px 25px rgba(78, 115, 223, 0.15);
    transform: translateY(-3px);
}

.action-icon-wrapper {
    position: relative;
    width: 60px;
    height: 60px;
    border-radius: 1rem;
    display: flex;
    align-items: center;
    justify-content: center;
    flex-shrink: 0;
}

.modern-action-card.courses .action-icon-wrapper {
    background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
}

.modern-action-card.assignments .action-icon-wrapper {
    background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
}

.modern-action-card.calendar .action-icon-wrapper {
    background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
}

.modern-action-card.certificates .action-icon-wrapper {
    background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%);
}

.action-icon-wrapper i {
    font-size: 1.8rem;
    color: white;
}

.action-badge {
    position: absolute;
    top: -5px;
    right: -5px;
    background: #dc3545;
    color: white;
    font-size: 0.7rem;
    font-weight: 700;
    padding: 0.2rem 0.5rem;
    border-radius: 1rem;
    min-width: 20px;
    text-align: center;
}

.action-content {
    flex: 1;
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.action-content h4 {
    margin: 0;
    color: #2c3e50;
    font-size: 1.1rem;
    font-weight: 600;
}

.action-content p {
    margin: 0;
    color: #5a5c69;
    font-size: 0.9rem;
    line-height: 1.4;
}

.action-arrow {
    color: #4e73df;
    font-size: 1.2rem;
    transition: transform 0.3s ease;
}

.modern-action-card:hover .action-arrow {
    transform: translateX(5px);
}

/* Dashboard Grid */
.dashboard-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
    gap: 2rem;
    margin-bottom: 2rem;
}

/* Responsive Design */
@media (max-width: 992px) {
    .dashboard-grid {
        grid-template-columns: 1fr;
    }
    
    .modern-actions-grid {
        grid-template-columns: 1fr;
    }
    
    .card-header {
        padding: 1rem 1.5rem 0.75rem 1.5rem;
    }
    
    .card-content {
        padding: 1.5rem;
    }
}

@media (max-width: 768px) {
    .modern-empty-state {
        padding: 2rem 1rem;
    }
    
    .empty-illustration {
        width: 60px;
        height: 60px;
    }
    
    .empty-illustration i {
        font-size: 2rem;
    }
    
    .modern-action-card {
        padding: 1rem;
    }
    
    .action-icon-wrapper {
        width: 50px;
        height: 50px;
    }
    
    .action-icon-wrapper i {
        font-size: 1.5rem;
    }
}

/* Statistics Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    border-radius: 0.75rem;
    padding: 1.5rem;
    box-shadow: 0 0.125rem 0.75rem rgba(0, 0, 0, 0.1);
    border: 1px solid #e3e6f0;
    transition: all 0.2s ease;
}

.stat-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 0.25rem 1.25rem rgba(0, 0, 0, 0.15);
}

.stat-icon {
    width: 3rem;
    height: 3rem;
    border-radius: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-bottom: 1rem;
    font-size: 1.25rem;
    color: white;
}

.stat-icon.bg-primary { background: #4e73df; }
.stat-icon.bg-success { background: #1cc88a; }
.stat-icon.bg-warning { background: #f6c23e; color: #333; }
.stat-icon.bg-info { background: #36b9cc; }

.stat-content {
    margin-bottom: 1rem;
}

.stat-number {
    font-size: 2rem;
    font-weight: 700;
    color: #5a5c69;
    line-height: 1;
}

.stat-label {
    font-size: 0.875rem;
    color: #858796;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    margin-top: 0.25rem;
}

.stat-action a {
    color: #4e73df;
    text-decoration: none;
    font-weight: 500;
    font-size: 0.875rem;
    display: flex;
    align-items: center;
    gap: 0.25rem;
}

.stat-action a:hover {
    color: #2e59d9;
}

.stat-progress {
    margin-top: 1rem;
}

.progress-bar {
    width: 100%;
    height: 6px;
    background: #eaecf4;
    border-radius: 3px;
    overflow: hidden;
    margin-bottom: 0.5rem;
}

.progress-fill {
    height: 100%;
    background: #1cc88a;
    transition: width 0.3s ease;
}

.progress-text {
    font-size: 0.65rem;
    color: #858796;
    margin-top: 0.25rem;
}

/* Dashboard Grid */
.dashboard-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 2rem;
    margin-bottom: 2rem;
}

.dashboard-section {
    background: white;
    border-radius: 0.75rem;
    border: 1px solid #e3e6f0;
    overflow: hidden;
}

.section-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1.25rem 1.5rem;
    border-bottom: 1px solid #e3e6f0;
    background: #f8f9fc;
}

.section-header h3 {
    margin: 0;
    font-size: 1.1rem;
    font-weight: 600;
    color: #5a5c69;
}

.view-all-link {
    color: #4e73df;
    text-decoration: none;
    font-size: 0.875rem;
    font-weight: 500;
}

.view-all-link:hover {
    color: #2e59d9;
}

.section-content {
    padding: 1.5rem;
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 2rem 1rem;
}

.empty-icon {
    width: 4rem;
    height: 4rem;
    margin: 0 auto 1rem;
    background: #f8f9fc;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #858796;
    font-size: 1.5rem;
}

.empty-state h4 {
    margin: 0 0 0.5rem 0;
    color: #5a5c69;
    font-weight: 600;
}

.empty-state p {
    margin: 0 0 1rem 0;
    color: #858796;
    font-size: 0.875rem;
}

.btn-sm {
    padding: 0.5rem 1rem;
    font-size: 0.875rem;
}

/* Quick Actions */
.quick-actions {
    background: white;
    border-radius: 0.75rem;
    padding: 1.5rem;
    border: 1px solid #e3e6f0;
}

.quick-actions h3 {
    margin: 0 0 1.5rem 0;
    color: #5a5c69;
    font-weight: 600;
}

.actions-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1rem;
}

.action-card {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 1rem;
    border: 1px solid #e3e6f0;
    border-radius: 0.5rem;
    text-decoration: none;
    color: inherit;
    transition: all 0.2s ease;
}

.action-card:hover {
    background: #f8f9fc;
    border-color: #4e73df;
    color: inherit;
    transform: translateY(-1px);
}

.action-icon {
    width: 2.5rem;
    height: 2.5rem;
    background: #4e73df;
    color: white;
    border-radius: 0.375rem;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.1rem;
    flex-shrink: 0;
}

.action-text h4 {
    margin: 0 0 0.25rem 0;
    font-size: 0.95rem;
    font-weight: 600;
    color: #5a5c69;
}

.action-text p {
    margin: 0;
    font-size: 0.8rem;
    color: #858796;
}

/* Responsive Design */
@media (max-width: 768px) {
    .dashboard-content {
        padding: 1rem;
    }
    
    .banner-content {
        flex-direction: column;
        text-align: center;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
    }
    
    .dashboard-grid {
        grid-template-columns: 1fr;
        gap: 1.5rem;
    }
    
    .actions-grid {
        grid-template-columns: 1fr;
    }
    
    .welcome-text h1 {
        font-size: 1.5rem;
    }
}

@media (max-width: 576px) {
    .welcome-actions {
        width: 100%;
        justify-content: center;
    }
    
    .welcome-actions .btn {
        flex: 1;
        justify-content: center;
    }
}

/* Modern Statistics Grid */
.modern-stats-grid {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 1rem;
    margin-bottom: 2rem;
}

.modern-stat-card {
    background: white;
    border-radius: 1rem;
    padding: 0;
    box-shadow: 0 0.25rem 1rem rgba(0, 0, 0, 0.08);
    border: 1px solid #e3e6f0;
    transition: all 0.4s ease;
    position: relative;
    overflow: hidden;
    cursor: pointer;
}

.modern-stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(90deg, #4e73df, #36b9cc);
    transform: scaleX(0);
    transition: transform 0.4s ease;
}

.modern-stat-card:hover::before {
    transform: scaleX(1);
}

.modern-stat-card:hover {
    transform: translateY(-8px);
    box-shadow: 0 1rem 3rem rgba(0, 0, 0, 0.15);
}

/* Stat Card Header */
.stat-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1rem 1rem 0 1rem;
    position: relative;
}

.stat-icon {
    width: 40px;
    height: 40px;
    border-radius: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    position: relative;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
}

.stat-icon i {
    font-size: 1.2rem;
    color: white;
}

/* Keep Original Icon Colors */
.bg-primary {
    background: linear-gradient(135deg, #4e73df 0%, #3c65d4 100%);
}

.bg-success {
    background: linear-gradient(135deg, #1cc88a 0%, #17a673 100%);
}

.bg-warning {
    background: linear-gradient(135deg, #f6c23e 0%, #dda20a 100%);
}

.bg-info {
    background: linear-gradient(135deg, #36b9cc 0%, #2c9faf 100%);
}

/* Stat Header Elements */
.stat-trend {
    color: #1cc88a;
    font-size: 1.2rem;
    opacity: 0.7;
    transition: all 0.3s ease;
}

.modern-stat-card:hover .stat-trend {
    opacity: 1;
    transform: scale(1.1);
}

.stat-badge {
    background: linear-gradient(135deg, #1cc88a 0%, #17a673 100%);
    color: white;
    padding: 0.3rem 0.6rem;
    border-radius: 0.8rem;
    font-size: 0.7rem;
    font-weight: 700;
    box-shadow: 0 2px 8px rgba(28, 200, 138, 0.3);
}

.stat-alert {
    color: #f6c23e;
    font-size: 1.2rem;
    animation: pulse-alert 2s infinite;
}

@keyframes pulse-alert {
    0%, 100% { opacity: 1; transform: scale(1); }
    50% { opacity: 0.6; transform: scale(1.1); }
}

.stat-celebration {
    color: #f1c40f;
    font-size: 1.2rem;
    animation: celebrate 3s infinite;
}

@keyframes celebrate {
    0%, 90%, 100% { transform: scale(1) rotate(0deg); }
    5% { transform: scale(1.2) rotate(10deg); }
    10% { transform: scale(1.1) rotate(-10deg); }
    15% { transform: scale(1.2) rotate(10deg); }
}

/* Stat Card Body */
.stat-body {
    padding: 0.75rem 1rem;
    text-align: left;
}

.stat-number {
    font-size: 1.8rem;
    font-weight: 700;
    color: #2c3e50;
    line-height: 1;
    margin-bottom: 0.25rem;
    background: linear-gradient(135deg, #2c3e50 0%, #4e73df 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.stat-label {
    font-size: 0.75rem;
    font-weight: 600;
    color: #5a5c69;
    margin-bottom: 0.1rem;
    text-transform: uppercase;
    letter-spacing: 0.3px;
}

.stat-description {
    font-size: 0.65rem;
    color: #858796;
    font-style: italic;
    opacity: 0.8;
}

/* Stat Progress */
.stat-progress {
    padding: 0 1rem 0.5rem 1rem;
}

.progress-container {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.progress-bar {
    width: 100%;
    height: 8px;
    background: #e3e6f0;
    border-radius: 4px;
    overflow: hidden;
    position: relative;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(90deg, #1cc88a 0%, #17a673 100%);
    border-radius: 4px;
    transition: width 1s ease;
    position: relative;
}

.progress-fill::after {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
    animation: shimmer 2s infinite;
}

@keyframes shimmer {
    0% { transform: translateX(-100%); }
    100% { transform: translateX(100%); }
}

.progress-text {
    font-size: 0.8rem;
    color: #5a5c69;
    font-weight: 600;
    text-align: right;
}

/* Stat Footer */
.stat-footer {
    padding: 0 1.5rem 1.5rem 1.5rem;
    border-top: 1px solid #f1f3f4;
    margin-top: 0.5rem;
    padding-top: 1rem;
}

.stat-link {
    display: flex;
    align-items: center;
    justify-content: space-between;
    color: #4e73df;
    text-decoration: none;
    font-weight: 600;
    font-size: 0.9rem;
    transition: all 0.3s ease;
    padding: 0.5rem 0;
}

.stat-link:hover {
    color: #2e59d9;
    transform: translateX(5px);
}

.stat-link i {
    font-size: 1.1rem;
    transition: transform 0.3s ease;
}

.stat-link:hover i {
    transform: translateX(3px);
}

/* Specific Card Styling */
.modern-stat-card.enrolled:hover {
    border-color: #4e73df;
}

.modern-stat-card.completed:hover {
    border-color: #1cc88a;
}

.modern-stat-card.assignments:hover {
    border-color: #f6c23e;
}

.modern-stat-card.certificates:hover {
    border-color: #36b9cc;
}

/* Responsive Design for Stats */
@media (max-width: 1200px) {
    .modern-stats-grid {
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 1.5rem;
    }
}

@media (max-width: 768px) {
    .modern-stats-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .stat-header {
        padding: 1rem 1rem 0 1rem;
    }
    
    .stat-body {
        padding: 0.75rem 1rem;
    }
    
    .stat-footer {
        padding: 0 1rem 1rem 1rem;
    }
    
    .stat-icon {
        width: 50px;
        height: 50px;
    }
    
    .stat-icon i {
        font-size: 1.5rem;
    }
    
    .stat-number {
        font-size: 2.5rem;
    }
}

@media (max-width: 480px) {
    .modern-stats-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 0.75rem;
    }
    
    .modern-stat-card {
        min-height: auto;
    }
    
    .stat-number {
        font-size: 2rem;
    }
    
    .stat-label {
        font-size: 0.85rem;
    }
    
    .stat-description {
        font-size: 0.75rem;
    }
}
</style>

        </div> <!-- Close home-content -->
    </section> <!-- Close home-section -->

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Dashboard JavaScript -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script>
        // Course Cycling Management
        class CourseCycling {
            constructor() {
                this.currentSet = 0;
                this.allCourses = [];
                this.isLoading = false;
                this.cyclingInterval = null;
                this.init();
            }
            
            init() {
                this.loadCourses();
                this.startCycling();
                this.setupResponsive();
            }
            
            // Determine how many courses to show based on screen size
            getCoursesPerSet() {
                const width = $(window).width();
                if (width <= 576) return 2; // Mobile small
                if (width <= 768) return 3; // Mobile
                if (width <= 1200) return 4; // Tablet
                return 6; // Desktop
            }
            
            // Load courses from API
            async loadCourses() {
                if (this.isLoading) return;
                this.isLoading = true;
                
                try {
                    const response = await $.ajax({
                        url: 'api/get_random_courses.php',
                        method: 'GET',
                        data: { limit: 18 }, // Load more courses for cycling
                        dataType: 'json',
                        timeout: 10000
                    });
                    
                    if (response.success && response.courses.length > 0) {
                        this.allCourses = response.courses;
                        this.displayCourses();
                    } else {
                        this.showError('No courses available');
                    }
                } catch (error) {
                    console.error('Course loading error:', error);
                    this.showError('Failed to load courses');
                } finally {
                    this.isLoading = false;
                }
            }
            
            // Display current set of courses
            displayCourses() {
                if (this.allCourses.length === 0) return;
                
                const coursesPerSet = this.getCoursesPerSet();
                const startIndex = (this.currentSet * coursesPerSet) % this.allCourses.length;
                const coursesToShow = [];
                
                // Get courses for current set (with wrapping)
                for (let i = 0; i < coursesPerSet; i++) {
                    const courseIndex = (startIndex + i) % this.allCourses.length;
                    coursesToShow.push(this.allCourses[courseIndex]);
                }
                
                // Create HTML for courses
                const coursesHtml = coursesToShow.map(course => this.createCourseCard(course)).join('');
                
                // Fade out, update content, fade in
                $('#cyclingCourses').fadeOut(300, () => {
                    $('#cyclingCourses').html(coursesHtml).fadeIn(500);
                });
                
                // Update indicators
                this.updateIndicators();
            }
            
            // Create individual course card HTML
            createCourseCard(course) {
                const priceDisplay = course.price > 0 ? `R${course.price}` : 'Free';
                
                return `
                    <div class="mini-course-card" data-course-id="${course.id}">
                        <img src="${course.thumbnail}" alt="${course.title}" class="course-thumbnail" 
                             onerror="this.src='/assets/img/courses/default-course.jpg'">
                        <div class="course-info">
                            <h4 title="${course.title}">${course.title}</h4>
                            <div class="course-meta">
                                <span class="course-level">${course.level}</span>
                                <span class="course-rating">★ ${course.rating}</span>
                            </div>
                            <div class="course-price">${priceDisplay}</div>
                        </div>
                    </div>
                `;
            }
            
            // Update cycling indicators
            updateIndicators() {
                const maxSets = Math.ceil(this.allCourses.length / this.getCoursesPerSet());
                const activeIndex = this.currentSet % 3; // Always show 3 dots
                
                $('.indicator-dot').removeClass('active').eq(activeIndex).addClass('active');
            }
            
            // Start automatic cycling
            startCycling() {
                this.stopCycling(); // Clear any existing interval
                
                this.cyclingInterval = setInterval(() => {
                    if (this.allCourses.length > 0) {
                        this.nextSet();
                    }
                }, 4000); // Change every 4 seconds
            }
            
            // Stop cycling
            stopCycling() {
                if (this.cyclingInterval) {
                    clearInterval(this.cyclingInterval);
                    this.cyclingInterval = null;
                }
            }
            
            // Move to next set of courses
            nextSet() {
                const maxSets = Math.ceil(this.allCourses.length / this.getCoursesPerSet());
                this.currentSet = (this.currentSet + 1) % Math.max(maxSets, 1);
                this.displayCourses();
            }
            
            // Handle responsive changes
            setupResponsive() {
                let resizeTimer;
                $(window).on('resize', () => {
                    clearTimeout(resizeTimer);
                    resizeTimer = setTimeout(() => {
                        this.displayCourses(); // Refresh layout on resize
                    }, 300);
                });
            }
            
            // Show error message
            showError(message) {
                $('#cyclingCourses').html(`
                    <div class="course-loading">
                        <i class='bx bx-error-circle' style="font-size: 2rem; margin-bottom: 0.5rem;"></i>
                        <p>${message}</p>
                    </div>
                `);
            }
        }

        // Circular Progress Animation
        function animateProgress() {
            const progressBars = document.querySelectorAll('.progress-bar');
            
            progressBars.forEach(bar => {
                const progress = parseInt(bar.getAttribute('data-progress'));
                const circumference = 2 * Math.PI * 54; // radius = 54
                const progressOffset = circumference - (progress / 100) * circumference;
                
                // Start with full offset (0% progress)
                bar.style.strokeDashoffset = circumference;
                
                // Animate to actual progress after a short delay
                setTimeout(() => {
                    bar.style.strokeDashoffset = progressOffset;
                }, 500);
            });
        }

        // Initialize floating icon animations
        function initFloatingIcons() {
            const floatingIcons = document.querySelectorAll('.floating-icon');
            
            floatingIcons.forEach(icon => {
                const delay = icon.getAttribute('data-delay') || '0s';
                icon.style.animationDelay = delay;
            });
        }

        // Load enrolled courses with grades
        function loadEnrolledCourses() {
            $.ajax({
                url: 'api/get_enrolled_courses.php',
                type: 'GET',
                dataType: 'json',
                success: function(response) {
                    if (response.success && response.courses) {
                        displayEnrolledCourses(response.courses);
                    } else {
                        showCoursesError('Failed to load courses');
                    }
                },
                error: function() {
                    showCoursesError('Error connecting to server');
                }
            });
        }

        // Helper function to get course icon based on title
        function getCourseIcon(title) {
            const titleLower = title.toLowerCase();
            if (titleLower.includes('programming') || titleLower.includes('code')) return 'bx-code-alt';
            if (titleLower.includes('design') || titleLower.includes('ui')) return 'bx-palette';
            if (titleLower.includes('data') || titleLower.includes('analytics')) return 'bx-bar-chart-alt-2';
            if (titleLower.includes('web') || titleLower.includes('html')) return 'bx-world';
            if (titleLower.includes('mobile') || titleLower.includes('app')) return 'bx-mobile-alt';
            if (titleLower.includes('security') || titleLower.includes('cyber')) return 'bx-shield-quarter';
            if (titleLower.includes('ai') || titleLower.includes('machine learning')) return 'bx-brain';
            if (titleLower.includes('network') || titleLower.includes('cloud')) return 'bx-cloud';
            if (titleLower.includes('database') || titleLower.includes('sql')) return 'bx-data';
            if (titleLower.includes('javascript') || titleLower.includes('js')) return 'bx-code-block';
            if (titleLower.includes('python')) return 'bx-code-curly';
            if (titleLower.includes('java')) return 'bx-coffee';
            if (titleLower.includes('business') || titleLower.includes('management')) return 'bx-briefcase';
            if (titleLower.includes('marketing') || titleLower.includes('social')) return 'bx-trending-up';
            if (titleLower.includes('powerbi') || titleLower.includes('power bi')) return 'bx-bar-chart-alt-2';
            if (titleLower.includes('cyber') || titleLower.includes('security')) return 'bx-shield-quarter';
            if (titleLower.includes('entrepreneur') || titleLower.includes('venture')) return 'bx-briefcase';
            if (titleLower.includes('assessment') || titleLower.includes('facilitator')) return 'bx-check-square';
            if (titleLower.includes('computer') || titleLower.includes('science')) return 'bx-desktop';
            return 'bx-book-open'; // Default icon
        }

        // Display enrolled courses - show completely empty state
        function displayEnrolledCourses(courses) {
            const container = $('#enrolledCoursesGrid');
            
            // Show completely empty state - no content
            container.html('');
        }

        $(document).ready(function() {
            const courseCycling = new CourseCycling();
            
            // Load enrolled courses
            loadEnrolledCourses();
            
            // Animate progress circles
            animateProgress();
            
            // Initialize floating icon animations
            initFloatingIcons();
            
            // Handle course card clicks
            $(document).on('click', '.mini-course-card', function() {
                const courseId = $(this).data('course-id');
                if (courseId) {
                    window.location.href = `course-details.php?id=${courseId}`;
                }
            });
            
            // Pause cycling on hover
            $(document).on('mouseenter', '.course-cycling-container', function() {
                courseCycling.stopCycling();
            });
            
            // Resume cycling when not hovering
            $(document).on('mouseleave', '.course-cycling-container', function() {
                courseCycling.startCycling();
            });
            
            // Manual indicator clicks
            $(document).on('click', '.indicator-dot', function() {
                const index = $(this).index();
                courseCycling.currentSet = index;
                courseCycling.displayCourses();
            });
        });

        // Smooth scrolling for internal links
        document.addEventListener('DOMContentLoaded', function() {
            document.querySelectorAll('a[href^="#"]').forEach(anchor => {
                anchor.addEventListener('click', function (e) {
                    e.preventDefault();
                    const target = document.querySelector(this.getAttribute('href'));
                    if (target) {
                        target.scrollIntoView({
                            behavior: 'smooth',
                            block: 'start'
                        });
                    }
                });
            });
            
            console.log('Clean dashboard loaded successfully!');
        });
    </script>

    <!-- Browse Courses Button Styling -->
    <style>
        /* Banner Action Section - Browse Courses Button */
        .banner-action-section {
            margin-top: 1.5rem;
            margin-bottom: 1rem;
            opacity: 0;
            transform: translateY(15px);
            animation: slideUp 0.8s ease-out forwards 0.7s;
        }

        .browse-courses-btn {
            display: inline-flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.875rem 1.75rem;
            background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
            color: #ffffff;
            border-radius: 12px;
            text-decoration: none;
            font-weight: 600;
            font-size: 0.95rem;
            letter-spacing: 0.025em;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            box-shadow: 
                0 4px 14px rgba(245, 158, 11, 0.3),
                0 2px 6px rgba(0, 0, 0, 0.1);
            position: relative;
            overflow: hidden;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .browse-courses-btn::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
            transition: left 0.5s ease;
        }

        .browse-courses-btn:hover {
            transform: translateY(-2px) scale(1.02);
            box-shadow: 
                0 8px 25px rgba(245, 158, 11, 0.4),
                0 4px 12px rgba(0, 0, 0, 0.15);
            background: linear-gradient(135deg, #d97706 0%, #b45309 100%);
            color: #ffffff;
            text-decoration: none;
        }

        .browse-courses-btn:hover::before {
            left: 100%;
        }

        .browse-courses-btn:active {
            transform: translateY(-1px) scale(1.01);
            box-shadow: 
                0 6px 20px rgba(245, 158, 11, 0.35),
                0 3px 8px rgba(0, 0, 0, 0.12);
        }

        .browse-courses-btn i {
            font-size: 1rem;
            transition: transform 0.3s ease;
        }

        .browse-courses-btn:hover i:first-child {
            animation: searchPulse 0.6s ease-in-out;
        }

        .browse-courses-btn:hover i:last-child {
            transform: translateX(3px);
        }

        /* Browse Button Animation */
        @keyframes searchPulse {
            0%, 100% {
                transform: scale(1);
            }
            50% {
                transform: scale(1.2);
            }
        }

        /* Responsive Design for Browse Button */
        @media (max-width: 768px) {
            .banner-action-section {
                margin-top: 1.25rem;
                text-align: center;
            }
            
            .browse-courses-btn {
                width: 100%;
                justify-content: center;
                padding: 1rem 1.5rem;
                font-size: 1rem;
            }
        }

        @media (max-width: 480px) {
            .browse-courses-btn {
                padding: 0.875rem 1.25rem;
                font-size: 0.9rem;
                gap: 0.5rem;
            }
        }
    </style>
</body>
</html>
