<?php
require_once '../auth/functions.php';
requireRole('student');
require_once '../config/db_connect.php';

header('Content-Type: text/html; charset=UTF-8');
?>
<!DOCTYPE html>
<html>
<head>
    <title>🔍 Accreditation Tables Debug</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body { background: #f8f9fa; font-family: monospace; }
        .debug-section { background: white; margin: 20px; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .table { font-size: 0.85rem; }
        .alert-danger { border-left: 4px solid #dc3545; }
        .alert-warning { border-left: 4px solid #ffc107; }
        .alert-success { border-left: 4px solid #198754; }
        pre { background: #f8f9fa; padding: 15px; border-radius: 5px; border: 1px solid #dee2e6; }
    </style>
</head>
<body>
    <div class="container-fluid">
        <h1 class="text-center my-4">🔍 Accreditation Security Debug</h1>
        
        <?php
        $student_id = $_SESSION['user_id'];
        $course_id = isset($_GET['course_id']) ? intval($_GET['course_id']) : 1; // Default to course 1
        
        echo "<div class='alert alert-info'>";
        echo "<strong>🔍 Debug Info:</strong><br>";
        echo "Student ID: $student_id<br>";
        echo "Course ID: $course_id<br>";
        echo "Session Data: " . print_r($_SESSION, true);
        echo "</div>";
        ?>

        <!-- 1. ACCREDITATION REQUESTS TABLE -->
        <div class="debug-section">
            <h3>🎓 Accreditation Requests Table</h3>
            <?php
            $requests_query = "SELECT * FROM accreditation_requests WHERE student_id = ? ORDER BY request_date DESC";
            if ($stmt = $conn->prepare($requests_query)) {
                $stmt->bind_param("i", $student_id);
                $stmt->execute();
                $requests_result = $stmt->get_result();
                
                if ($requests_result->num_rows > 0) {
                    echo "<div class='alert alert-danger'><strong>⚠️ SECURITY ISSUE:</strong> Found " . $requests_result->num_rows . " accreditation requests for this student!</div>";
                    echo "<div class='table-responsive'>";
                    echo "<table class='table table-bordered table-striped'>";
                    echo "<thead class='table-dark'><tr>";
                    echo "<th>Request ID</th><th>Course ID</th><th>Status</th><th>Payment Amount</th><th>Payment Status</th><th>Payment Reference</th><th>Request Date</th><th>Approved By</th>";
                    echo "</tr></thead><tbody>";
                    
                    while ($row = $requests_result->fetch_assoc()) {
                        $statusClass = $row['payment_status'] == 'pending' ? 'table-danger' : ($row['payment_status'] == 'completed' ? 'table-success' : 'table-warning');
                        echo "<tr class='$statusClass'>";
                        echo "<td><strong>" . htmlspecialchars($row['request_id']) . "</strong></td>";
                        echo "<td>" . htmlspecialchars($row['course_id']) . "</td>";
                        echo "<td><span class='badge bg-primary'>" . htmlspecialchars($row['status']) . "</span></td>";
                        echo "<td><strong>R" . htmlspecialchars($row['payment_amount']) . "</strong></td>";
                        echo "<td><span class='badge " . ($row['payment_status'] == 'pending' ? 'bg-danger' : ($row['payment_status'] == 'completed' ? 'bg-success' : 'bg-warning')) . "'>" . htmlspecialchars($row['payment_status']) . "</span></td>";
                        echo "<td>" . htmlspecialchars($row['payment_reference']) . "</td>";
                        echo "<td>" . htmlspecialchars($row['request_date']) . "</td>";
                        echo "<td>" . htmlspecialchars($row['approved_by'] ?? 'NULL') . "</td>";
                        echo "</tr>";
                    }
                    echo "</tbody></table></div>";
                } else {
                    echo "<div class='alert alert-success'>✅ No accreditation requests found - SECURE</div>";
                }
                $stmt->close();
            }
            ?>
        </div>

        <!-- 2. PAYMENT TRANSACTIONS -->
        <div class="debug-section">
            <h3>💳 Payment Transactions (for accreditation)</h3>
            <?php
            // Check payments table for accreditation payments
            $payments_query = "SELECT * FROM payments WHERE user_id = ? AND (item_description LIKE '%accreditation%' OR item_description LIKE '%SITA%' OR payment_reference LIKE 'ACC_%') ORDER BY created_at DESC";
            if ($stmt = $conn->prepare($payments_query)) {
                $stmt->bind_param("i", $student_id);
                $stmt->execute();
                $payments_result = $stmt->get_result();
                
                if ($payments_result->num_rows > 0) {
                    echo "<div class='alert alert-warning'><strong>💳 Found " . $payments_result->num_rows . " accreditation-related payments</strong></div>";
                    echo "<div class='table-responsive'>";
                    echo "<table class='table table-bordered table-striped'>";
                    echo "<thead class='table-dark'><tr>";
                    echo "<th>Payment ID</th><th>Amount</th><th>Status</th><th>Reference</th><th>Description</th><th>Created</th>";
                    echo "</tr></thead><tbody>";
                    
                    while ($row = $payments_result->fetch_assoc()) {
                        $statusClass = $row['status'] == 'pending' ? 'table-warning' : ($row['status'] == 'completed' ? 'table-success' : 'table-danger');
                        echo "<tr class='$statusClass'>";
                        echo "<td>" . htmlspecialchars($row['payment_id']) . "</td>";
                        echo "<td><strong>R" . htmlspecialchars($row['amount']) . "</strong></td>";
                        echo "<td><span class='badge " . ($row['status'] == 'completed' ? 'bg-success' : 'bg-warning') . "'>" . htmlspecialchars($row['status']) . "</span></td>";
                        echo "<td>" . htmlspecialchars($row['payment_reference']) . "</td>";
                        echo "<td>" . htmlspecialchars($row['item_description']) . "</td>";
                        echo "<td>" . htmlspecialchars($row['created_at']) . "</td>";
                        echo "</tr>";
                    }
                    echo "</tbody></table></div>";
                } else {
                    echo "<div class='alert alert-info'>ℹ️ No accreditation payments found</div>";
                }
                $stmt->close();
            }
            ?>
        </div>

        <!-- 3. CURRENT DISPLAY LOGIC DEBUG -->
        <div class="debug-section">
            <h3>🔒 Security Logic Analysis</h3>
            <?php
            // Replicate the exact logic from course-detail.php
            $accreditation_request = null;
            $request_sql = "SELECT ar.*, 
                                   u.first_name as admin_first_name, u.last_name as admin_last_name
                            FROM accreditation_requests ar
                            LEFT JOIN users u ON ar.approved_by = u.user_id
                            WHERE ar.student_id = ? AND ar.course_id = ? 
                            AND ar.status NOT IN ('cancelled', 'rejected', 'certificate_issued')
                            ORDER BY ar.request_date DESC LIMIT 1";
            
            if ($stmt = $conn->prepare($request_sql)) {
                $stmt->bind_param("ii", $student_id, $course_id);
                $stmt->execute();
                $result = $stmt->get_result();
                $accreditation_request = $result->fetch_assoc();
                $stmt->close();
            }

            echo "<pre>";
            echo "<strong>🔍 Current Logic Variables:</strong>\n";
            echo "accreditation_request exists: " . ($accreditation_request ? "YES" : "NO") . "\n";
            if ($accreditation_request) {
                echo "Request ID: " . $accreditation_request['request_id'] . "\n";
                echo "Status: " . $accreditation_request['status'] . "\n";
                echo "Payment Status: " . $accreditation_request['payment_status'] . "\n";
                echo "Payment Amount: R" . $accreditation_request['payment_amount'] . "\n";
                echo "Payment Reference: " . $accreditation_request['payment_reference'] . "\n";
            }
            
            // Check what interface should be displayed
            $show_payment_card = !$accreditation_request || ($accreditation_request && $accreditation_request['payment_status'] == 'pending');
            $show_progress_interface = $accreditation_request && $accreditation_request['payment_status'] == 'completed';
            
            echo "\n<strong>🎯 Interface Display Logic (Course ID $course_id):</strong>\n";
            echo "Show Payment Card: " . ($show_payment_card ? "YES" : "NO") . "\n";
            echo "Show Progress Interface: " . ($show_progress_interface ? "YES" : "NO") . "\n";
            
            if ($accreditation_request && $accreditation_request['payment_status'] == 'pending') {
                echo "\n<strong>⚠️ SECURITY ISSUE DETECTED:</strong>\n";
                echo "- Accreditation request exists but payment is PENDING\n";
                echo "- System may be showing progress interface without confirmed payment\n";
                echo "- This is a security vulnerability!\n";
            }
            
            // Check for requests in other courses
            $other_requests_stmt = $conn->prepare("SELECT request_id, course_id, status, payment_status FROM accreditation_requests WHERE student_id = ? AND course_id != ? ORDER BY request_date DESC");
            $other_requests_stmt->bind_param("ii", $student_id, $course_id);
            $other_requests_stmt->execute();
            $other_requests_result = $other_requests_stmt->get_result();
            
            if ($other_requests_result->num_rows > 0) {
                echo "\n<strong>🔍 REQUESTS IN OTHER COURSES:</strong>\n";
                while ($other_row = $other_requests_result->fetch_assoc()) {
                    echo "- Request ID: " . $other_row['request_id'] . " | Course ID: " . $other_row['course_id'] . " | Status: " . $other_row['status'] . " | Payment: " . $other_row['payment_status'] . "\n";
                }
            }
            echo "</pre>";
            ?>
        </div>

        <!-- 4. RECOMMENDED SECURITY FIX -->
        <div class="debug-section">
            <h3>🔒 Security Fix Recommendations</h3>
            <div class="alert alert-danger">
                <h5>⚠️ CRITICAL SECURITY ISSUES:</h5>
                <ul>
                    <li><strong>Unpaid Access:</strong> Progress interface visible without confirmed payment</li>
                    <li><strong>Weak Logic:</strong> Only checking if request exists, not if payment is completed</li>
                    <li><strong>Data Exposure:</strong> Showing sensitive accreditation workflow before payment</li>
                </ul>
            </div>
            
            <div class="alert alert-success">
                <h5>✅ Required Security Fixes:</h5>
                <ol>
                    <li><strong>Payment Verification:</strong> Only show progress after payment_status = 'completed'</li>
                    <li><strong>Database Cleanup:</strong> Remove unpaid accreditation requests</li>
                    <li><strong>Logic Update:</strong> Proper payment confirmation checks</li>
                    <li><strong>Access Control:</strong> Block all accreditation features until payment confirmed</li>
                </ol>
            </div>
        </div>

        <!-- 5. CLEANUP ACTIONS -->
        <div class="debug-section">
            <h3>🧹 Database Cleanup Actions</h3>
            
            <?php
            // Get all requests for this student to create specific deletion buttons
            $all_requests_stmt = $conn->prepare("SELECT request_id, course_id, status, payment_status FROM accreditation_requests WHERE student_id = ? ORDER BY request_date DESC");
            $all_requests_stmt->bind_param("i", $student_id);
            $all_requests_stmt->execute();
            $all_requests_result = $all_requests_stmt->get_result();
            $all_requests = $all_requests_result->fetch_all(MYSQLI_ASSOC);
            ?>
            
            <div class="row">
                <div class="col-md-4">
                    <a href="debug_accreditation_tables.php?action=delete_all_student&course_id=<?= $course_id ?>" 
                       class="btn btn-danger btn-lg w-100 mb-3"
                       onclick="return confirm('⚠️ Delete ALL accreditation requests for Student ID <?= $student_id ?>?')">
                        🗑️ Delete ALL Requests
                    </a>
                </div>
                <div class="col-md-4">
                    <a href="debug_accreditation_tables.php?action=delete_completed&course_id=<?= $course_id ?>" 
                       class="btn btn-warning btn-lg w-100 mb-3"
                       onclick="return confirm('⚠️ Delete all COMPLETED requests for testing?')">
                        🎯 Delete Completed Only
                    </a>
                </div>
                <div class="col-md-4">
                    <a href="course-detail.php?id=<?= base64_encode(14) ?>&tab=certificate" 
                       class="btn btn-primary btn-lg w-100 mb-3">
                        🔄 Test Course 14
                    </a>
                </div>
            </div>
            
            <?php if (!empty($all_requests)): ?>
            <h5>🎯 Individual Request Actions:</h5>
            <div class="row">
                <?php foreach ($all_requests as $req): ?>
                <div class="col-md-4 mb-2">
                    <a href="debug_accreditation_tables.php?action=delete_specific&request_id=<?= $req['request_id'] ?>&course_id=<?= $course_id ?>" 
                       class="btn btn-outline-danger w-100"
                       onclick="return confirm('Delete Request #<?= $req['request_id'] ?> (Course <?= $req['course_id'] ?>)?')">
                        🗑️ Delete Request #<?= $req['request_id'] ?><br>
                        <small>Course <?= $req['course_id'] ?> | <?= $req['status'] ?></small>
                    </a>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
        </div>

        <?php
        // Handle cleanup actions
        if (isset($_GET['action'])) {
            echo "<div class='debug-section'>";
            echo "<h3>🧹 Cleanup Results</h3>";
            
            $action = $_GET['action'];
            
            if ($action == 'delete_all_student') {
                // Delete ALL requests for this student (any status)
                $delete_query = "DELETE FROM accreditation_requests WHERE student_id = ?";
                if ($stmt = $conn->prepare($delete_query)) {
                    $stmt->bind_param("i", $student_id);
                    if ($stmt->execute()) {
                        $affected = $stmt->affected_rows;
                        echo "<div class='alert alert-success'>✅ Deleted $affected accreditation requests for Student ID $student_id</div>";
                        echo "<script>setTimeout(() => window.location = 'debug_accreditation_tables.php?course_id=$course_id', 2000);</script>";
                    } else {
                        echo "<div class='alert alert-danger'>❌ Failed to delete requests: " . $stmt->error . "</div>";
                    }
                    $stmt->close();
                }
            }
            elseif ($action == 'delete_completed') {
                // Delete only completed requests for this student
                $delete_query = "DELETE FROM accreditation_requests WHERE student_id = ? AND payment_status = 'completed'";
                if ($stmt = $conn->prepare($delete_query)) {
                    $stmt->bind_param("i", $student_id);
                    if ($stmt->execute()) {
                        $affected = $stmt->affected_rows;
                        echo "<div class='alert alert-success'>✅ Deleted $affected completed accreditation requests for Student ID $student_id</div>";
                        echo "<script>setTimeout(() => window.location = 'debug_accreditation_tables.php?course_id=$course_id', 2000);</script>";
                    } else {
                        echo "<div class='alert alert-danger'>❌ Failed to delete completed requests: " . $stmt->error . "</div>";
                    }
                    $stmt->close();
                }
            }
            elseif ($action == 'delete_specific') {
                // Delete specific request by ID
                $request_id_to_delete = intval($_GET['request_id'] ?? 0);
                if ($request_id_to_delete > 0) {
                    $delete_query = "DELETE FROM accreditation_requests WHERE request_id = ? AND student_id = ?";
                    if ($stmt = $conn->prepare($delete_query)) {
                        $stmt->bind_param("ii", $request_id_to_delete, $student_id);
                        if ($stmt->execute()) {
                            $affected = $stmt->affected_rows;
                            if ($affected > 0) {
                                echo "<div class='alert alert-success'>✅ Successfully deleted Request ID #$request_id_to_delete</div>";
                            } else {
                                echo "<div class='alert alert-warning'>⚠️ Request ID #$request_id_to_delete not found or doesn't belong to this student</div>";
                            }
                            echo "<script>setTimeout(() => window.location = 'debug_accreditation_tables.php?course_id=$course_id', 2000);</script>";
                        } else {
                            echo "<div class='alert alert-danger'>❌ Failed to delete Request #$request_id_to_delete: " . $stmt->error . "</div>";
                        }
                        $stmt->close();
                    }
                } else {
                    echo "<div class='alert alert-danger'>❌ Invalid request ID provided</div>";
                }
            }
            // Legacy support for old actions
            elseif ($action == 'delete_all_pending') {
                // Delete ALL pending requests for this student
                $delete_query = "DELETE FROM accreditation_requests WHERE student_id = ? AND payment_status = 'pending'";
                if ($stmt = $conn->prepare($delete_query)) {
                    $stmt->bind_param("i", $student_id);
                    if ($stmt->execute()) {
                        $affected = $stmt->affected_rows;
                        echo "<div class='alert alert-success'>✅ Deleted $affected pending accreditation requests for Student ID $student_id</div>";
                        echo "<script>setTimeout(() => window.location = 'debug_accreditation_tables.php?course_id=$course_id', 2000);</script>";
                    } else {
                        echo "<div class='alert alert-danger'>❌ Failed to delete requests: " . $stmt->error . "</div>";
                    }
                    $stmt->close();
                }
            }
            
            echo "</div>";
        }
        ?>
    </div>
</body>
</html>
