<?php
/**
 * DEBUG Certificate Download - Panadite Academy
 * Helps identify and fix certificate download issues
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h1>🔍 Certificate Download Debug</h1>";

// Start session and include database
require_once '../includes/session_start.php';

echo "<h2>📋 Step 1: Session Check</h2>";
echo "<p><strong>Session Status:</strong> " . (session_status() === PHP_SESSION_ACTIVE ? "✅ Active" : "❌ Inactive") . "</p>";
echo "<p><strong>User ID:</strong> " . ($_SESSION['user_id'] ?? "❌ Not set") . "</p>";
echo "<p><strong>Course ID (from URL):</strong> " . ($_GET['course_id'] ?? "❌ Not provided") . "</p>";

echo "<h2>🗄️ Step 2: Database Connection Test</h2>";

// Try different database connection approaches
$conn = null;
$db_approaches = [
    '../config/db_connect.php',
    '../includes/db_connection.php',
    '../config/database.php'
];

foreach ($db_approaches as $db_file) {
    if (file_exists($db_file)) {
        echo "<p>📁 Found: $db_file</p>";
        try {
            require_once $db_file;
            if (isset($conn) && $conn) {
                echo "<p>✅ Database connected via $db_file</p>";
                break;
            }
        } catch (Exception $e) {
            echo "<p>❌ Failed to connect via $db_file: " . $e->getMessage() . "</p>";
        }
    } else {
        echo "<p>❌ Not found: $db_file</p>";
    }
}

if (!$conn) {
    echo "<p style='color: red;'><strong>❌ NO DATABASE CONNECTION ESTABLISHED</strong></p>";
    echo "<h3>🔧 Manual Connection Attempt</h3>";
    
    // Try manual connection
    try {
        $conn = new mysqli("localhost", "your_username", "your_password", "your_database");
        if ($conn->connect_error) {
            throw new Exception("Connection failed: " . $conn->connect_error);
        }
        echo "<p>✅ Manual connection successful</p>";
    } catch (Exception $e) {
        echo "<p>❌ Manual connection failed: " . $e->getMessage() . "</p>";
        echo "<p><strong>Please update database credentials in the script or fix the database connection files.</strong></p>";
    }
}

echo "<h2>📊 Step 3: Database Structure Check</h2>";

if ($conn) {
    // Check if required tables exist
    $required_tables = ['users', 'courses', 'enrollments'];
    
    foreach ($required_tables as $table) {
        $check_query = "SHOW TABLES LIKE '$table'";
        $result = $conn->query($check_query);
        
        if ($result && $result->num_rows > 0) {
            echo "<p>✅ Table '$table' exists</p>";
            
            // Show table structure
            $describe = $conn->query("DESCRIBE $table");
            if ($describe) {
                echo "<details><summary>📋 View $table structure</summary>";
                echo "<table border='1' style='border-collapse: collapse; margin: 10px 0;'>";
                echo "<tr><th>Column</th><th>Type</th><th>Null</th><th>Key</th></tr>";
                while ($row = $describe->fetch_assoc()) {
                    echo "<tr>";
                    echo "<td>{$row['Field']}</td>";
                    echo "<td>{$row['Type']}</td>";
                    echo "<td>{$row['Null']}</td>";
                    echo "<td>{$row['Key']}</td>";
                    echo "</tr>";
                }
                echo "</table></details>";
            }
        } else {
            echo "<p>❌ Table '$table' missing</p>";
        }
    }
} else {
    echo "<p>❌ Cannot check database structure without connection</p>";
}

echo "<h2>🧪 Step 4: Test Query</h2>";

$user_id = $_SESSION['user_id'] ?? 39; // Default for testing
$course_id = intval($_GET['course_id'] ?? 14); // Default for testing

echo "<p><strong>Testing with:</strong> User ID = $user_id, Course ID = $course_id</p>";

if ($conn) {
    // Test the exact query from download_certificate.php
    $test_query = "
        SELECT c.title, c.description, u.first_name, u.last_name, e.enrollment_date
        FROM courses c
        JOIN enrollments e ON c.course_id = e.course_id
        JOIN users u ON e.user_id = u.user_id
        WHERE c.course_id = ? AND u.user_id = ?
    ";
    
    echo "<p><strong>Query:</strong></p>";
    echo "<pre>" . htmlspecialchars($test_query) . "</pre>";
    
    $stmt = $conn->prepare($test_query);
    
    if (!$stmt) {
        echo "<p style='color: red;'>❌ PREPARE FAILED: " . $conn->error . "</p>";
        
        // Try simpler query to isolate the issue
        echo "<h3>🔍 Testing Simpler Queries</h3>";
        
        $simple_queries = [
            "SELECT * FROM users WHERE user_id = $user_id",
            "SELECT * FROM courses WHERE course_id = $course_id",
            "SELECT * FROM enrollments WHERE user_id = $user_id AND course_id = $course_id"
        ];
        
        foreach ($simple_queries as $simple_query) {
            echo "<p><strong>Testing:</strong> " . htmlspecialchars($simple_query) . "</p>";
            $result = $conn->query($simple_query);
            if ($result) {
                echo "<p>✅ Query successful, rows: " . $result->num_rows . "</p>";
                if ($result->num_rows > 0) {
                    $row = $result->fetch_assoc();
                    echo "<pre>" . print_r($row, true) . "</pre>";
                }
            } else {
                echo "<p>❌ Query failed: " . $conn->error . "</p>";
            }
        }
        
    } else {
        echo "<p>✅ PREPARE SUCCESSFUL</p>";
        
        if ($stmt->bind_param("ii", $course_id, $user_id)) {
            echo "<p>✅ BIND PARAMETERS SUCCESSFUL</p>";
            
            if ($stmt->execute()) {
                echo "<p>✅ EXECUTE SUCCESSFUL</p>";
                
                $result = $stmt->get_result();
                if ($result && $result->num_rows > 0) {
                    echo "<p>✅ DATA FOUND</p>";
                    $data = $result->fetch_assoc();
                    echo "<pre>" . print_r($data, true) . "</pre>";
                    
                    echo "<h3>🎓 Certificate Would Be Generated For:</h3>";
                    echo "<p><strong>Student:</strong> " . htmlspecialchars($data['first_name'] . ' ' . $data['last_name']) . "</p>";
                    echo "<p><strong>Course:</strong> " . htmlspecialchars($data['title']) . "</p>";
                    echo "<p><strong>Completion Date:</strong> " . date('F j, Y', strtotime($data['enrollment_date'])) . "</p>";
                    
                } else {
                    echo "<p>❌ NO DATA FOUND - Student may not be enrolled in this course</p>";
                }
            } else {
                echo "<p>❌ EXECUTE FAILED: " . $stmt->error . "</p>";
            }
        } else {
            echo "<p>❌ BIND PARAMETERS FAILED</p>";
        }
        $stmt->close();
    }
}

echo "<h2>🔧 Step 5: Alternative Certificate Generation</h2>";

if ($conn) {
    echo '<p><a href="simple_certificate.php?course_id=' . $course_id . '" style="background: #4e73df; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;">🧪 Try Simple Certificate Generator</a></p>';
}

echo "<h2>📝 Recommendations</h2>";
echo "<ol>";
echo "<li>✅ <strong>Database Connection:</strong> " . ($conn ? "Working" : "❌ Fix database connection first") . "</li>";
echo "<li><strong>Table Structure:</strong> Ensure users, courses, and enrollments tables exist</li>";
echo "<li><strong>Test Data:</strong> Verify student enrollment in the course</li>";
echo "<li><strong>Permissions:</strong> Check file permissions for certificate generation</li>";
echo "</ol>";

?>
