<?php
// 🔍 DATABASE CONNECTION DEBUG SCRIPT
// This script will help us understand the database structure and connection issues

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h1>🔍 DATABASE CONNECTION & REVIEWS TABLE DEBUG</h1>";
echo "<style>
    body { font-family: Arial, sans-serif; background: #f5f5f5; padding: 20px; }
    .section { background: white; margin: 10px 0; padding: 15px; border-radius: 5px; border-left: 4px solid #007bff; }
    .error { border-left-color: #dc3545; background: #f8d7da; }
    .success { border-left-color: #28a745; background: #d4edda; }
    .warning { border-left-color: #ffc107; background: #fff3cd; }
    pre { background: #f8f9fa; padding: 10px; border-radius: 3px; overflow-x: auto; }
    table { width: 100%; border-collapse: collapse; }
    th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
    th { background: #f2f2f2; }
</style>";

session_start();

// 1. SESSION DEBUG
echo "<div class='section'>";
echo "<h2>📋 1. SESSION INFORMATION</h2>";
echo "<pre>";
echo "Session ID: " . session_id() . "\n";
echo "User ID: " . ($_SESSION['user_id'] ?? 'NOT SET') . "\n";
echo "Session Status: " . (isset($_SESSION['user_id']) ? 'LOGGED IN' : 'NOT LOGGED IN') . "\n";
echo "All Session Data:\n";
print_r($_SESSION);
echo "</pre>";
echo "</div>";

// 2. DATABASE CONNECTION TESTS
echo "<div class='section'>";
echo "<h2>🔌 2. DATABASE CONNECTION ATTEMPTS</h2>";

$connection_attempts = [
    [
        'name' => 'Method 1: Include database.php',
        'path' => '../config/database.php',
        'function' => 'getDbConnection'
    ],
    [
        'name' => 'Method 2: Include db_connect.php',
        'path' => '../includes/db_connect.php',
        'function' => 'getDbConnection'
    ],
    [
        'name' => 'Method 3: Direct PDO (panadite_academy)',
        'direct' => true,
        'params' => ['localhost', 'panadite_academy', 'root', '']
    ],
    [
        'name' => 'Method 4: Direct PDO (common variations)',
        'direct' => true,
        'params' => ['localhost', 'panadite_lms', 'root', '']
    ],
    [
        'name' => 'Method 5: Direct PDO (academy_db)',
        'direct' => true,
        'params' => ['localhost', 'academy_db', 'root', '']
    ]
];

$successful_connection = null;

foreach ($connection_attempts as $i => $attempt) {
    echo "<h3>🔍 " . $attempt['name'] . "</h3>";
    
    try {
        if (isset($attempt['direct']) && $attempt['direct']) {
            // Direct PDO connection
            list($host, $dbname, $username, $password) = $attempt['params'];
            echo "<p>Trying: mysql:host=$host;dbname=$dbname with user='$username'</p>";
            
            $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8", $username, $password);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            
            echo "<div class='success'>✅ SUCCESS: Connected to database '$dbname'</div>";
            
            if (!$successful_connection) {
                $successful_connection = $pdo;
                $connection_info = ['host' => $host, 'dbname' => $dbname, 'username' => $username];
            }
            
        } else {
            // Include file method
            echo "<p>File path: " . $attempt['path'] . "</p>";
            echo "<p>File exists: " . (file_exists($attempt['path']) ? 'YES' : 'NO') . "</p>";
            
            if (file_exists($attempt['path'])) {
                require_once $attempt['path'];
                
                if (function_exists($attempt['function'])) {
                    $pdo = call_user_func($attempt['function']);
                    echo "<div class='success'>✅ SUCCESS: Connected via " . $attempt['function'] . "()</div>";
                    
                    if (!$successful_connection) {
                        $successful_connection = $pdo;
                        $connection_info = ['method' => $attempt['name']];
                    }
                } else {
                    echo "<div class='warning'>⚠️ Function '" . $attempt['function'] . "()' not found in file</div>";
                }
            } else {
                echo "<div class='warning'>⚠️ File not found: " . $attempt['path'] . "</div>";
            }
        }
        
    } catch (Exception $e) {
        echo "<div class='error'>❌ ERROR: " . $e->getMessage() . "</div>";
    }
    
    echo "<hr>";
}

echo "</div>";

// 3. DATABASE STRUCTURE ANALYSIS
if ($successful_connection) {
    echo "<div class='section success'>";
    echo "<h2>🎉 3. SUCCESSFUL CONNECTION ANALYSIS</h2>";
    echo "<p><strong>Connection Details:</strong></p>";
    echo "<pre>" . print_r($connection_info, true) . "</pre>";
    
    try {
        // Get all tables
        echo "<h3>📊 Available Tables:</h3>";
        $stmt = $successful_connection->query("SHOW TABLES");
        $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        echo "<ul>";
        foreach ($tables as $table) {
            echo "<li>$table</li>";
        }
        echo "</ul>";
        
        // Check for specific tables we need
        $required_tables = ['courses', 'users', 'enrollments', 'course_reviews'];
        echo "<h3>🔍 Required Tables Check:</h3>";
        echo "<table>";
        echo "<tr><th>Table</th><th>Status</th><th>Action</th></tr>";
        
        foreach ($required_tables as $table) {
            $exists = in_array($table, $tables);
            $status = $exists ? '✅ EXISTS' : '❌ MISSING';
            $action = $exists ? 'Ready to use' : 'Will be created automatically';
            echo "<tr><td>$table</td><td>$status</td><td>$action</td></tr>";
        }
        echo "</table>";
        
        // If course_reviews exists, show its structure
        if (in_array('course_reviews', $tables)) {
            echo "<h3>📋 course_reviews Table Structure:</h3>";
            $stmt = $successful_connection->query("DESCRIBE course_reviews");
            $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo "<table>";
            echo "<tr><th>Column</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr>";
            foreach ($columns as $column) {
                echo "<tr>";
                echo "<td>{$column['Field']}</td>";
                echo "<td>{$column['Type']}</td>";
                echo "<td>{$column['Null']}</td>";
                echo "<td>{$column['Key']}</td>";
                echo "<td>{$column['Default']}</td>";
                echo "<td>{$column['Extra']}</td>";
                echo "</tr>";
            }
            echo "</table>";
            
            // Show existing reviews
            echo "<h3>📝 Existing Reviews:</h3>";
            $stmt = $successful_connection->query("SELECT COUNT(*) as count FROM course_reviews");
            $count = $stmt->fetch()['count'];
            echo "<p>Total reviews in database: <strong>$count</strong></p>";
            
            if ($count > 0) {
                $stmt = $successful_connection->query("SELECT * FROM course_reviews ORDER BY created_at DESC LIMIT 5");
                $reviews = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                echo "<table>";
                echo "<tr><th>ID</th><th>User ID</th><th>Course ID</th><th>Rating</th><th>Title</th><th>Status</th><th>Created</th></tr>";
                foreach ($reviews as $review) {
                    echo "<tr>";
                    echo "<td>{$review['id']}</td>";
                    echo "<td>{$review['user_id']}</td>";
                    echo "<td>{$review['course_id']}</td>";
                    echo "<td>{$review['rating']}</td>";
                    echo "<td>" . substr($review['review_title'], 0, 30) . "...</td>";
                    echo "<td>{$review['status']}</td>";
                    echo "<td>{$review['created_at']}</td>";
                    echo "</tr>";
                }
                echo "</table>";
            }
        }
        
        // Check enrollment for current user (if logged in)
        if (isset($_SESSION['user_id'])) {
            echo "<h3>👤 Current User Enrollment Check:</h3>";
            $user_id = $_SESSION['user_id'];
            
            $stmt = $successful_connection->prepare("SELECT * FROM enrollments WHERE user_id = ? ORDER BY created_at DESC LIMIT 5");
            $stmt->execute([$user_id]);
            $enrollments = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (count($enrollments) > 0) {
                echo "<table>";
                echo "<tr><th>Course ID</th><th>Status</th><th>Created</th></tr>";
                foreach ($enrollments as $enrollment) {
                    echo "<tr>";
                    echo "<td>{$enrollment['course_id']}</td>";
                    echo "<td>{$enrollment['status']}</td>";
                    echo "<td>{$enrollment['created_at']}</td>";
                    echo "</tr>";
                }
                echo "</table>";
            } else {
                echo "<p>❌ No enrollments found for user ID: $user_id</p>";
            }
        }
        
    } catch (Exception $e) {
        echo "<div class='error'>❌ Database analysis error: " . $e->getMessage() . "</div>";
    }
    
    echo "</div>";
    
} else {
    echo "<div class='section error'>";
    echo "<h2>❌ 4. NO SUCCESSFUL CONNECTION</h2>";
    echo "<p>Could not establish database connection with any method.</p>";
    echo "<p><strong>Recommendations:</strong></p>";
    echo "<ul>";
    echo "<li>Check if MySQL/MariaDB server is running</li>";
    echo "<li>Verify database name, username, and password</li>";
    echo "<li>Check if the database exists</li>";
    echo "<li>Verify PHP PDO MySQL extension is installed</li>";
    echo "</ul>";
    echo "</div>";
}

// 4. SYSTEM INFO
echo "<div class='section'>";
echo "<h2>⚙️ 5. SYSTEM INFORMATION</h2>";
echo "<pre>";
echo "PHP Version: " . phpversion() . "\n";
echo "PDO Available: " . (extension_loaded('pdo') ? 'YES' : 'NO') . "\n";
echo "PDO MySQL Available: " . (extension_loaded('pdo_mysql') ? 'YES' : 'NO') . "\n";
echo "Current Working Directory: " . getcwd() . "\n";
echo "Script Path: " . __FILE__ . "\n";

// Check if config files exist
$config_files = [
    '../config/database.php',
    '../includes/db_connect.php',
    '../config/db_connect.php',
    'includes/db_connect.php'
];

echo "\nConfig Files Check:\n";
foreach ($config_files as $file) {
    echo "$file: " . (file_exists($file) ? 'EXISTS' : 'NOT FOUND') . "\n";
}
echo "</pre>";
echo "</div>";

echo "<div class='section'>";
echo "<h2>🔧 6. NEXT STEPS</h2>";
echo "<p>Based on the results above:</p>";
echo "<ul>";
echo "<li>If a connection was successful, update submit_review.php with the working connection details</li>";
echo "<li>If no connection worked, check your database server and configuration</li>";
echo "<li>Ensure the course_reviews table exists or will be created</li>";
echo "<li>Verify user enrollment status for review submission</li>";
echo "</ul>";
echo "</div>";
?>
