<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
require_once '../config/fast_email_service.php';

// Set the missing session variables that requireRole needs
if (!isset($_SESSION['logged_in']) && isset($_SESSION['user_id'])) {
    $_SESSION['logged_in'] = true;
}
if (!isset($_SESSION['user_role']) && isset($_SESSION['role'])) {
    $_SESSION['user_role'] = $_SESSION['role'];
}

requireRole('student');
require_once '../config/db_connect.php';

// Bridge database connections - Email service needs PDO, but this page uses MySQLi - EXACT SAME AS ADMIN COURSES
if (!isset($pdo)) {
    try {
        // Create PDO connection using the same credentials as MySQLi
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        error_log("PDO connection created successfully for debug email service");
    } catch (Exception $e) {
        error_log("Failed to create PDO connection for debug email service: " . $e->getMessage());
        $pdo = null;
    }
}

/**
 * Send document submission completion email to student - EXACT SAME AS SUBMISSION PAGE
 */
function sendDocumentSubmissionEmail($request_id) {
    global $conn;
    
    try {
        // Get request, student and course details  
        $stmt = $conn->prepare("SELECT ar.request_id, ar.payment_amount, ar.assignment_document_path, ar.updated_at, ar.course_id, u.first_name, u.last_name, u.email, u.user_id, c.title as course_title, c.description FROM accreditation_requests ar JOIN users u ON ar.student_id = u.user_id JOIN courses c ON ar.course_id = c.course_id WHERE ar.request_id = ?");
        
        if (!$stmt) {
            error_log("Failed to prepare document submission email query: " . $conn->error);
            return false;
        }
        
        $stmt->bind_param("i", $request_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $details = $result->fetch_assoc();
        
        if (!$details) {
            error_log("Accreditation request not found for document submission email: $request_id");
            return false;
        }
        
        $student_name = $details['first_name'] . ' ' . $details['last_name'];
        $student_email = $details['email'];
        $course_title = $details['course_title'];
        $course_id = $details['course_id'];
        $request_id_formatted = str_pad($request_id, 6, '0', STR_PAD_LEFT);
        $submission_date = date('M j, Y', strtotime($details['updated_at']));
        
        // Create simple email content - avoid spam filters
        $subject = "Documents Submitted Successfully - Request #$request_id_formatted";
        $message = "Dear " . $student_name . ",

Your SITA accreditation documents have been successfully submitted.

Course: " . $course_title . "
Request ID: #" . $request_id_formatted . "
Submitted: " . $submission_date . "

IMPORTANT: Processing time is 3-5 months.

Your documents are now under review by SITA's accreditation board.

Next steps:
1. Document verification (1-2 weeks)
2. SITA board review (2-4 months) 
3. Final certification

You will receive updates via email.

Best regards,
Panadite Academy Team";
        
        // Try direct email sending - EXACT COPY FROM ADMIN COURSES.PHP LINES 81-94
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($student_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail - EXACT COPY FROM ADMIN COURSES.PHP
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($student_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending document submission email: " . $e->getMessage());
        return false;
    }
}

$student_id = $_SESSION['user_id'];

echo "<!DOCTYPE html>";
echo "<html><head>";
echo "<title>Debug Document Submission System</title>";
echo "<link href='https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css' rel='stylesheet'>";
echo "<style>.debug-section { background: #f8f9fa; padding: 15px; margin: 15px 0; border-radius: 8px; border-left: 4px solid #007bff; }</style>";
echo "</head><body>";
echo "<div class='container mt-4'>";

echo "<h1>🔍 Debug Document Submission System</h1>";
echo "<p class='text-muted'>Debugging document submission and email functionality for Student ID: $student_id</p>";

// 1. Check accreditation_documents table structure
echo "<div class='debug-section'>";
echo "<h3>📋 1. Document Table Structure</h3>";

echo "<h5>Accreditation Documents Table:</h5>";
$result = $conn->query("DESCRIBE accreditation_documents");
if ($result) {
    echo "<table class='table table-sm table-bordered'>";
    echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr>";
    while ($row = $result->fetch_assoc()) {
        echo "<tr>";
        echo "<td><strong>" . $row['Field'] . "</strong></td>";
        echo "<td>" . $row['Type'] . "</td>";
        echo "<td>" . $row['Null'] . "</td>";
        echo "<td>" . $row['Key'] . "</td>";
        echo "<td>" . $row['Default'] . "</td>";
        echo "<td>" . $row['Extra'] . "</td>";
        echo "</tr>";
    }
    echo "</table>";
} else {
    echo "<div class='alert alert-danger'>❌ Failed to describe accreditation_documents table: " . $conn->error . "</div>";
}
echo "</div>";

// 2. Show all accreditation requests for this student
echo "<div class='debug-section'>";
echo "<h3>📊 2. Student's Accreditation Requests</h3>";
$result = $conn->query("SELECT * FROM accreditation_requests WHERE student_id = $student_id ORDER BY request_date DESC");
if ($result && $result->num_rows > 0) {
    echo "<table class='table table-sm table-striped'>";
    echo "<tr><th>Request ID</th><th>Course ID</th><th>Status</th><th>Payment Status</th><th>Request Date</th><th>Documents Date</th></tr>";
    
    while ($row = $result->fetch_assoc()) {
        echo "<tr>";
        echo "<td><strong>" . $row['request_id'] . "</strong></td>";
        echo "<td>" . $row['course_id'] . "</td>";
        echo "<td><span class='badge bg-info'>" . $row['status'] . "</span></td>";
        echo "<td><span class='badge bg-success'>" . $row['payment_status'] . "</span></td>";
        echo "<td>" . $row['request_date'] . "</td>";
        echo "<td>" . ($row['documents_submitted_date'] ?? 'Not submitted') . "</td>";
        echo "</tr>";
    }
    echo "</table>";
} else {
    echo "<div class='alert alert-warning'>⚠️ No accreditation requests found for student ID: $student_id</div>";
}
echo "</div>";

// 3. Show submitted documents
echo "<div class='debug-section'>";
echo "<h3>📄 3. Submitted Documents</h3>";
$docs_result = $conn->query("
    SELECT ad.*, ar.request_id, ar.status as request_status, c.title as course_title 
    FROM accreditation_documents ad 
    JOIN accreditation_requests ar ON ad.request_id = ar.request_id 
    JOIN courses c ON ar.course_id = c.course_id 
    WHERE ad.student_id = $student_id 
    ORDER BY ad.request_id DESC, ad.document_type
");

if ($docs_result && $docs_result->num_rows > 0) {
    echo "<table class='table table-sm table-striped'>";
    echo "<tr><th>Request ID</th><th>Course</th><th>Document Type</th><th>File Name</th><th>Upload Date</th><th>Status</th><th>Request Status</th></tr>";
    
    while ($doc = $docs_result->fetch_assoc()) {
        echo "<tr>";
        echo "<td><strong>" . $doc['request_id'] . "</strong></td>";
        echo "<td>" . htmlspecialchars($doc['course_title']) . "</td>";
        echo "<td><span class='badge bg-primary'>" . $doc['document_type'] . "</span></td>";
        echo "<td>" . htmlspecialchars($doc['document_name']) . "</td>";
        echo "<td>" . $doc['upload_date'] . "</td>";
        echo "<td><span class='badge bg-warning'>" . $doc['verification_status'] . "</span></td>";
        echo "<td><span class='badge bg-info'>" . $doc['request_status'] . "</span></td>";
        echo "</tr>";
    }
    echo "</table>";
} else {
    echo "<div class='alert alert-warning'>⚠️ No submitted documents found for student ID: $student_id</div>";
}
echo "</div>";

// 4. Test document count query (the one used in the API)
echo "<div class='debug-section'>";
echo "<h3>🧪 4. Test Document Count Logic</h3>";

// Get latest request to test with
$latest_result = $conn->query("SELECT request_id FROM accreditation_requests WHERE student_id = $student_id ORDER BY request_date DESC LIMIT 1");
if ($latest_result && $latest_result->num_rows > 0) {
    $latest_row = $latest_result->fetch_assoc();
    $test_request_id = $latest_row['request_id'];
    
    echo "<p><strong>Testing with latest Request ID:</strong> " . $test_request_id . "</p>";
    
    // Test the exact count query from the API
    $count_query = "SELECT COUNT(DISTINCT document_type) as count FROM accreditation_documents WHERE request_id = ? AND student_id = ? AND document_type IN ('cv', 'qualifications', 'id_document', 'assignment')";
    
    echo "<div class='alert alert-info'>";
    echo "<strong>Count Query:</strong><br><code>" . htmlspecialchars($count_query) . "</code>";
    echo "</div>";
    
    $count_stmt = $conn->prepare($count_query);
    if ($count_stmt) {
        $count_stmt->bind_param("ii", $test_request_id, $student_id);
        if ($count_stmt->execute()) {
            $count_result = $count_stmt->get_result()->fetch_assoc();
            echo "<div class='alert alert-success'>✅ Document count: " . $count_result['count'] . "/4 required documents</div>";
            
            // Show which specific document types exist
            $detail_query = "SELECT DISTINCT document_type FROM accreditation_documents WHERE request_id = ? AND student_id = ?";
            $detail_stmt = $conn->prepare($detail_query);
            $detail_stmt->bind_param("ii", $test_request_id, $student_id);
            $detail_stmt->execute();
            $detail_result = $detail_stmt->get_result();
            
            echo "<h6>Document Types Found:</h6><ul>";
            while ($doc_type = $detail_result->fetch_assoc()) {
                echo "<li><strong>" . $doc_type['document_type'] . "</strong></li>";
            }
            echo "</ul>";
            
            // Check if should trigger email
            if ($count_result['count'] >= 4) {
                echo "<div class='alert alert-success'>✅ All documents submitted - EMAIL SHOULD BE SENT!</div>";
            } else {
                echo "<div class='alert alert-warning'>⚠️ Only " . $count_result['count'] . " documents - EMAIL NOT TRIGGERED</div>";
            }
            
        } else {
            echo "<div class='alert alert-danger'>❌ Count query failed: " . $count_stmt->error . "</div>";
        }
    } else {
        echo "<div class='alert alert-danger'>❌ Failed to prepare count query: " . $conn->error . "</div>";
    }
} else {
    echo "<div class='alert alert-warning'>⚠️ No requests found to test with</div>";
}
echo "</div>";

// 5. Test the email function query
echo "<div class='debug-section'>";
echo "<h3>📧 5. Test Email Function Query</h3>";

if (isset($test_request_id)) {
    // Test the exact query from the email function
    $email_query = "SELECT ar.request_id, ar.payment_amount, ar.assignment_document_path, ar.updated_at, ar.course_id, u.first_name, u.last_name, u.email, u.user_id, c.title as course_title, c.description FROM accreditation_requests ar JOIN users u ON ar.student_id = u.user_id JOIN courses c ON ar.course_id = c.course_id WHERE ar.request_id = ?";
    
    echo "<div class='alert alert-info'>";
    echo "<strong>Email Query:</strong><br><code>" . htmlspecialchars($email_query) . "</code>";
    echo "</div>";
    
    $email_stmt = $conn->prepare($email_query);
    if ($email_stmt) {
        echo "<div class='alert alert-success'>✅ Email query prepared successfully</div>";
        $email_stmt->bind_param("i", $test_request_id);
        if ($email_stmt->execute()) {
            echo "<div class='alert alert-success'>✅ Email query executed successfully</div>";
            $email_result = $email_stmt->get_result();
            if ($email_result && $email_result->num_rows > 0) {
                $email_details = $email_result->fetch_assoc();
                echo "<div class='alert alert-success'>✅ Email data retrieved successfully</div>";
                echo "<h6>Email Data:</h6>";
                echo "<table class='table table-sm table-bordered'>";
                foreach ($email_details as $key => $value) {
                    echo "<tr><td><strong>" . $key . "</strong></td><td>" . htmlspecialchars($value) . "</td></tr>";
                }
                echo "</table>";
                
            } else {
                echo "<div class='alert alert-warning'>⚠️ Email query returned no results</div>";
            }
        } else {
            echo "<div class='alert alert-danger'>❌ Email query execution failed: " . $email_stmt->error . "</div>";
        }
    } else {
        echo "<div class='alert alert-danger'>❌ Failed to prepare email query: " . $conn->error . "</div>";
    }
}
echo "</div>";

// 6. Check email service availability
echo "<div class='debug-section'>";
echo "<h3>⚙️ 6. Email Service Status</h3>";

echo "<h6>Function Availability:</h6>";
echo "<ul>";
echo "<li>sendEmailNow: " . (function_exists('sendEmailNow') ? '✅ Available' : '❌ Not Available') . "</li>";
echo "<li>mail: " . (function_exists('mail') ? '✅ Available' : '❌ Not Available') . "</li>";
echo "<li>sendDocumentSubmissionEmail: " . (function_exists('sendDocumentSubmissionEmail') ? '✅ Available' : '❌ Not Available') . "</li>";
echo "</ul>";

// Try to include the email service
try {
    require_once '../config/fast_email_service.php';
    echo "<div class='alert alert-success'>✅ Email service included successfully</div>";
} catch (Exception $e) {
    echo "<div class='alert alert-danger'>❌ Failed to include email service: " . $e->getMessage() . "</div>";
}
echo "</div>";

// 7. Manual email test (if we have data)
if (isset($test_request_id) && isset($email_details)) {
    echo "<div class='debug-section'>";
    echo "<h3>✉️ 7. Manual Email Test</h3>";
    
    echo "<p><strong>Testing email for Request ID:</strong> " . $test_request_id . "</p>";
    echo "<p><strong>Student Email:</strong> " . htmlspecialchars($email_details['email']) . "</p>";
    
    // Test basic email
    $test_subject = "Test Email - Document Submission Debug";
    $test_message = "<h1>Test Email</h1><p>This is a test email for request ID: " . $test_request_id . "</p>";
    
    if (function_exists('mail')) {
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        $mail_result = mail($email_details['email'], $test_subject, $test_message, $headers);
        if ($mail_result) {
            echo "<div class='alert alert-success'>✅ Test email sent successfully via PHP mail!</div>";
        } else {
            echo "<div class='alert alert-danger'>❌ Test email failed via PHP mail!</div>";
        }
    } else {
        echo "<div class='alert alert-warning'>⚠️ PHP mail function not available</div>";
    }
    
    echo "</div>";
}

echo "<div class='mt-4'>";
echo "<a href='assignments.php' class='btn btn-primary'>← Back to Assignments</a>";
echo " ";
echo "<a href='sita_assignment_submission.php?request_id=" . ($test_request_id ?? '11') . "' class='btn btn-secondary'>→ Assignment Submission</a>";
echo "</div>";

echo "</div></body></html>";
?>
