<?php
// Session is started in header.php
require_once '../auth/functions.php';

// Require student role authentication - no exceptions
requireRole('student');

// Fetch student information
$userId = $_SESSION['user_id'] ?? null;
$studentId = null;
$debug = [];

try {
    require_once '../config/db_connect.php';
    
    if ($conn) {
        // Debug header
        $debug[] = "===== ENROLLMENT DEBUGGING =====";
        $debug[] = "User ID: $userId";
        
        // First, get the student_id from the students table
        $stmt = $conn->prepare("SELECT student_id FROM students WHERE user_id = ?");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($row = $result->fetch_assoc()) {
            $studentId = $row['student_id'];
            $debug[] = "Student ID found: $studentId";
            
            // Debug query 1: Check raw enrollments table
            $debug[] = "\n=== Raw Enrollments Table Data ===";
            $enrollQuery = "SELECT * FROM enrollments WHERE student_id = ?";
            $stmt = $conn->prepare($enrollQuery);
            $stmt->bind_param("i", $studentId);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                $debug[] = "Found " . $result->num_rows . " enrollment records";
                
                while ($row = $result->fetch_assoc()) {
                    $debug[] = "- Enrollment ID: " . $row['enrollment_id'] . 
                               ", Course ID: " . $row['course_id'] . 
                               ", Progress: " . $row['progress'] . "%";
                }
            } else {
                $debug[] = "NO ENROLLMENTS FOUND in enrollments table";
            }
            
            // Debug query 2: Check course enrollments with course details
            $debug[] = "\n=== Enrollments Joined With Courses ===";
            $courseQuery = "
                SELECT 
                    e.enrollment_id,
                    e.student_id,
                    e.course_id,
                    e.progress,
                    c.title as course_title
                FROM 
                    enrollments e
                JOIN 
                    courses c ON e.course_id = c.course_id
                WHERE 
                    e.student_id = ?
            ";
            $stmt = $conn->prepare($courseQuery);
            $stmt->bind_param("i", $studentId);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                $debug[] = "Found " . $result->num_rows . " course enrollments with details";
                
                while ($row = $result->fetch_assoc()) {
                    $debug[] = "- Course: " . $row['course_title'] . 
                               " (ID: " . $row['course_id'] . ")" .
                               ", Progress: " . $row['progress'] . "%";
                }
            } else {
                $debug[] = "NO COURSE ENROLLMENTS found in join query";
            }
            
            // Debug query 3: Check the exact query used in dashboard.php
            $debug[] = "\n=== Dashboard Query Results ===";
            $dashboardQuery = "
                SELECT 
                    COUNT(*) as total_courses,
                    SUM(CASE WHEN e.progress >= 100 THEN 1 ELSE 0 END) as completed
                FROM 
                    enrollments e
                WHERE 
                    e.student_id = ?
            ";
            $stmt = $conn->prepare($dashboardQuery);
            $stmt->bind_param("i", $studentId);
            $stmt->execute();
            $result = $stmt->get_result();
            $row = $result->fetch_assoc();
            
            $debug[] = "Dashboard query results:";
            $debug[] = "- Total courses count: " . $row['total_courses'];
            $debug[] = "- Completed courses count: " . ($row['completed'] ?: 0);
            $debug[] = "- Raw values (for debugging type issues):";
            $debug[] = "  * total_courses (raw): " . var_export($row['total_courses'], true) . " (type: " . gettype($row['total_courses']) . ")";
            $debug[] = "  * completed (raw): " . var_export($row['completed'], true) . " (type: " . gettype($row['completed']) . ")";
            
        } else {
            $debug[] = "ERROR: Could not find student_id for user_id $userId";
            
            // Check if the user exists at all
            $stmt = $conn->prepare("SELECT * FROM users WHERE user_id = ?");
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                $debug[] = "User exists in users table, but no matching student record found";
            } else {
                $debug[] = "User does not exist in users table";
            }
        }
    } else {
        $debug[] = "ERROR: Database connection failed";
    }
} catch (Exception $e) {
    $debug[] = "EXCEPTION: " . $e->getMessage();
}

// Display debug information
echo "<html><head><title>Enrollment Debug</title>";
echo "<style>
    body { font-family: monospace; padding: 20px; }
    h1 { color: #333; }
    pre { background: #f5f5f5; padding: 15px; border-radius: 5px; }
</style>";
echo "</head><body>";
echo "<h1>Enrollment Debug Information</h1>";
echo "<pre>" . implode("\n", $debug) . "</pre>";
echo "</body></html>";
?>
