<?php
session_start();
require_once '../includes/config.php';
require_once '../includes/functions.php';

// Ensure user is logged in as student
requireRole('student');

$student_id = $_SESSION['user_id'] ?? 15; // Fallback for testing
$course_id = 13; // Computer science course

echo "<!DOCTYPE html>
<html lang='en'>
<head>
    <meta charset='UTF-8'>
    <meta name='viewport' content='width=device-width, initial-scale=1.0'>
    <title>🔧 Progress System Debug Tool</title>
    <style>
        body { font-family: monospace; padding: 20px; background: #1a1a1a; color: #00ff00; }
        .debug-section { 
            margin: 20px 0; 
            padding: 15px; 
            border: 1px solid #333; 
            border-radius: 8px;
            background: #2a2a2a;
        }
        .success { color: #00ff00; }
        .error { color: #ff4444; }
        .warning { color: #ffaa00; }
        .info { color: #4488ff; }
        .header { color: #ffffff; font-size: 18px; font-weight: bold; }
        table { width: 100%; border-collapse: collapse; margin: 10px 0; }
        th, td { padding: 8px; border: 1px solid #555; text-align: left; }
        th { background: #333; color: #fff; }
        .test-button { 
            background: #007acc; 
            color: white; 
            padding: 10px 20px; 
            border: none; 
            border-radius: 4px; 
            cursor: pointer; 
            margin: 5px;
            font-family: monospace;
        }
        .test-button:hover { background: #0099ff; }
        #log-output { 
            background: #000; 
            color: #00ff00; 
            padding: 15px; 
            height: 300px; 
            overflow-y: scroll; 
            font-family: monospace;
            border: 1px solid #333;
        }
    </style>
</head>
<body>";

echo "<div class='header'>🔧 PANADITE ACADEMY - PROGRESS SYSTEM DEBUG TOOL</div>";
echo "<div class='info'>🎯 Testing Student ID: $student_id | Course ID: $course_id</div>";

// 1. DATABASE ANALYSIS
echo "<div class='debug-section'>";
echo "<div class='header'>📊 DATABASE ANALYSIS</div>";

// Check video_progress table structure
$structure_query = "DESCRIBE video_progress";
$result = $conn->query($structure_query);
echo "<h3 class='info'>🗄️ video_progress Table Structure:</h3>";
echo "<table><tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
while ($row = $result->fetch_assoc()) {
    echo "<tr>";
    echo "<td>{$row['Field']}</td>";
    echo "<td>{$row['Type']}</td>";
    echo "<td>{$row['Null']}</td>";
    echo "<td>{$row['Key']}</td>";
    echo "<td>{$row['Default']}</td>";
    echo "</tr>";
}
echo "</table>";

// Check current progress data
$progress_query = "SELECT * FROM video_progress WHERE student_id = ? ORDER BY video_id";
$stmt = $conn->prepare($progress_query);
$stmt->bind_param("i", $student_id);
$stmt->execute();
$progress_data = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

echo "<h3 class='info'>📈 Current Progress Data for Student $student_id:</h3>";
if (empty($progress_data)) {
    echo "<div class='warning'>⚠️ NO PROGRESS DATA FOUND</div>";
} else {
    echo "<table>";
    echo "<tr><th>Video ID</th><th>Completion %</th><th>Last Position</th><th>Completed</th><th>Last Watched</th></tr>";
    foreach ($progress_data as $progress) {
        $completed_status = $progress['completed'] ? "✅ YES" : "❌ NO";
        echo "<tr>";
        echo "<td>{$progress['video_id']}</td>";
        echo "<td class='success'>{$progress['completion_percentage']}%</td>";
        echo "<td>{$progress['last_position']}</td>";
        echo "<td>$completed_status</td>";
        echo "<td>{$progress['last_watched']}</td>";
        echo "</tr>";
    }
    echo "</table>";
}
echo "</div>";

// 2. VIDEOS ANALYSIS
echo "<div class='debug-section'>";
echo "<div class='header'>🎬 VIDEOS ANALYSIS</div>";

$videos_query = "SELECT * FROM videos WHERE course_id = ? ORDER BY video_order";
$stmt = $conn->prepare($videos_query);
$stmt->bind_param("i", $course_id);
$stmt->execute();
$videos = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

echo "<h3 class='info'>🎥 Videos in Course $course_id:</h3>";
echo "<table>";
echo "<tr><th>Video ID</th><th>Title</th><th>Type</th><th>File Path/URL</th><th>Duration</th><th>Published</th></tr>";
foreach ($videos as $video) {
    $type = !empty($video['video_file']) ? "📁 FILE UPLOAD" : "🔗 URL VIDEO";
    $source = !empty($video['video_file']) ? $video['video_file'] : $video['video_url'];
    $published = $video['is_published'] ? "✅ YES" : "❌ NO";
    
    echo "<tr>";
    echo "<td class='info'>{$video['video_id']}</td>";
    echo "<td>{$video['title']}</td>";
    echo "<td>$type</td>";
    echo "<td style='max-width: 300px; word-wrap: break-word;'>$source</td>";
    echo "<td>{$video['duration']}</td>";
    echo "<td>$published</td>";
    echo "</tr>";
}
echo "</table>";
echo "</div>";

// 3. PROGRESS TRACKING TEST INTERFACE
echo "<div class='debug-section'>";
echo "<div class='header'>🧪 PROGRESS TRACKING TEST INTERFACE</div>";

echo "<div style='margin: 15px 0;'>";
echo "<button class='test-button' onclick='testVideoProgress(4, 25)'>📁 Test File Upload Progress (Video 4 → 25%)</button>";
echo "<button class='test-button' onclick='testVideoProgress(24, 75)'>🔗 Test URL Video Progress (Video 24 → 75%)</button>";
echo "<button class='test-button' onclick='testVideoProgress(4, 100)'>📁 Complete File Upload (Video 4 → 100%)</button>";
echo "<button class='test-button' onclick='testVideoProgress(24, 100)'>🔗 Complete URL Video (Video 24 → 100%)</button>";
echo "</div>";

echo "<div style='margin: 15px 0;'>";
echo "<button class='test-button' onclick='getCourseProgress()'>📊 Get Course Progress</button>";
echo "<button class='test-button' onclick='clearAllProgress()'>🗑️ Clear All Progress</button>";
echo "<button class='test-button' onclick='refreshData()'>🔄 Refresh Database View</button>";
echo "</div>";

echo "<h3 class='info'>📝 Live Test Log:</h3>";
echo "<div id='log-output'>🚀 Debug console ready. Click buttons above to test progress tracking...\n</div>";
echo "</div>";

echo "<script>
const logOutput = document.getElementById('log-output');

function log(message, type = 'info') {
    const timestamp = new Date().toLocaleTimeString();
    const colors = {
        success: '#00ff00',
        error: '#ff4444',
        warning: '#ffaa00',
        info: '#4488ff'
    };
    
    logOutput.innerHTML += `<div style='color: \${colors[type]}'>[{\${timestamp}}] \${message}</div>`;
    logOutput.scrollTop = logOutput.scrollHeight;
}

// Test video progress tracking
async function testVideoProgress(videoId, percentage) {
    log(`🧪 TESTING: Video \${videoId} → \${percentage}%`, 'info');
    
    try {
        const response = await fetch('ajax/video-progress-handler.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                action: 'save_progress',
                video_id: videoId,
                lesson_id: videoId,
                course_id: $course_id,
                progress_percentage: percentage,
                current_time: Math.round((percentage / 100) * 120), // Assume 2min video
                completed: percentage >= 100 ? 1 : 0,
                student_id: $student_id,
                tracking_id: videoId
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            log(`✅ SUCCESS: Video \${videoId} progress saved → \${percentage}%`, 'success');
            log(`📊 Database Response: \${JSON.stringify(result.debug_data)}`, 'info');
        } else {
            log(`❌ ERROR: \${result.error || result.message}`, 'error');
        }
    } catch (error) {
        log(`❌ AJAX ERROR: \${error.message}`, 'error');
    }
}

// Get course progress
async function getCourseProgress() {
    log('📊 FETCHING: Course progress...', 'info');
    
    try {
        const response = await fetch('ajax/get_course_progress.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded'
            },
            body: 'course_id=$course_id'
        });
        
        const result = await response.json();
        
        if (result.success) {
            log(`✅ COURSE PROGRESS: \${result.video_progress}%`, 'success');
            log(`📈 Completed: \${result.completed_lessons}/\${result.total_lessons} videos`, 'info');
            log(`📊 Full Response: \${JSON.stringify(result)}`, 'info');
        } else {
            log(`❌ ERROR: \${result.error}`, 'error');
        }
    } catch (error) {
        log(`❌ AJAX ERROR: \${error.message}`, 'error');
    }
}

// Clear all progress
async function clearAllProgress() {
    if (!confirm('⚠️ Clear all progress data for student $student_id?')) return;
    
    log('🗑️ CLEARING: All progress data...', 'warning');
    
    try {
        const response = await fetch('debug_clear_progress.php', {
            method: 'POST',
            body: 'student_id=$student_id'
        });
        
        const result = await response.text();
        log(`🗑️ CLEARED: \${result}`, 'warning');
    } catch (error) {
        log(`❌ CLEAR ERROR: \${error.message}`, 'error');
    }
}

// Refresh data view
function refreshData() {
    log('🔄 REFRESHING: Page data...', 'info');
    window.location.reload();
}

// Auto-refresh every 30 seconds
setInterval(() => {
    log('🔄 Auto-refreshing course progress...', 'info');
    getCourseProgress();
}, 30000);

log('🚀 Progress Debug Tool initialized successfully!', 'success');
</script>";

echo "</body></html>";
?>
