<?php
/**
 * 🔍 VIDEO COMPLETION DATABASE DEBUG SCRIPT
 * Isolated script to verify video completion is being saved and retrieved correctly
 */

session_start();
require_once 'config/connection.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    die('<h1 style="color: red;">❌ Please log in first to debug video completion data</h1>');
}

$user_id = $_SESSION['user_id'];
$course_id = isset($_GET['course_id']) ? intval($_GET['course_id']) : null;

if (!$course_id) {
    die('<h1 style="color: red;">❌ Please provide course_id parameter: debug_video_completion.php?course_id=X</h1>');
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>🔍 Video Completion Debug</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .debug-section { background: white; padding: 15px; margin: 10px 0; border-radius: 8px; border-left: 4px solid #007bff; }
        .success { border-left-color: #28a745; }
        .warning { border-left-color: #ffc107; }
        .error { border-left-color: #dc3545; }
        table { width: 100%; border-collapse: collapse; margin: 10px 0; }
        th, td { padding: 8px; border: 1px solid #ddd; text-align: left; }
        th { background: #f8f9fa; }
        .completed { background: #d4edda; color: #155724; }
        .incomplete { background: #fff3cd; color: #856404; }
        .not-started { background: #f8d7da; color: #721c24; }
        code { background: #f8f9fa; padding: 2px 4px; border-radius: 3px; }
    </style>
</head>
<body>
    <h1>🔍 Video Completion Database Debug</h1>
    <p><strong>User ID:</strong> <?= $user_id ?> | <strong>Course ID:</strong> <?= $course_id ?></p>
    <p><strong>Time:</strong> <?= date('Y-m-d H:i:s') ?></p>

<?php

// 1. Check video_progress table structure
echo '<div class="debug-section">';
echo '<h2>🗄️ Database Table Structure</h2>';

$structure_query = "DESCRIBE video_progress";
$result = $conn->query($structure_query);

if ($result) {
    echo '<h3>✅ video_progress table structure:</h3>';
    echo '<table>';
    echo '<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr>';
    while ($row = $result->fetch_assoc()) {
        echo '<tr>';
        echo '<td><code>' . htmlspecialchars($row['Field']) . '</code></td>';
        echo '<td>' . htmlspecialchars($row['Type']) . '</td>';
        echo '<td>' . htmlspecialchars($row['Null']) . '</td>';
        echo '<td>' . htmlspecialchars($row['Key']) . '</td>';
        echo '<td>' . htmlspecialchars($row['Default']) . '</td>';
        echo '<td>' . htmlspecialchars($row['Extra']) . '</td>';
        echo '</tr>';
    }
    echo '</table>';
} else {
    echo '<p class="error">❌ Error checking table structure: ' . $conn->error . '</p>';
}
echo '</div>';

// 2. Get all videos for this course
echo '<div class="debug-section">';
echo '<h2>📹 Course Videos & Completion Status</h2>';

$videos_query = "
    SELECT 
        v.video_id,
        v.title,
        v.video_order,
        vp.completion_percentage,
        vp.completed,
        vp.last_position,
        vp.last_watched,
        vp.created_at as progress_created,
        vp.updated_at as progress_updated
    FROM videos v
    LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = ?
    WHERE v.course_id = ? AND v.is_published = 1
    ORDER BY v.video_order ASC, v.created_at ASC
";

if ($stmt = $conn->prepare($videos_query)) {
    $stmt->bind_param("ii", $user_id, $course_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    echo '<table>';
    echo '<tr>';
    echo '<th>Video ID</th>';
    echo '<th>Title</th>';
    echo '<th>Order</th>';
    echo '<th>Progress %</th>';
    echo '<th>Completed</th>';
    echo '<th>Last Position</th>';
    echo '<th>Last Watched</th>';
    echo '<th>Progress Created</th>';
    echo '<th>Status</th>';
    echo '</tr>';
    
    $total_videos = 0;
    $completed_videos = 0;
    $in_progress_videos = 0;
    $not_started_videos = 0;
    
    while ($row = $result->fetch_assoc()) {
        $total_videos++;
        
        $progress = $row['completion_percentage'] ?? 0;
        $completed = $row['completed'] ?? 0;
        
        if ($completed == 1) {
            $status_class = 'completed';
            $status_text = '🏆 COMPLETED';
            $completed_videos++;
        } elseif ($progress > 0) {
            $status_class = 'incomplete';
            $status_text = '⏳ IN PROGRESS';
            $in_progress_videos++;
        } else {
            $status_class = 'not-started';
            $status_text = '🔴 NOT STARTED';
            $not_started_videos++;
        }
        
        echo '<tr class="' . $status_class . '">';
        echo '<td><strong>' . $row['video_id'] . '</strong></td>';
        echo '<td>' . htmlspecialchars($row['title']) . '</td>';
        echo '<td>' . $row['video_order'] . '</td>';
        echo '<td>' . $progress . '%</td>';
        echo '<td>' . ($completed ? '✅ YES' : '❌ NO') . '</td>';
        echo '<td>' . ($row['last_position'] ?? 'N/A') . '</td>';
        echo '<td>' . ($row['last_watched'] ?? 'Never') . '</td>';
        echo '<td>' . ($row['progress_created'] ?? 'Never') . '</td>';
        echo '<td><strong>' . $status_text . '</strong></td>';
        echo '</tr>';
    }
    
    echo '</table>';
    
    echo '<div class="debug-section success">';
    echo '<h3>📊 Summary Statistics</h3>';
    echo '<ul>';
    echo '<li><strong>Total Videos:</strong> ' . $total_videos . '</li>';
    echo '<li><strong>🏆 Completed:</strong> ' . $completed_videos . ' videos</li>';
    echo '<li><strong>⏳ In Progress:</strong> ' . $in_progress_videos . ' videos</li>';
    echo '<li><strong>🔴 Not Started:</strong> ' . $not_started_videos . ' videos</li>';
    echo '<li><strong>📈 Course Progress:</strong> ' . ($total_videos > 0 ? round(($completed_videos / $total_videos) * 100) : 0) . '%</li>';
    echo '</ul>';
    echo '</div>';
    
    $stmt->close();
} else {
    echo '<p class="error">❌ Error fetching videos: ' . $conn->error . '</p>';
}

echo '</div>';

// 3. Test completion detection query (same as used in course-detail.php)
echo '<div class="debug-section">';
echo '<h2>🔍 Completion Detection Query Test</h2>';

$completion_test_query = "
    SELECT 
        v.video_id,
        v.title,
        CASE WHEN vp.completed = 1 THEN 1 ELSE 0 END as completed,
        CASE WHEN vp.last_position > 0 THEN vp.last_position ELSE 0 END as last_position
    FROM videos v
    LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = ?
    WHERE v.course_id = ? AND v.is_published = 1
    ORDER BY v.video_order ASC
";

if ($stmt = $conn->prepare($completion_test_query)) {
    $stmt->bind_param("ii", $user_id, $course_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    echo '<p>✅ <strong>This query matches the one used in course-detail.php:</strong></p>';
    echo '<table>';
    echo '<tr><th>Video ID</th><th>Title</th><th>Completed Flag</th><th>Last Position</th><th>Result</th></tr>';
    
    while ($row = $result->fetch_assoc()) {
        $completed_flag = $row['completed'];
        $result_text = $completed_flag ? '🏆 COMPLETED' : '❌ NOT COMPLETED';
        $row_class = $completed_flag ? 'completed' : 'incomplete';
        
        echo '<tr class="' . $row_class . '">';
        echo '<td>' . $row['video_id'] . '</td>';
        echo '<td>' . htmlspecialchars($row['title']) . '</td>';
        echo '<td><strong>' . $completed_flag . '</strong></td>';
        echo '<td>' . $row['last_position'] . '</td>';
        echo '<td><strong>' . $result_text . '</strong></td>';
        echo '</tr>';
    }
    
    echo '</table>';
    $stmt->close();
}

echo '</div>';

// 4. Raw database query for debugging
echo '<div class="debug-section warning">';
echo '<h2>🛠️ Raw Database Data</h2>';
echo '<p>Direct query from video_progress table:</p>';

$raw_query = "SELECT * FROM video_progress WHERE student_id = ? ORDER BY video_id";
if ($stmt = $conn->prepare($raw_query)) {
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        echo '<table>';
        echo '<tr>';
        echo '<th>ID</th><th>Student ID</th><th>Video ID</th><th>Completion %</th>';
        echo '<th>Last Position</th><th>Completed</th><th>Created</th><th>Updated</th>';
        echo '</tr>';
        
        while ($row = $result->fetch_assoc()) {
            $row_class = $row['completed'] ? 'completed' : 'incomplete';
            echo '<tr class="' . $row_class . '">';
            echo '<td>' . $row['id'] . '</td>';
            echo '<td>' . $row['student_id'] . '</td>';
            echo '<td><strong>' . $row['video_id'] . '</strong></td>';
            echo '<td>' . $row['completion_percentage'] . '%</td>';
            echo '<td>' . $row['last_position'] . '</td>';
            echo '<td><strong>' . ($row['completed'] ? '✅ 1' : '❌ 0') . '</strong></td>';
            echo '<td>' . $row['created_at'] . '</td>';
            echo '<td>' . $row['updated_at'] . '</td>';
            echo '</tr>';
        }
        echo '</table>';
    } else {
        echo '<p class="warning">⚠️ No progress records found for user ' . $user_id . '</p>';
    }
    
    $stmt->close();
}

echo '</div>';

?>

<div class="debug-section">
    <h2>🎯 Debug Analysis</h2>
    <h3>✅ What This Script Verified:</h3>
    <ul>
        <li><strong>Database Structure:</strong> Confirmed video_progress table exists and has correct fields</li>
        <li><strong>Completion Detection:</strong> Tested the exact query used in course-detail.php</li>
        <li><strong>Badge Logic:</strong> Verified which videos should show completed badges</li>
        <li><strong>Data Integrity:</strong> Checked raw database records for consistency</li>
    </ul>
    
    <h3>🔍 Key Points to Check:</h3>
    <ul>
        <li><strong>Completed Field:</strong> Videos with <code>completed = 1</code> should show "COMPLETED" badges</li>
        <li><strong>Progress Tracking:</strong> Videos with <code>completion_percentage > 0</code> have been started</li>
        <li><strong>Badge Display:</strong> The JavaScript reads PHP debug data to determine completion status</li>
    </ul>
    
    <p><strong>🎯 Next Steps:</strong> If the completed badges are showing correctly in the UI and this debug shows <code>completed = 1</code> for those videos, then the database integration is working perfectly!</p>
</div>

</body>
</html>
