<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';

// Require student role to access this page
requireRole('student');

// Check if assignment ID is provided
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    $_SESSION['error'] = "Invalid assignment ID.";
    header('Location: courses.php');
    exit();
}

$student_id = $_SESSION['user_id'];
$assignment_id = (int)$_GET['id'];

// Include database connection
require_once '../config/db_connect.php';

// Helper function to format file size
function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}

// Get assignment details and check if student is enrolled in the course
$assignmentQuery = "
    SELECT a.*, c.course_id, c.title as course_title,
           (SELECT COUNT(*) FROM enrollments WHERE course_id = a.course_id AND student_id = ?) as is_enrolled
    FROM assignments a
    JOIN courses c ON a.course_id = c.course_id
    WHERE a.assignment_id = ?
";

$stmt = $conn->prepare($assignmentQuery);
if (!$stmt) {
    $_SESSION['error'] = "Database error. Please try again later.";
    header("Location: courses.php");
    exit();
}

$stmt->bind_param("ii", $student_id, $assignment_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result && $result->num_rows > 0) {
    $assignment = $result->fetch_assoc();
    
    // Check if student is enrolled in the course
    if ($assignment['is_enrolled'] <= 0) {
        $_SESSION['error'] = "You must be enrolled in this course to download assignment materials.";
        header("Location: course-detail.php?id=" . $assignment['course_id']);
        exit();
    }
    
    // Check if assignment has a file attachment
    if (empty($assignment['file_path'])) {
        $_SESSION['error'] = "This assignment does not have any downloadable files.";
        header("Location: assignment_view.php?id=" . $assignment_id);
        exit();
    }
    
    $filePath = $assignment['file_path'];
    $assignmentTitle = $assignment['title'];
    $courseTitle = $assignment['course_title'];
    
    // Determine the actual file path
    // The student directory is inside the root, so we need to go up one level to access uploads
    if (strpos($filePath, 'uploads/assignments/') === 0) {
        // Path already includes the full structure from root, prepend ../
        $actualFilePath = '../' . $filePath;
    } elseif (strpos($filePath, '../uploads/assignments/') === 0) {
        // Path already includes the correct relative path
        $actualFilePath = $filePath;
    } else {
        // Otherwise, prepend the upload directory structure with correct relative path
        $actualFilePath = '../uploads/assignments/' . $filePath;
    }
    
    // Check if the file actually exists
    if (!file_exists($actualFilePath)) {
        $_SESSION['error'] = "The assignment file could not be found. Please contact your instructor.";
        header("Location: assignment_view.php?id=" . $assignment_id);
        exit();
    }
    
    // Get file info
    $fileSize = filesize($actualFilePath);
    $fileExtension = strtolower(pathinfo($actualFilePath, PATHINFO_EXTENSION));
    
    // Generate a clean filename for display to user
    $fileName = str_replace(' ', '_', strtolower($assignmentTitle)) . '.' . $fileExtension;
    
    // Set up MIME types for different file types
    $mimeTypes = [
        'pdf' => 'application/pdf',
        'doc' => 'application/msword',
        'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        'xls' => 'application/vnd.ms-excel',
        'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        'ppt' => 'application/vnd.ms-powerpoint',
        'pptx' => 'application/vnd.openxmlformats-officedocument.presentationml.presentation',
        'txt' => 'text/plain',
        'zip' => 'application/zip',
        'rar' => 'application/x-rar-compressed',
        'jpg' => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'png' => 'image/png',
        'gif' => 'image/gif',
        'mp3' => 'audio/mpeg',
        'mp4' => 'video/mp4',
    ];
    
    // Set the appropriate content type
    if (array_key_exists($fileExtension, $mimeTypes)) {
        header('Content-Type: ' . $mimeTypes[$fileExtension]);
    } else {
        header('Content-Type: application/octet-stream');
    }
    
    // Properly encode filename for Content-Disposition header
    $encodedFilename = rawurlencode($fileName);
    header("Content-Disposition: attachment; filename*=UTF-8''$encodedFilename; filename=\"$fileName\"");
    header('Expires: 0');
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    header('Content-Length: ' . $fileSize);
    
    // Record the download in user_downloads table for history tracking
    $insertDownloadSql = "INSERT INTO user_downloads (user_id, material_id, material_title, material_type, file_size, course_id, course_name, file_path, category) 
                         VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'Assignment')";
    
    $downloadStmt = $conn->prepare($insertDownloadSql);
    if ($downloadStmt) {
        $formattedSize = formatFileSize($fileSize);
        $fileType = 'Assignment - ' . strtoupper($fileExtension);
        $downloadStmt->bind_param("iisssiis", $student_id, $assignment_id, $assignmentTitle, $fileType, $formattedSize, $assignment['course_id'], $courseTitle, $filePath);
        $downloadStmt->execute();
        $downloadStmt->close();
    }
    
    // Disable output buffering
    while (ob_get_level()) ob_end_clean();
    
    // Output the file
    readfile($actualFilePath);
    exit();
    
} else {
    // Assignment not found or student not enrolled
    $_SESSION['error'] = "Assignment not found or you don't have permission to access it.";
    header('Location: courses.php');
    exit();
}
?>
