<?php
// Session is started in header.php
require_once '../auth/functions.php';

// Require student role authentication - no exceptions
requireRole('student');

// Get error information from URL parameters
$errorType = $_GET['type'] ?? 'not_found';
$resourceId = $_GET['resource_id'] ?? 'Unknown';
$fileName = $_GET['file_name'] ?? 'Unknown';
$resourceTitle = $_GET['title'] ?? 'Resource';

// Define error messages and details
$errors = [
    'not_found' => [
        'icon' => 'bx-file-blank',
        'title' => 'Document Not Found',
        'message' => 'The requested document could not be located on our servers.',
        'description' => 'This could happen if the file has been moved, deleted, or the link is incorrect.',
        'color' => '#e74c3c'
    ],
    'corrupted' => [
        'icon' => 'bx-error-alt',
        'title' => 'Document Corrupted',
        'message' => 'The document appears to be corrupted or damaged.',
        'description' => 'The file may have been damaged during upload or storage. Please contact support.',
        'color' => '#f39c12'
    ],
    'access_denied' => [
        'icon' => 'bx-lock-alt',
        'title' => 'Access Denied',
        'message' => 'You don\'t have permission to download this document.',
        'description' => 'This resource may require special permissions or enrollment in a specific course.',
        'color' => '#e67e22'
    ],
    'server_error' => [
        'icon' => 'bx-server',
        'title' => 'Server Error',
        'message' => 'There was a problem processing your download request.',
        'description' => 'Our servers encountered an issue. Please try again later.',
        'color' => '#9b59b6'
    ]
];

$error = $errors[$errorType] ?? $errors['not_found'];

// Set page title
$page_title = 'Download Error';

// Include header
include_once 'components/header.php';

// Direct CSS inclusion
echo '<style>
/* Main container styling */
.container-fluid {
    background: whitesmoke;
    min-height: calc(100vh - 70px);
    padding: 2rem 1rem;
}

.error-container {
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    border-radius: 25px;
    padding: 4rem 3rem;
    box-shadow: 
        0 20px 60px rgba(52, 152, 219, 0.1),
        0 8px 25px rgba(0, 0, 0, 0.08),
        inset 0 1px 0 rgba(255, 255, 255, 0.9);
    text-align: center;
    max-width: 800px;
    margin: 2rem auto;
    border: 1px solid rgba(52, 152, 219, 0.1);
    position: relative;
    overflow: hidden;
}

.error-container::before {
    content: "";
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 6px;
    background: linear-gradient(90deg, #3498db 0%, #2980b9 50%, #3498db 100%);
    background-size: 200% 100%;
    animation: shimmer 3s ease-in-out infinite;
}

@keyframes shimmer {
    0%, 100% { background-position: 200% 0; }
    50% { background-position: -200% 0; }
}

.error-icon {
    font-size: 5rem;
    color: ' . $error['color'] . ';
    margin-bottom: 2rem;
    display: inline-block;
    animation: bounce 2.5s ease-in-out infinite;
    background: linear-gradient(135deg, ' . $error['color'] . ' 0%, ' . $error['color'] . '80 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    filter: drop-shadow(0 4px 8px rgba(0, 0, 0, 0.1));
}

@keyframes bounce {
    0%, 20%, 50%, 80%, 100% { transform: translateY(0); }
    40% { transform: translateY(-10px); }
    60% { transform: translateY(-5px); }
}

.error-title {
    font-size: 2.8rem;
    font-weight: 800;
    background: linear-gradient(135deg, #2c3e50 0%, #3498db 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    margin-bottom: 1.2rem;
    letter-spacing: -0.5px;
}

.error-message {
    font-size: 1.3rem;
    color: #5a6c7d;
    margin-bottom: 1.8rem;
    font-weight: 500;
    line-height: 1.5;
}

.error-description {
    color: #7f8c8d;
    margin-bottom: 2.5rem;
    line-height: 1.7;
    font-size: 1.05rem;
}

.error-details {
    background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);
    border-radius: 20px;
    padding: 2rem;
    margin: 2.5rem 0;
    border: 1px solid #e3f2fd;
    border-left: 5px solid #3498db;
    text-align: left;
    box-shadow: 
        0 8px 25px rgba(52, 152, 219, 0.08),
        inset 0 1px 0 rgba(255, 255, 255, 0.9);
}

.error-details h5 {
    color: #2c3e50;
    margin-bottom: 1.5rem;
    font-weight: 700;
    text-align: center;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
    font-size: 1.2rem;
}

.error-details h5 i {
    color: #3498db;
    font-size: 1.3rem;
}

.detail-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.8rem 0;
    border-bottom: 1px solid rgba(52, 152, 219, 0.1);
    transition: all 0.3s ease;
}

.detail-item:hover {
    background: rgba(52, 152, 219, 0.02);
    padding-left: 0.5rem;
    border-radius: 8px;
}

.detail-item:last-child {
    border-bottom: none;
}

.detail-label {
    font-weight: 600;
    color: #2c3e50;
    font-size: 1rem;
}

.detail-value {
    color: #3498db;
    max-width: 60%;
    overflow-wrap: break-word;
    font-weight: 500;
    font-family: "Courier New", monospace;
    font-size: 0.9rem;
    background: rgba(52, 152, 219, 0.05);
    padding: 0.3rem 0.8rem;
    border-radius: 8px;
    border: 1px solid rgba(52, 152, 219, 0.1);
}

.action-buttons {
    display: flex;
    gap: 1.2rem;
    justify-content: center;
    flex-wrap: wrap;
    margin-top: 2rem;
}

.btn-custom {
    border-radius: 50px;
    padding: 15px 35px;
    font-weight: 600;
    text-decoration: none;
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    border: none;
    font-size: 1rem;
    position: relative;
    overflow: hidden;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.btn-custom::before {
    content: "";
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.3), transparent);
    transition: left 0.5s;
}

.btn-custom:hover::before {
    left: 100%;
}

.btn-primary-custom {
    background: linear-gradient(135deg, #3498db 0%, #2980b9 100%);
    color: white;
    box-shadow: 0 6px 20px rgba(52, 152, 219, 0.3);
}

.btn-primary-custom:hover {
    transform: translateY(-3px) scale(1.02);
    box-shadow: 0 12px 35px rgba(52, 152, 219, 0.4);
    color: white;
}

.btn-secondary-custom {
    background: linear-gradient(135deg, #ecf0f1 0%, #bdc3c7 100%);
    color: #2c3e50;
    border: 2px solid #bdc3c7;
    box-shadow: 0 6px 20px rgba(189, 195, 199, 0.2);
}

.btn-secondary-custom:hover {
    transform: translateY(-3px) scale(1.02);
    box-shadow: 0 12px 35px rgba(189, 195, 199, 0.3);
    color: #2c3e50;
    background: linear-gradient(135deg, #ffffff 0%, #ecf0f1 100%);
}

.btn-contact {
    background: linear-gradient(135deg, ' . $error['color'] . ' 0%, ' . $error['color'] . 'dd 100%);
    color: white;
    box-shadow: 0 6px 20px rgba(' . hexdec(substr($error['color'], 1, 2)) . ', ' . hexdec(substr($error['color'], 3, 2)) . ', ' . hexdec(substr($error['color'], 5, 2)) . ', 0.3);
}

.btn-contact:hover {
    transform: translateY(-3px) scale(1.02);
    box-shadow: 0 12px 35px rgba(' . hexdec(substr($error['color'], 1, 2)) . ', ' . hexdec(substr($error['color'], 3, 2)) . ', ' . hexdec(substr($error['color'], 5, 2)) . ', 0.4);
    color: white;
}

/* Floating elements for visual appeal */
.floating-elements {
    position: absolute;
    width: 100%;
    height: 100%;
    top: 0;
    left: 0;
    pointer-events: none;
    overflow: hidden;
}

.floating-element {
    position: absolute;
    opacity: 0.03;
    animation: float 8s ease-in-out infinite;
}

.floating-element:nth-child(1) {
    top: 15%;
    left: 10%;
    font-size: 2rem;
    animation-delay: 0s;
}

.floating-element:nth-child(2) {
    top: 70%;
    right: 15%;
    font-size: 1.5rem;
    animation-delay: 3s;
}

.floating-element:nth-child(3) {
    bottom: 25%;
    left: 25%;
    font-size: 1.8rem;
    animation-delay: 6s;
}

@keyframes float {
    0%, 100% { transform: translateY(0px) rotate(0deg); }
    25% { transform: translateY(-15px) rotate(2deg); }
    50% { transform: translateY(-25px) rotate(-2deg); }
    75% { transform: translateY(-10px) rotate(1deg); }
}

/* Responsive design */
@media (max-width: 768px) {
    .container-fluid {
        padding: 1rem 0.5rem;
    }
    
    .error-container {
        padding: 2.5rem 2rem;
        margin: 1rem;
        border-radius: 20px;
    }
    
    .error-title {
        font-size: 2.2rem;
    }
    
    .error-icon {
        font-size: 4rem;
    }
    
    .error-message {
        font-size: 1.1rem;
    }
    
    .action-buttons {
        flex-direction: column;
        gap: 1rem;
    }
    
    .btn-custom {
        padding: 12px 25px;
        font-size: 0.95rem;
    }
    
    .error-details {
        padding: 1.5rem;
    }
}

@media (max-width: 480px) {
    .error-container {
        padding: 2rem 1.5rem;
    }
    
    .error-title {
        font-size: 1.8rem;
    }
    
    .detail-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.3rem;
    }
    
    .detail-value {
        max-width: 100%;
        font-size: 0.85rem;
    }
}
</style>';

?>
<!-- Error Container -->
<div class="container-fluid">

    <div class="error-container">
        <div class="floating-elements">
            <div class="floating-element">
                <i class="bx bx-file-pdf" style="color: #3498db;"></i>
            </div>
            <div class="floating-element">
                <i class="bx bx-cloud-download" style="color: #2980b9;"></i>
            </div>
            <div class="floating-element">
                <i class="bx bx-shield-x" style="color: #e74c3c;"></i>
            </div>
        </div>

        <div class="error-icon">
            <i class="bx <?= $error['icon'] ?>"></i>
        </div>

        <h1 class="error-title"><?= $error['title'] ?></h1>
        <p class="error-message"><?= $error['message'] ?></p>
        <p class="error-description"><?= $error['description'] ?></p>

        <div class="error-details">
            <h5><i class="bx bx-info-circle me-2"></i>Request Details</h5>
            <div class="detail-item">
                <span class="detail-label">Resource Title:</span>
                <span class="detail-value"><?= htmlspecialchars($resourceTitle) ?></span>
            </div>
            <div class="detail-item">
                <span class="detail-label">Resource ID:</span>
                <span class="detail-value"><?= htmlspecialchars($resourceId) ?></span>
            </div>
            <div class="detail-item">
                <span class="detail-label">File Name:</span>
                <span class="detail-value"><?= htmlspecialchars($fileName) ?></span>
            </div>
            <div class="detail-item">
                <span class="detail-label">Date & Time:</span>
                <span class="detail-value"><?= date('M d, Y - H:i:s') ?></span>
            </div>
        </div>

        <div class="action-buttons">
            <a href="library.php" class="btn btn-custom btn-primary-custom">
                <i class="bx bx-arrow-back me-2"></i>Back to Library
            </a>
            
            <a href="javascript:history.back()" class="btn btn-custom btn-secondary-custom">
                <i class="bx bx-undo me-2"></i>Go Back
            </a>
            
            <a href="messages.php" class="btn btn-custom btn-contact">
                <i class="bx bx-support me-2"></i>Contact Support
            </a>
        </div>
    </div>
</div>

<script>
// Add some interactive effects
document.addEventListener('DOMContentLoaded', function() {
    const errorIcon = document.querySelector('.error-icon');
    
    // Add click animation to icon
    errorIcon.addEventListener('click', function() {
        this.style.animation = 'none';
        setTimeout(() => {
            this.style.animation = 'pulse 0.5s ease-in-out';
            setTimeout(() => {
                this.style.animation = 'pulse 2s ease-in-out infinite alternate';
            }, 500);
        }, 10);
    });

    // Auto-refresh suggestion after 30 seconds
    let refreshTimer;
    if ('<?= $errorType ?>' === 'server_error') {
        refreshTimer = setTimeout(() => {
            if (confirm('Would you like to try downloading again?')) {
                window.history.back();
            }
        }, 30000);
    }
});
</script>
