<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';

// Require student role to access this page
requireRole('student');

// Check if material ID is provided
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    $_SESSION['error'] = "Invalid material ID.";
    header('Location: courses.php');
    exit();
}

$student_id = $_SESSION['user_id'];
$material_id = (int)$_GET['id'];
$filePath = '';
$fileName = '';
$fileTitle = '';

// Include database connection
require_once '../config/db_connect.php';

// Helper function to format file size
function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}

// Get material details and check if student is enrolled in the course
$materialQuery = "
    SELECT m.*, c.course_id,
           (SELECT COUNT(*) FROM enrollments WHERE course_id = m.course_id AND student_id = ?) as is_enrolled
    FROM course_materials m
    JOIN courses c ON m.course_id = c.course_id
    WHERE m.material_id = ? AND m.status = 'active'
";

$stmt = $conn->prepare($materialQuery);
if (!$stmt) {
    $_SESSION['error'] = "Database error. Please try again later.";
    header("Location: courses.php");
    exit();
}

$stmt->bind_param("ii", $student_id, $material_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result && $result->num_rows > 0) {
    $material = $result->fetch_assoc();
    
    // Check if student is enrolled in the course
    if ($material['is_enrolled'] <= 0) {
        $_SESSION['error'] = "You must be enrolled in this course to download its materials.";
        header("Location: course-detail.php?id=" . $material['course_id']);
        exit();
    }
    
    $filePath = $material['file_path'];
    $fileTitle = $material['title'];
    $fileType = $material['file_type'];
    
    // Generate a clean filename for display to user
    $fileExt = pathinfo($filePath, PATHINFO_EXTENSION);
    $fileName = str_replace(' ', '_', strtolower($fileTitle)) . '.' . strtolower($fileExt);
    
    // Log the download in the database if you want to track downloads
    // This could be a separate table for material_downloads
    
    // Determine the actual file path
    // If the file_path already contains the full path (starts with '../uploads/materials/')
    if (strpos($filePath, '../uploads/materials/') === 0) {
        $actualFilePath = $filePath;
    } else {
        // Otherwise, prepend the upload directory
        $actualFilePath = '../uploads/materials/' . $filePath;
    }
    
    // Check if the file actually exists
    if (!file_exists($actualFilePath)) {
        // If file doesn't exist, create a placeholder for demo purposes
        $materialsDir = '../uploads/materials';
        if (!file_exists($materialsDir)) {
            mkdir($materialsDir, 0755, true);
        }
        
        // Create a placeholder file based on file type
        $placeholderPath = $materialsDir . '/' . uniqid() . '_placeholder';
        if (strpos($fileType, 'pdf') !== false) {
            $placeholderPath .= '.pdf';
            
            // Instead of creating an invalid PDF, redirect to a message page
            $_SESSION['warning'] = "The requested PDF file '{$fileTitle}' could not be found in the system. Please contact your instructor.";
            header("Location: course-materials.php?course_id=" . $material['course_id']);
            exit();
        } else if (strpos($fileType, 'image') !== false) {
            $placeholderPath .= '.png';
            // Use a placeholder image from assets
            copy('../assets/img/placeholder.png', $placeholderPath);
        } else {
            $placeholderPath .= '.html';
            // Create an HTML placeholder
            $content = "<!DOCTYPE html>
<html>
<head>
    <title>{$fileTitle}</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 40px; }
        h1 { color: #2c3e50; }
        .material-info { background-color: #f8f9fa; padding: 20px; border-radius: 5px; }
        .note { color: #6c757d; font-style: italic; }
    </style>
</head>
<body>
    <h1>{$fileTitle}</h1>
    <div class='material-info'>
        <p>This is a placeholder file for course material.</p>
        <p>In a production environment, this would be the actual course material content.</p>
    </div>
    <p class='note'>Material ID: {$material_id}</p>
    <p class='note'>Downloaded on: " . date('Y-m-d H:i:s') . "</p>
</body>
</html>";
            file_put_contents($placeholderPath, $content);
        }
        
        // Update the path to the placeholder
        $actualFilePath = $placeholderPath;
    }
    
    // Set appropriate headers for the download
    header('Content-Description: File Transfer');
    
    // Set content type based on file extension
    $fileExtension = strtolower(pathinfo($actualFilePath, PATHINFO_EXTENSION));
    
    // Map file extensions to MIME types
    $mimeTypes = [
        'pdf' => 'application/pdf',
        'doc' => 'application/msword',
        'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        'xls' => 'application/vnd.ms-excel',
        'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        'ppt' => 'application/vnd.ms-powerpoint',
        'pptx' => 'application/vnd.openxmlformats-officedocument.presentationml.presentation',
        'txt' => 'text/plain',
        'zip' => 'application/zip',
        'rar' => 'application/x-rar-compressed',
        'jpg' => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'png' => 'image/png',
        'gif' => 'image/gif',
        'mp3' => 'audio/mpeg',
        'mp4' => 'video/mp4',
    ];
    
    // Set the appropriate content type
    if (array_key_exists($fileExtension, $mimeTypes)) {
        header('Content-Type: ' . $mimeTypes[$fileExtension]);
    } else {
        header('Content-Type: application/octet-stream');
    }
    
    // Properly encode filename for Content-Disposition header
    $encodedFilename = rawurlencode($fileName);
    header("Content-Disposition: attachment; filename*=UTF-8''$encodedFilename; filename=\"$fileName\"");
    header('Expires: 0');
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    
    // Check one more time if file exists before attempting to output
    if (file_exists($actualFilePath)) {
        // Get the file size
        $fileSize = filesize($actualFilePath);
        header('Content-Length: ' . $fileSize);
        
        // Record the download in user_downloads table for history tracking
        $insertDownloadSql = "INSERT INTO user_downloads (user_id, material_id, material_title, material_type, file_size, course_id, course_name, file_path, category) 
                             SELECT ?, m.material_id, m.title, m.file_type, ?, m.course_id, c.title, m.file_path, 'Course Materials'
                             FROM course_materials m 
                             JOIN courses c ON m.course_id = c.course_id 
                             WHERE m.material_id = ?";
        
        $downloadStmt = $conn->prepare($insertDownloadSql);
        if ($downloadStmt) {
            $formattedSize = formatFileSize($fileSize);
            $downloadStmt->bind_param("isi", $student_id, $formattedSize, $material_id);
            $downloadStmt->execute();
            $downloadStmt->close();
        }
        
        // For debugging - store path information in session
        $_SESSION['debug_file_info'] = [
            'path' => $actualFilePath,
            'size' => $fileSize,
            'type' => $mimeTypes[$fileExtension] ?? 'unknown'
        ];
        
        // Disable output buffering
        while (ob_get_level()) ob_end_clean();
        
        // Output the file
        readfile($actualFilePath);
        exit();
    } else {
        // File doesn't exist, redirect with error message
        $_SESSION['error'] = "Material file could not be found at: {$actualFilePath}";
        header("Location: course-materials.php?course_id=" . $material['course_id']);
        exit();
    }
    } else {
        // Material not found or student not enrolled
        $_SESSION['error'] = "Material not found or you don't have permission to access it.";
        header('Location: courses.php');
        exit();
    }


// If we get here, something went wrong with the download process
$_SESSION['error'] = "An unexpected error occurred during download.";
header('Location: courses.php');
exit();
?>
