<?php
/**
 * Student Download Script for Teacher Feedback Files
 * Allows students to download teacher feedback files for their own submissions
 */

// Start session with proper settings
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);
session_start();

// Include database connection first
require_once '../includes/db_connection.php';

// Debug mode
$debug = true;

if ($debug) {
    echo "DEBUG: Starting student download script<br>";
    echo "DEBUG: Session ID: " . session_id() . "<br>";
    echo "DEBUG: Session data: " . print_r($_SESSION, true) . "<br>";
    echo "DEBUG: GET params: " . print_r($_GET, true) . "<br>";
}

// Get parameters
$file_name = $_GET['file'] ?? '';
$submission_id = intval($_GET['submission'] ?? 0);
$student_id = intval($_GET['student'] ?? 0);

if ($debug) {
    echo "DEBUG: File: $file_name<br>";
    echo "DEBUG: Submission ID: $submission_id<br>";
    echo "DEBUG: Student ID: $student_id<br>";
}

if (empty($file_name) || $submission_id <= 0) {
    die("Invalid request - Missing file or submission parameters");
}

// If no session, check if student ID provided and valid
$current_student_id = null;

if (!empty($_SESSION['user_id'])) {
    $current_student_id = $_SESSION['user_id'];
    if ($debug) echo "DEBUG: Using session student ID: $current_student_id<br>";
} elseif ($student_id > 0) {
    // Verify student ID matches submission
    $verify_query = "SELECT student_id FROM assignment_submissions WHERE submission_id = ? AND student_id = ?";
    $stmt = $conn->prepare($verify_query);
    $stmt->bind_param("ii", $submission_id, $student_id);
    $stmt->execute();
    $verify_result = $stmt->get_result()->fetch_assoc();
    
    if ($verify_result) {
        $current_student_id = $student_id;
        if ($debug) echo "DEBUG: Using verified student ID: $current_student_id<br>";
    }
}

if (!$current_student_id) {
    die("Authentication required - Please access this file from your assignment details page");
}

// Verify this submission belongs to the student
$permission_query = "
    SELECT s.assignment_id, s.teacher_feedback_files, s.student_id
    FROM assignment_submissions s
    WHERE s.submission_id = ? AND s.student_id = ?
";

if ($debug) {
    echo "DEBUG: Querying submission with ID $submission_id for student $current_student_id<br>";
}

$stmt = $conn->prepare($permission_query);
$stmt->bind_param("ii", $submission_id, $current_student_id);
$stmt->execute();
$result = $stmt->get_result()->fetch_assoc();

if ($debug) {
    echo "DEBUG: Query result: " . print_r($result, true) . "<br>";
}

if (!$result) {
    die("Access denied - This submission does not belong to you");
}

// Verify file exists in teacher feedback files JSON
$teacher_files = json_decode($result['teacher_feedback_files'], true) ?: [];
$file_found = false;
$original_name = '';

foreach ($teacher_files as $file) {
    if ($file['file_path'] === $file_name) {
        $file_found = true;
        $original_name = $file['original_name'];
        break;
    }
}

if (!$file_found) {
    die("File not found in teacher feedback");
}

// Build file path
$file_path = __DIR__ . '/../uploads/teacher_feedback/' . $file_name;

if (!file_exists($file_path)) {
    die("File does not exist on server");
}

// Get file info
$file_size = filesize($file_path);
$file_extension = pathinfo($original_name, PATHINFO_EXTENSION);
$mime_type = 'application/octet-stream';

// Set appropriate MIME type
$mime_types = [
    'pdf' => 'application/pdf',
    'doc' => 'application/msword',
    'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'txt' => 'text/plain',
    'jpg' => 'image/jpeg',
    'jpeg' => 'image/jpeg',
    'png' => 'image/png',
    'gif' => 'image/gif',
    'mp4' => 'video/mp4',
    'mov' => 'video/quicktime',
    'avi' => 'video/x-msvideo',
    'mp3' => 'audio/mpeg',
    'wav' => 'audio/wav',
    'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
    'xls' => 'application/vnd.ms-excel',
    'ppt' => 'application/vnd.ms-powerpoint',
    'pptx' => 'application/vnd.openxmlformats-officedocument.presentationml.presentation'
];

if (isset($mime_types[strtolower($file_extension)])) {
    $mime_type = $mime_types[strtolower($file_extension)];
}

// Check if this should be displayed inline or as download
$is_inline_view = isset($_GET['inline']) && $_GET['inline'] === '1';
$is_image = in_array(strtolower($file_extension), ['jpg', 'jpeg', 'png', 'gif']);
$is_video = in_array(strtolower($file_extension), ['mp4', 'mov', 'avi', 'webm']);
$is_audio = in_array(strtolower($file_extension), ['mp3', 'wav']);
$is_pdf = strtolower($file_extension) === 'pdf';

// Send appropriate headers
header('Content-Type: ' . $mime_type);

if ($is_inline_view || $is_image || $is_video || $is_audio || $is_pdf) {
    // Display inline for viewing
    header('Content-Disposition: inline; filename="' . $original_name . '"');
} else {
    // Force download for other file types
    header('Content-Disposition: attachment; filename="' . $original_name . '"');
}

header('Content-Length: ' . $file_size);
header('Cache-Control: public, max-age=3600');
header('Accept-Ranges: bytes');

// Output file
readfile($file_path);
exit;
?>
