<?php
// 🚀 DYNAMIC LESSON DATA ENDPOINT - No page reload required
// CRITICAL: Start output buffering to prevent any HTML output before JSON
ob_start();

// Suppress all error output to prevent breaking JSON
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

header('Content-Type: application/json');
header('Cache-Control: no-cache, must-revalidate');

try {
    // Use the SAME authentication approach as course-detail.php
    require_once '../includes/session_start.php';
    require_once '../auth/functions.php';
    require_once '../includes/id_encryption.php';  // CRITICAL: Add ID encryption support
    require_once '../config/db_connect.php';
} catch (Exception $e) {
    // Clear any output buffer and send error
    ob_clean();
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Include error: ' . $e->getMessage()
    ]);
    exit;
}

// Security: Use the SAME requireRole function as course-detail.php
// BUT catch any redirects and return JSON instead
try {
    // Check authentication manually to avoid redirects
    if (!isset($_SESSION['user_id'])) {
        throw new Exception('No user session found');
    }
    
    $userRole = $_SESSION['user_role'] ?? $_SESSION['role'] ?? '';
    if ($userRole !== 'student') {
        throw new Exception('Invalid role: ' . $userRole);
    }
    
    // Authentication successful
    $user_id = $_SESSION['user_id'];
    
} catch (Exception $e) {
    // Clear any output buffer and send JSON error
    ob_clean();
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => 'Authentication required: ' . $e->getMessage(),
        'debug' => [
            'session_exists' => isset($_SESSION),
            'user_id' => $_SESSION['user_id'] ?? 'not set',
            'user_role' => $_SESSION['user_role'] ?? 'not set',
            'role' => $_SESSION['role'] ?? 'not set',
            'all_session' => array_keys($_SESSION ?? [])
        ]
    ]);
    exit;
}

try {
    // Clear any stray output that might have been generated
    ob_clean();
    
    // 🔐 CRITICAL: Handle encrypted IDs properly - EXACT SAME AS COURSE-DETAIL.PHP
    $raw_course_id = $_GET['course_id'] ?? '';
    $raw_lesson_id = $_GET['lesson_id'] ?? '';
    
    // Try to decrypt IDs if they appear to be encrypted
    if (!empty($raw_course_id)) {
        if (is_numeric($raw_course_id)) {
            // Already a plain number
            $course_id = (int)$raw_course_id;
        } else {
            // Try to decrypt encrypted ID
            $decrypted = decryptId($raw_course_id);
            if ($decrypted === false) {
                throw new Exception('Invalid encrypted course_id: ' . $raw_course_id);
            }
            $course_id = $decrypted;
        }
    } else {
        $course_id = 0;
    }
    
    if (!empty($raw_lesson_id)) {
        if (is_numeric($raw_lesson_id)) {
            // Already a plain number
            $lesson_id = (int)$raw_lesson_id;
        } else {
            // Try to decrypt encrypted ID
            $decrypted = decryptId($raw_lesson_id);
            if ($decrypted === false) {
                throw new Exception('Invalid encrypted lesson_id: ' . $raw_lesson_id);
            }
            $lesson_id = $decrypted;
        }
    } else {
        $lesson_id = 0;
    }

    if (!$course_id || !$lesson_id) {
        throw new Exception('Missing or invalid required parameters - course_id: ' . $course_id . ', lesson_id: ' . $lesson_id);
    }

    // Verify student is enrolled in this course
    $enrollment_check = $conn->prepare("SELECT enrollment_id FROM enrollments WHERE student_id = ? AND course_id = ? AND status = 'active'");
    $enrollment_check->bind_param("ii", $user_id, $course_id);
    $enrollment_check->execute();
    $enrollment_result = $enrollment_check->get_result();
    
    if ($enrollment_result->num_rows === 0) {
        throw new Exception('Access denied: Not enrolled in this course');
    }
    $enrollment_check->close();

    // Fetch lesson data from videos table
    $lesson_query = $conn->prepare("
        SELECT 
            v.video_id,
            v.lesson_id,
            v.title,
            v.description,
            v.video_url,
            v.video_path,
            v.thumbnail,
            v.duration,
            v.created_at,
            v.updated_at,
            v.video_order,
            v.is_published,
            v.last_position,
            CASE 
                WHEN v.video_url LIKE '%youtube.com%' OR 
                     v.video_url LIKE '%youtu.be%' OR 
                     v.video_url LIKE '%vimeo.com%' OR
                     v.video_url LIKE 'http%' THEN 1 
                ELSE 0 
            END as is_external,
            -- Check if lesson is completed by this student
            CASE 
                WHEN uvc.completion_percentage >= 100 THEN 1
                WHEN vp.completed = 1 THEN 1
                ELSE 0
            END as completed,
            -- Get watch progress
            COALESCE(uvc.completion_percentage, vp.progress_percentage, 0) as watch_progress
        FROM videos v
        LEFT JOIN user_video_completions uvc ON v.video_id = uvc.video_id AND uvc.user_id = ?
        LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = ?
        WHERE v.course_id = ? AND v.lesson_id = ? AND v.is_published = 1
        ORDER BY v.video_order ASC, v.created_at ASC
        LIMIT 1
    ");

    if (!$lesson_query) {
        throw new Exception('Database prepare failed: ' . $conn->error);
    }

    $lesson_query->bind_param("iiii", $user_id, $user_id, $course_id, $lesson_id);
    $lesson_query->execute();
    $lesson_result = $lesson_query->get_result();

    if ($lesson_result->num_rows === 0) {
        throw new Exception('Lesson not found or not published');
    }

    $lesson = $lesson_result->fetch_assoc();
    $lesson_query->close();

    // Process lesson data for frontend
    $lesson['lesson_id'] = (int)$lesson['lesson_id'];
    $lesson['video_id'] = (int)$lesson['video_id'];
    $lesson['is_external'] = (bool)$lesson['is_external'];
    $lesson['completed'] = (bool)$lesson['completed'];
    $lesson['watch_progress'] = (float)$lesson['watch_progress'];

    // Handle video URL/path based on type
    if (!empty($lesson['video_url'])) {
        // Check if this is a local file path (starts with 'uploads/') or external URL
        if (strpos($lesson['video_url'], 'uploads/') === 0) {
            // Local video file stored in video_url field - construct correct relative path
            $video_source = '../' . $lesson['video_url']; // '../uploads/videos/filename.mp4'
            $lesson['video_url'] = $video_source;
        } elseif (preg_match('/^https?:\/\//', $lesson['video_url'])) {
            // External video URL (YouTube, Vimeo, etc.) - use as-is
            $video_source = $lesson['video_url'];
        } else {
            // Fallback: treat as local file path
            $video_source = '../uploads/videos/' . basename($lesson['video_url']);
            $lesson['video_url'] = $video_source;
        }
    } elseif (!empty($lesson['video_path'])) {
        // Local video file in separate video_path field - construct full path
        $video_source = '../uploads/videos/' . $lesson['video_path'];
        $lesson['video_url'] = $video_source;
    } else {
        $lesson['video_url'] = null;
    }

    // Handle thumbnail path
    if (!empty($lesson['thumbnail']) && !preg_match('/^https?:\/\//', $lesson['thumbnail'])) {
        $lesson['thumbnail'] = '../uploads/thumbnails/' . $lesson['thumbnail'];
    }

    // Add additional course context if needed
    $course_query = $conn->prepare("SELECT title, description FROM courses WHERE course_id = ?");
    $course_query->bind_param("i", $course_id);
    $course_query->execute();
    $course_result = $course_query->get_result();
    $course_data = $course_result->fetch_assoc();
    $course_query->close();

    // Check if lesson has assignments, quizzes, materials
    $lesson['has_assignment'] = false;
    $lesson['has_quiz'] = false;
    $lesson['has_materials'] = false;

    // Check for assignments
    $assignment_check = $conn->prepare("SELECT COUNT(*) as count FROM assignments WHERE course_id = ?");
    if ($assignment_check) {
        $assignment_check->bind_param("i", $course_id);
        $assignment_check->execute();
        $assignment_result = $assignment_check->get_result();
        $assignment_data = $assignment_result->fetch_assoc();
        $lesson['has_assignment'] = $assignment_data['count'] > 0;
        $assignment_check->close();
    }

    // Check for quizzes
    $quiz_check = $conn->prepare("SELECT COUNT(*) as count FROM quizzes WHERE course_id = ?");
    if ($quiz_check) {
        $quiz_check->bind_param("i", $course_id);
        $quiz_check->execute();
        $quiz_result = $quiz_check->get_result();
        $quiz_data = $quiz_result->fetch_assoc();
        $lesson['has_quiz'] = $quiz_data['count'] > 0;
        $quiz_check->close();
    }

    // Check for materials
    $material_check = $conn->prepare("SELECT COUNT(*) as count FROM course_materials WHERE course_id = ?");
    if ($material_check) {
        $material_check->bind_param("i", $course_id);
        $material_check->execute();
        $material_result = $material_check->get_result();
        $material_data = $material_result->fetch_assoc();
        $lesson['has_materials'] = $material_data['count'] > 0;
        $material_check->close();
    }

    // Clear output buffer one final time before JSON response
    ob_clean();
    
    // Return successful response
    echo json_encode([
        'success' => true,
        'message' => 'Lesson data loaded successfully',
        'lesson' => $lesson,
        'course' => $course_data,
        'timestamp' => date('Y-m-d H:i:s')
    ]);

} catch (Exception $e) {
    // Log the full error for debugging
    error_log("DYNAMIC LESSON DATA ERROR: " . $e->getMessage() . " | File: " . __FILE__ . " | Line: " . __LINE__);
    
    // Clear any output and send clean JSON error
    ob_clean();
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'error_code' => 'LESSON_FETCH_ERROR',
        'debug' => [
            'file' => basename(__FILE__),
            'course_id' => $course_id ?? 'not set',
            'lesson_id' => $lesson_id ?? 'not set',
            'user_id' => $user_id ?? 'not set'
        ]
    ]);
}

// Final cleanup - close database connection and flush output
if (isset($conn)) {
    $conn->close();
}

// End output buffering and flush clean JSON
ob_end_flush();
?>
