<?php
// 🚀 CLEAN LESSON DATA ENDPOINT - Following exact project session pattern
ob_start();
error_reporting(0);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

header('Content-Type: application/json');
header('Cache-Control: no-cache, must-revalidate');

try {
    // ✅ CRITICAL: Use EXACT same session pattern as course-detail.php
    require_once '../includes/session_start.php';
    require_once '../auth/functions.php';
    require_once '../includes/id_encryption.php';
    
    // Use the same authentication check as the main page
    requireRole('student');
    
    // Get user data from session (same way as main page)
    $user_id = $_SESSION['user_id'];
    
    // ✅ Use same database connection as main page
    require_once '../config/db_connect.php';
    
    // Handle course_id from POST JSON data or GET parameter
    $input_data = json_decode(file_get_contents('php://input'), true) ?? [];
    $raw_course_id = $input_data['course_id'] ?? $_GET['course_id'] ?? '';
    
    if (empty($raw_course_id)) {
        error_log("❌ COURSE ID ERROR: No course_id found in POST JSON or GET");
        error_log("POST data: " . json_encode($input_data));
        error_log("GET data: " . json_encode($_GET));
        throw new Exception('No course_id provided');
    }
    
    error_log("✅ COURSE ID: Received course_id: " . substr($raw_course_id, 0, 30) . "...");
    
    // 🧪 CRITICAL: Test decryption function first
    if (!function_exists('decryptId')) {
        throw new Exception('decryptId function not available - check if id_encryption.php was loaded');
    }
    
    if (is_numeric($raw_course_id)) {
        $course_id = (int)$raw_course_id;
        error_log("✅ DECRYPT: course_id is already numeric: $course_id");
    } else {
        // This is encrypted - decrypt it
        error_log("🔐 DECRYPT: Attempting to decrypt course_id: " . substr($raw_course_id, 0, 30) . "...");
        
        // 🧪 Test the decryption step by step
        try {
            $course_id = decryptId($raw_course_id);
            error_log("🔓 DECRYPT: Raw result: " . var_export($course_id, true));
            
            if ($course_id === false) {
                throw new Exception('decryptId returned FALSE for: ' . substr($raw_course_id, 0, 20) . '...');
            }
            if (empty($course_id)) {
                throw new Exception('decryptId returned empty result for: ' . substr($raw_course_id, 0, 20) . '...');
            }
            if (!is_numeric($course_id)) {
                throw new Exception('decryptId returned non-numeric result: ' . var_export($course_id, true));
            }
            
            $course_id = (int)$course_id;
            error_log("✅ DECRYPT: Successfully decrypted to: $course_id");
            
        } catch (Exception $e) {
            throw new Exception('Decryption failed: ' . $e->getMessage() . ' (Input: ' . substr($raw_course_id, 0, 20) . '...)');
        }
    }
    
    // Handle lesson_id from POST JSON data or GET parameter
    $raw_lesson_id = $input_data['lesson_id'] ?? $_GET['lesson_id'] ?? '';
    
    if (empty($raw_lesson_id)) {
        error_log("❌ LESSON ID ERROR: No lesson_id found in POST JSON or GET");
        throw new Exception('No lesson_id provided');
    }
    
    error_log("✅ LESSON ID: Received lesson_id: $raw_lesson_id");
    
    if (is_numeric($raw_lesson_id)) {
        $lesson_id = (int)$raw_lesson_id;
        error_log("✅ DECRYPT: lesson_id is already numeric: $lesson_id");
    } else {
        error_log("🔐 DECRYPT: Attempting to decrypt lesson_id: $raw_lesson_id");
        $lesson_id = decryptId($raw_lesson_id);
        if ($lesson_id === false) {
            throw new Exception('Invalid encrypted lesson_id: ' . $raw_lesson_id);
        }
        $lesson_id = (int)$lesson_id;
    }
    
    if (!$course_id || !$lesson_id) {
        throw new Exception('Invalid parameters after decryption: course_id=' . $course_id . ', lesson_id=' . $lesson_id);
    }
    
    error_log("🎯 FINAL IDs: course_id=$course_id, lesson_id=$lesson_id");
    
    if (!isset($conn) || !$conn) {
        throw new Exception('Database connection failed');
    }
    
    // 🔍 DEBUG: Check database connection
    error_log("🔍 DATABASE: Connection status: " . ($conn ? 'Connected' : 'NOT Connected'));
    error_log("🔍 DATABASE: Connection error: " . $conn->error);
    
    // Fixed lesson query - using ACTUAL column names from videos table
    $sql = "
        SELECT 
            v.video_id,
            v.video_id as lesson_id,
            v.title,
            v.description,
            v.video_url,
            v.external_url,
            v.platform,
            v.thumbnail,
            v.duration,
            v.video_type,
            v.embed_code,
            v.video_id_external,
            CASE 
                WHEN v.video_url LIKE '%youtube.com%' OR 
                     v.video_url LIKE '%youtu.be%' OR 
                     v.video_url LIKE '%vimeo.com%' OR
                     v.external_url LIKE 'http%' OR
                     v.video_type = 'url' THEN 1 
                ELSE 0 
            END as is_external
        FROM videos v
        WHERE v.course_id = ? AND v.video_id = ? AND v.is_published = 1
        ORDER BY v.video_order ASC, v.created_at ASC
        LIMIT 1
    ";
    
    error_log("🔍 QUERY: Preparing SQL: " . preg_replace('/\s+/', ' ', $sql));
    error_log("🔍 QUERY: Parameters - course_id: $course_id, lesson_id: $lesson_id");
    
    $lesson_query = $conn->prepare($sql);

    if (!$lesson_query) {
        error_log("❌ QUERY PREPARE FAILED: " . $conn->error);
        error_log("❌ QUERY ERRNO: " . $conn->errno);
        throw new Exception('Query prepare failed: ' . $conn->error);
    }
    
    error_log("✅ QUERY: Successfully prepared");

    $lesson_query->bind_param("ii", $course_id, $lesson_id);
    
    if (!$lesson_query->execute()) {
        throw new Exception('Query execution failed');
    }
    
    $lesson_result = $lesson_query->get_result();

    if ($lesson_result->num_rows === 0) {
        throw new Exception('Lesson not found');
    }

    $lesson = $lesson_result->fetch_assoc();
    $lesson_query->close();

    // Process lesson data - return comprehensive lesson info with proper URL handling
    $lesson_data = [
        'lesson_id' => (int)$lesson['lesson_id'],
        'video_id' => (int)$lesson['video_id'],
        'title' => $lesson['title'],
        'description' => $lesson['description'],
        'video_url' => $lesson['external_url'] ?: $lesson['video_url'], // Use external_url first
        'external_url' => $lesson['external_url'],
        'platform' => $lesson['platform'],
        'video_type' => $lesson['video_type'],
        'embed_code' => $lesson['embed_code'],
        'video_id_external' => $lesson['video_id_external'],
        'thumbnail' => $lesson['thumbnail'],
        'duration' => $lesson['duration'],
        'is_external' => (bool)$lesson['is_external']
    ]; // Simple default for now

    // 🚨 CRITICAL FIX: Detect video platform while preserving database video_type
    error_log("🔍 BACKEND DEBUG: Raw video_url = " . ($lesson['video_url'] ?? 'NULL'));
    error_log("🔍 BACKEND DEBUG: Raw video_path = " . ($lesson['video_path'] ?? 'NULL'));
    error_log("🔍 BACKEND DEBUG: Database video_type = " . ($lesson['video_type'] ?? 'NULL'));
    
    // Add platform detection for frontend (separate from database video_type)
    $lesson['video_platform'] = 'unknown';
    
    if (!empty($lesson['video_url'])) {
        $video_url = $lesson['video_url'];
        error_log("🔍 BACKEND DEBUG: Processing video_url: $video_url");
        
        if (strpos($video_url, 'youtube.com') !== false || strpos($video_url, 'youtu.be') !== false) {
            $lesson['video_platform'] = 'youtube';
            // Keep database video_type as 'url' - don't change it!
            error_log("🎬 BACKEND SUCCESS: Detected YouTube video: $video_url");
            error_log("🎬 BACKEND SUCCESS: Set video_platform = 'youtube' (keeping video_type = '" . $lesson['video_type'] . "')");
        } elseif (strpos($video_url, 'vimeo.com') !== false) {
            $lesson['video_platform'] = 'vimeo';
            error_log("🎬 BACKEND SUCCESS: Detected Vimeo video: $video_url");
        } else {
            $lesson['video_platform'] = 'external_url';
            error_log("🎬 BACKEND WARNING: Detected other external URL video: $video_url");
        }
    } elseif (!empty($lesson['video_path'])) {
        $lesson['video_platform'] = 'local_file';
        error_log("🎬 BACKEND SUCCESS: Detected local file video: " . $lesson['video_path']);
    } else {
        $lesson['video_platform'] = 'none';
        error_log("🎬 BACKEND ERROR: No video source found");
    }
    
    error_log("🔍 BACKEND DEBUG: Final video_type = " . ($lesson['video_type'] ?? 'NULL') . ", video_platform = " . $lesson['video_platform']);

    // Handle video URL/path based on type
    if (!empty($lesson['video_url'])) {
        // Check if this is a local file path (starts with 'uploads/') or external URL
        if (strpos($lesson['video_url'], 'uploads/') === 0) {
            // Local video file stored in video_url field - construct correct relative path
            $video_source = '../' . $lesson['video_url']; // '../uploads/videos/filename.mp4'
            $lesson['video_url'] = $video_source;
            error_log("🔧 BACKEND FIX: Corrected local video URL path: " . $video_source);
        } elseif (preg_match('/^https?:\/\//', $lesson['video_url'])) {
            // External video URL (YouTube, Vimeo, etc.) - use as-is
            $video_source = $lesson['video_url'];
            error_log("🔗 BACKEND: Using external video URL: " . $video_source);
        } else {
            // Fallback: treat as local file path
            $video_source = '../uploads/videos/' . basename($lesson['video_url']);
            $lesson['video_url'] = $video_source;
            error_log("🔧 BACKEND FALLBACK: Constructed video path: " . $video_source);
        }
    } elseif (!empty($lesson['video_path'])) {
        // Local video file in separate video_path field - construct full path
        $video_source = '../uploads/videos/' . $lesson['video_path'];
        $lesson['video_url'] = $video_source;
        error_log("🔧 BACKEND: Using video_path field: " . $video_source);
    } else {
        $lesson['video_url'] = null;
        error_log("❌ BACKEND: No video URL or path found");
    }

    // Handle thumbnail path
    if (!empty($lesson['thumbnail']) && !preg_match('/^https?:\/\//', $lesson['thumbnail'])) {
        $lesson['thumbnail'] = '../uploads/thumbnails/' . $lesson['thumbnail'];
    }

    // Simple additional flags
    $lesson['has_assignment'] = false;
    $lesson['has_quiz'] = false;
    $lesson['has_materials'] = false;

    // Clear any stray output
    ob_clean();
    
    // Return clean JSON with proper lesson_data structure
    echo json_encode([
        'success' => true,
        'message' => 'Lesson data loaded successfully',
        'lesson' => $lesson_data,
        'course' => ['title' => 'Course'],
        'timestamp' => date('Y-m-d H:i:s')
    ]);

} catch (Exception $e) {
    // Clear any output and log error
    ob_clean();
    error_log("❌ CLEAN LESSON DATA ERROR: " . $e->getMessage());
    
    // Get POST data for debugging
    $input_data = json_decode(file_get_contents('php://input'), true) ?? [];
    
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => '🔴 BACKEND ERROR: ' . $e->getMessage(),
        'error_code' => 'LESSON_FETCH_ERROR',
        'detailed_error' => $e->getMessage(),
        'database_error' => isset($conn) ? $conn->error : 'No database connection',
        'database_errno' => isset($conn) ? $conn->errno : 'N/A',
        'debug_data' => [
            'post_course_id' => $input_data['course_id'] ?? 'NOT_IN_POST',
            'post_lesson_id' => $input_data['lesson_id'] ?? 'NOT_IN_POST',
            'get_course_id' => $_GET['course_id'] ?? 'NOT_IN_GET',
            'get_lesson_id' => $_GET['lesson_id'] ?? 'NOT_IN_GET',
            'user_id' => $_SESSION['user_id'] ?? 'NOT_SET',
            'user_role' => $_SESSION['user_role'] ?? $_SESSION['role'] ?? 'NOT_SET',
            'database_connected' => isset($conn) && $conn ? 'YES' : 'NO',
            'post_data_raw' => $input_data,
            'get_data_raw' => $_GET,
            'session_data' => [
                'user_id' => $_SESSION['user_id'] ?? 'MISSING',
                'user_role' => $_SESSION['user_role'] ?? $_SESSION['role'] ?? 'MISSING'
            ]
        ],
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}

// Final cleanup
if (isset($conn)) {
    $conn->close();
}

ob_end_flush();
?>
