<?php
// 🚀 FIXED LESSON DATA ENDPOINT - Working version
ob_start();
header('Content-Type: application/json');
header('Cache-Control: no-cache, must-revalidate');

try {
    // ✅ CRITICAL: Use EXACT same session pattern as course-detail.php
    require_once '../includes/session_start.php';
    require_once '../auth/functions.php';
    require_once '../includes/id_encryption.php';
    require_once '../config/db_connect.php';
    
    // Use the same authentication check as the main page
    requireRole('student');
    
    // Get user data from session (same way as main page)
    $user_id = $_SESSION['user_id'];
    
    // Handle course_id - WE KNOW DECRYPTION WORKS!
    $raw_course_id = $_GET['course_id'] ?? '';
    if (empty($raw_course_id)) {
        throw new Exception('No course_id provided');
    }
    
    if (is_numeric($raw_course_id)) {
        $course_id = (int)$raw_course_id;
    } else {
        // Decrypt it - we confirmed this works!
        $course_id = decryptId($raw_course_id);
        if ($course_id === false || !is_numeric($course_id)) {
            throw new Exception('Decryption failed for course_id');
        }
        $course_id = (int)$course_id;
    }
    
    // Handle lesson_id
    $lesson_id = (int)($_GET['lesson_id'] ?? 0);
    if (!$lesson_id) {
        throw new Exception('Invalid lesson_id');
    }
    
    if (!isset($conn) || !$conn) {
        throw new Exception('Database connection failed');
    }
    
    // Simple lesson query - using EXACT same structure as working main page
    $lesson_query = $conn->prepare("
        SELECT v.*,
               v.video_id as lesson_id,
               COALESCE(uvc.completion_percentage, 0) as watch_progress,
               CASE WHEN uvc.completed_at IS NOT NULL THEN 1 ELSE 0 END as completed,
               CASE WHEN uvc.watch_time > 0 THEN uvc.watch_time ELSE 0 END as last_position
        FROM videos v
        LEFT JOIN user_video_completions uvc ON v.video_id = uvc.video_id AND uvc.user_id = ?
        WHERE v.course_id = ? AND v.video_id = ? AND v.is_published = 1
        ORDER BY v.video_order ASC, v.created_at ASC
        LIMIT 1
    ");

    if (!$lesson_query) {
        throw new Exception('Query prepare failed: ' . $conn->error);
    }

    $lesson_query->bind_param("iii", $user_id, $course_id, $lesson_id);
    
    if (!$lesson_query->execute()) {
        throw new Exception('Query execution failed: ' . $lesson_query->error);
    }
    
    $lesson_result = $lesson_query->get_result();

    if ($lesson_result->num_rows === 0) {
        throw new Exception('Lesson not found: course_id=' . $course_id . ', lesson_id=' . $lesson_id);
    }

    $lesson = $lesson_result->fetch_assoc();
    $lesson_query->close();

    // Process video data EXACTLY like the working main page
    if (!empty($lesson['external_url'])) {
        // URL video - convert to proper embed format (same as main page)
        $external_url = $lesson['external_url'];
        
        // Convert YouTube URLs to embed format
        if (strpos($external_url, 'youtube.com/watch') !== false) {
            parse_str(parse_url($external_url, PHP_URL_QUERY), $query);
            if (isset($query['v'])) {
                $lesson['video_url'] = 'https://www.youtube.com/embed/' . $query['v'];
            } else {
                $lesson['video_url'] = $external_url;
            }
        } elseif (strpos($external_url, 'youtu.be') !== false) {
            $video_id_from_url = basename(parse_url($external_url, PHP_URL_PATH));
            $lesson['video_url'] = 'https://www.youtube.com/embed/' . $video_id_from_url;
        } elseif (strpos($external_url, 'youtube.com/embed') !== false) {
            $lesson['video_url'] = $external_url;
        } else {
            $lesson['video_url'] = $external_url;
        }
        
        $lesson['is_external'] = true;
    } elseif (!empty($lesson['video_url'])) {
        // File upload - fix video URL path for web access (same as main page)
        if (strpos($lesson['video_url'], 'http') === 0) {
            $lesson['video_url'] = $lesson['video_url'];
        } else {
            $lesson['video_url'] = '/' . ltrim($lesson['video_url'], './');
        }
        $lesson['is_external'] = false;
    } else {
        throw new Exception('No video source available');
    }

    // Ensure proper data types
    $lesson['lesson_id'] = (int)$lesson['lesson_id'];
    $lesson['video_id'] = (int)$lesson['video_id'];
    $lesson['completed'] = (bool)$lesson['completed'];
    $lesson['watch_progress'] = (float)$lesson['watch_progress'];

    // Clear any output buffer
    ob_clean();
    
    // Return successful response
    echo json_encode([
        'success' => true,
        'lesson' => $lesson,
        'debug_info' => [
            'decrypted_course_id' => $course_id,
            'lesson_id' => $lesson_id,
            'user_id' => $user_id,
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ]);

} catch (Exception $e) {
    // Clear any output buffer
    ob_clean();
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'error_code' => 'LESSON_FETCH_ERROR',
        'debug_data' => [
            'raw_course_id' => $_GET['course_id'] ?? 'not_set',
            'raw_lesson_id' => $_GET['lesson_id'] ?? 'not_set',
            'user_id' => $_SESSION['user_id'] ?? 'not_set',
            'user_role' => $_SESSION['role'] ?? $_SESSION['user_role'] ?? 'not_set'
        ]
    ]);
} finally {
    ob_end_flush();
}
?>
