/**
 * 🎓 Accreditation PayFast Payment Integration
 * Handles SITA accreditation payment processing through PayFast
 */

// Wait for DOM to be ready
document.addEventListener('DOMContentLoaded', function() {
    initializeAccreditationPayment();
});

function initializeAccreditationPayment() {
    const accreditationForm = document.querySelector('#accreditationPaymentForm');
    if (!accreditationForm) {
        console.log('🎓 Accreditation payment form not found on this page');
        return;
    }
    
    console.log('🎓 Accreditation PayFast integration initialized');
    
    accreditationForm.addEventListener('submit', function(e) {
        e.preventDefault();
        handleAccreditationSubmission(this);
    });
}

function handleAccreditationSubmission(form) {
    // Show loading state
    const submitBtn = form.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    submitBtn.innerHTML = '<i class="spinner-border spinner-border-sm me-2"></i>Processing...';
    submitBtn.disabled = true;
    
    // Prepare form data
    const formData = new FormData(form);
    
    // Submit accreditation request
    fetch('api/submit_accreditation_request.php', {
        method: 'POST',
        body: formData
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Network response was not ok');
        }
        return response.json();
    })
    .then(data => {
        console.log('🎓 Accreditation response:', data);
        
        if (data.success) {
            if (data.requires_payment && data.payfast_data) {
                // Show payment confirmation modal
                showPaymentModal(data);
            } else {
                // Handle success without payment (shouldn't happen for accreditation)
                showSuccessMessage(data.message);
                setTimeout(() => location.reload(), 2000);
            }
        } else {
            throw new Error(data.message || 'Failed to submit request');
        }
    })
    .catch(error => {
        console.error('🎓 Accreditation request error:', error);
        showErrorMessage(error.message || 'Failed to submit accreditation request');
    })
    .finally(() => {
        // Reset button state
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    });
}

function showPaymentModal(data) {
    // Remove existing modal if any
    const existingModal = document.getElementById('paymentModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    // Create payment modal
    const modal = document.createElement('div');
    modal.className = 'modal fade show';
    modal.style.display = 'block';
    modal.style.backgroundColor = 'rgba(0,0,0,0.8)';
    modal.style.zIndex = '9999';
    modal.id = 'paymentModal';
    
    modal.innerHTML = `
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title">
                        <i class="bx bx-credit-card me-2"></i>
                        Complete Payment
                    </h5>
                    <button type="button" class="btn-close btn-close-white" onclick="closePaymentModal()"></button>
                </div>
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="payment-summary p-4 bg-light rounded">
                                <h6 class="fw-bold mb-3">Payment Summary</h6>
                                <div class="d-flex justify-content-between mb-2">
                                    <span>SITA Accreditation Certificate</span>
                                    <span class="fw-bold">R${data.payment_amount}</span>
                                </div>
                                <hr>
                                <div class="d-flex justify-content-between mb-3">
                                    <span class="fw-bold">Total Amount:</span>
                                    <span class="fw-bold text-primary fs-5">R${data.payment_amount}</span>
                                </div>
                                <small class="text-muted">
                                    <i class="bx bx-info-circle me-1"></i>
                                    Secure payment processed by PayFast
                                </small>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="payment-info">
                                <h6 class="fw-bold mb-3">What's Included</h6>
                                <ul class="list-unstyled">
                                    <li class="mb-2">
                                        <i class="bx bx-check text-success me-2"></i>
                                        Official SITA-accredited certificate
                                    </li>
                                    <li class="mb-2">
                                        <i class="bx bx-check text-success me-2"></i>
                                        Professional skills assessment
                                    </li>
                                    <li class="mb-2">
                                        <i class="bx bx-check text-success me-2"></i>
                                        Industry recognition
                                    </li>
                                    <li class="mb-2">
                                        <i class="bx bx-check text-success me-2"></i>
                                        Career advancement support
                                    </li>
                                </ul>
                                
                                <div class="alert alert-info mt-3">
                                    <small>
                                        <i class="bx bx-time me-1"></i>
                                        Processing time: Up to 5 months from payment
                                    </small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-outline-secondary" onclick="closePaymentModal()">
                        <i class="bx bx-x me-2"></i>Cancel
                    </button>
                    <button type="button" class="btn btn-success" onclick="proceedToPayment()">
                        <i class="bx bx-credit-card me-2"></i>
                        Pay R${data.payment_amount} with PayFast
                    </button>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Store payment data globally for the payment function
    window.payfastPaymentData = data;
    
    console.log('🎓 Payment modal displayed for amount: R' + data.payment_amount);
}

function closePaymentModal() {
    const modal = document.getElementById('paymentModal');
    if (modal) {
        modal.remove();
    }
    delete window.payfastPaymentData;
    console.log('🎓 Payment modal closed');
}

function proceedToPayment() {
    if (!window.payfastPaymentData) {
        showErrorMessage('Payment data not found. Please try again.');
        return;
    }
    
    const data = window.payfastPaymentData;
    
    console.log('🎓 Proceeding to PayFast with data:', {
        amount: data.payment_amount,
        reference: data.payfast_data.m_payment_id,
        url: data.payfast_url
    });
    
    // Create PayFast form and submit
    const payfastForm = document.createElement('form');
    payfastForm.method = 'POST';
    payfastForm.action = data.payfast_url;
    payfastForm.style.display = 'none';
    
    // Add all PayFast parameters as hidden inputs
    Object.entries(data.payfast_data).forEach(([key, value]) => {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = key;
        input.value = value;
        payfastForm.appendChild(input);
    });
    
    document.body.appendChild(payfastForm);
    
    // Show loading message
    const modal = document.getElementById('paymentModal');
    if (modal) {
        modal.querySelector('.modal-body').innerHTML = `
            <div class="text-center py-5">
                <div class="spinner-border text-primary mb-3" role="status"></div>
                <h5>Redirecting to PayFast...</h5>
                <p class="text-muted">Please wait while we redirect you to complete your payment.</p>
                <div class="alert alert-warning mt-3">
                    <small><i class="bx bx-info-circle me-1"></i>
                    If the redirect doesn't happen automatically, please click the PayFast button again.</small>
                </div>
            </div>
        `;
    }
    
    // Submit form after brief delay
    setTimeout(() => {
        console.log('🎓 Submitting PayFast form to:', data.payfast_url);
        payfastForm.submit();
    }, 1500);
}

function showSuccessMessage(message) {
    // Remove existing alerts
    removeExistingAlerts();
    
    const alert = document.createElement('div');
    alert.className = 'alert alert-success alert-dismissible fade show position-fixed';
    alert.style.top = '20px';
    alert.style.right = '20px';
    alert.style.zIndex = '9999';
    alert.style.maxWidth = '400px';
    alert.innerHTML = `
        <i class="bx bx-check-circle me-2"></i>
        ${message}
        <button type="button" class="btn-close" onclick="this.parentElement.remove()"></button>
    `;
    document.body.appendChild(alert);
    
    // Auto-remove after 5 seconds
    setTimeout(() => {
        if (alert.parentElement) {
            alert.remove();
        }
    }, 5000);
}

function showErrorMessage(message) {
    // Remove existing alerts
    removeExistingAlerts();
    
    const alert = document.createElement('div');
    alert.className = 'alert alert-danger alert-dismissible fade show position-fixed';
    alert.style.top = '20px';
    alert.style.right = '20px';
    alert.style.zIndex = '9999';
    alert.style.maxWidth = '400px';
    alert.innerHTML = `
        <i class="bx bx-x-circle me-2"></i>
        ${message}
        <button type="button" class="btn-close" onclick="this.parentElement.remove()"></button>
    `;
    document.body.appendChild(alert);
    
    // Auto-remove after 8 seconds for errors
    setTimeout(() => {
        if (alert.parentElement) {
            alert.remove();
        }
    }, 8000);
}

function removeExistingAlerts() {
    const existingAlerts = document.querySelectorAll('.alert.position-fixed');
    existingAlerts.forEach(alert => alert.remove());
}

// Make functions globally available for onclick handlers
window.closePaymentModal = closePaymentModal;
window.proceedToPayment = proceedToPayment;

console.log('🎓 Accreditation PayFast integration script loaded');
