<?php
require_once '../../auth/functions.php';
require_once '../../config/fast_email_service.php';

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Start output buffering to catch any unwanted output
ob_start();

try {
    // Apply advanced session fix first - before ANY output (same as working API files)
    require_once '../../includes/advanced_session.php';
    require_once '../../config/db_connect.php';

    // Bridge database connections - Email service needs PDO, but API uses MySQLi
    if (!isset($pdo)) {
        try {
            // Create PDO connection using the same credentials as MySQLi
            $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
            error_log("PDO connection created successfully for assignment submission email service");
        } catch (Exception $e) {
            error_log("Failed to create PDO connection for assignment submission email service: " . $e->getMessage());
            $pdo = null;
        }
    }

    /**
     * Send document submission completion email to student
     */
    function sendDocumentSubmissionEmail($request_id) {
        global $conn;
        
        try {
            // Get request, student and course details  
            $stmt = $conn->prepare("SELECT ar.request_id, ar.payment_amount, ar.assignment_document_path, ar.updated_at, ar.course_id, u.first_name, u.last_name, u.email, u.user_id, c.title as course_title, c.description FROM accreditation_requests ar JOIN users u ON ar.student_id = u.user_id JOIN courses c ON ar.course_id = c.course_id WHERE ar.request_id = ?");
            
            if (!$stmt) {
                error_log("Failed to prepare document submission email query: " . $conn->error);
                return false;
            }
            
            $stmt->bind_param("i", $request_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $details = $result->fetch_assoc();
            
            if (!$details) {
                error_log("Accreditation request not found for document submission email: $request_id");
                return false;
            }
            
            $student_name = $details['first_name'] . ' ' . $details['last_name'];
            $student_email = $details['email'];
            $course_title = $details['course_title'];
            $course_id = $details['course_id'];
            $request_id_formatted = str_pad($request_id, 6, '0', STR_PAD_LEFT);
            $submission_date = date('M j, Y', strtotime($details['updated_at']));
            
            // Create simple email content - avoid spam filters
            $subject = "Documents Submitted Successfully - Request #$request_id_formatted";
            $message = "Dear " . $student_name . ",

Your SITA accreditation documents have been successfully submitted.

Course: " . $course_title . "
Request ID: #" . $request_id_formatted . "
Submitted: " . $submission_date . "

IMPORTANT: Processing time is 3-5 months.

Your documents are now under review by SITA's accreditation board.

Next steps:
1. Document verification (1-2 weeks)
2. SITA board review (2-4 months) 
3. Final certification

You will receive updates via email.

Best regards,
Panadite Academy Team";
            
            // Try direct email sending - EXACT COPY FROM ADMIN COURSES.PHP LINES 81-94
            if (function_exists('sendEmailNow')) {
                $result = sendEmailNow($student_email, $subject, $message);
                if ($result) {
                    return true;
                }
            }
            
            // Fallback to native PHP mail - EXACT COPY FROM ADMIN COURSES.PHP
            $headers = "MIME-Version: 1.0" . "\r\n";
            $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
            $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
            
            return mail($student_email, $subject, $message, $headers);
            
        } catch (Exception $e) {
            error_log("Error sending document submission email: " . $e->getMessage());
            return false;
        }
    }

    // Set the missing session variables for compatibility
    if (!isset($_SESSION['logged_in']) && isset($_SESSION['user_id'])) {
        $_SESSION['logged_in'] = true;
    }
    if (!isset($_SESSION['user_role']) && isset($_SESSION['role'])) {
        $_SESSION['user_role'] = $_SESSION['role'];
    }

    // Direct check for API authentication - don't use requireRole which redirects (same as working APIs)
    if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'student') {
        ob_clean();
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Authentication required']);
    }
} catch (Exception $e) {
    ob_clean();
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'Initialization error: ' . $e->getMessage(),
        'debug' => ['error' => $e->getMessage(), 'file' => $e->getFile(), 'line' => $e->getLine()]
    ]);
    exit();
}

$student_id = $_SESSION['user_id'];
$request_id = isset($_POST['request_id']) ? intval($_POST['request_id']) : 0;

if (!$request_id) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid request ID']);
    exit();
}

try {
    // Verify ownership of accreditation request
    $verify_query = "SELECT * FROM accreditation_requests WHERE request_id = ? AND student_id = ? AND status = 'assignment_created'";
    $verify_stmt = $conn->prepare($verify_query);
    $verify_stmt->bind_param("ii", $request_id, $student_id);
    $verify_stmt->execute();
    $verify_result = $verify_stmt->get_result();
    
    if ($verify_result->num_rows === 0) {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'Access denied or assignment not found']);
        exit();
    }
    
    // Create upload directory
    $upload_dir = "../../uploads/accreditation/request_{$request_id}/";
    if (!is_dir($upload_dir)) {
        if (!mkdir($upload_dir, 0755, true)) {
            throw new Exception('Failed to create upload directory');
        }
    }
    
    $uploaded_files = [];
    $allowed_types = [
        'cv' => ['pdf', 'doc', 'docx'],
        'qualifications' => ['pdf', 'jpg', 'jpeg', 'png'],
        'id_document' => ['pdf', 'jpg', 'jpeg', 'png'],
        'assignment_work' => ['pdf', 'doc', 'docx', 'zip']
    ];
    
    // Map form field names to database enum values
    $doc_type_mapping = [
        'cv' => 'cv',
        'qualifications' => 'qualifications', 
        'id_document' => 'id_document',
        'assignment_work' => 'assignment'  // Map to 'assignment' enum value
    ];
    
    // Debug: Log what we received
    error_log("API Debug - Files received: " . print_r($_FILES, true));
    error_log("API Debug - Request ID: $request_id, Student ID: $student_id");
    error_log("API Debug - Upload directory: $upload_dir");

    // Process uploaded files
    if (isset($_FILES['documents']) && is_array($_FILES['documents']['name'])) {
        error_log("API Debug - Processing documents array with " . count($_FILES['documents']['name']) . " files");
        foreach ($_FILES['documents']['name'] as $doc_type => $filename) {
            error_log("API Debug - Processing document type: $doc_type, filename: $filename");
            if ($_FILES['documents']['error'][$doc_type] === UPLOAD_ERR_OK) {
                error_log("API Debug - File upload OK for: $doc_type");
                $tmp_name = $_FILES['documents']['tmp_name'][$doc_type];
                $file_size = $_FILES['documents']['size'][$doc_type];
                $file_ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
                
                // Validate file type
                if (!isset($allowed_types[$doc_type]) || !in_array($file_ext, $allowed_types[$doc_type])) {
                    throw new Exception("Invalid file type for {$doc_type}: {$file_ext}");
                }
                
                // Validate file size (10MB limit)
                if ($file_size > 10485760) {
                    throw new Exception("File too large for {$doc_type}: " . round($file_size / 1048576, 2) . "MB");
                }
                
                // Generate safe filename
                $safe_filename = $doc_type . '_' . time() . '.' . $file_ext;
                $file_path = $upload_dir . $safe_filename;
                
                // Move uploaded file
                if (!move_uploaded_file($tmp_name, $file_path)) {
                    throw new Exception("Failed to upload {$doc_type}");
                }
                
                // Map document type to database enum value
                $db_doc_type = $doc_type_mapping[$doc_type] ?? $doc_type;
                
                // Check if document already exists
                $check_query = "SELECT document_id FROM accreditation_documents WHERE request_id = ? AND student_id = ? AND document_type = ?";
                $check_stmt = $conn->prepare($check_query);
                if (!$check_stmt) {
                    error_log("API Error - Failed to prepare check query: " . $conn->error);
                    throw new Exception("Database error: " . $conn->error);
                }
                $check_stmt->bind_param("iis", $request_id, $student_id, $db_doc_type);
                if (!$check_stmt->execute()) {
                    error_log("API Error - Failed to execute check query: " . $check_stmt->error);
                    throw new Exception("Database query failed: " . $check_stmt->error);
                }
                $existing = $check_stmt->get_result()->fetch_assoc();
                
                // Get file MIME type for database
                $mime_type = mime_content_type($file_path) ?: 'application/octet-stream';
                
                if ($existing) {
                    // Update existing document
                    $update_query = "UPDATE accreditation_documents 
                                   SET document_name = ?, file_path = ?, file_size = ?, mime_type = ?, 
                                       upload_date = NOW(), verification_status = 'pending' 
                                   WHERE document_id = ?";
                    $update_stmt = $conn->prepare($update_query);
                    if (!$update_stmt) {
                        throw new Exception("Failed to prepare update statement: " . $conn->error);
                    }
                    $update_stmt->bind_param("ssssi", $filename, $file_path, $file_size, $mime_type, $existing['document_id']);
                    if (!$update_stmt->execute()) {
                        throw new Exception("Failed to update document: " . $update_stmt->error);
                    }
                    error_log("API Debug - Updated existing document: $doc_type (ID: {$existing['document_id']})");
                } else {
                    // Insert new document with all required fields
                    $insert_query = "INSERT INTO accreditation_documents 
                                   (request_id, student_id, document_type, document_name, file_path, file_size, mime_type, verification_status) 
                                   VALUES (?, ?, ?, ?, ?, ?, ?, 'pending')";
                    $insert_stmt = $conn->prepare($insert_query);
                    if (!$insert_stmt) {
                        throw new Exception("Failed to prepare insert statement: " . $conn->error);
                    }
                    $insert_stmt->bind_param("iisssss", $request_id, $student_id, $db_doc_type, $filename, $file_path, $file_size, $mime_type);
                    if (!$insert_stmt->execute()) {
                        throw new Exception("Failed to insert document: " . $insert_stmt->error);
                    }
                    error_log("API Debug - Inserted new document: $doc_type -> $db_doc_type, insert_id: " . $conn->insert_id);
                }
                
                $uploaded_files[] = $doc_type;
            }
        }
    }
    
    if (empty($uploaded_files)) {
        error_log("API Debug - No files were uploaded. FILES array: " . print_r($_FILES, true));
        echo json_encode(['success' => false, 'message' => 'No files were uploaded']);
        exit();
    }
    
    error_log("API Debug - Successfully processed " . count($uploaded_files) . " files: " . implode(", ", $uploaded_files));
    
    // Check if all required documents are now submitted (using database enum values)
    $required_db_docs = ['cv', 'qualifications', 'id_document', 'assignment'];
    $submitted_count_query = "SELECT COUNT(DISTINCT document_type) as count FROM accreditation_documents WHERE request_id = ? AND student_id = ? AND document_type IN ('cv', 'qualifications', 'id_document', 'assignment')";
    $count_stmt = $conn->prepare($submitted_count_query);
    $count_stmt->bind_param("ii", $request_id, $student_id);
    $count_stmt->execute();
    $count_result = $count_stmt->get_result()->fetch_assoc();
    
    error_log("API Debug - Document count check: {$count_result['count']}/4 required documents submitted");
    
    // Update accreditation request status if all documents submitted
    if ($count_result['count'] >= 4) {
        $status_update_query = "UPDATE accreditation_requests SET status = 'documents_submitted', documents_submitted_date = NOW() WHERE request_id = ?";
        $status_stmt = $conn->prepare($status_update_query);
        if (!$status_stmt) {
            error_log("API Error - Failed to prepare status update: " . $conn->error);
        } else {
            $status_stmt->bind_param("i", $request_id);
            if ($status_stmt->execute()) {
                error_log("API Debug - Successfully updated status to 'documents_submitted' for request $request_id");
                
                // Send document submission completion email
                try {
                    $email_sent = sendDocumentSubmissionEmail($request_id);
                    if ($email_sent) {
                        error_log("Document submission email sent successfully for request ID: " . $request_id);
                    } else {
                        error_log("Failed to send document submission email for request ID: " . $request_id);
                    }
                } catch (Exception $e) {
                    error_log("Error sending document submission email for request " . $request_id . ": " . $e->getMessage());
                }
                
            } else {
                error_log("API Error - Failed to update status: " . $status_stmt->error);
            }
        }
        
        $message = 'All documents submitted successfully! Your submission is now under review for SITA accreditation (3-5 months processing time).';
    } else {
        $message = 'Documents uploaded successfully. Please submit remaining required documents (' . $count_result['count'] . '/4 completed).';
    }
    
    echo json_encode([
        'success' => true, 
        'message' => $message,
        'uploaded_files' => $uploaded_files,
        'total_submitted' => $count_result['count']
    ]);
    
} catch (Exception $e) {
    error_log("Document submission error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>
