<?php
/**
 * Mark Video Completed API Endpoint
 * Handles AJAX requests to mark a video as completed
 */

// 🛡️ NUCLEAR-LEVEL ERROR SUPPRESSION
error_reporting(0);
ini_set('display_errors', 0);
ini_set('log_errors', 0);
ini_set('html_errors', 0);
ini_set('ignore_repeated_errors', 1);
ini_set('ignore_repeated_source', 1);

// 🛡️ CAPTURE ALL OUTPUT BEFORE STARTING
ob_start();
ob_implicit_flush(false);

// 🛡️ SUPPRESS ALL WARNINGS AND NOTICES
set_error_handler(function($severity, $message, $file, $line) {
    // Silently ignore all errors
    return true;
});

// 🛡️ CATCH FATAL ERRORS AND SHOW ACTUAL ERROR DETAILS
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error && $error['type'] === E_ERROR) {
        // Fatal error occurred - force clean JSON output with REAL error details
        ob_clean();
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false, 
            'message' => 'FATAL ERROR: ' . $error['message'],
            'error_file' => $error['file'],
            'error_line' => $error['line'],
            'error_type' => $error['type'],
            'fallback' => true
        ]);
        exit;
    }
});

// 🛡️ CLEAN SLATE - DISCARD ANY PREVIOUS OUTPUT
if (ob_get_level()) {
    ob_end_clean();
}
ob_start();

// 🔐 START SESSION TO ACCESS USER AUTHENTICATION  
require_once '../includes/session_start.php';

// Include database connection with maximum suppression
try {
    @require_once '../config/db_connect.php';
} catch (Exception $dbError) {
    ob_clean();
    ob_start();
    echo json_encode(['success' => false, 'message' => 'Database connection failed']);
    exit;
}

// 🛡️ FINAL OUTPUT BUFFER FLUSH AND CLEAN
while (ob_get_level()) {
    ob_end_clean();
}
ob_start();

// Set response header to JSON
header('Content-Type: application/json');

// Check if request is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

// Use the same session system as the main application
require_once '../includes/session_start.php';

// Check if user is authenticated and is a student
$user_id = null;
$is_authenticated = false;

// Try to get user ID from session
if (isset($_SESSION['user_id']) && !empty($_SESSION['user_id'])) {
    $user_id = $_SESSION['user_id'];
}

// Check multiple possible role session variables
$role_check = false;
if ((isset($_SESSION['role']) && $_SESSION['role'] === 'student') ||
    (isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'student') ||
    (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'student')) {
    $role_check = true;
}

// Set authenticated if both user_id and role check pass
if ($user_id && $role_check) {
    $is_authenticated = true;
}

// If not authenticated, return detailed debug info
if (!$is_authenticated) {
    @header('Content-Type: application/json');
    echo json_encode([
        'success' => false, 
        'message' => 'User not authenticated',
        'debug' => [
            'session_exists' => !empty($_SESSION),
            'all_session_keys' => array_keys($_SESSION ?? []),
            'user_id' => $_SESSION['user_id'] ?? 'NOT SET',
            'role' => $_SESSION['role'] ?? 'NOT SET',
            'user_role' => $_SESSION['user_role'] ?? 'NOT SET', 
            'user_type' => $_SESSION['user_type'] ?? 'NOT SET',
            'role_check_passed' => $role_check,
            'user_id_check_passed' => !empty($user_id),
            'final_authenticated' => $is_authenticated
        ]
    ]);
    exit;
}

// Get video ID and course ID from POST data
if (!isset($_POST['video_id']) || !isset($_POST['course_id'])) {
    echo json_encode(['success' => false, 'message' => 'Missing required parameters']);
    exit;
}

$video_id = $_POST['video_id'];
$course_id = $_POST['course_id'];
$auto_complete = isset($_POST['auto_complete']) ? (int)$_POST['auto_complete'] : 0;

try {
    // Check if user is enrolled in the course
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM enrollments WHERE user_id = ? AND course_id = ? LIMIT 1");
    if (!$stmt) {
        echo json_encode(['success' => false, 'message' => 'Database prepare error - enrollments check']);
        exit;
    }
    $stmt->execute([$user_id, $course_id]);
    $enrollment_count = $stmt->fetchColumn();
    
    if ($enrollment_count == 0) {
        ob_clean();
        echo json_encode(['success' => false, 'message' => 'User not enrolled in this course']);
        exit;
    }
    
    // Check if video belongs to course  
    $stmt = $pdo->prepare("SELECT video_id, title FROM videos WHERE video_id = ? AND course_id = ? LIMIT 1");
    if (!$stmt) {
        echo json_encode(['success' => false, 'message' => 'Database prepare error - video check']);
        exit;
    }
    $stmt->execute([$video_id, $course_id]);
    $video = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$video) {
        ob_clean();
        echo json_encode(['success' => false, 'message' => 'Video does not belong to this course']);
        exit;
    }
    
    // Begin transaction
    @$pdo->beginTransaction();
    
    // 🛡️ SET PDO ERROR MODE TO SILENT
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_SILENT);
    
    // 🛡️ SAFELY CHECK AND CREATE TABLE WITH ERROR HANDLING
    try {
        $stmt = @$pdo->query("SHOW TABLES LIKE 'user_video_completions'");
        if ($stmt && $stmt->rowCount() == 0) {
            // Create the table with comprehensive error handling
            $createTableQuery = "CREATE TABLE IF NOT EXISTS `user_video_completions` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `user_id` int(11) NOT NULL,
                `video_id` int(11) NOT NULL,
                `completed_at` datetime NOT NULL,
                `created_at` datetime NOT NULL,
                `updated_at` datetime NOT NULL,
                PRIMARY KEY (`id`),
                UNIQUE KEY `user_video_unique` (`user_id`, `video_id`),
                KEY `user_id` (`user_id`),
                KEY `video_id` (`video_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
            @$pdo->exec($createTableQuery);
        }
    } catch (Exception $tableError) {
        // 🛡️ TABLE CREATION FAILED - RETURN CLEAN JSON ERROR
        ob_clean();
        ob_start();
        echo json_encode([
            'success' => false, 
            'message' => 'Database table error',
            'fallback' => true
        ]);
        exit;
    }
    
    // Check if completion record already exists
    $stmt = $pdo->prepare("SELECT id, completed_at FROM user_video_completions 
                          WHERE user_id = ? AND video_id = ? LIMIT 1");
    $existing = false;
    if ($stmt) {
        $stmt->execute([$user_id, $video_id]);
        $existing = $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    $now = date('Y-m-d H:i:s');
    
    if ($existing) {
        // Update existing record
        $stmt = $pdo->prepare("UPDATE user_video_completions 
                              SET completed_at = ?, updated_at = ? 
                              WHERE id = ?");
        if ($stmt) {
            $stmt->execute([$now, $now, $existing['id']]);
        }
    } else {
        // Create new completion record
        $stmt = $pdo->prepare("INSERT INTO user_video_completions 
                              (user_id, video_id, completion_percentage, watch_time, completed_at, updated_at) 
                              VALUES (?, ?, ?, ?, ?, ?)");
        if ($stmt) {
            $stmt->execute([$user_id, $video_id, 100.00, 0.00, $now, $now]);
        }
    }
    
    // Check if user_video_completions table exists for progress calculation
    $stmt = @$pdo->query("SHOW TABLES LIKE 'user_video_completions'");
    $table_exists = $stmt && $stmt->rowCount() > 0;
    
    // Calculate progress percentage - simplified without is_published column
    if ($table_exists) {
        $stmt = $pdo->prepare("SELECT 
                              (SELECT COUNT(*) FROM videos WHERE course_id = ?) AS total_videos,
                              (SELECT COUNT(*) FROM user_video_completions uvc 
                               JOIN videos v ON uvc.video_id = v.video_id 
                               WHERE uvc.user_id = ? AND v.course_id = ?) AS completed_videos");
        $stmt->execute([$course_id, $user_id, $course_id]);
    } else {
        // If table doesn't exist, set completed_videos to 0
        $stmt = $pdo->prepare("SELECT 
                              COUNT(*) AS total_videos,
                              0 AS completed_videos
                              FROM videos 
                              WHERE course_id = ?");
        $stmt->execute([$course_id]);
    }
    $progress = @$stmt->fetch(PDO::FETCH_ASSOC);
    
    $total_videos = (int)$progress['total_videos'];
    $completed_videos = (int)$progress['completed_videos'];
    $progress_percent = $total_videos > 0 ? round(($completed_videos / $total_videos) * 100) : 0;
    
    // Skip updating progress in enrollments table if column doesn't exist
    try {
        $stmt = $pdo->prepare("UPDATE enrollments SET updated_at = ? WHERE user_id = ? AND course_id = ?");
        if ($stmt) {
            $stmt->execute([$now, $user_id, $course_id]);
        }
    } catch (Exception $e) {
        // Ignore if columns don't exist
    }
    
    // Check if there's a next video - simplified without is_published
    $stmt = $pdo->prepare("SELECT v.video_id FROM videos v
                          WHERE v.course_id = ?
                          AND v.video_id > ?
                          ORDER BY v.video_id ASC
                          LIMIT 1");
    $next_video = false;
    $has_next_video = false;
    $next_video_id = null;
    
    if ($stmt) {
        $stmt->execute([$course_id, $video_id]);
        $next_video = $stmt->fetch(PDO::FETCH_ASSOC);
        $has_next_video = ($next_video !== false);
        $next_video_id = $has_next_video ? $next_video['video_id'] : null;
    }
    
    // Commit transaction
    @$pdo->commit();
    
    // 🛡️ FINAL GUARANTEED CLEAN JSON OUTPUT
    ob_clean();
    ob_start();
    
    echo json_encode([
        'success' => true,
        'message' => 'Video marked as completed',
        'video_id' => $video_id,
        'video_title' => $video['title'],
        'completed_at' => $now,
        'progress_percent' => $progress_percent,
        'has_next_video' => $has_next_video,
        'next_video_id' => $next_video_id,
        'total_videos' => $total_videos,
        'completed_videos' => $completed_videos
    ]);
    
    ob_end_flush();
    
} catch (Exception $e) {
    // Rollback transaction on error
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    // 🛡️ ENSURE CLEAN JSON ERROR RESPONSE
    ob_clean();
    echo json_encode([
        'success' => false, 
        'message' => 'Database error: ' . $e->getMessage(),
        'error_file' => $e->getFile(),
        'error_line' => $e->getLine()
    ]);
}

// 🛡️ ENSURE NO TRAILING OUTPUT
ob_end_flush();
?>
