<?php
/**
 * 🏆 MARK VIDEO AS COMPLETED ENDPOINT
 * Marks a video as 100% completed in user_video_completions table
 */

require_once '../includes/config.php';
require_once '../includes/session_start.php';

header('Content-Type: application/json');

try {
    // Start session if not already started
    if (session_status() == PHP_SESSION_NONE) {
        session_start();
    }
    
    // Debug session data
    error_log("DEBUG - Session data: " . json_encode($_SESSION));
    error_log("DEBUG - POST data: " . json_encode($_POST));
    
    // Check if user is logged in
    if (!isset($_SESSION['user_id'])) {
        echo json_encode([
            'success' => false, 
            'message' => 'User not logged in',
            'debug' => [
                'session_status' => session_status(),
                'session_id' => session_id(),
                'session_data' => $_SESSION ?? 'No session data'
            ]
        ]);
        exit;
    }
    
    $user_id = $_SESSION['user_id'];
    $video_id = (int)$_POST['video_id'];
    $completion_percentage = 100.00;
    $watch_time = (float)($_POST['watch_time'] ?? 0);
    $completed_at = $_POST['completed_at'] ?? date('Y-m-d H:i:s');
    
    // Validate input
    if (!$video_id) {
        echo json_encode(['success' => false, 'message' => 'Invalid video ID']);
        exit;
    }
    
    // Insert or update completion record
    $stmt = $conn->prepare("
        INSERT INTO user_video_completions 
        (user_id, video_id, completion_percentage, watch_time, completed_at, updated_at) 
        VALUES (?, ?, ?, ?, ?, NOW())
        ON DUPLICATE KEY UPDATE 
        completion_percentage = 100.00,
        watch_time = GREATEST(watch_time, VALUES(watch_time)),
        completed_at = COALESCE(completed_at, VALUES(completed_at)),
        updated_at = NOW()
    ");
    
    $stmt->bind_param("iidds", $user_id, $video_id, $completion_percentage, $watch_time, $completed_at);
    
    if ($stmt->execute()) {
        echo json_encode([
            'success' => true,
            'message' => 'Video marked as completed successfully',
            'video_id' => $video_id,
            'completion_percentage' => $completion_percentage,
            'completed_at' => $completed_at
        ]);
    } else {
        throw new Exception('Failed to mark as completed: ' . $stmt->error);
    }
    
    $stmt->close();
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
