<?php
// Session is started in header.php
require_once '../auth/functions.php';
require_once '../config/db_connect.php';

// Require student role authentication - no exceptions
requireRole('student');

$userId = $_SESSION['user_id'];

// Check if this is an AJAX request
$isAjaxRequest = isset($_SERVER['HTTP_X_REQUESTED_WITH']) && 
                 strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';

// Handle AJAX requests for loading conversations
if ($isAjaxRequest && isset($_GET['ajax']) && $_GET['ajax'] === 'load_conversation') {
    $contactId = isset($_GET['contact']) ? intval($_GET['contact']) : 0;
    
    if ($contactId > 0) {
        // Get contact information
        $contactQuery = "SELECT u.user_id, u.first_name, u.last_name, u.profile_picture, c.title as course_name
                        FROM users u
                        JOIN teachers t ON u.user_id = t.user_id
                        JOIN courses c ON t.teacher_id = c.teacher_id
                        JOIN enrollments e ON c.course_id = e.course_id
                        WHERE u.user_id = ? AND e.student_id = ?
                        LIMIT 1";
        
        $stmt = $conn->prepare($contactQuery);
        $stmt->bind_param("ii", $contactId, $userId);
        $stmt->execute();
        $contact = $stmt->get_result()->fetch_assoc();
        
        if ($contact) {
            // Get messages for this conversation
            $messagesQuery = "SELECT m.*, u.first_name, u.last_name
                             FROM messages m
                             JOIN users u ON m.sender_id = u.user_id
                             WHERE (m.sender_id = ? AND m.recipient_id = ?) 
                                OR (m.sender_id = ? AND m.recipient_id = ?)
                             ORDER BY m.sent_at ASC";
            
            $stmt = $conn->prepare($messagesQuery);
            $stmt->bind_param("iiii", $userId, $contactId, $contactId, $userId);
            $stmt->execute();
            $messages = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            
            // Mark messages as read
            $markReadQuery = "UPDATE messages SET is_read = 1 
                             WHERE sender_id = ? AND recipient_id = ? AND is_read = 0";
            $stmt = $conn->prepare($markReadQuery);
            $stmt->bind_param("ii", $contactId, $userId);
            $stmt->execute();
            
            // Output the conversation HTML
            ?>
            <div class="message-header">
                <div class="d-flex align-items-center">
                    <button class="btn btn-sm btn-outline-secondary d-md-none me-3" onclick="showContactsList()">
                        <i class="fas fa-arrow-left"></i>
                    </button>
                    <div class="contact-avatar me-3">
                        <?php if ($contact['profile_picture']): ?>
                            <img src="<?= htmlspecialchars($contact['profile_picture']) ?>" 
                                 alt="<?= htmlspecialchars($contact['first_name']) ?>" class="rounded-circle">
                        <?php else: ?>
                            <div class="avatar-placeholder">
                                <?= strtoupper(substr($contact['first_name'], 0, 1)) ?>
                            </div>
                        <?php endif; ?>
                    </div>
                    <div>
                        <h5 class="mb-1"><?= htmlspecialchars($contact['first_name'] . ' ' . $contact['last_name']) ?></h5>
                        <small class="text-muted">
                            <i class="bx bx-book-alt me-1"></i>
                            <?= htmlspecialchars($contact['course_name']) ?>
                        </small>
                    </div>
                </div>
            </div>
            
            <div class="messages-container">
                <div class="messages-list" id="messages-list">
                    <?php if (!empty($messages)): ?>
                        <?php
                        $currentDate = '';
                        foreach ($messages as $message):
                            $messageDate = date('Y-m-d', strtotime($message['sent_at']));
                            if ($messageDate !== $currentDate):
                                $currentDate = $messageDate;
                        ?>
                            <div class="date-separator">
                                <span><?= date('F j, Y', strtotime($message['sent_at'])) ?></span>
                            </div>
                        <?php endif; ?>
                        
                        <div class="message-bubble <?= $message['sender_id'] == $userId ? 'message-sent' : 'message-received' ?>">
                            <?= nl2br(htmlspecialchars($message['message_text'])) ?>
                            <div class="message-time">
                                <?= date('g:i A', strtotime($message['sent_at'])) ?>
                                <?php if ($message['sender_id'] == $userId): ?>
                                    <i class="bx bx-check-double <?= $message['is_read'] ? 'text-primary' : 'text-muted' ?>"></i>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <div class="no-messages">
                            <i class="bx bx-message-square-dots"></i>
                            <p>No messages yet. Start a conversation!</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="message-input-area">
                <form id="message-form" class="d-flex w-100">
                    <input type="hidden" id="recipient-id" value="<?= $contactId ?>">
                    <div class="message-input-wrapper flex-grow-1 me-2">
                        <textarea class="form-control" id="message-input" 
                                  placeholder="Type your message..." rows="1" required></textarea>
                    </div>
                    <button type="submit" class="btn btn-primary" id="send-btn">
                        <i class="fas fa-paper-plane"></i>
                    </button>
                </form>
            </div>
            <?php
        }
    }
    exit;
}

// Get contacts (teachers/facilitators) for this student
$contacts = [];
$selectedContactId = isset($_GET['contact']) ? intval($_GET['contact']) : null;

try {
    // Get all teachers/facilitators that the student can message
    $contactsQuery = "SELECT DISTINCT 
                        u.user_id, 
                        u.first_name, 
                        u.last_name, 
                        u.email, 
                        u.profile_picture,
                        c.course_id,
                        c.title as course_name,
                        (SELECT COUNT(*) FROM messages WHERE sender_id = u.user_id AND recipient_id = ? AND is_read = 0) as unread_count,
                        (SELECT MAX(sent_at) FROM messages WHERE (sender_id = u.user_id AND recipient_id = ?) OR (sender_id = ? AND recipient_id = u.user_id)) as last_message_date
                      FROM users u
                      JOIN teachers t ON u.user_id = t.user_id
                      JOIN courses c ON t.teacher_id = c.teacher_id
                      JOIN enrollments e ON c.course_id = e.course_id
                      WHERE e.student_id = ? AND u.user_role = 'teacher'
                      ORDER BY last_message_date DESC, u.last_name, u.first_name";
    
    $stmt = $conn->prepare($contactsQuery);
    $stmt->bind_param("iiii", $userId, $userId, $userId, $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $contactsMap = [];
    while ($row = $result->fetch_assoc()) {
        $contactId = $row['user_id'];
        if (!isset($contactsMap[$contactId])) {
            $contactsMap[$contactId] = [
                'user_id' => $contactId,
                'first_name' => $row['first_name'],
                'last_name' => $row['last_name'],
                'email' => $row['email'],
                'profile_picture' => $row['profile_picture'],
                'courses' => [],
                'unread_count' => $row['unread_count'],
                'last_message_date' => $row['last_message_date']
            ];
        }
        $contactsMap[$contactId]['courses'][] = [
            'id' => $row['course_id'],
            'name' => $row['course_name']
        ];
    }
    
    $contacts = array_values($contactsMap);
    
} catch (Exception $e) {
    error_log("Messages error: " . $e->getMessage());
}

// Function to format message time
function formatMessageTime($datetime) {
    if (!$datetime) return '';
    
    $time = strtotime($datetime);
    $now = time();
    $diff = $now - $time;
    
    if ($diff < 60) return 'Just now';
    if ($diff < 3600) return floor($diff / 60) . 'm ago';
    if ($diff < 86400) return floor($diff / 3600) . 'h ago';
    if ($diff < 604800) return floor($diff / 86400) . 'd ago';
    
    return date('M j, Y', $time);
}

// Include header for regular page requests
include_once 'components/header.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Messages - Panadite Academy</title>
    <style>
        /* Premium Modern Messaging Interface */
        .messaging-container {
            height: calc(100vh - 200px);
            min-height: 600px;
            background: #F5F5F5;
            border-radius: 16px;
            overflow: hidden;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.08);
            border: 1px solid rgba(255, 255, 255, 0.8);
        }
        
        .contacts-sidebar {
            border-right: 1px solid rgba(230, 236, 241, 0.8);
            background: linear-gradient(145deg, #ffffff 0%, #fafbfc 100%);
            backdrop-filter: blur(10px);
        }
        
        .contacts-header {
            padding: 2rem 1.5rem;
            border-bottom: 1px solid rgba(230, 236, 241, 0.6);
            background: linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%);
            color: white;
            position: relative;
        }
        
        .contacts-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.1) 0%, rgba(255, 255, 255, 0.05) 100%);
            pointer-events: none;
        }
        
        .contacts-header h5 {
            margin: 0;
            font-weight: 700;
            font-size: 1.1rem;
            letter-spacing: 0.5px;
            position: relative;
            z-index: 1;
        }
        
        .search-box {
            margin-top: 1.5rem;
            position: relative;
            z-index: 1;
        }
        
        .search-box .form-control {
            border-radius: 25px;
            background: rgba(255, 255, 255, 0.95);
            border: 2px solid rgba(255, 255, 255, 0.3);
            padding: 12px 20px;
            font-size: 0.9rem;
            backdrop-filter: blur(10px);
            transition: all 0.3s ease;
        }
        
        .search-box .form-control:focus {
            background: white;
            border-color: rgba(255, 255, 255, 0.8);
            box-shadow: 0 4px 15px rgba(255, 255, 255, 0.2);
        }
        
        .contacts-list {
            height: calc(100% - 140px);
            overflow-y: auto;
        }
        
        .contact-item {
            padding: 1.2rem 1.5rem;
            cursor: pointer;
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            border-bottom: 1px solid rgba(240, 244, 248, 0.8);
            margin: 0.3rem 0.8rem;
            border-radius: 12px;
            position: relative;
        }
        
        .contact-item:hover {
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.9) 0%, rgba(248, 250, 252, 0.95) 100%);
            transform: translateY(-2px) translateX(3px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.08);
            border-bottom-color: transparent;
        }
        
        .contact-item.active {
            background: linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%);
            color: white;
            transform: translateY(-1px);
            box-shadow: 0 12px 35px rgba(99, 102, 241, 0.25);
            border-bottom-color: transparent;
        }
        
        .contact-item.active::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.1) 0%, rgba(255, 255, 255, 0.05) 100%);
            border-radius: 12px;
            pointer-events: none;
        }
        
        .contact-avatar {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            background: linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%);
            color: white;
            font-weight: 600;
            font-size: 1.1rem;
            box-shadow: 0 4px 12px rgba(99, 102, 241, 0.2);
            border: 2px solid rgba(255, 255, 255, 0.8);
            transition: all 0.3s ease;
        }
        
        .contact-item:hover .contact-avatar {
            transform: scale(1.05);
            box-shadow: 0 6px 18px rgba(99, 102, 241, 0.3);
        }
        
        .contact-item.active .contact-avatar {
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.9) 0%, rgba(248, 250, 252, 0.9) 100%);
            color: #6366f1;
            border-color: rgba(255, 255, 255, 1);
        }
        
        .contact-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .contact-info h6 {
            margin: 0;
            font-weight: 600;
        }
        
        .contact-info p {
            margin: 0;
            font-size: 0.875rem;
            opacity: 0.8;
        }
        
        .unread-badge {
            background: #dc3545;
            color: white;
            border-radius: 50%;
            width: 20px;
            height: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.75rem;
            font-weight: bold;
        }
        
        .messages-area {
            display: flex;
            flex-direction: column;
            height: 100%;
            overflow: hidden;
            position: relative;
            background: linear-gradient(145deg, #F5F5F5 0%, #fafbfc 100%);
        }
        
        .message-header {
            padding: 1.8rem 2rem;
            border-bottom: 1px solid rgba(230, 236, 241, 0.6);
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.95) 0%, rgba(248, 250, 252, 0.9) 100%);
            backdrop-filter: blur(15px);
            position: relative;
        }
        
        .message-header::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            height: 1px;
            background: linear-gradient(90deg, transparent 0%, rgba(230, 236, 241, 0.8) 50%, transparent 100%);
        }
        
        .messages-container {
            flex: 1;
            display: flex;
            flex-direction: column;
            height: 100%;
            overflow: hidden;
        }
        
        .messages-list {
            flex: 1;
            padding: 2rem 1.5rem;
            overflow-y: scroll;
            background: linear-gradient(145deg, #F5F5F5 0%, rgba(250, 251, 252, 0.8) 100%);
            min-height: 300px;
            max-height: calc(100vh - 400px);
            scrollbar-width: thin;
            scrollbar-color: rgba(156, 163, 175, 0.5) transparent;
            position: relative;
        }
        
        .messages-list::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: 
                radial-gradient(circle at 20% 80%, rgba(120, 119, 198, 0.05) 0%, transparent 50%),
                radial-gradient(circle at 80% 20%, rgba(255, 255, 255, 0.8) 0%, transparent 50%);
            pointer-events: none;
        }
        
        .messages-list::-webkit-scrollbar {
            width: 6px;
        }
        
        .messages-list::-webkit-scrollbar-track {
            background: #f7fafc;
        }
        
        .messages-list::-webkit-scrollbar-thumb {
            background: #cbd5e0;
            border-radius: 3px;
        }
        
        .messages-list::-webkit-scrollbar-thumb:hover {
            background: #a0aec0;
        }
        
        .date-separator {
            text-align: center;
            margin: 1rem 0;
        }
        
        .date-separator span {
            background: #6c757d;
            color: white;
            padding: 0.25rem 0.75rem;
            border-radius: 12px;
            font-size: 0.75rem;
        }
        
        .message-bubble {
            max-width: 75%;
            padding: 1rem 1.4rem;
            margin: 0.8rem 0;
            border-radius: 20px;
            position: relative;
            word-wrap: break-word;
            overflow-wrap: break-word;
            display: block;
            backdrop-filter: blur(10px);
            transition: all 0.3s ease;
            z-index: 1;
        }
        
        .message-bubble:hover {
            transform: translateY(-1px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
        }
        
        .message-sent {
            background: linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%);
            color: white;
            margin-left: auto;
            box-shadow: 0 4px 15px rgba(99, 102, 241, 0.2);
        }
        
        .message-sent::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.1) 0%, rgba(255, 255, 255, 0.05) 100%);
            border-radius: 20px;
            pointer-events: none;
        }
        
        .message-received {
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.95) 0%, rgba(248, 250, 252, 0.9) 100%);
            color: #374151;
            border: 1px solid rgba(230, 236, 241, 0.6);
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
        }
        
        .message-time {
            font-size: 0.75rem;
            opacity: 0.7;
            margin-top: 0.25rem;
        }
        
        .message-input-area {
            padding: 1.8rem 2rem;
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.98) 0%, rgba(248, 250, 252, 0.95) 100%);
            border-top: 1px solid rgba(230, 236, 241, 0.6);
            flex-shrink: 0;
            position: sticky;
            bottom: 0;
            z-index: 10;
            backdrop-filter: blur(20px);
            box-shadow: 0 -4px 20px rgba(0, 0, 0, 0.05);
        }
        
        .message-input-area::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 1px;
            background: linear-gradient(90deg, transparent 0%, rgba(230, 236, 241, 0.8) 50%, transparent 100%);
        }
        
        .message-input-wrapper {
            position: relative;
        }
        
        #message-input {
            border-radius: 25px;
            border: 2px solid rgba(230, 236, 241, 0.8);
            padding: 1rem 1.5rem;
            resize: none;
            min-height: 50px;
            max-height: 120px;
            background: rgba(255, 255, 255, 0.9);
            font-size: 0.95rem;
            line-height: 1.5;
            transition: all 0.3s ease;
            backdrop-filter: blur(10px);
        }
        
        #message-input:focus {
            border-color: rgba(99, 102, 241, 0.5);
            background: white;
            box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1), 0 4px 15px rgba(0, 0, 0, 0.05);
            outline: none;
        }
        
        #send-btn {
            border-radius: 50%;
            width: 50px;
            height: 50px;
            display: flex;
            align-items: center;
            justify-content: center;
            background: linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%);
            border: none;
            box-shadow: 0 4px 15px rgba(99, 102, 241, 0.3);
            transition: all 0.3s ease;
        }
        
        #send-btn:hover {
            transform: translateY(-2px) scale(1.05);
            box-shadow: 0 8px 25px rgba(99, 102, 241, 0.4);
        }
        
        #send-btn:active {
            transform: translateY(0) scale(0.98);
        }
        
        .welcome-message {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            height: 100%;
            text-align: center;
            color: #6b7280;
            padding: 3rem;
        }
        
        .welcome-message i {
            font-size: 5rem;
            margin-bottom: 1.5rem;
            background: linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            opacity: 0.8;
        }
        
        .welcome-message h4 {
            color: #374151;
            font-weight: 600;
            margin-bottom: 0.8rem;
            font-size: 1.3rem;
        }
        
        .welcome-message p {
            color: #6b7280;
            font-size: 1rem;
            opacity: 0.8;
        }
        
        .loading-state {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            height: 100%;
        }
        
        .no-messages {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            height: 200px;
            text-align: center;
            color: #6c757d;
        }
        
        .no-messages i {
            font-size: 3rem;
            margin-bottom: 1rem;
            opacity: 0.5;
        }
        
        @media (max-width: 768px) {
            .messaging-container {
                height: calc(100vh - 120px);
            }
            
            .contacts-sidebar {
                position: absolute;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                z-index: 1000;
                transition: transform 0.3s ease;
            }
            
            .contacts-sidebar.d-none {
                transform: translateX(-100%);
            }
            
            .messages-area {
                width: 100%;
            }
        }
    </style>
</head>
<body>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="page-header mb-4">
                <h2><i class="bx bx-message-square-detail me-2"></i>Messages</h2>
                <p class="text-muted">Communicate with your course facilitators</p>
                <div class="d-flex gap-3 mt-3">
                    <div class="stat-card bg-primary text-white p-3 rounded">
                        <div class="d-flex align-items-center">
                            <i class="bx bx-user-voice fs-2 me-2"></i>
                            <div>
                                <div class="fs-4 fw-bold"><?= count($contacts) ?></div>
                                <div>Facilitators</div>
                            </div>
                        </div>
                    </div>
                    <div class="stat-card bg-info text-white p-3 rounded">
                        <div class="d-flex align-items-center">
                            <i class="bx bx-book-open fs-2 me-2"></i>
                            <div>
                                <div class="fs-4 fw-bold"><?= array_sum(array_map(function($c) { return count($c['courses']); }, $contacts)) ?></div>
                                <div>Courses</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="messaging-container">
        <div class="row g-0 h-100">
            <!-- Contacts Sidebar -->
            <div class="col-lg-4 contacts-sidebar" id="contacts-sidebar">
                <div class="contacts-header">
                    <h5><i class="bx bx-user-voice me-2"></i>Course Facilitators</h5>
                    <div class="search-box">
                        <input type="text" class="form-control" id="contact-search" 
                               placeholder="Search facilitators...">
                    </div>
                </div>
                
                <div class="contacts-list" id="contacts-list">
                    <?php if (!empty($contacts)): ?>
                        <?php foreach ($contacts as $contact): ?>
                            <div class="contact-item <?= ($selectedContactId == $contact['user_id']) ? 'active' : '' ?>" 
                                 data-contact-id="<?= $contact['user_id'] ?>" 
                                 onclick="loadContact(<?= $contact['user_id'] ?>)">
                                <div class="d-flex align-items-center">
                                    <div class="contact-avatar me-3">
                                        <?php if ($contact['profile_picture']): ?>
                                            <img src="<?= htmlspecialchars($contact['profile_picture']) ?>" 
                                                alt="<?= htmlspecialchars($contact['first_name']) ?>">
                                        <?php else: ?>
                                            <?= strtoupper(substr($contact['first_name'], 0, 1) . substr($contact['last_name'], 0, 1)) ?>
                                        <?php endif; ?>
                                    </div>
                                    <div class="contact-info flex-grow-1">
                                        <h6 class="contact-name"><?= htmlspecialchars($contact['first_name'] . ' ' . $contact['last_name']) ?></h6>
                                        <p class="contact-course">
                                            <?= count($contact['courses']) > 1 ? 
                                                count($contact['courses']) . ' courses' : 
                                                htmlspecialchars($contact['courses'][0]['name']) ?>
                                        </p>
                                        <?php if ($contact['last_message_date']): ?>
                                            <small class="text-muted">
                                                <?= formatMessageTime($contact['last_message_date']) ?>
                                            </small>
                                        <?php endif; ?>
                                    </div>
                                    <div class="ms-2">
                                        <?php if ($contact['unread_count'] > 0): ?>
                                            <span class="unread-badge"><?= $contact['unread_count'] ?></span>
                                        <?php endif; ?>
                                        <i class="bx bx-chevron-right text-muted"></i>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <div class="p-4 text-center text-muted">
                            <i class="bx bx-user-x fs-1 mb-3"></i>
                            <h5>No Facilitators Found</h5>
                            <p>You don't have any course facilitators to message yet.</p>
                            <a href="courses.php" class="btn btn-primary">
                                <i class="bx bx-book-open me-2"></i>Browse Courses
                            </a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Messages Area -->
            <div class="col-lg-8 messages-area" id="messages-area">
                <div id="messages-panel">
                    <?php if ($selectedContactId): ?>
                        <div class="loading-state">
                            <div class="spinner-border text-primary" role="status">
                                <span class="visually-hidden">Loading...</span>
                            </div>
                            <p class="mt-3">Loading conversation...</p>
                        </div>
                    <?php else: ?>
                        <div class="welcome-message">
                            <i class="bx bx-message-square-dots"></i>
                            <h4>Select a Facilitator</h4>
                            <p>Choose a course facilitator from the list to start messaging</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    let currentContactId = <?= $selectedContactId ? $selectedContactId : 'null' ?>;
    
    // Load conversation if contact is selected
    if (currentContactId) {
        loadConversation(currentContactId);
    }
    
    // Contact search functionality
    const searchInput = document.getElementById('contact-search');
    if (searchInput) {
        searchInput.addEventListener('input', function() {
            const query = this.value.toLowerCase();
            document.querySelectorAll('.contact-item').forEach(item => {
                const name = item.querySelector('.contact-name').textContent.toLowerCase();
                if (name.includes(query)) {
                    item.style.display = '';
                } else {
                    item.style.display = 'none';
                }
            });
        });
    }
    
    // Load contact function
    window.loadContact = function(contactId) {
        if (contactId === currentContactId) return;
        
        // Update active state
        document.querySelectorAll('.contact-item').forEach(item => {
            item.classList.remove('active');
        });
        
        const contactItem = document.querySelector(`.contact-item[data-contact-id="${contactId}"]`);
        if (contactItem) {
            contactItem.classList.add('active');
        }
        
        // Update URL
        const url = new URL(window.location);
        url.searchParams.set('contact', contactId);
        window.history.pushState({}, '', url);
        
        // Load conversation
        currentContactId = contactId;
        loadConversation(contactId);
        
        // Hide contacts on mobile
        const contactsSidebar = document.getElementById('contacts-sidebar');
        if (window.innerWidth <= 768) {
            contactsSidebar.classList.add('d-none');
        }
    };
    
    // Load conversation function
    function loadConversation(contactId) {
        const messagesPanel = document.getElementById('messages-panel');
        if (!messagesPanel) return;
        
        messagesPanel.innerHTML = `
            <div class="loading-state">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
                <p class="mt-3">Loading messages...</p>
            </div>
        `;
        
        fetch(`messages.php?ajax=load_conversation&contact=${contactId}`, {
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => response.text())
        .then(html => {
            messagesPanel.innerHTML = html;
            setupMessageForm();
            // Multiple scroll attempts to ensure it works
            setTimeout(scrollToBottom, 100);
            setTimeout(scrollToBottom, 500);
            setTimeout(scrollToBottom, 1000);
        })
        .catch(error => {
            console.error('Error loading conversation:', error);
            messagesPanel.innerHTML = `
                <div class="welcome-message">
                    <i class="bx bx-error-circle text-danger"></i>
                    <h4>Error Loading Messages</h4>
                    <p>Please try again later.</p>
                </div>
            `;
        });
    }
    
    // Setup message form
    function setupMessageForm() {
        const form = document.getElementById('message-form');
        if (!form) return;
        
        // Remove existing listeners
        const newForm = form.cloneNode(true);
        form.parentNode.replaceChild(newForm, form);
        
        // Add submit listener
        newForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const messageInput = document.getElementById('message-input');
            const recipientId = document.getElementById('recipient-id').value;
            const messageText = messageInput.value.trim();
            
            if (!messageText) return;
            
            // Add message to UI immediately
            const messagesList = document.getElementById('messages-list');
            if (messagesList) {
                const messageDiv = document.createElement('div');
                messageDiv.className = 'message-bubble message-sent';
                messageDiv.innerHTML = `
                    ${messageText.replace(/\n/g, '<br>')}
                    <div class="message-time">Just now</div>
                `;
                messagesList.appendChild(messageDiv);
                scrollToBottom();
            }
            
            // Clear input and disable button
            const sendBtn = document.getElementById('send-btn');
            sendBtn.disabled = true;
            messageInput.value = '';
            messageInput.style.height = 'auto';
            
            // Send via AJAX (you'll need to create ajax_messages.php)
            const formData = new FormData();
            formData.append('recipient_id', recipientId);
            formData.append('message', messageText);
            formData.append('action', 'send_message');
            
            fetch('ajax_messages.php', {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            })
            .then(response => response.json())
            .then(data => {
                if (!data.success) {
                    alert('Error sending message: ' + (data.message || 'Unknown error'));
                }
            })
            .catch(error => {
                console.error('Error sending message:', error);
                alert('Error sending message. Please try again.');
            })
            .finally(() => {
                sendBtn.disabled = false;
                messageInput.focus();
            });
        });
        
        // Auto-resize textarea
        const messageInput = document.getElementById('message-input');
        if (messageInput) {
            messageInput.addEventListener('input', function() {
                this.style.height = 'auto';
                this.style.height = (this.scrollHeight) + 'px';
            });
        }
    }
    
    // Scroll to bottom function
    function scrollToBottom() {
        const messagesList = document.getElementById('messages-list');
        if (messagesList) {
            setTimeout(() => {
                messagesList.scrollTop = messagesList.scrollHeight + 100;
            }, 100);
            // Force scroll again after a longer delay
            setTimeout(() => {
                messagesList.scrollTop = messagesList.scrollHeight + 100;
            }, 300);
        }
    }
    
    // Show contacts list (for mobile back button)
    window.showContactsList = function() {
        const contactsSidebar = document.getElementById('contacts-sidebar');
        contactsSidebar.classList.remove('d-none');
    };
});
</script>

</body>
</html>
