<?php
require_once '../includes/session_start.php';
require_once '../config/database.php';

// Check if user is logged in and is a student
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'student') {
    header("Location: ../login.php");
    exit;
}

$userId = $_SESSION['user_id'];

// Check if this is an AJAX request
$isAjaxRequest = isset($_SERVER['HTTP_X_REQUESTED_WITH']) && 
                 strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';

// Function to format message time
function formatMessageTime($datetime) {
    $time = strtotime($datetime);
    $now = time();
    $diff = $now - $time;
    
    if ($diff < 60) return 'Just now';
    if ($diff < 3600) return floor($diff / 60) . 'm ago';
    if ($diff < 86400) return floor($diff / 3600) . 'h ago';
    if ($diff < 604800) return floor($diff / 86400) . 'd ago';
    
    return date('M j, Y', $time);
}

// Get contacts (teachers/facilitators) for this student
$contacts = [];
$messages = [];
$selectedContact = null;
$selectedContactId = isset($_GET['contact']) ? intval($_GET['contact']) : null;

try {
    // Get all teachers/facilitators that the student can message
    $contactsQuery = "SELECT DISTINCT 
                        u.user_id, 
                        u.first_name, 
                        u.last_name, 
                        u.email, 
                        u.profile_picture,
                        c.course_id,
                        c.name as course_name
                      FROM users u
                      JOIN course_instructors ci ON u.user_id = ci.instructor_id
                      JOIN courses c ON ci.course_id = c.course_id
                      JOIN course_enrollments ce ON c.course_id = ce.course_id
                      WHERE ce.student_id = ? AND u.role = 'teacher' AND u.status = 'active'
                      ORDER BY u.last_name, u.first_name";
    
    $stmt = $conn->prepare($contactsQuery);
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $contactsMap = [];
    while ($row = $result->fetch_assoc()) {
        $contactId = $row['user_id'];
        if (!isset($contactsMap[$contactId])) {
            $contactsMap[$contactId] = [
                'user_id' => $contactId,
                'first_name' => $row['first_name'],
                'last_name' => $row['last_name'],
                'email' => $row['email'],
                'profile_picture' => $row['profile_picture'],
                'courses' => [],
                'unread_count' => 0,
                'last_message_date' => null
            ];
        }
        $contactsMap[$contactId]['courses'][] = [
            'id' => $row['course_id'],
            'name' => $row['course_name']
        ];
    }
    
    $contacts = array_values($contactsMap);
    
    // Get unread message counts and last message dates
    if (!empty($contacts)) {
        $contactIds = array_column($contacts, 'user_id');
        $placeholders = str_repeat('?,', count($contactIds) - 1) . '?';
        
        $unreadQuery = "SELECT sender_id, COUNT(*) as unread_count, MAX(sent_at) as last_message_date
                        FROM messages 
                        WHERE recipient_id = ? AND sender_id IN ($placeholders) AND is_read = 0
                        GROUP BY sender_id";
        
        $params = array_merge(['i', $userId], array_fill(0, count($contactIds), 'i'), $contactIds);
        $stmt = $conn->prepare($unreadQuery);
        $stmt->bind_param(...$params);
        $stmt->execute();
        $result = $stmt->get_result();
        
        while ($row = $result->fetch_assoc()) {
            foreach ($contacts as &$contact) {
                if ($contact['user_id'] == $row['sender_id']) {
                    $contact['unread_count'] = $row['unread_count'];
                    $contact['last_message_date'] = $row['last_message_date'];
                    break;
                }
            }
        }
        
        // Sort contacts by recent activity
        usort($contacts, function($a, $b) {
            if (isset($a['last_message_date']) && isset($b['last_message_date'])) {
                return strtotime($b['last_message_date']) - strtotime($a['last_message_date']);
            } elseif (isset($a['last_message_date'])) {
                return -1;
            } elseif (isset($b['last_message_date'])) {
                return 1;
            } else {
                return strcmp($a['last_name'], $b['last_name']);
            }
        });
    }
    
    // If a contact is selected, get messages
    if ($selectedContactId && !empty($contacts)) {
        foreach ($contacts as $contact) {
            if ($contact['user_id'] == $selectedContactId) {
                $selectedContact = $contact;
                break;
            }
        }
        
        if ($selectedContact) {
            $messagesQuery = "SELECT 
                                m.message_id, 
                                m.sender_id, 
                                m.recipient_id, 
                                m.message_text, 
                                m.sent_at, 
                                m.is_read,
                                CASE 
                                    WHEN m.sender_id = ? THEN 'sent' 
                                    ELSE 'received' 
                                END as direction,
                                u.first_name as sender_first_name,
                                u.last_name as sender_last_name,
                                u.profile_picture as sender_profile_picture
                              FROM messages m
                              JOIN users u ON m.sender_id = u.user_id
                              WHERE (m.sender_id = ? AND m.recipient_id = ?) 
                                 OR (m.sender_id = ? AND m.recipient_id = ?)
                              ORDER BY m.sent_at ASC";
            
            $stmt = $conn->prepare($messagesQuery);
            $stmt->bind_param("iiiii", $userId, $userId, $selectedContactId, $selectedContactId, $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result) {
                $messages = $result->fetch_all(MYSQLI_ASSOC);
            }
            
            // Mark messages as read
            $markReadQuery = "UPDATE messages SET is_read = 1 
                              WHERE sender_id = ? AND recipient_id = ? AND is_read = 0";
            $stmt = $conn->prepare($markReadQuery);
            $stmt->bind_param("ii", $selectedContactId, $userId);
            $stmt->execute();
        }
    }
    
} catch (Exception $e) {
    error_log("Messages error: " . $e->getMessage());
}

// If this is an AJAX request, return only the conversation content
if ($isAjaxRequest && isset($_GET['ajax'])) {
    if ($selectedContact): ?>
        <div class="message-header">
            <div class="d-flex align-items-center">
                <button class="btn btn-sm btn-outline-secondary d-md-none me-3" onclick="showContactsList()">
                    <i class="bx bx-arrow-back"></i>
                </button>
                <div class="contact-avatar me-3">
                    <?php if ($selectedContact['profile_picture']): ?>
                        <img src="<?= htmlspecialchars($selectedContact['profile_picture']) ?>" 
                             alt="<?= htmlspecialchars($selectedContact['first_name']) ?>" class="rounded-circle">
                    <?php else: ?>
                        <div class="avatar-placeholder">
                            <?= strtoupper(substr($selectedContact['first_name'], 0, 1)) ?>
                        </div>
                    <?php endif; ?>
                </div>
                <div>
                    <h5 class="mb-1"><?= htmlspecialchars($selectedContact['first_name'] . ' ' . $selectedContact['last_name']) ?></h5>
                    <?php if (!empty($selectedContact['courses'])): ?>
                        <small class="text-muted">
                            <i class="bx bx-book-alt me-1"></i>
                            <?= htmlspecialchars($selectedContact['courses'][0]['name']) ?>
                        </small>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <div class="messages-container">
            <div class="messages-list" id="messages-list">
                <?php if (!empty($messages)): ?>
                    <?php
                    $currentDate = '';
                    foreach ($messages as $message):
                        $messageDate = date('Y-m-d', strtotime($message['sent_at']));
                        if ($messageDate !== $currentDate):
                            $currentDate = $messageDate;
                    ?>
                        <div class="date-separator">
                            <span><?= date('F j, Y', strtotime($message['sent_at'])) ?></span>
                        </div>
                    <?php endif; ?>
                    
                    <div class="message <?= $message['direction'] ?>">
                        <div class="message-content">
                            <p><?= nl2br(htmlspecialchars($message['message_text'])) ?></p>
                            <div class="message-time">
                                <?= date('g:i A', strtotime($message['sent_at'])) ?>
                                <?php if ($message['direction'] === 'sent'): ?>
                                    <i class="bx bx-check-double <?= $message['is_read'] ? 'text-primary' : 'text-muted' ?>"></i>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div class="no-messages">
                        <i class="bx bx-message-square-dots"></i>
                        <p>No messages yet. Start a conversation!</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="message-input-area">
            <form id="message-form" class="d-flex gap-2">
                <input type="hidden" id="recipient-id" value="<?= $selectedContact['user_id'] ?>">
                <div class="flex-grow-1">
                    <textarea id="message-input" class="form-control" 
                              placeholder="Type your message..." rows="1" required></textarea>
                </div>
                <button type="submit" class="btn btn-primary" id="send-btn">
                    <i class="bx bx-send"></i>
                </button>
            </form>
        </div>
    <?php else: ?>
        <div class="no-contact-selected">
            <i class="bx bx-message-square-dots"></i>
            <h4>Contact not found</h4>
            <p>The selected contact could not be found or you don't have permission to message them.</p>
        </div>
    <?php endif;
    exit;
}

// For regular page requests, include the full page
if (!$isAjaxRequest):
    include_once 'components/header.php';
?>

<link rel="stylesheet" href="../css/messages.css?v=<?= time() ?>">

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="page-header mb-4">
                <h2><i class="bx bx-message-square-detail me-2"></i>Messages</h2>
                <p class="text-muted">Communicate with your course facilitators</p>
                <div class="stats-cards d-flex gap-3 mt-3">
                    <div class="stat-card">
                        <div class="stat-icon">
                            <i class="bx bx-user-voice"></i>
                        </div>
                        <div class="stat-info">
                            <span class="stat-number"><?= count($contacts) ?></span>
                            <span class="stat-label">Facilitators</span>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon">
                            <i class="bx bx-book-open"></i>
                        </div>
                        <div class="stat-info">
                            <span class="stat-number"><?= array_sum(array_map(function($c) { return count($c['courses']); }, $contacts)) ?></span>
                            <span class="stat-label">Courses</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="messaging-container">
        <div class="row g-0">
            <!-- Contacts Sidebar -->
            <div class="col-lg-4 contacts-column" id="contacts-column">
                <div class="contacts-panel">
                    <div class="contacts-header">
                        <h5><i class="bx bx-user-voice me-2"></i>Course Facilitators</h5>
                        <div class="search-box">
                            <div class="input-group">
                                <span class="input-group-text"><i class="bx bx-search"></i></span>
                                <input type="text" class="form-control" id="contact-search" 
                                       placeholder="Search facilitators...">
                            </div>
                        </div>
                    </div>
                    
                    <div class="contacts-list" id="contacts-list">
                        <?php if (!empty($contacts)): ?>
                            <?php foreach ($contacts as $contact): ?>
                                <div class="contact-item <?= ($selectedContactId == $contact['user_id']) ? 'active' : '' ?>" 
                                     data-contact-id="<?= $contact['user_id'] ?>"
                                     onclick="selectContact(<?= $contact['user_id'] ?>)">
                                    <div class="contact-avatar">
                                        <?php if ($contact['profile_picture']): ?>
                                            <img src="<?= htmlspecialchars($contact['profile_picture']) ?>" 
                                                 alt="<?= htmlspecialchars($contact['first_name']) ?>">
                                        <?php else: ?>
                                            <div class="avatar-placeholder">
                                                <?= strtoupper(substr($contact['first_name'], 0, 1) . substr($contact['last_name'], 0, 1)) ?>
                                            </div>
                                        <?php endif; ?>
                                        <?php if ($contact['unread_count'] > 0): ?>
                                            <span class="unread-badge"><?= $contact['unread_count'] ?></span>
                                        <?php endif; ?>
                                    </div>
                                    <div class="contact-info">
                                        <h6 class="contact-name"><?= htmlspecialchars($contact['first_name'] . ' ' . $contact['last_name']) ?></h6>
                                        <?php if (!empty($contact['courses'])): ?>
                                            <p class="contact-course">
                                                <i class="bx bx-book-alt"></i>
                                                <?= count($contact['courses']) > 1 ? 
                                                    count($contact['courses']) . ' courses' : 
                                                    htmlspecialchars($contact['courses'][0]['name']) ?>
                                            </p>
                                        <?php endif; ?>
                                        <?php if (isset($contact['last_message_date'])): ?>
                                            <small class="last-message-time text-muted">
                                                <?= formatMessageTime($contact['last_message_date']) ?>
                                            </small>
                                        <?php endif; ?>
                                    </div>
                                    <div class="contact-actions">
                                        <i class="bx bx-chevron-right"></i>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <div class="no-contacts">
                                <i class="bx bx-user-x"></i>
                                <h5>No Facilitators Found</h5>
                                <p>You don't have any course facilitators to message yet.</p>
                                <a href="courses.php" class="btn btn-primary">
                                    <i class="bx bx-book-open me-2"></i>Browse Courses
                                </a>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <!-- Messages Content -->
            <div class="col-lg-8 messages-column" id="messages-column">
                <div class="messages-panel" id="messages-panel">
                    <?php if ($selectedContact): ?>
                        <!-- This will be loaded via AJAX -->
                        <div class="loading-state">
                            <div class="spinner-border text-primary" role="status">
                                <span class="visually-hidden">Loading...</span>
                            </div>
                            <p>Loading conversation...</p>
                        </div>
                    <?php else: ?>
                        <div class="no-contact-selected">
                            <div class="welcome-message">
                                <i class="bx bx-message-square-dots"></i>
                                <h4>Select a Facilitator</h4>
                                <p>Choose a course facilitator from the list to start messaging</p>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
let currentContactId = <?= $selectedContactId ? $selectedContactId : 'null' ?>;

// Initialize the page
document.addEventListener('DOMContentLoaded', function() {
    // Load initial conversation if contact is selected
    if (currentContactId) {
        loadConversation(currentContactId);
    }
    
    // Setup search functionality
    setupContactSearch();
    
    // Setup responsive handlers
    setupResponsiveHandlers();
});

// Select contact function
function selectContact(contactId) {
    if (contactId === currentContactId) return;
    
    currentContactId = contactId;
    
    // Update URL
    const newUrl = `messages.php?contact=${contactId}`;
    window.history.pushState({contactId}, '', newUrl);
    
    // Update active state
    document.querySelectorAll('.contact-item').forEach(item => {
        item.classList.remove('active');
    });
    document.querySelector(`[data-contact-id="${contactId}"]`).classList.add('active');
    
    // Show messages column on mobile
    showMessagesColumn();
    
    // Load conversation
    loadConversation(contactId);
}

// Load conversation via AJAX
function loadConversation(contactId) {
    const messagesPanel = document.getElementById('messages-panel');
    
    // Show loading state
    messagesPanel.innerHTML = `
        <div class="loading-state">
            <div class="spinner-border text-primary" role="status">
                <span class="visually-hidden">Loading...</span>
            </div>
            <p>Loading conversation...</p>
        </div>
    `;
    
    fetch(`messages.php?contact=${contactId}&ajax=true`, {
        method: 'GET',
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(response => response.text())
    .then(html => {
        messagesPanel.innerHTML = html;
        setupMessageForm();
        scrollToBottom();
    })
    .catch(error => {
        console.error('Error loading conversation:', error);
        messagesPanel.innerHTML = `
            <div class="error-state">
                <i class="bx bx-error-circle"></i>
                <h4>Error Loading Conversation</h4>
                <p>Please try again.</p>
                <button class="btn btn-primary" onclick="loadConversation(${contactId})">Retry</button>
            </div>
        `;
    });
}

// Setup message form
function setupMessageForm() {
    const messageForm = document.getElementById('message-form');
    const messageInput = document.getElementById('message-input');
    const sendBtn = document.getElementById('send-btn');
    
    if (!messageForm) return;
    
    // Auto-resize textarea
    messageInput.addEventListener('input', function() {
        this.style.height = 'auto';
        this.style.height = Math.min(this.scrollHeight, 120) + 'px';
    });
    
    // Handle form submission
    messageForm.addEventListener('submit', function(e) {
        e.preventDefault();
        sendMessage();
    });
    
    // Send on Enter, new line on Shift+Enter
    messageInput.addEventListener('keydown', function(e) {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            sendMessage();
        }
    });
}

// Send message function
function sendMessage() {
    const messageInput = document.getElementById('message-input');
    const sendBtn = document.getElementById('send-btn');
    const recipientId = document.getElementById('recipient-id').value;
    
    const messageText = messageInput.value.trim();
    if (!messageText || !recipientId) return;
    
    // Show sending state
    sendBtn.disabled = true;
    sendBtn.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i>';
    
    const formData = new FormData();
    formData.append('recipient_id', recipientId);
    formData.append('message_text', messageText);
    
    fetch('api/send_message.php', {
        method: 'POST',
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        },
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            messageInput.value = '';
            messageInput.style.height = 'auto';
            loadConversation(currentContactId);
        } else {
            alert('Error sending message: ' + (data.message || 'Unknown error'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error sending message. Please try again.');
    })
    .finally(() => {
        sendBtn.disabled = false;
        sendBtn.innerHTML = '<i class="bx bx-send"></i>';
        messageInput.focus();
    });
}

// Setup contact search
function setupContactSearch() {
    const searchInput = document.getElementById('contact-search');
    if (!searchInput) return;
    
    searchInput.addEventListener('input', function() {
        const searchTerm = this.value.toLowerCase();
        const contactItems = document.querySelectorAll('.contact-item');
        
        contactItems.forEach(item => {
            const name = item.querySelector('.contact-name').textContent.toLowerCase();
            const course = item.querySelector('.contact-course')?.textContent.toLowerCase() || '';
            
            if (name.includes(searchTerm) || course.includes(searchTerm)) {
                item.style.display = '';
            } else {
                item.style.display = 'none';
            }
        });
    });
}

// Responsive handlers
function setupResponsiveHandlers() {
    // Handle browser back/forward
    window.addEventListener('popstate', function(e) {
        const contactId = new URLSearchParams(window.location.search).get('contact');
        if (contactId && contactId !== currentContactId) {
            currentContactId = parseInt(contactId);
            loadConversation(currentContactId);
        }
    });
}

function showContactsList() {
    document.getElementById('contacts-column').classList.add('show');
    document.getElementById('messages-column').classList.remove('show');
}

function showMessagesColumn() {
    document.getElementById('contacts-column').classList.remove('show');
    document.getElementById('messages-column').classList.add('show');
}

function scrollToBottom() {
    const messagesList = document.getElementById('messages-list');
    if (messagesList) {
        setTimeout(() => {
            messagesList.scrollTop = messagesList.scrollHeight;
        }, 100);
    }
}
</script>

<?php endif; // End of full page check ?>
