<?php
// Session is started in header.php
require_once '../auth/functions.php';

// Require student role authentication - no exceptions
requireRole('student');

// Set custom CSS for this page
$customCss = 'notifications';

// Fetch notifications
$userId = $_SESSION['user_id'] ?? null;
$notifications = [];
$unreadCount = 0;

try {
    require_once '../config/db_connect.php';
    
    if ($conn) {
        // Get notifications with proper filtering
        $typeFilter = '';
        $params = [$userId];
        $paramTypes = 'i';
        
        // Apply type filter if set
        if (isset($_GET['type'])) {
            $filterType = $_GET['type'];
            
            if ($filterType === 'unread') {
                $typeFilter = ' AND n.is_read = 0';
            } elseif ($filterType === 'course') {
                $typeFilter = ' AND (n.related_to_type = ? OR n.type IN ("course_new", "enrollment"))';
                $params[] = $filterType;
                $paramTypes .= 's';
            } elseif ($filterType === 'assignment') {
                $typeFilter = ' AND (n.related_to_type = ? OR n.type IN ("assignment_due", "grade_posted"))';
                $params[] = $filterType;
                $paramTypes .= 's';
            } elseif ($filterType === 'message') {
                $typeFilter = ' AND (n.related_to_type = ? OR n.type IN ("message_new"))';
                $params[] = $filterType;
                $paramTypes .= 's';
            } elseif ($filterType === 'system') {
                $typeFilter = ' AND (n.related_to_type = ? OR n.type IN ("system_announcement", "payment", "enrollment"))';
                $params[] = $filterType;
                $paramTypes .= 's';
            }
        }
        
        $query = "SELECT 
                n.*,
                CASE
                    WHEN n.related_to_type = 'course' THEN c.title
                    WHEN n.related_to_type = 'assignment' THEN a.title
                    WHEN n.related_to_type = 'message' AND n.sender_id IS NOT NULL THEN CONCAT('Message from ', u.first_name, ' ', u.last_name)
                    ELSE NULL
                END as related_item_name
            FROM notifications n
            LEFT JOIN courses c ON n.related_to_type = 'course' AND n.related_to_id = c.course_id
            LEFT JOIN assignments a ON n.related_to_type = 'assignment' AND n.related_to_id = a.assignment_id
            LEFT JOIN users u ON n.sender_id = u.user_id
            WHERE n.user_id = ?{$typeFilter}
            ORDER BY n.created_at DESC
            LIMIT 50";
        
        $stmt = $conn->prepare($query);
        $stmt->bind_param($paramTypes, ...$params);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result) {
            $notifications = $result->fetch_all(MYSQLI_ASSOC);
            
            // Count unread notifications
            $unreadQuery = "SELECT COUNT(*) as unread FROM notifications WHERE user_id = ? AND is_read = 0";
            $stmt = $conn->prepare($unreadQuery);
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $unreadResult = $stmt->get_result()->fetch_assoc();
            $unreadCount = $unreadResult['unread'] ?? 0;
            
            // Mark as read if the mark_read parameter is set
            if (isset($_GET['mark_read']) && $_GET['mark_read'] === 'all') {
                $updateQuery = "UPDATE notifications SET is_read = 1 WHERE user_id = ?";
                $stmt = $conn->prepare($updateQuery);
                $stmt->bind_param("i", $userId);
                $stmt->execute();
                
                // Update read status in our current array too
                foreach ($notifications as $key => $notification) {
                    $notifications[$key]['is_read'] = 1;
                }
                $unreadCount = 0;
            } 
            // Mark single notification as read
            elseif (isset($_GET['mark_read']) && is_numeric($_GET['mark_read'])) {
                $notificationId = intval($_GET['mark_read']);
                $updateQuery = "UPDATE notifications SET is_read = 1 WHERE notification_id = ? AND user_id = ?";
                $stmt = $conn->prepare($updateQuery);
                $stmt->bind_param("ii", $notificationId, $userId);
                $stmt->execute();
                
                // Update read status in our current array too
                foreach ($notifications as $key => $notification) {
                    if ($notification['notification_id'] == $notificationId) {
                        $notifications[$key]['is_read'] = 1;
                        if ($unreadCount > 0) $unreadCount--;
                        break;
                    }
                }
            }
        }
    }
} catch (Exception $e) {
    $error = "Database error: " . $e->getMessage();
}

// No demo data - use only database data

// Function to format notification time
function formatNotificationTime($dateTime) {
    $now = new DateTime();
    $date = new DateTime($dateTime);
    $diff = $now->diff($date);
    
    if ($diff->y > 0) {
        return $diff->y . ' year' . ($diff->y > 1 ? 's' : '') . ' ago';
    } elseif ($diff->m > 0) {
        return $diff->m . ' month' . ($diff->m > 1 ? 's' : '') . ' ago';
    } elseif ($diff->d > 0) {
        if ($diff->d == 1) {
            return 'Yesterday';
        } else {
            return $diff->d . ' days ago';
        }
    } elseif ($diff->h > 0) {
        return $diff->h . ' hour' . ($diff->h > 1 ? 's' : '') . ' ago';
    } elseif ($diff->i > 0) {
        return $diff->i . ' minute' . ($diff->i > 1 ? 's' : '') . ' ago';
    } else {
        return 'Just now';
    }
}

// Function to get notification icon and color based on type
function getNotificationTypeInfo($notificationType) {
    $types = [
        'course_new' => ['icon' => 'bx-book-open', 'color' => 'primary'],
        'assignment_due' => ['icon' => 'bx-task', 'color' => 'warning'],
        'grade_posted' => ['icon' => 'bx-star', 'color' => 'success'],
        'message_new' => ['icon' => 'bx-envelope', 'color' => 'info'],
        'forum_reply' => ['icon' => 'bx-chat', 'color' => 'secondary'],
        'system_announcement' => ['icon' => 'bx-bell', 'color' => 'danger']
    ];
    
    return $types[$notificationType] ?? ['icon' => 'bx-bell', 'color' => 'secondary'];
}

// Function to get the action URL for a notification
function getNotificationActionUrl($notification) {
    switch ($notification['related_to_type']) {
        case 'course':
            return 'courses.php?course_id=' . $notification['related_to_id'];
        case 'assignment':
            return 'assignments.php?assignment_id=' . $notification['related_to_id'];
        case 'message':
            return 'messages.php' . ($notification['sender_id'] ? '?contact=' . $notification['sender_id'] : '');
        default:
            return '#';
    }
}

// Include header
include_once 'components/header.php';

// Force direct CSS inclusion for notifications page to bypass detection issues
echo '<link rel="stylesheet" href="assets/css/pages/notifications.css?v=' . time() . '">';
?>

<!-- Notifications Container -->
<div class="container-fluid notifications-container">
    <!-- Page Header -->
    <div class="page-header">
        <h1 class="page-title">Notifications</h1>
        <p class="page-subtitle">Stay updated with important announcements and activity</p>
        <?php if ($unreadCount > 0): ?>
        <div class="ms-auto">
            <a href="?<?= http_build_query(array_merge($_GET, ['mark_read' => 'all'])) ?>" class="btn btn-outline-primary rounded-pill">
                <i class="bx bx-check-double me-1"></i> Mark All as Read
            </a>
        </div>
        <?php endif; ?>
    </div>
    
    <div class="notifications-wrapper">
        <!-- Filters Sidebar -->
        <div class="notifications-filters">
            <div class="filter-card">
                <div class="filter-header">
                    <h5><i class="bx bx-filter me-2"></i>Filters</h5>
                </div>
                <div class="filter-body">
                    <a href="notifications.php" class="filter-item <?= !isset($_GET['type']) ? 'active' : '' ?>">
                        <span class="filter-icon"><i class="bx bx-bell"></i></span>
                        <span class="filter-name">All Notifications</span>
                        <span class="filter-badge"><?= count($notifications) ?></span>
                    </a>
                    <a href="notifications.php?type=unread" class="filter-item <?= isset($_GET['type']) && $_GET['type'] === 'unread' ? 'active' : '' ?>">
                        <span class="filter-icon"><i class="bx bx-envelope"></i></span>
                        <span class="filter-name">Unread</span>
                        <?php if ($unreadCount > 0): ?>
                        <span class="filter-badge"><?= $unreadCount ?></span>
                        <?php endif; ?>
                    </a>
                    <a href="notifications.php?type=course" class="filter-item <?= isset($_GET['type']) && $_GET['type'] === 'course' ? 'active' : '' ?>">
                        <span class="filter-icon"><i class="bx bx-book-open"></i></span>
                        <span class="filter-name">Courses</span>
                    </a>
                    <a href="notifications.php?type=assignment" class="filter-item <?= isset($_GET['type']) && $_GET['type'] === 'assignment' ? 'active' : '' ?>">
                        <span class="filter-icon"><i class="bx bx-task"></i></span>
                        <span class="filter-name">Assignments</span>
                    </a>
                    <a href="notifications.php?type=message" class="filter-item <?= isset($_GET['type']) && $_GET['type'] === 'message' ? 'active' : '' ?>">
                        <span class="filter-icon"><i class="bx bx-message-square-detail"></i></span>
                        <span class="filter-name">Messages</span>
                    </a>
                    <a href="notifications.php?type=system" class="filter-item <?= isset($_GET['type']) && $_GET['type'] === 'system' ? 'active' : '' ?>">
                        <span class="filter-icon"><i class="bx bx-cog"></i></span>
                        <span class="filter-name">System</span>
                    </a>
                </div>
            </div>
        </div>
        
        <!-- Notifications List -->
        <div class="notifications-content">
            <div class="notifications-card">
                <?php if (count($notifications) > 0): ?>
                <div class="notifications-header">
                    <h5>
                        <span><?= count($notifications) ?> Notifications</span>
                        <?php if ($unreadCount > 0): ?>
                        <span class="unread-count"><?= $unreadCount ?> unread</span>
                        <?php endif; ?>
                    </h5>
                </div>
                <div class="notifications-list">
                    <?php foreach ($notifications as $notification): 
                        $properties = getNotificationTypeInfo($notification['type']);
                        $actionLink = getNotificationActionUrl($notification);
                    ?>
                    <div class="notification-item <?= $notification['is_read'] ? '' : 'unread' ?>">
                        <div class="notification-icon bg-<?= $properties['color'] ?>-light text-<?= $properties['color'] ?>">
                            <i class="bx <?= $properties['icon'] ?>"></i>
                        </div>
                        <div class="notification-content">
                            <h6 class="notification-title"><?= $notification['title'] ?></h6>
                            <p class="notification-text"><?= $notification['content'] ?></p>
                            <?php if ($notification['related_item_name']): ?>
                            <div class="related-items">
                                <a href="<?= $actionLink ?>" class="related-item">
                                    <i class="bx bx-link-alt"></i> <?= $notification['related_item_name'] ?>
                                </a>
                            </div>
                            <?php endif; ?>
                            <div class="notification-time">
                                <?= formatNotificationTime($notification['created_at']) ?>
                            </div>
                        </div>
                        <div class="notification-actions">
                            <?php if (!$notification['is_read']): ?>
                            <a href="?<?= http_build_query(array_merge($_GET, ['mark_read' => $notification['notification_id']])) ?>" class="btn btn-icon" title="Mark as Read">
                                <i class="bx bx-check"></i>
                            </a>
                            <?php endif; ?>
                            <?php if ($actionLink != '#'): ?>
                            <a href="<?= $actionLink ?>" class="btn btn-sm btn-primary rounded-pill px-3">View</a>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                
                <!-- Pagination if needed -->
                <?php if (count($notifications) >= 50): ?>
                <div class="notifications-pagination">
                    <nav>
                        <ul class="pagination justify-content-center mt-4">
                            <li class="page-item disabled"><a class="page-link" href="#">Previous</a></li>
                            <li class="page-item active"><a class="page-link" href="#">1</a></li>
                            <li class="page-item"><a class="page-link" href="#">2</a></li>
                            <li class="page-item"><a class="page-link" href="#">3</a></li>
                            <li class="page-item"><a class="page-link" href="#">Next</a></li>
                        </ul>
                    </nav>
                </div>
                <?php endif; ?>
                <?php else: ?>
                <div class="empty-state">
                    <i class="bx bx-bell-off"></i>
                    <h4>No Notifications</h4>
                    <p>You don't have any notifications at this time. We'll notify you about important course updates, assignments, and messages.</p>
                    <a href="dashboard.php" class="btn btn-primary">Go to Dashboard</a>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- JavaScript for Interactive Notifications -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Handle notification click to mark as read
    const notifications = document.querySelectorAll('.notification-item.unread');
    notifications.forEach(notification => {
        notification.style.cursor = 'pointer';
        notification.addEventListener('click', function(e) {
            // Don't trigger if clicking on action buttons
            if (e.target.closest('.notification-actions')) return;
            
            // Find the mark as read button and click it
            const markReadBtn = this.querySelector('a[href*="mark_read="]');
            if (markReadBtn) {
                markReadBtn.click();
            }
        });
    });
    
    // Handle mark as read button confirmation
    const markReadButtons = document.querySelectorAll('a[href*="mark_read=all"]');
    markReadButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            if (!confirm('Are you sure you want to mark all notifications as read?')) {
                e.preventDefault();
            }
        });
    });
});
</script>


