<?php
/**
 * Payment Success Handler with Referral Processing
 * Panadite Academy - Processes referral rewards after successful payments
 */

require_once '../config/db_connect.php';
require_once '../includes/referral_processor.php';

/**
 * Process referral rewards after successful payment
 */
function processPaymentReferralRewards($user_id, $purchase_id, $course_ids, $total_amount) {
    global $conn;
    
    if (!file_exists('../includes/referral_processor.php')) {
        return ['success' => false, 'message' => 'Referral processor not available'];
    }
    
    // FIX: Update any 'pending' referrals to 'registered' status for this user
    $fix_status_query = "UPDATE referrals SET status = 'registered' WHERE referred_user_id = ? AND status = 'pending'";
    $stmt = $conn->prepare($fix_status_query);
    if ($stmt) {
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $updated_rows = $stmt->affected_rows;
        if ($updated_rows > 0) {
            error_log("Fixed referral status: Updated {$updated_rows} referrals from 'pending' to 'registered' for user {$user_id}");
        }
    }
    
    $referral_processor = new ReferralProcessor($conn);
    $rewards_processed = [];
    
    // Process rewards for each course purchased
    foreach ($course_ids as $course_id) {
        $result = $referral_processor->processPurchaseRewards($user_id, $purchase_id, $course_id, $total_amount);
        if ($result['success']) {
            $rewards_processed[] = $result;
        }
    }
    
    // Apply referral discount if it was used
    if (isset($_SESSION['applied_referral_discount'])) {
        $discount_info = $_SESSION['applied_referral_discount'];
        $referral_processor->applyStudentDiscount($discount_info['discount_id'], $purchase_id);
        unset($_SESSION['applied_referral_discount']);
    }
    
    return [
        'success' => true,
        'rewards_processed' => count($rewards_processed),
        'details' => $rewards_processed
    ];
}

/**
 * Send referral reward notifications
 */
function sendReferralNotifications($user_id, $purchase_amount) {
    global $conn;
    
    // Get referrals where this user was the referred person
    $query = "
        SELECT r.*, 
               referrer.first_name as referrer_first_name, 
               referrer.last_name as referrer_last_name,
               referrer.email as referrer_email
        FROM referrals r 
        JOIN users referrer ON r.referrer_id = referrer.user_id
        WHERE r.referred_user_id = ? AND r.status = 'completed'
    ";
    
    $stmt = $conn->prepare($query);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $referrals = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    
    foreach ($referrals as $referral) {
        // In a real application, you would send email notifications here
        // For now, we'll just log the notification
        error_log("Referral reward notification: {$referral['referrer_first_name']} {$referral['referrer_last_name']} earned rewards from referral");
    }
}

// If this file is called directly (for testing)
if (basename($_SERVER['PHP_SELF']) == basename(__FILE__)) {
    // Test the referral processing
    if (isset($_GET['test']) && $_GET['test'] === '1') {
        echo "<h2>🧪 Testing Referral Processing</h2>";
        
        // Test data
        $test_user_id = 1;
        $test_purchase_id = 123;
        $test_course_ids = [1, 2];
        $test_amount = 1000;
        
        $result = processPaymentReferralRewards($test_user_id, $test_purchase_id, $test_course_ids, $test_amount);
        
        echo "<pre>";
        print_r($result);
        echo "</pre>";
        
        echo "<p>Test completed. Check the database for referral reward entries.</p>";
    } else {
        echo "<h2>Payment Success Handler</h2>";
        echo "<p>This file handles referral processing after successful payments.</p>";
        echo "<p><a href='?test=1'>Run Test</a></p>";
    }
}
?>
