<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';

// Require student role to access this page
requireRole('student');

// Include PayFast functions
require_once '../payments/payfast.php';

// Include database connection
require_once '../config/db_connect.php';

// Include email service for purchase confirmation emails - EXACT SAME AS ADMIN COURSE PAGE
require_once '../config/fast_email_service.php';

// Bridge database connections - Email service needs PDO, but this page uses MySQLi - EXACT SAME AS ADMIN COURSE PAGE
if (!isset($pdo)) {
    try {
        // Create PDO connection using the same credentials as MySQLi
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        error_log("PDO connection created successfully for email service");
    } catch (Exception $e) {
        error_log("Failed to create PDO connection for email service: " . $e->getMessage());
        $pdo = null;
    }
}

// Include revenue sharing processor - use absolute path to ensure it's found
$purchase_processor_path = __DIR__ . '/../includes/purchase_processor.php';
if (file_exists($purchase_processor_path)) {
    require_once $purchase_processor_path;
} else {
    error_log("Purchase processor not found at: $purchase_processor_path");
}

// Initialize variables
$success = false;
$course = null;
$error = '';

// Get current user
$user_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 0;

// Function to create notifications
function createNotification($user_id, $type, $message) {
    global $conn;
    
    $insertNotification = "
        INSERT INTO notifications (user_id, type, message, created_at) 
        VALUES (?, ?, ?, NOW())
    ";
    
    $stmt = $conn->prepare($insertNotification);
    $stmt->bind_param("iss", $user_id, $type, $message);
    $stmt->execute();
}

/**
 * Send purchase confirmation notification to student - EXACT SAME APPROACH AS ADMIN COURSE EMAILS
 */
function sendPurchaseConfirmationEmail($user_id, $course_id, $payment_id, $amount) {
    global $conn;
    
    try {
        // Get course and student details - SEND EMAIL TO STUDENT WHO PURCHASED THE COURSE
        $stmt = $conn->prepare("
            SELECT c.title, c.price, 
                   u.first_name, u.last_name, u.email,
                   t.first_name as teacher_first_name, t.last_name as teacher_last_name
            FROM courses c 
            JOIN users u ON u.user_id = ?
            JOIN users t ON c.teacher_id = t.user_id 
            WHERE c.course_id = ?
        ");
        $stmt->bind_param("ii", $user_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $purchase = $result->fetch_assoc();
        
        if (!$purchase) {
            error_log("Course or user not found for purchase confirmation email: course_id=$course_id, user_id=$user_id");
            return false;
        }
        
        $student_name = $purchase['first_name'] . ' ' . $purchase['last_name'];
        $student_email = $purchase['email'];
        $course_title = $purchase['title'];
        $teacher_name = $purchase['teacher_first_name'] . ' ' . $purchase['teacher_last_name'];
        $purchase_amount = number_format(floatval($amount), 2);
        $purchase_date = date('M j, Y \a\t g:i A');
        
        // Create email content - EXACT SAME STRUCTURE AS WORKING ADMIN EMAILS
        $subject = "🎉 Purchase Confirmed: \"$course_title\" - Access Now Available!";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#28a745; color:white; padding:20px; text-align:center;'>
                <h1>🎉 Purchase Confirmed!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Congratulations, " . htmlspecialchars($student_name) . "! 🎓</h2>
                <p>Your purchase of <strong>\"" . htmlspecialchars($course_title) . "\"</strong> has been confirmed and you now have full access!</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Purchase Details:</h3>
                    <p><strong>Course:</strong> " . htmlspecialchars($course_title) . "</p>
                    <p><strong>Instructor:</strong> " . htmlspecialchars($teacher_name) . "</p>
                    <p><strong>Amount Paid:</strong> R" . $purchase_amount . "</p>
                    <p><strong>Purchase Date:</strong> " . $purchase_date . "</p>
                    <p><strong>Payment ID:</strong> " . htmlspecialchars($payment_id) . "</p>
                </div>
                <div style='background:#e8f5e8; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h4>🚀 Ready to Start Learning?</h4>
                    <p>You can now access all course materials, videos, and resources immediately!</p>
                    <p>Start your learning journey right away and unlock your potential.</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/student/my-courses.php' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        Access Your Course Now
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending - EXACT SAME PATTERN AS WORKING ADMIN EMAILS
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($student_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($student_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending purchase confirmation email: " . $e->getMessage());
        return false;
    }
}

/**
 * Send course sale notification to teacher - EXACT COPY FROM ADMIN COURSES.PHP
 */
function sendTeacherSaleNotificationEmail($course_id) {
    global $conn;
    
    try {
        // COMPREHENSIVE DEBUGGING - SAME APPROACH THAT FIXED WITHDRAWAL APPROVALS
        error_log("=== TEACHER EMAIL DEBUG START ===");
        error_log("TEACHER EMAIL DEBUG: Function called with course_id: $course_id");
        error_log("TEACHER EMAIL DEBUG: MySQLi connection available: " . (isset($conn) && $conn ? "YES" : "NO"));
        error_log("TEACHER EMAIL DEBUG: sendEmailNow function exists: " . (function_exists('sendEmailNow') ? "YES" : "NO"));
        
        // Get course and teacher details - EXACT COPY FROM ADMIN COURSES.PHP
        $stmt = $conn->prepare("
            SELECT c.title, c.description, c.price, c.created_at, 
                   u.first_name, u.last_name, u.email, u.user_id,
                   cat.category_name
            FROM courses c 
            JOIN users u ON c.teacher_id = u.user_id 
            LEFT JOIN categories cat ON c.category_id = cat.category_id 
            WHERE c.course_id = ?
        ");
        
        error_log("TEACHER EMAIL DEBUG: SQL query prepared successfully");
        
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $course = $result->fetch_assoc();
        
        error_log("TEACHER EMAIL DEBUG: Query executed, rows found: " . ($result->num_rows));
        
        if (!$course) {
            error_log("TEACHER EMAIL DEBUG: CRITICAL ERROR - Course not found for course_id: $course_id");
            return false;
        }
        
        error_log("TEACHER EMAIL DEBUG: Course found - Title: " . ($course['title'] ?? 'NULL'));
        error_log("TEACHER EMAIL DEBUG: Teacher name: " . ($course['first_name'] ?? 'NULL') . " " . ($course['last_name'] ?? 'NULL'));
        error_log("TEACHER EMAIL DEBUG: Teacher email: " . ($course['email'] ?? 'NULL'));
        
        $teacher_name = $course['first_name'] . ' ' . $course['last_name'];
        $teacher_email = $course['email'];
        $course_title = $course['title'];
        $category = $course['category_name'] ?? 'General';
        $price = floatval($course['price']);
        $created_date = date('M j, Y', strtotime($course['created_at']));
        
        // Create email content
        $subject = "🎉 Course Sale: \"$course_title\" - New Student Enrolled!";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#f39c12; color:white; padding:20px; text-align:center;'>
                <h1>🎉 Course Sale!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Congratulations, " . htmlspecialchars($teacher_name) . "! 💰</h2>
                <p>Great news! A student has just purchased your course <strong>\"" . htmlspecialchars($course_title) . "\"</strong>!</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Sale Details:</h3>
                    <p><strong>Category:</strong> " . htmlspecialchars($category) . "</p>
                    <p><strong>Price:</strong> R" . number_format($price, 2) . "</p>
                    <p><strong>Sale Date:</strong> " . date('M j, Y') . "</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/teacher/courses.php' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        View Dashboard
                    </a>
                </p>
            </div>
        </div>";
        
        error_log("TEACHER EMAIL DEBUG: About to send email to: $teacher_email");
        error_log("TEACHER EMAIL DEBUG: Email subject: $subject");
        
        // Try direct email sending - EXACT COPY FROM ADMIN COURSES.PHP
        if (function_exists('sendEmailNow')) {
            error_log("TEACHER EMAIL DEBUG: sendEmailNow function is available, calling it...");
            $result = sendEmailNow($teacher_email, $subject, $message);
            error_log("TEACHER EMAIL DEBUG: sendEmailNow returned: " . ($result ? "TRUE" : "FALSE"));
            if ($result) {
                error_log("TEACHER EMAIL DEBUG: SUCCESS - Email sent via sendEmailNow!");
                error_log("=== TEACHER EMAIL DEBUG END ===");
                return true;
            }
            error_log("TEACHER EMAIL DEBUG: sendEmailNow failed, trying mail() fallback...");
        } else {
            error_log("TEACHER EMAIL DEBUG: sendEmailNow function NOT available!");
        }
        
        // Fallback to native PHP mail - EXACT COPY FROM ADMIN COURSES.PHP
        error_log("TEACHER EMAIL DEBUG: Trying native mail() function...");
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        $mail_result = mail($teacher_email, $subject, $message, $headers);
        error_log("TEACHER EMAIL DEBUG: mail() function returned: " . ($mail_result ? "TRUE" : "FALSE"));
        
        if ($mail_result) {
            error_log("TEACHER EMAIL DEBUG: SUCCESS - Email sent via mail()!");
        } else {
            error_log("TEACHER EMAIL DEBUG: FAILED - Both sendEmailNow and mail() failed!");
        }
        
        error_log("=== TEACHER EMAIL DEBUG END ===");
        return $mail_result;
        
    } catch (Exception $e) {
        error_log("TEACHER EMAIL DEBUG: EXCEPTION - " . $e->getMessage());
        error_log("=== TEACHER EMAIL DEBUG END ===");
        return false;
    }
}

// Log all GET parameters for debugging
error_log("Payment success page accessed with parameters: " . json_encode($_GET));

// Process the payment return - PayFast sends different parameters
// PayFast may send: pf_payment_id, payment_status, m_payment_id, etc.
$pf_payment_id = $_GET['pf_payment_id'] ?? '';
$m_payment_id = $_GET['m_payment_id'] ?? '';
$payment_status = $_GET['payment_status'] ?? '';

// Try to extract course info from our custom payment ID format: PFA-timestamp-userid-random
if ($m_payment_id && preg_match('/PFA-\d+-(\d+)-\d+/', $m_payment_id, $matches)) {
    $payment_user_id = $matches[1];
    
    // Verify this matches the current user
    if ($user_id == $payment_user_id) {
        // For successful payments, we'll assume it worked if PayFast returned here
        // The ITN handler will process the actual enrollment
        
        // Extract course information from PayFast custom fields or session
        $course_id = null;
        $amount = 0;
        
        // Method 1: Check for course_id in PayFast custom fields
        $course_id = $_GET['custom_str1'] ?? $_GET['item_name'] ?? null;
        
        // Method 2: Check session for recently selected course (multiple possible session keys)
        if (!$course_id && isset($_SESSION['payment_course_id'])) {
            $course_id = $_SESSION['payment_course_id'];
        }
        if (!$course_id && isset($_SESSION['last_viewed_course'])) {
            $course_id = $_SESSION['last_viewed_course'];
        }
        if (!$course_id && isset($_SESSION['selected_course_id'])) {
            $course_id = $_SESSION['selected_course_id'];
        }
        
        // Method 3: Check for course_id in the payment reference
        if (!$course_id && $m_payment_id) {
            // Look for course info in recent payment sessions or database
            $payment_lookup = "SELECT item_id, amount FROM payments WHERE pf_payment_id = ? OR payment_id = ? LIMIT 1";
            $stmt = $conn->prepare($payment_lookup);
            $stmt->bind_param("ss", $pf_payment_id, $m_payment_id);
            $stmt->execute();
            $payment_result = $stmt->get_result();
            if ($payment_result->num_rows > 0) {
                $payment_data = $payment_result->fetch_assoc();
                $course_id = $payment_data['item_id'];
                $amount = $payment_data['amount'];
            }
        }
        
        // If we have course_id, get the course details and correct price
        if ($course_id) {
            $courseQuery = "SELECT course_id, title, price FROM courses WHERE course_id = ?";
            $stmt = $conn->prepare($courseQuery);
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
        } else {
            // Fallback: Try to identify course by amount if provided
            $amount = $_GET['amount_gross'] ?? $_GET['amount'] ?? 199.00;
            $courseQuery = "SELECT course_id, title, price FROM courses WHERE price = ? ORDER BY course_id DESC LIMIT 1";
            $stmt = $conn->prepare($courseQuery);
            $stmt->bind_param("d", $amount);
            $stmt->execute();
            $result = $stmt->get_result();
        }
        
        if ($result->num_rows > 0) {
            $courseData = $result->fetch_assoc();
            $course = ['title' => $courseData['title'], 'course_id' => $courseData['course_id']];
            
            // Use the correct course price from database, not the guessed amount
            $amount = $courseData['price'];
            
            // Record the payment in our database
            $recordPayment = "
                INSERT IGNORE INTO payments (payment_id, pf_payment_id, user_id, item_type, item_id, amount, payment_status, created_at) 
                VALUES (?, ?, ?, 'course', ?, ?, 'COMPLETE', NOW())
            ";
            
            $stmt = $conn->prepare($recordPayment);
            $stmt->bind_param("ssidd", $m_payment_id, $pf_payment_id, $user_id, $courseData['course_id'], $amount);
            $stmt->execute();
            
            // Check if user is already enrolled
            $enrollmentQuery = "SELECT enrollment_id FROM enrollments WHERE user_id = ? AND course_id = ?";
            $stmt = $conn->prepare($enrollmentQuery);
            $stmt->bind_param("ii", $user_id, $courseData['course_id']);
            $stmt->execute();
            $enrollmentResult = $stmt->get_result();
            
            if ($enrollmentResult->num_rows == 0) {
                // Enroll the user with proper revenue tracking
                $enrollQuery = "
                    INSERT INTO enrollments (user_id, course_id, course_price, enrolled_date, progress) 
                    VALUES (?, ?, ?, NOW(), 0)
                ";
                
                $stmt = $conn->prepare($enrollQuery);
                $stmt->bind_param("iid", $user_id, $courseData['course_id'], $amount);
                
                if ($stmt->execute()) {
                    $enrollment_id = $conn->insert_id;
                    
                    // Process revenue sharing for this purchase
                    try {
                        $purchase_processor = new PurchaseProcessor($conn);
                        $revenue_result = $purchase_processor->processPurchase(
                            $user_id,
                            $courseData['course_id'], 
                            $amount,
                            'payfast',
                            $pf_payment_id
                        );
                        
                        if ($revenue_result['success']) {
                            // Update enrollment with purchase_id
                            $update_enrollment = "UPDATE enrollments SET purchase_id = ? WHERE enrollment_id = ?";
                            $stmt = $conn->prepare($update_enrollment);
                            $stmt->bind_param("ii", $revenue_result['purchase_id'], $enrollment_id);
                            $stmt->execute();
                            
                            $success = true;
                            
                            // Create success notification with revenue info
                            createNotification($user_id, 'enrollment', "Successfully enrolled in course: " . $course['title'] . " after payment confirmation.");
                            
                            // Create payment confirmation notification
                            createNotification($user_id, 'payment', "Payment of R" . number_format($amount, 2) . " confirmed for course: " . $course['title'] . ". Revenue has been distributed.");
                            
                            // Log successful revenue processing
                            error_log("Revenue sharing processed successfully for purchase ID: " . $revenue_result['purchase_id']);
                            
                            // COMPREHENSIVE DEBUGGING - SAME APPROACH THAT SUCCESSFULLY FIXED WITHDRAWAL APPROVALS
                            error_log("=== PAYMENT SUCCESS DEBUGGING START ===");
                            error_log("PAYMENT SUCCESS DEBUG: User ID: $user_id");
                            error_log("PAYMENT SUCCESS DEBUG: Course ID: " . $courseData['course_id']);
                            error_log("PAYMENT SUCCESS DEBUG: Course Title: " . $courseData['title']);
                            error_log("PAYMENT SUCCESS DEBUG: Amount: $amount");
                            error_log("PAYMENT SUCCESS DEBUG: Payment ID: " . ($pf_payment_id ?: 'NONE'));
                            error_log("PAYMENT SUCCESS DEBUG: PDO available: " . (isset($pdo) && $pdo ? "YES" : "NO"));
                            error_log("PAYMENT SUCCESS DEBUG: MySQLi conn available: " . (isset($conn) && $conn ? "YES" : "NO"));
                            error_log("PAYMENT SUCCESS DEBUG: sendEmailNow function exists: " . (function_exists('sendEmailNow') ? "YES" : "NO"));
                            error_log("PAYMENT SUCCESS DEBUG: fast_email_service.php included: " . (function_exists('sendEmailNow') ? "YES" : "NO"));
                            
                            // Get student email for debugging
                            $student_email_query = "SELECT email, first_name, last_name FROM users WHERE user_id = ?";
                            $stmt_debug = $conn->prepare($student_email_query);
                            if ($stmt_debug) {
                                $stmt_debug->bind_param("i", $user_id);
                                $stmt_debug->execute();
                                $student_result = $stmt_debug->get_result();
                                if ($student_data = $student_result->fetch_assoc()) {
                                    error_log("PAYMENT SUCCESS DEBUG: Student email: " . $student_data['email']);
                                    error_log("PAYMENT SUCCESS DEBUG: Student name: " . $student_data['first_name'] . ' ' . $student_data['last_name']);
                                } else {
                                    error_log("PAYMENT SUCCESS DEBUG: CRITICAL ERROR - Student not found in database!");
                                }
                            } else {
                                error_log("PAYMENT SUCCESS DEBUG: CRITICAL ERROR - Could not prepare student query!");
                            }
                            
                            // Send purchase confirmation email - EXACT SAME APPROACH AS ADMIN COURSE EMAILS
                            error_log("PAYMENT SUCCESS DEBUG: About to call sendPurchaseConfirmationEmail...");
                            $email_result = sendPurchaseConfirmationEmail($user_id, $courseData['course_id'], $pf_payment_id ?: ('SUCCESS_' . time()), $amount);
                            error_log("PAYMENT SUCCESS DEBUG: Email function returned: " . ($email_result ? "TRUE" : "FALSE"));
                            error_log("PAYMENT SUCCESS DEBUG: Purchase confirmation email sent: " . ($email_result ? "SUCCESS" : "FAILED") . " for course " . $courseData['course_id'] . " to user $user_id");
                            
                            // Send teacher sale notification email - EXACT COPY FROM ADMIN COURSES.PHP
                            error_log("PAYMENT SUCCESS DEBUG: About to call sendTeacherSaleNotificationEmail...");
                            $teacher_email_result = sendTeacherSaleNotificationEmail($courseData['course_id']);
                            error_log("PAYMENT SUCCESS DEBUG: Teacher email function returned: " . ($teacher_email_result ? "TRUE" : "FALSE"));
                            error_log("PAYMENT SUCCESS DEBUG: Teacher sale notification email sent: " . ($teacher_email_result ? "SUCCESS" : "FAILED") . " for course " . $courseData['course_id'] . " to teacher");
                            
                            // Process referral commissions - THIS WAS MISSING!
                            error_log("REFERRAL DEBUG: Starting commission processing for user $user_id, course " . $courseData['course_id'] . ", amount $amount");
                            
                            require_once '../student/payment-success-handler.php';
                            $referral_result = processPaymentReferralRewards(
                                $user_id, 
                                $revenue_result['purchase_id'], 
                                [$courseData['course_id']], 
                                $amount
                            );
                            
                            if ($referral_result && $referral_result['success']) {
                                error_log("REFERRAL DEBUG: Successfully processed {$referral_result['rewards_processed']} referral rewards");
                            } else {
                                error_log("REFERRAL DEBUG: Referral processing failed: " . ($referral_result['message'] ?? 'Unknown error'));
                            }
                            
                            error_log("=== PAYMENT SUCCESS DEBUGGING END ===");
                        } else {
                            // Enrollment succeeded but revenue processing failed - log for manual processing
                            error_log("Revenue sharing failed for enrollment $enrollment_id: " . $revenue_result['error']);
                            
                            $success = true; // Still show success to user
                            createNotification($user_id, 'enrollment', "Successfully enrolled in course: " . $course['title'] . " after payment confirmation.");
                            createNotification($user_id, 'payment', "Payment of R" . number_format($amount, 2) . " confirmed for course: " . $course['title']);
                        }
                    } catch (Exception $e) {
                        // Revenue processing failed but enrollment succeeded
                        error_log("Revenue sharing exception for enrollment $enrollment_id: " . $e->getMessage());
                        
                        $success = true; // Still show success to user
                        createNotification($user_id, 'enrollment', "Successfully enrolled in course: " . $course['title'] . " after payment confirmation.");
                        createNotification($user_id, 'payment', "Payment of R" . number_format($amount, 2) . " confirmed for course: " . $course['title']);
                    }
                } else {
                    $error = "Payment successful but enrollment failed. Please contact support.";
                }
            } else {
                $success = true; // Already enrolled
            }
        } else {
            // If we can't find the specific course, but payment was successful,
            // set a generic success message
            $success = true;
            $course = ['title' => 'Course', 'course_id' => 0];
            createNotification($user_id, 'payment', "Payment of R" . number_format($amount, 2) . " was successful. Course enrollment may take a few minutes to process.");
        }
    } else {
        $error = "Invalid payment information received. User ID mismatch.";
    }
} else if ($pf_payment_id || $m_payment_id) {
    // If we have payment IDs but couldn't parse the format, still consider it successful
    // This handles cases where PayFast returns successfully but with different ID formats
    $success = true;
    $course = ['title' => 'Course', 'course_id' => 0];
    createNotification($user_id, 'payment', "Payment was successful. Course enrollment may take a few minutes to process.");
} else {
    $error = "Payment was not completed successfully.";
}

// Include header
include_once 'components/header.php';
?>

<style>
.success-container {
    max-width: 600px;
    margin: 50px auto;
    padding: 20px;
    text-align: center;
}

.success-icon {
    width: 80px;
    height: 80px;
    background: linear-gradient(45deg, #28a745, #20c997);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 20px;
    font-size: 40px;
    color: white;
}

.error-icon {
    width: 80px;
    height: 80px;
    background: linear-gradient(45deg, #dc3545, #c82333);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 20px;
    font-size: 40px;
    color: white;
}

.success-card {
    background: white;
    border-radius: 15px;
    padding: 40px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    border: 1px solid #e9ecef;
}

.celebration {
    animation: celebration 0.8s ease-in-out;
}

@keyframes celebration {
    0% { transform: scale(0.8); opacity: 0; }
    50% { transform: scale(1.05); }
    100% { transform: scale(1); opacity: 1; }
}

.action-buttons {
    display: flex;
    gap: 15px;
    justify-content: center;
    flex-wrap: wrap;
    margin-top: 30px;
}

.action-buttons .btn {
    min-width: 180px;
    padding: 12px 24px;
    font-weight: 600;
    border-radius: 8px;
    transition: all 0.3s ease;
}

.action-buttons .btn:hover {
    transform: translateY(-2px);
}
</style>

<div class="main-wrapper">
    <div class="success-container">
        <?php if ($success): ?>
            <div class="success-card celebration">
                <div class="success-icon">
                    <i class="bx bx-check"></i>
                </div>
                
                <h2 class="text-success mb-3">Payment Successful!</h2>
                
                <p class="lead mb-4">
                    Congratulations! You have successfully purchased and enrolled in:
                </p>
                
                <h4 class="text-primary mb-4"><?= htmlspecialchars($course['title']) ?></h4>
                
                <div class="alert alert-success">
                    <i class="bx bx-info-circle me-2"></i>
                    <strong>What's Next?</strong><br>
                    You can now access your course content and start learning immediately. 
                    Check your notifications for confirmation details.
                </div>
                
                <div class="action-buttons">
                    <a href="course-detail.php?id=<?= $course['course_id'] ?>" class="btn btn-primary">
                        <i class="bx bx-play-circle me-2"></i>
                        Start Learning
                    </a>
                    
                    <a href="courses.php" class="btn btn-outline-secondary">
                        <i class="bx bx-book-content me-2"></i>
                        My Courses
                    </a>
                </div>
                
                <div class="mt-4">
                    <small class="text-muted">
                        <i class="bx bx-receipt me-1"></i>
                        Payment confirmation has been sent to your registered email address.
                    </small>
                </div>
            </div>
            
        <?php else: ?>
            <div class="success-card">
                <div class="error-icon">
                    <i class="bx bx-x"></i>
                </div>
                
                <h2 class="text-danger mb-3">Payment Issue</h2>
                
                <p class="lead mb-4"><?= htmlspecialchars($error) ?></p>
                
                <div class="alert alert-warning">
                    <i class="bx bx-warning me-2"></i>
                    <strong>Need Help?</strong><br>
                    If you believe you have completed the payment successfully, please contact our support team 
                    with your payment reference number.
                </div>
                
                <div class="action-buttons">
                    <a href="courses.php" class="btn btn-primary">
                        <i class="bx bx-arrow-back me-2"></i>
                        Back to Courses
                    </a>
                    
                    <a href="contact.php" class="btn btn-outline-secondary">
                        <i class="bx bx-support me-2"></i>
                        Contact Support
                    </a>
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>

<script>
// Add confetti animation for successful payments
<?php if ($success): ?>
document.addEventListener('DOMContentLoaded', function() {
    // Create confetti effect
    function createConfetti() {
        const colors = ['#007bff', '#28a745', '#ffc107', '#dc3545', '#6f42c1'];
        
        for (let i = 0; i < 50; i++) {
            setTimeout(() => {
                const confetti = document.createElement('div');
                confetti.style.position = 'fixed';
                confetti.style.width = '10px';
                confetti.style.height = '10px';
                confetti.style.background = colors[Math.floor(Math.random() * colors.length)];
                confetti.style.left = Math.random() * window.innerWidth + 'px';
                confetti.style.top = '-10px';
                confetti.style.zIndex = '9999';
                confetti.style.pointerEvents = 'none';
                confetti.style.borderRadius = '50%';
                
                document.body.appendChild(confetti);
                
                const animation = confetti.animate([
                    { transform: 'translateY(0px) rotate(0deg)', opacity: 1 },
                    { transform: `translateY(${window.innerHeight + 100}px) rotate(720deg)`, opacity: 0 }
                ], {
                    duration: 3000,
                    easing: 'cubic-bezier(0.25, 0.46, 0.45, 0.94)'
                });
                
                animation.addEventListener('finish', () => {
                    confetti.remove();
                });
            }, i * 100);
        }
    }
    
    setTimeout(createConfetti, 500);
});
<?php endif; ?>
</script>

<?php include_once 'components/footer.php'; ?>
