<?php
/**
 * Enhanced Student Referral Program - Panadite Academy
 * Clean, modern referral system with rewards and beautiful UI
 */

// Set current page for navigation
$currentPage = 'referrals';

// Authentication - Require student role
require_once '../auth/functions.php';
requireRole('student');

require_once '../config/db_connect.php';

require_once '../includes/referral_processor.php';

$student_id = $_SESSION['user_id'];

// Get user information
$user_query = "SELECT first_name, last_name, email, username FROM users WHERE user_id = ?";
$stmt = $conn->prepare($user_query);
$stmt->bind_param("i", $student_id);
$stmt->execute();
$user_data = $stmt->get_result()->fetch_assoc();

$student_name = trim($user_data['first_name'] . ' ' . $user_data['last_name']);
$student_email = $user_data['email'];

// Try to use secure referral system, fallback to legacy if it fails
$referral_code = null;

try {
    // Include secure referral generator
    require_once '../includes/secure_referral_generator.php';
    
    // Generate secure referral code (new format) or get existing one
    $referral_code = SecureReferralGenerator::getOrCreateSecureCode($student_id, $conn);
} catch (Exception $e) {
    // Log the error but continue with legacy system
    error_log("Secure referral system error: " . $e->getMessage());
}

// Fallback to legacy format if secure generation fails
if (!$referral_code) {
    $referral_code = strtoupper(substr($user_data['first_name'], 0, 2) . substr($user_data['last_name'], 0, 2) . $student_id . rand(10, 99));
}

// Initialize referral processor with database connection
$referral_processor = new ReferralProcessor($conn);

// Get enhanced referral statistics (exclude incomplete records)
$stats_query = "
    SELECT 
        COUNT(*) as total_referrals,
        COUNT(CASE WHEN status = 'completed' THEN 1 END) as successful_referrals,
        COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_referrals,
        COUNT(CASE WHEN status = 'registered' THEN 1 END) as registered_referrals,
        SUM(reward_amount) as total_rewards_earned
    FROM referrals 
    WHERE referrer_id = ? AND referrer_type = 'student' 
    AND referred_user_id IS NOT NULL AND referred_user_id != '' 
    AND status IS NOT NULL AND status != ''
";
$stmt = $conn->prepare($stats_query);
if ($stmt) {
    $stmt->bind_param("i", $student_id);
    $stmt->execute();
    $stats = $stmt->get_result()->fetch_assoc();
} else {
    $stats = [
        'total_referrals' => 0,
        'successful_referrals' => 0,
        'pending_referrals' => 0,
        'registered_referrals' => 0,
        'total_rewards_earned' => 0
    ];
}

// Get available discounts
$available_discounts = $referral_processor->getStudentDiscounts($student_id);

// Get recent referrals with enhanced data (exclude incomplete records)
$recent_query = "
    SELECT 
        r.*,
        u.first_name,
        u.last_name,
        u.email,
    FROM referrals r
    LEFT JOIN users u ON r.referred_user_id = u.user_id
    WHERE r.referrer_id = ? AND r.referrer_type = 'student'
    AND r.referred_user_id IS NOT NULL AND r.referred_user_id != '' 
    AND r.status IS NOT NULL AND r.status != ''
    ORDER BY r.created_at DESC
    LIMIT 10
";
$stmt = $conn->prepare($recent_query);
if ($stmt) {
    $stmt->bind_param("i", $student_id);
    $stmt->execute();
    $recent_referrals = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
} else {
    $recent_referrals = [];
}

// Include header
include_once __DIR__ . '/components/header.php';
?>


<style>
    /* Clean Panadite Academy Referral Page Styling */
    body {
        background: #f8fafc;
        font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }

    .dashboard-content {
        padding: 1.5rem;
        max-width: 1200px;
        margin: 0 auto;
    }

    /* Beautiful Header Section */
    .referral-header {
        background: linear-gradient(135deg, #3366ff 0%, #4facfe 100%);
        color: white;
        padding: 3rem 2rem;
        border-radius: 1rem;
        margin-bottom: 2rem;
        text-align: center;
        position: relative;
        overflow: hidden;
    }

    .referral-header::before {
        content: '';
        position: absolute;
        top: -50%;
        right: -50%;
        width: 200px;
        height: 200px;
        background: rgba(255, 255, 255, 0.1);
        border-radius: 50%;
        animation: float 6s ease-in-out infinite;
    }

    .referral-header::after {
        content: '';
        position: absolute;
        bottom: -30%;
        left: -20%;
        width: 150px;
        height: 150px;
        background: rgba(255, 255, 255, 0.08);
        border-radius: 50%;
        animation: float 8s ease-in-out infinite reverse;
    }

    @keyframes float {
        0%, 100% { transform: translateY(0px) rotate(0deg); }
        50% { transform: translateY(-20px) rotate(180deg); }
    }

    .referral-header h1 {
        font-size: 2.5rem;
        font-weight: 700;
        margin-bottom: 0.5rem;
        position: relative;
        z-index: 2;
    }

    .referral-header p {
        font-size: 1.1rem;
        opacity: 0.9;
        margin-bottom: 0;
        position: relative;
        z-index: 2;
    }

    /* Stats Cards */
    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 1.5rem;
        margin-bottom: 2rem;
    }

    .stat-card {
        background: white;
        padding: 2rem;
        border-radius: 0.75rem;
        border: 1px solid #e8ecf4;
        text-align: center;
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }

    .stat-card:hover {
        transform: translateY(-4px);
        border-color: #3366ff;
        box-shadow: 0 10px 30px rgba(51, 102, 255, 0.15);
    }

    .stat-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: linear-gradient(90deg, #3366ff, #4facfe);
    }

    .stat-icon {
        width: 60px;
        height: 60px;
        background: linear-gradient(135deg, #3366ff, #4facfe);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto 1rem auto;
        font-size: 1.5rem;
        color: white;
    }

    .stat-number {
        font-size: 2.5rem;
        font-weight: 700;
        color: #1a2332;
        margin-bottom: 0.5rem;
    }

    .stat-label {
        font-size: 0.9rem;
        color: #6b7280;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    /* Referral Code Section */
    .referral-code-section {
        background: white;
        padding: 2rem;
        border-radius: 0.75rem;
        border: 1px solid #e8ecf4;
        margin-bottom: 2rem;
        text-align: center;
    }

    .referral-code-section h3 {
        font-size: 1.5rem;
        font-weight: 700;
        color: #1a2332;
        margin-bottom: 1rem;
    }

    .referral-code-display {
        background: #f0f4ff;
        border: 2px dashed #3366ff;
        border-radius: 0.5rem;
        padding: 1.5rem;
        margin: 1.5rem 0;
        position: relative;
    }

    .referral-code {
        font-size: 2rem;
        font-weight: 700;
        color: #3366ff;
        font-family: 'Courier New', monospace;
        letter-spacing: 2px;
        margin-bottom: 0.5rem;
    }

    .referral-link {
        font-size: 0.9rem;
        color: #6b7280;
        word-break: break-all;
        background: white;
        padding: 0.75rem;
        border-radius: 0.375rem;
        border: 1px solid #e8ecf4;
        margin-top: 1rem;
    }

    .copy-buttons {
        display: flex;
        gap: 1rem;
        justify-content: center;
        margin-top: 1.5rem;
    }

    .copy-btn {
        background: #3366ff;
        color: white;
        border: none;
        padding: 0.75rem 1.5rem;
        border-radius: 0.5rem;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }

    .copy-btn:hover {
        background: #2563eb;
        transform: translateY(-1px);
        box-shadow: 0 4px 12px rgba(51, 102, 255, 0.3);
    }

    .copy-btn.copied {
        background: #22c55e;
    }

    /* Share Options */
    .share-options {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 1rem;
        margin-top: 2rem;
    }

    /* Enhanced Rewards Section */
    .rewards-section {
        background: white;
        padding: 2rem;
        border-radius: 1rem;
        box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
        margin-bottom: 2rem;
        border: 1px solid #e5e7eb;
    }

    .rewards-section h3 {
        color: #374151;
        margin-bottom: 1.5rem;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }

    .discounts-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
        gap: 1rem;
        margin-bottom: 1rem;
    }

    .discount-card {
        background: linear-gradient(135deg, #10b981 0%, #059669 100%);
        color: white;
        padding: 1.5rem;
        border-radius: 12px;
        position: relative;
        overflow: hidden;
        box-shadow: 0 4px 6px rgba(16, 185, 129, 0.25);
    }

    .discount-card::before {
        content: '';
        position: absolute;
        top: -50%;
        right: -50%;
        width: 100px;
        height: 100px;
        background: rgba(255, 255, 255, 0.1);
        border-radius: 50%;
    }

    .discount-amount {
        font-size: 2rem;
        font-weight: bold;
        margin-bottom: 0.5rem;
    }

    .discount-details p {
        margin: 0.25rem 0;
    }

    .expires {
        opacity: 0.9;
        font-size: 0.9rem;
    }

    .discount-status {
        position: absolute;
        top: 0.5rem;
        right: 0.5rem;
        background: rgba(255, 255, 255, 0.2);
        padding: 0.25rem 0.75rem;
        border-radius: 12px;
        font-size: 0.8rem;
        font-weight: 600;
    }

    .discount-info {
        background: #f0f9ff;
        border: 1px solid #bae6fd;
        padding: 1rem;
        border-radius: 8px;
        display: flex;
        align-items: center;
        gap: 0.5rem;
        color: #0369a1;
        font-size: 0.9rem;
    }

    .share-btn {
        padding: 1rem;
        border-radius: 0.5rem;
        border: 1px solid #e8ecf4;
        background: white;
        text-decoration: none;
        color: #1a2332;
        display: flex;
        align-items: center;
        gap: 0.75rem;
        transition: all 0.3s ease;
    }

    .share-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        text-decoration: none;
        color: #1a2332;
    }

    .share-btn i {
        font-size: 1.5rem;
        width: 40px;
        height: 40px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 50%;
    }

    .share-btn.whatsapp i { background: #25d366; color: white; }
    .share-btn.email i { background: #ea4335; color: white; }
    .share-btn.twitter i { background: #1da1f2; color: white; }
    .share-btn.linkedin i { background: #0077b5; color: white; }

    /* Recent Referrals Table */
    .recent-referrals {
        background: white;
        border-radius: 0.75rem;
        border: 1px solid #e8ecf4;
        overflow: hidden;
    }

    .recent-referrals h3 {
        padding: 1.5rem 2rem;
        margin: 0;
        font-size: 1.3rem;
        font-weight: 700;
        color: #1a2332;
        border-bottom: 1px solid #e8ecf4;
        background: #fafbff;
    }

    .referrals-table {
        width: 100%;
        border-collapse: collapse;
    }

    .referrals-table th {
        background: #f8fafc;
        padding: 1rem 1.5rem;
        text-align: left;
        font-weight: 600;
        color: #6b7280;
        font-size: 0.8rem;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .referrals-table td {
        padding: 1rem 1.5rem;
        border-bottom: 1px solid #f1f5f9;
    }

    .referrals-table tr:hover {
        background: #fafbff;
    }

    .status-badge {
        padding: 0.375rem 0.75rem;
        border-radius: 1rem;
        font-size: 0.75rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .status-completed {
        background: #dcfce7;
        color: #166534;
    }

    .status-pending {
        background: #fef3c7;
        color: #92400e;
    }

    .status-expired {
        background: #fee2e2;
        color: #dc2626;
    }

    .empty-state {
        text-align: center;
        padding: 3rem 2rem;
        color: #6b7280;
    }

    .empty-state i {
        font-size: 3rem;
        color: #d1d5db;
        margin-bottom: 1rem;
    }

    /* Responsive Design */
    @media (max-width: 768px) {
        .dashboard-content {
            padding: 1rem;
        }
        
        .referral-header {
            padding: 2rem 1.5rem;
        }
        
        .referral-header h1 {
            font-size: 2rem;
        }
        
        .copy-buttons {
            flex-direction: column;
        }
        
        .share-options {
            grid-template-columns: 1fr;
        }
        
        .referrals-table {
            font-size: 0.875rem;
        }
        
        .referrals-table th,
        .referrals-table td {
            padding: 0.75rem 1rem;
        }
    }
</style>

<div class="dashboard-content">
    <!-- Beautiful Header -->
    <div class="referral-header">
        <h1>🎯 Referral Program</h1>
        <p>Share Panadite Academy with friends and earn rewards together!</p>
    </div>

    <!-- Stats Grid -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon">
                <i class='bx bx-user-plus'></i>
            </div>
            <div class="stat-number"><?= $stats['total_referrals'] ?></div>
            <div class="stat-label">Total Referrals</div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon">
                <i class='bx bx-check-circle'></i>
            </div>
            <div class="stat-number"><?= $stats['successful_referrals'] ?></div>
            <div class="stat-label">Successful</div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon">
                <i class='bx bx-time-five'></i>
            </div>
            <div class="stat-number"><?= $stats['pending_referrals'] ?></div>
            <div class="stat-label">Pending</div>
        </div>

        <div class="stat-card">
            <div class="stat-icon">
                <i class='bx bx-gift'></i>
            </div>
            <div class="stat-number">
                <?php if (!empty($available_discounts)): ?>
                    <?= $available_discounts[0]['discount_percentage'] ?>% OFF
                <?php else: ?>
                    No Discounts
                <?php endif; ?>
            </div>
            <div class="stat-label">
                <?php if (!empty($available_discounts)): ?>
                    Course Discount Available
                <?php else: ?>
                    Discount Credits Available
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Available Discounts Section -->
    <?php if (!empty($available_discounts)): ?>
    <div class="rewards-section">
        <h3>🎁 Your Available Discount Credits</h3>
        <div class="discounts-grid">
            <?php foreach ($available_discounts as $discount): ?>
            <div class="discount-card">
                <div class="discount-amount"><?= $discount['discount_percentage'] ?>% OFF</div>
                <div class="discount-details">
                    <p><strong>Save up to R<?= number_format($discount['discount_amount'], 2) ?></strong></p>
                    <p class="discount-type">Course Purchase Discount</p>
                    <p class="expires">Expires: <?= date('M j, Y', strtotime($discount['expires_at'])) ?></p>
                    <small>💡 Applied automatically at checkout</small>
                </div>
                <div class="discount-status active">Ready to Use</div>
            </div>
            <?php endforeach; ?>
        </div>
        <div class="discount-info">
            <i class='bx bx-info-circle'></i>
            <span>💳 These are course purchase discounts, not cash rewards. They'll be automatically applied when you buy courses!</span>
        </div>
    </div>
    <?php endif; ?>

    <!-- Referral Code Section -->
    <div class="referral-code-section">
        <h3>🔗 Your Referral Code</h3>
        <p>Share this code with friends to invite them to Panadite Academy</p>
        
        <div class="referral-code-display">
            <div class="referral-code" id="referralCode"><?= $referral_code ?></div>
            <div class="referral-link" id="referralLink">
                <?= $_SERVER['HTTP_HOST'] ?>/auth/register.php?ref=<?= $referral_code ?>
            </div>
        </div>

        <div class="copy-buttons">
            <button class="copy-btn" onclick="copyCode(this)">
                <i class='bx bx-copy'></i>
                Copy Code
            </button>
            <button class="copy-btn" onclick="copyLink(this)">
                <i class='bx bx-link'></i>
                Copy Link
            </button>
        </div>

        <!-- Share Options -->
        <div class="share-options">
            <a href="https://wa.me/?text=Join%20me%20at%20Panadite%20Academy!%20Use%20code%20<?= $referral_code ?>%20to%20get%20started:%20<?= $_SERVER['HTTP_HOST'] ?>/register.php?ref=<?= $referral_code ?>" class="share-btn whatsapp" target="_blank">
                <i class='bx bxl-whatsapp'></i>
                <span>Share on WhatsApp</span>
            </a>
            
            <a href="mailto:?subject=Join%20Panadite%20Academy&body=Hi!%20I%20wanted%20to%20invite%20you%20to%20join%20Panadite%20Academy.%20Use%20my%20referral%20code%20<?= $referral_code ?>%20to%20get%20started:%20<?= $_SERVER['HTTP_HOST'] ?>/register.php?ref=<?= $referral_code ?>" class="share-btn email">
                <i class='bx bx-envelope'></i>
                <span>Share via Email</span>
            </a>
            
            <a href="https://twitter.com/intent/tweet?text=Join%20me%20at%20Panadite%20Academy!%20Use%20code%20<?= $referral_code ?>%20<?= $_SERVER['HTTP_HOST'] ?>/register.php?ref=<?= $referral_code ?>" class="share-btn twitter" target="_blank">
                <i class='bx bxl-twitter'></i>
                <span>Share on Twitter</span>
            </a>
            
            <a href="https://linkedin.com/sharing/share-offsite/?url=<?= $_SERVER['HTTP_HOST'] ?>/register.php?ref=<?= $referral_code ?>" class="share-btn linkedin" target="_blank">
                <i class='bx bxl-linkedin'></i>
                <span>Share on LinkedIn</span>
            </a>
        </div>
    </div>

    <!-- Anonymous Referral Tracking -->
    <div class="recent-referrals">
        <h3>📊 Your Referral Activity</h3>
        
        <?php if (empty($recent_referrals)): ?>
            <div class="empty-state">
                <i class='bx bx-user-plus'></i>
                <h4>No referrals yet</h4>
                <p>Start sharing your referral code to see your activity here!</p>
            </div>
        <?php else: ?>
            <table class="referrals-table">
                <thead>
                    <tr>
                        <th>Referral</th>
                        <th>Status</th>
                        <th>Date</th>
                        <th>Reward</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    $referralCount = 1;
                    foreach ($recent_referrals as $referral): 
                    ?>
                        <tr>
                            <td>
                                <strong>Referral #<?= $referralCount ?></strong>
                            </td>
                            <td>
                                <span class="status-badge status-<?= $referral['status'] ?>">
                                    <?= ucfirst($referral['status']) ?>
                                </span>
                            </td>
                            <td><?= date('M j, Y', strtotime($referral['created_at'])) ?></td>
                            <td>
                                <?php if ($referral['status'] === 'completed'): ?>
                                    <span style="color: #22c55e; font-weight: 600;">+10 points</span>
                                <?php else: ?>
                                    <span style="color: #6b7280;">Pending</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php 
                    $referralCount++;
                    endforeach; 
                    ?>
                 </tbody>
            </table>
        <?php endif; ?>
        </div>
    </div>

</div>

<script>
function copyCode(element) {
    const code = document.getElementById('referralCode').textContent;
    const btn = element;
    
    navigator.clipboard.writeText(code).then(() => {
        const originalHTML = btn.innerHTML;
        btn.innerHTML = '<i class="bx bx-check"></i> Copied!';
        btn.classList.add('copied');
        
        setTimeout(() => {
            btn.innerHTML = originalHTML;
            btn.classList.remove('copied');
        }, 2000);
    }).catch(err => {
        console.error('Failed to copy code: ', err);
        btn.innerHTML = '<i class="bx bx-error"></i> Error';
        setTimeout(() => {
            btn.innerHTML = originalHTML;
        }, 2000);
    });
}

function copyLink(element) {
    const link = document.getElementById('referralLink').textContent;
    const fullLink = link.startsWith('http') ? link : 'https://' + link;
    const btn = element;
    
    navigator.clipboard.writeText(fullLink).then(() => {
        const originalHTML = btn.innerHTML;
        btn.innerHTML = '<i class="bx bx-check"></i> Copied!';
        btn.classList.add('copied');
        
        setTimeout(() => {
            btn.innerHTML = originalHTML;
            btn.classList.remove('copied');
        }, 2000);
    }).catch(err => {
        console.error('Failed to copy link: ', err);
        btn.innerHTML = '<i class="bx bx-error"></i> Error';
        setTimeout(() => {
            btn.innerHTML = originalHTML;
        }, 2000);
    });
}
</script>

<?php include_once __DIR__ . '/components/footer.php'; ?>
