<?php
require_once __DIR__ . '/../auth/functions.php';
requireRole('student');

require_once __DIR__ . '/../config/db_connect.php';

$student_id = $_SESSION['user_id'];
$message = '';
$messageType = '';

// Handle file upload
if ($_POST && isset($_FILES)) {
    $assessment_id = intval($_POST['assessment_id']);
    
    // Verify this assessment belongs to the current student
    $verify_stmt = $conn->prepare("SELECT sa.*, ar.request_id, c.title as course_title FROM sita_assessments sa JOIN accreditation_requests ar ON sa.accreditation_request_id = ar.request_id JOIN courses c ON sa.course_id = c.course_id WHERE sa.assessment_id = ? AND sa.student_id = ?");
    $verify_stmt->bind_param("ii", $assessment_id, $student_id);
    $verify_stmt->execute();
    $assessment = $verify_stmt->get_result()->fetch_assoc();
    
    if ($assessment && $assessment['status'] === 'passed') {
        // Create uploads directory if it doesn't exist
        $upload_dir = __DIR__ . '/../uploads/required-documents/' . $student_id . '/' . $assessment_id . '/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }
        
        $uploaded_files = [];
        $upload_errors = [];
        
        // Define required document types
        $required_docs = ['cv', 'qualifications', 'id_document', 'course_assignments'];
        
        foreach ($required_docs as $doc_type) {
            if (isset($_FILES[$doc_type]) && $_FILES[$doc_type]['error'] === UPLOAD_ERR_OK) {
                $file = $_FILES[$doc_type];
                $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
                
                // Validate file type
                $allowed_extensions = ['pdf', 'doc', 'docx', 'jpg', 'jpeg', 'png'];
                if (!in_array($file_extension, $allowed_extensions)) {
                    $upload_errors[] = ucfirst(str_replace('_', ' ', $doc_type)) . ": Only PDF, DOC, DOCX, JPG, JPEG, PNG files are allowed";
                    continue;
                }
                
                // Validate file size (5MB max)
                if ($file['size'] > 5 * 1024 * 1024) {
                    $upload_errors[] = ucfirst(str_replace('_', ' ', $doc_type)) . ": File size must be less than 5MB";
                    continue;
                }
                
                // Generate unique filename
                $filename = $doc_type . '_' . time() . '_' . uniqid() . '.' . $file_extension;
                $file_path = $upload_dir . $filename;
                
                if (move_uploaded_file($file['tmp_name'], $file_path)) {
                    // Save file info to database
                    $insert_stmt = $conn->prepare("INSERT INTO student_documents (assessment_id, student_id, document_type, original_filename, stored_filename, file_path, file_size, upload_date) VALUES (?, ?, ?, ?, ?, ?, ?, NOW()) ON DUPLICATE KEY UPDATE original_filename = VALUES(original_filename), stored_filename = VALUES(stored_filename), file_path = VALUES(file_path), file_size = VALUES(file_size), upload_date = NOW()");
                    $relative_path = 'uploads/required-documents/' . $student_id . '/' . $assessment_id . '/' . $filename;
                    $insert_stmt->bind_param("iissssi", $assessment_id, $student_id, $doc_type, $file['name'], $filename, $relative_path, $file['size']);
                    $insert_stmt->execute();
                    
                    $uploaded_files[] = ucfirst(str_replace('_', ' ', $doc_type));
                } else {
                    $upload_errors[] = ucfirst(str_replace('_', ' ', $doc_type)) . ": Failed to save file";
                }
            }
        }
        
        // Update assessment status if all documents uploaded
        if (count($uploaded_files) > 0 && count($upload_errors) === 0) {
            $update_stmt = $conn->prepare("UPDATE accreditation_requests SET status = 'documents_submitted' WHERE request_id = ?");
            $update_stmt->bind_param("i", $assessment['request_id']);
            $update_stmt->execute();
            
            // Log audit trail
            $audit_stmt = $conn->prepare("INSERT INTO accreditation_audit_log (request_id, action, performed_by, details) VALUES (?, 'documents_submitted', ?, ?)");
            $doc_list = implode(', ', $uploaded_files);
            $audit_details = "Student uploaded required documents: " . $doc_list;
            $audit_stmt->bind_param("iis", $assessment['request_id'], $student_id, $audit_details);
            $audit_stmt->execute();
        }
        
        if (count($uploaded_files) > 0) {
            $message = 'Documents uploaded successfully: ' . implode(', ', $uploaded_files);
            $messageType = 'success';
        }
        
        if (count($upload_errors) > 0) {
            if ($messageType === 'success') {
                $message .= ' However, some files had errors: ' . implode('; ', $upload_errors);
            } else {
                $message = 'Upload errors: ' . implode('; ', $upload_errors);
                $messageType = 'error';
            }
        }
    } else {
        $message = 'Invalid assessment or assessment not ready for document submission.';
        $messageType = 'error';
    }
}

// Get student's assessments that are ready for document submission
$assessments_query = "SELECT sa.*, 
                             c.title as course_title,
                             ar.request_id, ar.status as request_status
                      FROM sita_assessments sa
                      JOIN accreditation_requests ar ON sa.accreditation_request_id = ar.request_id
                      JOIN courses c ON sa.course_id = c.course_id
                      WHERE sa.student_id = ? AND sa.status = 'passed'
                      ORDER BY sa.created_date DESC";

$assessments_stmt = $conn->prepare($assessments_query);
$assessments_stmt->bind_param("i", $student_id);
$assessments_stmt->execute();
$assessments_result = $assessments_stmt->get_result();
$assessments = [];

while ($row = $assessments_result->fetch_assoc()) {
    $assessments[] = $row;
}

// Get current assessment if viewing one
$current_assessment = null;
$uploaded_documents = [];

if (isset($_GET['id'])) {
    $assessment_id = intval($_GET['id']);
    
    // Get assessment details
    $current_stmt = $conn->prepare("SELECT sa.*, c.title as course_title, ar.request_id, ar.status as request_status FROM sita_assessments sa JOIN accreditation_requests ar ON sa.accreditation_request_id = ar.request_id JOIN courses c ON sa.course_id = c.course_id WHERE sa.assessment_id = ? AND sa.student_id = ? AND sa.status = 'passed'");
    $current_stmt->bind_param("ii", $assessment_id, $student_id);
    $current_stmt->execute();
    $current_assessment = $current_stmt->get_result()->fetch_assoc();
    
    if ($current_assessment) {
        // Get uploaded documents
        $docs_stmt = $conn->prepare("SELECT * FROM student_documents WHERE assessment_id = ? AND student_id = ? ORDER BY upload_date DESC");
        $docs_stmt->bind_param("ii", $assessment_id, $student_id);
        $docs_stmt->execute();
        $docs_result = $docs_stmt->get_result();
        
        while ($doc = $docs_result->fetch_assoc()) {
            $uploaded_documents[$doc['document_type']] = $doc;
        }
    }
}

// Check if student_documents table exists, if not create it
$table_check_query = "CREATE TABLE IF NOT EXISTS student_documents (
    document_id INT AUTO_INCREMENT PRIMARY KEY,
    assessment_id INT NOT NULL,
    student_id INT NOT NULL,
    document_type VARCHAR(50) NOT NULL,
    original_filename VARCHAR(255) NOT NULL,
    stored_filename VARCHAR(255) NOT NULL,
    file_path VARCHAR(500) NOT NULL,
    file_size INT NOT NULL,
    upload_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (assessment_id) REFERENCES sita_assessments(assessment_id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES users(user_id) ON DELETE CASCADE,
    UNIQUE KEY unique_doc (assessment_id, student_id, document_type),
    INDEX idx_assessment_student (assessment_id, student_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

$conn->query($table_check_query);

function formatFileSize($bytes) {
    if ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}

function getDocumentTitle($doc_type) {
    switch ($doc_type) {
        case 'cv': return 'Curriculum Vitae (CV)';
        case 'qualifications': return 'Academic Qualifications';
        case 'id_document': return 'Identity Document';
        case 'course_assignments': return 'Course Assignments';
        default: return ucfirst(str_replace('_', ' ', $doc_type));
    }
}

function getDocumentDescription($doc_type) {
    switch ($doc_type) {
        case 'cv': return 'Upload your current CV or resume (PDF preferred)';
        case 'qualifications': return 'Upload copies of your academic certificates and qualifications';
        case 'id_document': return 'Upload a clear copy of your ID/passport';
        case 'course_assignments': return 'Upload your completed course assignments and projects';
        default: return 'Upload the required document';
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $current_assessment ? 'Required Documents - ' . htmlspecialchars($current_assessment['course_title']) : 'Required Documents' ?> - Panadite Academy</title>
    
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        body {
            font-family: 'Poppins', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        
        .main-container {
            padding: 2rem 0;
        }
        
        .documents-card {
            background: white;
            border-radius: 16px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            overflow: hidden;
            margin-bottom: 2rem;
        }
        
        .documents-header {
            background: linear-gradient(135deg, #16a085 0%, #1abc9c 100%);
            color: white;
            padding: 2rem;
            text-align: center;
        }
        
        .documents-body {
            padding: 2rem;
        }
        
        .document-item {
            background: #f8fafc;
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            padding: 1.5rem;
            margin-bottom: 1.5rem;
            transition: all 0.3s ease;
        }
        
        .document-item.uploaded {
            border-color: #10b981;
            background: #f0fdf4;
        }
        
        .document-item.required {
            border-color: #f59e0b;
            background: #fefbf0;
        }
        
        .document-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            margin-bottom: 1rem;
        }
        
        .document-icon.uploaded {
            background: rgba(16, 185, 129, 0.1);
            color: #10b981;
        }
        
        .document-icon.required {
            background: rgba(245, 158, 11, 0.1);
            color: #f59e0b;
        }
        
        .file-upload-area {
            border: 2px dashed #d1d5db;
            border-radius: 8px;
            padding: 2rem;
            text-align: center;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .file-upload-area:hover {
            border-color: #3b82f6;
            background: #f8fafc;
        }
        
        .file-upload-area.dragover {
            border-color: #10b981;
            background: #f0fdf4;
        }
        
        .file-info {
            background: white;
            border: 1px solid #e5e7eb;
            border-radius: 8px;
            padding: 1rem;
            margin-top: 1rem;
        }
        
        .back-link {
            color: white;
            text-decoration: none;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            margin-bottom: 2rem;
            padding: 0.5rem 1rem;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 8px;
            backdrop-filter: blur(10px);
        }
        
        .back-link:hover {
            color: white;
            background: rgba(255, 255, 255, 0.2);
        }
        
        .progress-indicator {
            background: #f0fdf4;
            border: 2px solid #10b981;
            border-radius: 12px;
            padding: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #16a085 0%, #1abc9c 100%);
            border: none;
            border-radius: 8px;
            padding: 0.75rem 2rem;
            font-weight: 600;
        }
        
        .btn-success {
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
            border: none;
        }
    </style>
</head>
<body>
    <?php include 'components/header.php'; ?>
    
    <div class="main-container">
        <div class="container">
            <a href="assessments.php" class="back-link">
                <i class="bx bx-arrow-back"></i> Back to Assessments
            </a>
            
            <?php if ($message): ?>
                <div class="alert alert-<?= $messageType === 'success' ? 'success' : 'danger' ?> alert-dismissible fade show" role="alert">
                    <i class="bx <?= $messageType === 'success' ? 'bx-check-circle' : 'bx-error-circle' ?> me-2"></i>
                    <?= htmlspecialchars($message) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>
            
            <?php if ($current_assessment): ?>
                <!-- Document Upload View -->
                <div class="documents-card">
                    <div class="documents-header">
                        <h2><i class="bx bx-folder-open me-2"></i>Required Documents</h2>
                        <p class="mb-0 opacity-90"><?= htmlspecialchars($current_assessment['course_title']) ?></p>
                    </div>
                    
                    <div class="documents-body">
                        <?php if ($current_assessment['request_status'] === 'documents_submitted'): ?>
                            <div class="progress-indicator">
                                <h5><i class="bx bx-check-circle me-2"></i>Documents Submitted Successfully!</h5>
                                <p class="mb-0">Your documents have been submitted and are under review. You will be notified of the final decision.</p>
                            </div>
                        <?php else: ?>
                            <div class="alert alert-info">
                                <h5><i class="bx bx-info-circle me-2"></i>Document Submission Requirements</h5>
                                <p class="mb-2">Please upload all required documents to complete your accreditation process:</p>
                                <ul class="mb-0">
                                    <li>All files must be in PDF, DOC, DOCX, JPG, JPEG, or PNG format</li>
                                    <li>Maximum file size: 5MB per document</li>
                                    <li>Ensure all documents are clear and legible</li>
                                    <li>You can replace documents by uploading new ones</li>
                                </ul>
                            </div>
                        <?php endif; ?>
                        
                        <form method="POST" enctype="multipart/form-data" id="documentsForm">
                            <input type="hidden" name="assessment_id" value="<?= $current_assessment['assessment_id'] ?>">
                            
                            <?php
                            $required_document_types = ['cv', 'qualifications', 'id_document', 'course_assignments'];
                            foreach ($required_document_types as $doc_type):
                                $is_uploaded = isset($uploaded_documents[$doc_type]);
                            ?>
                                <div class="document-item <?= $is_uploaded ? 'uploaded' : 'required' ?>">
                                    <div class="row align-items-center">
                                        <div class="col-md-6">
                                            <div class="document-icon <?= $is_uploaded ? 'uploaded' : 'required' ?>">
                                                <i class="bx <?= $is_uploaded ? 'bx-check-circle' : 'bx-upload' ?>"></i>
                                            </div>
                                            <h5><?= getDocumentTitle($doc_type) ?></h5>
                                            <p class="text-muted mb-0"><?= getDocumentDescription($doc_type) ?></p>
                                            
                                            <?php if ($is_uploaded): ?>
                                                <div class="file-info mt-3">
                                                    <div class="d-flex justify-content-between align-items-center">
                                                        <div>
                                                            <strong><?= htmlspecialchars($uploaded_documents[$doc_type]['original_filename']) ?></strong><br>
                                                            <small class="text-muted">
                                                                <?= formatFileSize($uploaded_documents[$doc_type]['file_size']) ?> • 
                                                                Uploaded <?= date('M j, Y g:i A', strtotime($uploaded_documents[$doc_type]['upload_date'])) ?>
                                                            </small>
                                                        </div>
                                                        <span class="badge bg-success">
                                                            <i class="bx bx-check me-1"></i>Uploaded
                                                        </span>
                                                    </div>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <div class="col-md-6">
                                            <?php if ($current_assessment['request_status'] !== 'documents_submitted'): ?>
                                                <div class="file-upload-area" onclick="document.getElementById('<?= $doc_type ?>_input').click()">
                                                    <i class="bx bx-cloud-upload" style="font-size: 2rem; color: #6b7280; margin-bottom: 0.5rem;"></i>
                                                    <p class="mb-2"><?= $is_uploaded ? 'Click to replace document' : 'Click to upload or drag & drop' ?></p>
                                                    <small class="text-muted">PDF, DOC, DOCX, JPG, PNG (Max 5MB)</small>
                                                    <input type="file" id="<?= $doc_type ?>_input" name="<?= $doc_type ?>" accept=".pdf,.doc,.docx,.jpg,.jpeg,.png" style="display: none;" onchange="updateFileName(this, '<?= $doc_type ?>')">
                                                </div>
                                                <div id="<?= $doc_type ?>_filename" class="mt-2 text-center" style="display: none;">
                                                    <small class="text-success">
                                                        <i class="bx bx-check me-1"></i>
                                                        <span></span>
                                                    </small>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                            
                            <?php if ($current_assessment['request_status'] !== 'documents_submitted'): ?>
                                <div class="text-center mt-4">
                                    <button type="submit" class="btn btn-primary btn-lg">
                                        <i class="bx bx-upload me-2"></i>Upload Documents
                                    </button>
                                </div>
                            <?php endif; ?>
                        </form>
                    </div>
                </div>
            <?php else: ?>
                <!-- Documents List View -->
                <div class="documents-card">
                    <div class="documents-header">
                        <h2><i class="bx bx-folder-open me-2"></i>Required Documents</h2>
                        <p class="mb-0 opacity-90">Upload documents for your completed assessments</p>
                    </div>
                    
                    <div class="documents-body">
                        <?php if (count($assessments) > 0): ?>
                            <div class="row">
                                <?php foreach ($assessments as $assessment): ?>
                                    <div class="col-md-6 col-lg-4 mb-4">
                                        <div class="card h-100">
                                            <div class="card-body">
                                                <h5 class="card-title"><?= htmlspecialchars($assessment['course_title']) ?></h5>
                                                <p class="card-text">Assessment completed successfully. Upload your required documents to proceed.</p>
                                                <div class="mb-3">
                                                    <span class="badge bg-success">Assessment Passed</span>
                                                    <?php if ($assessment['request_status'] === 'documents_submitted'): ?>
                                                        <span class="badge bg-info">Documents Submitted</span>
                                                    <?php else: ?>
                                                        <span class="badge bg-warning">Documents Pending</span>
                                                    <?php endif; ?>
                                                </div>
                                                <div class="small text-muted mb-3">
                                                    <div><i class="bx bx-trophy me-1"></i>Score: <?= $assessment['score'] ?>%</div>
                                                    <div><i class="bx bx-calendar me-1"></i>Completed: <?= date('M j, Y', strtotime($assessment['submitted_date'])) ?></div>
                                                </div>
                                                <a href="required-documents.php?id=<?= $assessment['assessment_id'] ?>" class="btn btn-primary">
                                                    <i class="bx bx-folder-open me-1"></i>
                                                    <?= $assessment['request_status'] === 'documents_submitted' ? 'View Documents' : 'Upload Documents' ?>
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php else: ?>
                            <div class="text-center py-5">
                                <i class="bx bx-folder" style="font-size: 4rem; color: #cbd5e1;"></i>
                                <h4 class="text-muted mt-3">No Documents Required Yet</h4>
                                <p class="text-muted">Complete your assessments first, then return here to upload required documents.</p>
                                <a href="assessments.php" class="btn btn-primary">
                                    <i class="bx bx-edit me-2"></i>View Assessments
                                </a>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function updateFileName(input, docType) {
            const filenameDiv = document.getElementById(docType + '_filename');
            const filenameSpan = filenameDiv.querySelector('span');
            
            if (input.files.length > 0) {
                const file = input.files[0];
                filenameSpan.textContent = file.name + ' (' + formatFileSize(file.size) + ')';
                filenameDiv.style.display = 'block';
            } else {
                filenameDiv.style.display = 'none';
            }
        }
        
        function formatFileSize(bytes) {
            if (bytes >= 1048576) {
                return (bytes / 1048576).toFixed(2) + ' MB';
            } else if (bytes >= 1024) {
                return (bytes / 1024).toFixed(2) + ' KB';
            } else {
                return bytes + ' bytes';
            }
        }
        
        // Drag and drop functionality
        document.querySelectorAll('.file-upload-area').forEach(area => {
            area.addEventListener('dragover', function(e) {
                e.preventDefault();
                this.classList.add('dragover');
            });
            
            area.addEventListener('dragleave', function(e) {
                e.preventDefault();
                this.classList.remove('dragover');
            });
            
            area.addEventListener('drop', function(e) {
                e.preventDefault();
                this.classList.remove('dragover');
                
                const fileInput = this.querySelector('input[type="file"]');
                if (e.dataTransfer.files.length > 0) {
                    fileInput.files = e.dataTransfer.files;
                    fileInput.dispatchEvent(new Event('change'));
                }
            });
        });
        
        // Form validation
        document.getElementById('documentsForm')?.addEventListener('submit', function(e) {
            const fileInputs = this.querySelectorAll('input[type="file"]');
            let hasFiles = false;
            
            fileInputs.forEach(input => {
                if (input.files.length > 0) {
                    hasFiles = true;
                }
            });
            
            if (!hasFiles) {
                e.preventDefault();
                alert('Please select at least one document to upload.');
                return false;
            }
            
            // Check file sizes
            let oversizedFiles = [];
            fileInputs.forEach(input => {
                if (input.files.length > 0) {
                    const file = input.files[0];
                    if (file.size > 5 * 1024 * 1024) { // 5MB
                        oversizedFiles.push(file.name);
                    }
                }
            });
            
            if (oversizedFiles.length > 0) {
                e.preventDefault();
                alert('The following files are too large (max 5MB): ' + oversizedFiles.join(', '));
                return false;
            }
        });
    </script>
</body>
</html>
