<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
requireRole('student');
require_once '../config/db_connect.php';

// Set content type to JSON
header('Content-Type: application/json');

// Check if request is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

// Get JSON input
$input = file_get_contents('php://input');
$data = json_decode($input, true);

// Validate input
if (!$data || !isset($data['lesson_id']) || !isset($data['progress'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid input data']);
    exit;
}

$lesson_id = (int)$data['lesson_id'];
$progress = (int)$data['progress'];
$current_time = isset($data['current_time']) ? (int)$data['current_time'] : 0;
$student_id = $_SESSION['user_id'];

// Validate progress percentage
if ($progress < 0 || $progress > 100) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid progress percentage']);
    exit;
}

try {
    // Check if database connection exists
    if (!isset($conn)) {
        throw new Exception('Database connection not available');
    }

    // Check if progress record already exists
    $check_query = "SELECT id FROM video_progress WHERE video_id = ? AND student_id = ?";
    $check_stmt = $conn->prepare($check_query);
    
    if (!$check_stmt) {
        throw new Exception('Database prepare error: ' . $conn->error);
    }
    
    $check_stmt->bind_param("ii", $lesson_id, $student_id);
    $check_stmt->execute();
    $result = $check_stmt->get_result();
    $existing_record = $result->fetch_assoc();
    $check_stmt->close();

    $completed = ($progress >= 90) ? 1 : 0; // Mark as completed if 90% or more

    if ($existing_record) {
        // Update existing progress record
        $update_query = "UPDATE video_progress SET 
                        completion_percentage = ?, 
                        last_position = ?, 
                        completed = ?,
                        completed_at = CASE WHEN ? = 1 THEN CURRENT_TIMESTAMP ELSE completed_at END
                        WHERE video_id = ? AND student_id = ?";
        
        $update_stmt = $conn->prepare($update_query);
        if (!$update_stmt) {
            throw new Exception('Database prepare error: ' . $conn->error);
        }
        
        $update_stmt->bind_param("iiiiii", $progress, $current_time, $completed, $completed, $lesson_id, $student_id);
        
        if (!$update_stmt->execute()) {
            throw new Exception('Failed to update progress: ' . $update_stmt->error);
        }
        
        $update_stmt->close();
        $action = 'updated';
        
    } else {
        // Insert new progress record
        $insert_query = "INSERT INTO video_progress (video_id, student_id, completion_percentage, last_position, completed) 
                        VALUES (?, ?, ?, ?, ?)";
        
        $insert_stmt = $conn->prepare($insert_query);
        if (!$insert_stmt) {
            throw new Exception('Database prepare error: ' . $conn->error);
        }
        
        $insert_stmt->bind_param("iiiii", $lesson_id, $student_id, $progress, $current_time, $completed);
        
        if (!$insert_stmt->execute()) {
            throw new Exception('Failed to insert progress: ' . $insert_stmt->error);
        }
        
        $insert_stmt->close();
        $action = 'created';
    }

    // Update student's overall course progress if needed
    updateCourseProgress($conn, $student_id, $lesson_id);

    // Return success response
    echo json_encode([
        'success' => true,
        'message' => 'Progress saved successfully',
        'data' => [
            'lesson_id' => $lesson_id,
            'progress' => $progress,
            'current_time' => $current_time,
            'completed' => ($progress >= 90),
            'action' => $action
        ]
    ]);

} catch (Exception $e) {
    // Log error (you might want to implement proper logging)
    error_log('Save Progress Error: ' . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Failed to save progress',
        'message' => $e->getMessage()
    ]);
}

/**
 * Update overall course progress for the student
 */
function updateCourseProgress($conn, $student_id, $video_id) {
    try {
        // Get course_id from video
        $course_query = "SELECT course_id FROM videos WHERE video_id = ?";
        $course_stmt = $conn->prepare($course_query);
        if (!$course_stmt) return;
        
        $course_stmt->bind_param("i", $video_id);
        $course_stmt->execute();
        $course_result = $course_stmt->get_result();
        $course_data = $course_result->fetch_assoc();
        $course_stmt->close();
        
        if (!$course_data) return;
        $course_id = $course_data['course_id'];
        
        // Calculate overall progress for this course
        $progress_query = "
            SELECT 
                COUNT(*) as total_videos,
                SUM(CASE WHEN vp.completed = 1 THEN 1 ELSE 0 END) as completed_videos,
                AVG(COALESCE(vp.completion_percentage, 0)) as avg_progress
            FROM videos v
            LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = ?
            WHERE v.course_id = ? AND v.is_published = 1
        ";
        
        $progress_stmt = $conn->prepare($progress_query);
        if (!$progress_stmt) return;
        
        $progress_stmt->bind_param("ii", $student_id, $course_id);
        $progress_stmt->execute();
        $progress_result = $progress_stmt->get_result();
        $progress_data = $progress_result->fetch_assoc();
        $progress_stmt->close();
        
        if ($progress_data && $progress_data['total_videos'] > 0) {
            $overall_progress = round($progress_data['avg_progress']);
            
            // Update enrollment record with overall progress
            $update_enrollment = "UPDATE enrollments SET progress_percentage = ? WHERE student_id = ? AND course_id = ?";
            $enrollment_stmt = $conn->prepare($update_enrollment);
            if ($enrollment_stmt) {
                $enrollment_stmt->bind_param("iii", $overall_progress, $student_id, $course_id);
                $enrollment_stmt->execute();
                $enrollment_stmt->close();
            }
        }
        
    } catch (Exception $e) {
        // Silently handle course progress update errors
        error_log('Course progress update error: ' . $e->getMessage());
    }
}
?>
