<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';

// Require student role to access this page
requireRole('student');

// Include database connection
require_once '../config/db_connect.php';

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Check if tables already exist
$tablesExist = [];
$requiredTables = ['quizzes', 'quiz_questions', 'question_options', 'quiz_attempts', 'quiz_answers'];

foreach ($requiredTables as $table) {
    $result = $conn->query("SHOW TABLES LIKE '{$table}'");
    $tablesExist[$table] = ($result->num_rows > 0);
}

$results = [];

// Use transactions to ensure data consistency
$conn->begin_transaction();

try {
    // Step 1: Create quizzes table if it doesn't exist
    if (!$tablesExist['quizzes']) {
        $sql = "CREATE TABLE quizzes (
            quiz_id INT AUTO_INCREMENT PRIMARY KEY,
            course_id INT NOT NULL,
            title VARCHAR(255) NOT NULL,
            description TEXT,
            instructions TEXT,
            start_time DATETIME,
            end_time DATETIME,
            time_limit INT,
            total_points INT,
            max_attempts INT,
            status ENUM('draft', 'published', 'archived') DEFAULT 'published',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (course_id) REFERENCES courses(course_id) ON DELETE CASCADE
        )";
        
        if ($conn->query($sql) === TRUE) {
            $results['quizzes'] = "Quizzes table created successfully";
        } else {
            throw new Exception("Error creating quizzes table: " . $conn->error);
        }
    }
    
    // Step 2: Add sample quiz data
    $quiz_id = null;
    $quizCheck = $conn->query("SELECT quiz_id FROM quizzes WHERE title = 'App Development Quiz' AND course_id = 2");
    if ($quizCheck->num_rows == 0) {
        $sampleData = "INSERT INTO quizzes (course_id, title, description, instructions, start_time, end_time, time_limit, total_points, max_attempts) VALUES
        (2, 'App Development Quiz', 'Write an exam on app development', 'Complete all questions within the time limit', '2025-07-16 00:00:00', '2025-07-17 23:59:59', 30, 100, 3)";
        
        if ($conn->query($sampleData) === TRUE) {
            $quiz_id = $conn->insert_id;
            $results['sample_quiz'] = "Sample quiz data added successfully (ID: $quiz_id)";
        } else {
            throw new Exception("Error adding sample quiz data: " . $conn->error);
        }
    } else {
        $quiz_id = $quizCheck->fetch_assoc()['quiz_id'];
        $results['sample_quiz'] = "Sample quiz already exists (ID: $quiz_id)";
    }
    
    // Step 3: Create quiz_questions table if it doesn't exist
    if (!$tablesExist['quiz_questions']) {
        $sql = "CREATE TABLE quiz_questions (
            question_id INT AUTO_INCREMENT PRIMARY KEY,
            quiz_id INT NOT NULL,
            question_text TEXT NOT NULL,
            question_type ENUM('multiple_choice', 'multiple_answer', 'true_false', 'short_answer') DEFAULT 'multiple_choice',
            question_order INT DEFAULT 0,
            points INT DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (quiz_id) REFERENCES quizzes(quiz_id) ON DELETE CASCADE
        )";
        
        if ($conn->query($sql) === TRUE) {
            $results['quiz_questions'] = "Quiz questions table created successfully";
        } else {
            throw new Exception("Error creating quiz questions table: " . $conn->error);
        }
    }
    
    // Step 4: Add sample question data
    $questions_added = false;
    $questionsCheck = $conn->query("SELECT COUNT(*) as count FROM quiz_questions WHERE quiz_id = $quiz_id");
    $questionCount = $questionsCheck->fetch_assoc()['count'];
    
    if ($questionCount == 0 && $quiz_id) {
        $sampleData = "INSERT INTO quiz_questions (quiz_id, question_text, question_type, question_order, points) VALUES
        ($quiz_id, 'What is the primary purpose of an app manifest file?', 'multiple_choice', 1, 10),
        ($quiz_id, 'Which of these is NOT a common mobile app development framework?', 'multiple_choice', 2, 10),
        ($quiz_id, 'True or False: React Native can be used to develop both iOS and Android apps.', 'true_false', 3, 5),
        ($quiz_id, 'Explain briefly what is meant by responsive design in mobile app development.', 'short_answer', 4, 15)";
        
        if ($conn->query($sampleData) === TRUE) {
            $results['sample_questions'] = "Sample question data added successfully";
            $questions_added = true;
        } else {
            throw new Exception("Error adding sample question data: " . $conn->error);
        }
    } else {
        $results['sample_questions'] = "Sample questions already exist";
    }
    
    // Step 5: Create question_options table if it doesn't exist
    if (!$tablesExist['question_options']) {
        $sql = "CREATE TABLE question_options (
            option_id INT AUTO_INCREMENT PRIMARY KEY,
            question_id INT NOT NULL,
            option_text TEXT NOT NULL,
            is_correct BOOLEAN DEFAULT FALSE,
            option_order INT DEFAULT 0,
            feedback TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (question_id) REFERENCES quiz_questions(question_id) ON DELETE CASCADE
        )";
        
        if ($conn->query($sql) === TRUE) {
            $results['question_options'] = "Question options table created successfully";
        } else {
            throw new Exception("Error creating question options table: " . $conn->error);
        }
    }
    
    // Step 6: Add sample options data (only if we added questions)
    if ($questions_added) {
        // Get the question IDs that were just inserted
        $questionIdsQuery = $conn->query("SELECT question_id, question_type FROM quiz_questions WHERE quiz_id = $quiz_id ORDER BY question_order");
        $questionIds = [];
        while ($row = $questionIdsQuery->fetch_assoc()) {
            $questionIds[] = $row;
        }
        
        if (count($questionIds) >= 3) {
            // For the first question (multiple choice)
            $q1Id = $questionIds[0]['question_id'];
            $conn->query("INSERT INTO question_options (question_id, option_text, is_correct, option_order) VALUES
                ($q1Id, 'To define app permissions', TRUE, 1),
                ($q1Id, 'To create app icons', FALSE, 2),
                ($q1Id, 'To write app logic', FALSE, 3),
                ($q1Id, 'To style app elements', FALSE, 4)");
            
            // For the second question (multiple choice)
            $q2Id = $questionIds[1]['question_id'];
            $conn->query("INSERT INTO question_options (question_id, option_text, is_correct, option_order) VALUES
                ($q2Id, 'React Native', FALSE, 1),
                ($q2Id, 'Flutter', FALSE, 2),
                ($q2Id, 'Xamarin', FALSE, 3),
                ($q2Id, 'AppLaunch Pro', TRUE, 4)");
            
            // For the third question (true/false)
            $q3Id = $questionIds[2]['question_id'];
            $conn->query("INSERT INTO question_options (question_id, option_text, is_correct, option_order) VALUES
                ($q3Id, 'true', TRUE, 1),
                ($q3Id, 'false', FALSE, 2)");
            
            $results['sample_options'] = "Sample options data added successfully";
        } else {
            $results['sample_options'] = "Not enough questions found to add options";
        }
    } else {
        // Check if options already exist
        $optionsCheck = $conn->query("SELECT COUNT(*) as count FROM question_options");
        if ($optionsCheck->fetch_assoc()['count'] > 0) {
            $results['sample_options'] = "Sample options already exist";
        } else {
            $results['sample_options'] = "No new questions to add options for";
        }
    }
    
    // Step 7: Create quiz_attempts table if it doesn't exist
    if (!$tablesExist['quiz_attempts']) {
        $sql = "CREATE TABLE quiz_attempts (
            attempt_id INT AUTO_INCREMENT PRIMARY KEY,
            quiz_id INT NOT NULL,
            student_id INT NOT NULL,
            attempt_number INT DEFAULT 1,
            attempt_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            completion_time INT,
            score DECIMAL(10,2) DEFAULT 0,
            FOREIGN KEY (quiz_id) REFERENCES quizzes(quiz_id) ON DELETE CASCADE,
            FOREIGN KEY (student_id) REFERENCES users(user_id) ON DELETE CASCADE
        )";
        
        if ($conn->query($sql) === TRUE) {
            $results['quiz_attempts'] = "Quiz attempts table created successfully";
        } else {
            throw new Exception("Error creating quiz attempts table: " . $conn->error);
        }
    }
    
    // Step 8: Create quiz_answers table if it doesn't exist
    if (!$tablesExist['quiz_answers']) {
        $sql = "CREATE TABLE quiz_answers (
            answer_id INT AUTO_INCREMENT PRIMARY KEY,
            attempt_id INT NOT NULL,
            question_id INT NOT NULL,
            given_answer TEXT,
            correct_answer TEXT,
            is_correct BOOLEAN DEFAULT FALSE,
            points_earned DECIMAL(10,2) DEFAULT 0,
            feedback TEXT,
            FOREIGN KEY (attempt_id) REFERENCES quiz_attempts(attempt_id) ON DELETE CASCADE,
            FOREIGN KEY (question_id) REFERENCES quiz_questions(question_id) ON DELETE CASCADE
        )";
        
        if ($conn->query($sql) === TRUE) {
            $results['quiz_answers'] = "Quiz answers table created successfully";
        } else {
            throw new Exception("Error creating quiz answers table: " . $conn->error);
        }
    }

    // Step 9: Verify enrollments table exists and has correct records
    $enrollmentCheck = $conn->query("SELECT * FROM enrollments WHERE student_id = {$_SESSION['user_id']} AND course_id = 2");
    if ($enrollmentCheck->num_rows === 0) {
        // Insert enrollment record for the current student and course 2
        $insertEnrollment = "INSERT INTO enrollments (student_id, course_id, enrollment_date, status) 
                            VALUES ({$_SESSION['user_id']}, 2, NOW(), 'active')";
        if ($conn->query($insertEnrollment) === TRUE) {
            $results['enrollment'] = "Student enrollment record created";
        } else {
            throw new Exception("Error creating enrollment record: " . $conn->error);
        }
    } else {
        $results['enrollment'] = "Enrollment record already exists";
    }
    
    // Step 10: Update quiz status in the database
    if ($quiz_id) {
        $updateQuizStatus = "UPDATE quizzes SET start_time = '2025-07-16 00:00:00', end_time = '2025-07-17 23:59:59' WHERE quiz_id = $quiz_id";
        if ($conn->query($updateQuizStatus) === TRUE) {
            $results['quiz_status'] = "Quiz dates updated successfully";
        } else {
            throw new Exception("Error updating quiz dates: " . $conn->error);
        }
    }
    
    // Commit the transaction
    $conn->commit();
} catch (Exception $e) {
    // Roll back the transaction if something failed
    $conn->rollback();
    $results['error'] = $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Quiz Tables Setup</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body>
    <div class="container mt-5">
        <div class="row">
            <div class="col-md-8 offset-md-2">
                <div class="card">
                    <div class="card-header bg-primary text-white">
                        <h3 class="mb-0">Quiz System Setup</h3>
                    </div>
                    <div class="card-body">
                        <div class="alert alert-info">
                            <h4>Setup Results</h4>
                            <ul class="mb-0">
                                <?php foreach ($results as $key => $message): ?>
                                <li><strong><?= ucfirst($key) ?>:</strong> <?= $message ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                        
                        <h5 class="mt-4">Current Tables Status:</h5>
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>Table</th>
                                    <th>Status</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($requiredTables as $table): ?>
                                <tr>
                                    <td><?= $table ?></td>
                                    <td>
                                        <?php if ($tablesExist[$table]): ?>
                                            <span class="badge bg-success">Exists</span>
                                        <?php else: ?>
                                            <span class="badge bg-danger">Missing</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                        
                        <div class="mt-4">
                            <a href="course-quizzes.php?id=2" class="btn btn-primary">Return to Course Quizzes</a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
