<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
requireRole('student');
require_once '../includes/db_connection.php';

// Also create PDO connection for existing submission code
try {
    $db_host = 'localhost';
    $db_name = 'twpurnyc_panadite_DB';
    $db_user = 'twpurnyc_149586Alain';
    $db_pass = '149586Alain#';
    $db_charset = 'utf8mb4';
    $dsn = "mysql:host=$db_host;dbname=$db_name;charset=$db_charset";
    $options = [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES   => false,
    ];
    $pdo = new PDO($dsn, $db_user, $db_pass, $options);
} catch (Exception $e) {
    error_log("PDO connection failed: " . $e->getMessage());
    die("Database connection failed");
}

$student_id = $_SESSION['user_id'];
$assignment_id = intval($_GET['id'] ?? 0);

if ($assignment_id <= 0) {
    header('Location: assignments.php');
    exit;
}

// Get enhanced assignment details with all features
$query = "
    SELECT 
        a.assignment_id,
        a.title,
        a.description,
        a.instructions,
        a.due_date,
        a.start_date,
        a.points as max_points,
        a.submission_type,
        a.allowed_formats,
        a.max_file_size,
        a.allow_late,
        a.late_penalty,
        a.group_assignment,
        a.max_group_size,
        a.grading_rubric,
        a.contributes_to_final_grade,
        c.course_id,
        c.title as course_name,
        CONCAT(u.first_name, ' ', u.last_name) as instructor_name,
        ass.submission_id,
        ass.submission_text,
        ass.file_url,
        ass.submission_date,
        ass.score,
        ass.feedback,
        CASE 
            WHEN NOW() < a.start_date THEN 1 
            ELSE 0 
        END as not_yet_available,
        CASE 
            WHEN NOW() > a.due_date THEN 1 
            ELSE 0 
        END as is_overdue
    FROM assignments a
    JOIN courses c ON a.course_id = c.course_id
    JOIN users u ON a.teacher_id = u.user_id
    LEFT JOIN assignment_submissions ass ON a.assignment_id = ass.assignment_id AND ass.student_id = ?
    WHERE a.assignment_id = ? AND c.course_id IN (SELECT course_id FROM enrollments WHERE student_id = ?) AND a.is_published = 1
";

$stmt = $conn->prepare($query);
$stmt->bind_param("iii", $student_id, $assignment_id, $student_id);
$stmt->execute();
$result = $stmt->get_result();
$assignment = $result->fetch_assoc();

if (!$assignment) {
    header('Location: assignments.php');
    exit;
}

$page_title = "Submit Assignment - " . $assignment['title'];

// Parse assignment data safely with debugging
$submission_types = json_decode($assignment['submission_type'] ?? '[]', true) ?: [];
$allowed_formats = json_decode($assignment['allowed_formats'] ?? '[]', true) ?: [];

// FALLBACK: If no submission types are configured, provide default options
if (empty($submission_types)) {
    $submission_types = ['text_entry', 'file_upload']; // Default to text + file upload
    error_log("No submission types configured for assignment {$assignment_id}, using fallback: text_entry + file_upload");
}

// FALLBACK: If allowed_formats is double-encoded JSON, fix it
if (!empty($allowed_formats) && is_array($allowed_formats) && is_string($allowed_formats[0])) {
    $allowed_formats = json_decode($allowed_formats[0], true) ?: [];
    error_log("Fixed double-encoded allowed_formats for assignment {$assignment_id}");
}

// Logging for production debugging (can be removed later)
if (empty($assignment['submission_type'])) {
    error_log("Assignment {$assignment_id}: Using fallback submission types (text_entry + file_upload)");
}

// Handle enhanced form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $text_submission = trim($_POST['text_submission'] ?? '');
    $uploaded_files = [];
    $errors = [];
    
    // Check if assignment is available for submission
    if ($assignment['not_yet_available']) {
        $errors[] = "Assignment is not yet available for submission.";
    } elseif ($assignment['is_overdue'] && !$assignment['allow_late']) {
        $errors[] = "This assignment no longer accepts submissions.";
    }
    
    // Handle multiple file uploads based on submission types
    $upload_dir = '../uploads/assignments/';
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    // Process each submission type
    foreach ($submission_types as $type) {
        if (isset($_FILES[$type]) && !empty($_FILES[$type]['name'][0])) {
            $file_count = count($_FILES[$type]['name']);
            
            for ($i = 0; $i < $file_count; $i++) {
                if ($_FILES[$type]['error'][$i] === UPLOAD_ERR_OK) {
                    $file_name = $_FILES[$type]['name'][$i];
                    $file_tmp = $_FILES[$type]['tmp_name'][$i];
                    $file_size = $_FILES[$type]['size'][$i];
                    $file_extension = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
                    
                    // Validate file extension
                    if (!empty($allowed_formats) && !in_array($file_extension, $allowed_formats)) {
                        $errors[] = "File '{$file_name}' has an invalid format. Allowed: " . implode(', ', $allowed_formats);
                        continue;
                    }
                    
                    // Validate file size
                    $max_size = ($assignment['max_file_size'] ?? 50) * 1024 * 1024; // Convert MB to bytes
                    if ($file_size > $max_size) {
                        $errors[] = "File '{$file_name}' exceeds maximum size of " . ($assignment['max_file_size'] ?? 50) . "MB";
                        continue;
                    }
                    
                    // Generate unique filename
                    $unique_name = $student_id . '_' . $assignment_id . '_' . $type . '_' . time() . '_' . $i . '.' . $file_extension;
                    $upload_path = $upload_dir . $unique_name;
                    
                    if (move_uploaded_file($file_tmp, $upload_path)) {
                        $uploaded_files[] = [
                            'type' => $type,
                            'original_name' => $file_name,
                            'file_path' => $unique_name
                        ];
                    } else {
                        $errors[] = "Failed to upload file '{$file_name}'";
                    }
                }
            }
        }
    }
    
    // Only proceed if no errors
    if (empty($errors)) {
        try {
            $files_json = json_encode($uploaded_files);
            
            if ($assignment['submission_id']) {
                // Update existing submission
                $update_query = "
                    UPDATE assignment_submissions 
                    SET submission_text = ?, file_url = ?, submission_date = NOW()
                    WHERE assignment_id = ? AND student_id = ?
                ";
                $stmt = $conn->prepare($update_query);
                $stmt->bind_param("ssii", $text_submission, $files_json, $assignment_id, $student_id);
                $stmt->execute();
            } else {
                // Create new submission
                $insert_query = "
                    INSERT INTO assignment_submissions (assignment_id, student_id, submission_text, file_url, submission_date)
                    VALUES (?, ?, ?, ?, NOW())
                ";
                $stmt = $conn->prepare($insert_query);
                $stmt->bind_param("iiss", $assignment_id, $student_id, $text_submission, $files_json);
                $stmt->execute();
            }
            
            $success_message = "Assignment submitted successfully!";
            
            // Refresh assignment data to show updated submission
            $stmt = $conn->prepare($query);
            $stmt->bind_param("iii", $student_id, $assignment_id, $student_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $assignment = $result->fetch_assoc();
            
        } catch (Exception $e) {
            $errors[] = "Error submitting assignment. Please try again.";
            error_log("Assignment submission error: " . $e->getMessage());
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $page_title ?> - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/boxicons@2.1.4/css/boxicons.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        :root {
            --panadite-primary: #2563eb;
            --panadite-secondary: #1d4ed8;
            --panadite-light: #dbeafe;
            --panadite-accent: #f59e0b;
            --text-primary: #1f2937;
            --text-secondary: #6b7280;
            --border-color: #e5e7eb;
            --success-color: #10b981;
            --error-color: #ef4444;
            --bg-light: #f8fafc;
        }

        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: linear-gradient(135deg, var(--bg-light) 0%, #e2e8f0 100%);
            color: var(--text-primary);
            line-height: 1.6;
            min-height: 100vh;
        }

        .page-header {
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
            box-shadow: 0 4px 20px rgba(37, 99, 235, 0.15);
        }

        .header-content {
            max-width: 800px;
            margin: 0 auto;
            padding: 0 1rem;
            text-align: center;
        }

        .page-title {
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
        }

        .page-subtitle {
            font-size: 1rem;
            opacity: 0.9;
        }

        .container {
            max-width: 800px;
            margin: 0 auto;
            padding: 0 1rem 2rem;
        }

        .assignment-info {
            background: white;
            border-radius: 16px;
            padding: 2rem;
            margin-bottom: 2rem;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        }

        .assignment-title {
            font-size: 1.5rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 1rem;
        }

        .assignment-meta {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 1.5rem;
            padding: 1rem;
            background: var(--bg-light);
            border-radius: 12px;
        }

        .meta-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.9rem;
        }

        .meta-icon {
            color: var(--panadite-primary);
            font-size: 1.1rem;
        }

        .overdue-badge {
            background: #fee2e2;
            color: #dc2626;
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
            text-transform: uppercase;
            display: inline-flex;
            align-items: center;
            gap: 0.3rem;
            margin-bottom: 1rem;
        }

        .assignment-description {
            color: var(--text-secondary);
            margin-bottom: 1rem;
            line-height: 1.6;
        }

        .instructions {
            background: #fef3c7;
            border-left: 4px solid var(--panadite-accent);
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
        }

        .instructions-title {
            font-weight: 600;
            color: #d97706;
            margin-bottom: 0.5rem;
        }

        .submission-form {
            background: white;
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-label {
            display: block;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
        }

        .form-input, .form-textarea {
            width: 100%;
            padding: 1rem;
            border: 2px solid var(--border-color);
            border-radius: 12px;
            font-size: 0.95rem;
            transition: all 0.3s ease;
            font-family: inherit;
        }

        .form-textarea {
            min-height: 150px;
            resize: vertical;
        }

        .form-input:focus, .form-textarea:focus {
            outline: none;
            border-color: var(--panadite-primary);
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        .file-upload {
            position: relative;
            display: inline-block;
            cursor: pointer;
            width: 100%;
        }

        .file-upload input[type=file] {
            position: absolute;
            opacity: 0;
            width: 100%;
            height: 100%;
            cursor: pointer;
        }

        .file-upload-label {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            padding: 2rem;
            border: 2px dashed var(--border-color);
            border-radius: 12px;
            background: var(--bg-light);
            color: var(--text-secondary);
            font-weight: 500;
            transition: all 0.3s ease;
        }

        .file-upload:hover .file-upload-label {
            border-color: var(--panadite-primary);
            color: var(--panadite-primary);
            background: var(--panadite-light);
        }

        .submit-btn {
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            color: white;
            padding: 1rem 2rem;
            border: none;
            border-radius: 12px;
            font-weight: 600;
            font-size: 1rem;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin-top: 1rem;
        }

        .submit-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(37, 99, 235, 0.4);
        }

        .back-link {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            color: var(--panadite-primary);
            text-decoration: none;
            font-weight: 500;
            margin-bottom: 1rem;
        }

        .back-link:hover {
            text-decoration: underline;
        }

        .alert {
            padding: 1rem;
            border-radius: 12px;
            margin-bottom: 1rem;
            font-weight: 500;
        }

        .alert-success {
            background: #d1fae5;
            color: #059669;
            border: 1px solid #a7f3d0;
        }

        .alert-error {
            background: #fee2e2;
            color: #dc2626;
            border: 1px solid #fecaca;
        }

        @media (max-width: 768px) {
            .page-title {
                font-size: 1.5rem;
            }
            
            .assignment-info, .submission-form {
                padding: 1.5rem;
            }
            
            .assignment-meta {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <?php include_once 'components/sidebar.php'; ?>
    
    <div class="page-header">
        <div class="header-content">
            <h1 class="page-title">Submit Assignment</h1>
            <p class="page-subtitle"><?= htmlspecialchars($assignment['course_title']) ?></p>
        </div>
    </div>

    <div class="container">
        <a href="assignments.php" class="back-link">
            <i class="bx bx-arrow-back"></i>
            Back to Assignments
        </a>

        <?php if (isset($success_message)): ?>
            <div class="alert alert-success">
                <i class="bx bx-check-circle"></i>
                <?= $success_message ?>
            </div>
        <?php endif; ?>

        <?php if (isset($error_message)): ?>
            <div class="alert alert-error">
                <i class="bx bx-error-circle"></i>
                <?= $error_message ?>
            </div>
        <?php endif; ?>

        <!-- Enhanced Assignment Information Display -->
        <div class="assignment-info">
            <div style="display: flex; justify-content: between; align-items: start; margin-bottom: 1rem;">
                <div>
                    <h2 style="margin-bottom: 0.5rem; font-weight: 700; color: var(--text-primary);">
                        <?= htmlspecialchars($assignment['title']) ?>
                    </h2>
                    <div style="display: flex; gap: 0.5rem; margin-bottom: 1rem;">
                        <?php if ($assignment['contributes_to_final_grade']): ?>
                            <span style="background: #10b981; color: white; font-size: 0.75rem; padding: 0.25rem 0.5rem; border-radius: 12px;">
                                <i class="bx bx-certification"></i> Counts Toward Final Grade
                            </span>
                        <?php else: ?>
                            <span style="background: #6b7280; color: white; font-size: 0.75rem; padding: 0.25rem 0.5rem; border-radius: 12px;">
                                <i class="bx bx-practice"></i> Practice Assignment
                            </span>
                        <?php endif; ?>
                        
                        <?php if ($assignment['group_assignment']): ?>
                            <span style="background: #3b82f6; color: white; font-size: 0.75rem; padding: 0.25rem 0.5rem; border-radius: 12px;">
                                <i class="bx bx-group"></i> Group Assignment (Max <?= $assignment['max_group_size'] ?> members)
                            </span>
                        <?php endif; ?>
                        
                        <?php if ($assignment['is_overdue']): ?>
                            <span class="overdue-badge">
                                <?php if ($assignment['allow_late']): ?>
                                    <i class="bx bx-time"></i> Late Submission 
                                    <?= $assignment['late_penalty'] > 0 ? '(-'.$assignment['late_penalty'].'% penalty)' : '' ?>
                                <?php else: ?>
                                    <i class="bx bx-x-circle"></i> Submission Closed
                                <?php endif; ?>
                            </span>
                        <?php elseif ($assignment['not_yet_available']): ?>
                            <span style="background: #fef3c7; color: #92400e; padding: 0.5rem 1rem; border-radius: 20px; font-size: 0.8rem; font-weight: 600;">
                                <i class="bx bx-time"></i> Available <?= date('M d, g:i A', strtotime($assignment['start_date'])) ?>
                            </span>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <div class="assignment-meta">
                <?php if ($assignment['start_date'] && !$assignment['not_yet_available']): ?>
                    <div class="meta-item">
                        <i class="meta-icon bx bx-play-circle"></i>
                        <span>Available Since: <?= date('M d, Y g:i A', strtotime($assignment['start_date'])) ?></span>
                    </div>
                <?php endif; ?>
                
                <div class="meta-item <?= $assignment['is_overdue'] ? 'text-danger' : '' ?>">
                    <i class="meta-icon bx bx-calendar"></i>
                    <span>Due: <?= date('M d, Y g:i A', strtotime($assignment['due_date'])) ?></span>
                </div>
                
                <div class="meta-item">
                    <i class="meta-icon bx bx-award"></i>
                    <span><?= $assignment['max_points'] ?> Points</span>
                </div>
                
                <div class="meta-item">
                    <i class="meta-icon bx bx-user"></i>
                    <span>Instructor: <?= htmlspecialchars($assignment['first_name'] . ' ' . $assignment['last_name']) ?></span>
                </div>
                
                <?php if ($assignment['submission_date']): ?>
                    <div class="meta-item">
                        <i class="meta-icon bx bx-check"></i>
                        <span>Last Submitted: <?= date('M d, Y g:i A', strtotime($assignment['submission_date'])) ?></span>
                    </div>
                <?php endif; ?>
                
                <?php if ($assignment['grade'] !== null): ?>
                    <div class="meta-item">
                        <i class="meta-icon bx bx-trophy"></i>
                        <span>Grade: <strong><?= $assignment['grade'] ?>/<?= $assignment['max_points'] ?></strong> 
                            (<?= round(($assignment['grade'] / $assignment['max_points']) * 100, 1) ?>%)</span>
                    </div>
                <?php endif; ?>
            </div>
            
            <div class="assignment-description">
                <?= nl2br(htmlspecialchars($assignment['description'])) ?>
            </div>
            
            <!-- Enhanced Submission Requirements -->
            <?php if (!empty($submission_types)): ?>
                <div style="background: #f0f9ff; border: 1px solid #0ea5e9; border-radius: 12px; padding: 1.5rem; margin: 1.5rem 0;">
                    <h4 style="color: #0369a1; margin-bottom: 1rem; font-weight: 600;">
                        <i class="bx bx-list-check"></i> Submission Requirements
                    </h4>
                    
                    <div style="display: grid; gap: 1rem;">
                        <div>
                            <strong style="color: var(--text-primary);">Required Submission Types:</strong>
                            <div style="display: flex; flex-wrap: wrap; gap: 0.5rem; margin-top: 0.5rem;">
                                <?php foreach($submission_types as $type): ?>
                                    <span style="background: #dbeafe; color: #1e40af; padding: 0.25rem 0.75rem; border-radius: 20px; font-size: 0.85rem; border: 1px solid #93c5fd;">
                                        <?php 
                                            $type_icons = [
                                                'file_upload' => 'bx-upload',
                                                'video' => 'bx-video',
                                                'document' => 'bx-file-doc', 
                                                'image' => 'bx-image',
                                                'excel' => 'bx-spreadsheet',
                                                'pdf' => 'bx-file-pdf',
                                                'presentation' => 'bx-slideshow',
                                                'code' => 'bx-code-alt',
                                                'text_entry' => 'bx-edit-alt',
                                                'multiple_files' => 'bx-folder-plus'
                                            ];
                                            $icon = $type_icons[$type] ?? 'bx-file';
                                        ?>
                                        <i class="bx <?= $icon ?>"></i> <?= ucfirst(str_replace('_', ' ', $type)) ?>
                                    </span>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        
                        <?php if (!empty($allowed_formats)): ?>
                            <div>
                                <strong style="color: var(--text-primary);">Allowed File Formats:</strong>
                                <span style="color: var(--text-secondary); margin-left: 0.5rem;">
                                    <?= strtoupper(implode(', ', $allowed_formats)) ?>
                                </span>
                            </div>
                        <?php endif; ?>
                        
                        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; font-size: 0.9rem;">
                            <?php if ($assignment['max_file_size']): ?>
                                <div>
                                    <strong>Maximum File Size:</strong> <?= $assignment['max_file_size'] ?>MB per file
                                </div>
                            <?php endif; ?>
                            
                            <?php if ($assignment['group_assignment']): ?>
                                <div>
                                    <strong>Team Submission:</strong> One submission per team
                                </div>
                            <?php endif; ?>
                            
                            <?php if (!$assignment['allow_late']): ?>
                                <div style="color: #dc2626;">
                                    <strong>⚠️ No Late Submissions Allowed</strong>
                                </div>
                            <?php elseif ($assignment['late_penalty'] > 0): ?>
                                <div style="color: #f59e0b;">
                                    <strong>Late Penalty:</strong> <?= $assignment['late_penalty'] ?>% per day
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if ($assignment['instructions']): ?>
                <div class="instructions">
                    <div class="instructions-title">
                        <i class="bx bx-info-circle"></i>
                        Detailed Instructions
                    </div>
                    <div><?= nl2br(htmlspecialchars($assignment['instructions'])) ?></div>
                </div>
            <?php endif; ?>
            
            <?php if ($assignment['grading_rubric']): ?>
                <div style="background: #fefce8; border: 1px solid #eab308; border-radius: 12px; padding: 1.5rem; margin: 1rem 0;">
                    <h5 style="color: #a16207; margin-bottom: 0.75rem; font-weight: 600;">
                        <i class="bx bx-award"></i> Grading Rubric
                    </h5>
                    <div><?= nl2br(htmlspecialchars($assignment['grading_rubric'])) ?></div>
                </div>
            <?php endif; ?>
        </div>



        <!-- Enhanced Dynamic Submission Form -->
        <?php if ($assignment['not_yet_available']): ?>
            <div style="background: #fef3c7; border: 1px solid #f59e0b; border-radius: 12px; padding: 2rem; text-align: center; margin-top: 2rem;">
                <i class="bx bx-time" style="font-size: 3rem; color: #d97706; margin-bottom: 1rem;"></i>
                <h3 style="color: #92400e; margin-bottom: 0.5rem;">Assignment Not Yet Available</h3>
                <p style="color: #a16207;">This assignment will be available for submission on <?= date('M d, Y \a\t g:i A', strtotime($assignment['start_date'])) ?></p>
            </div>
        <?php elseif ($assignment['is_overdue'] && !$assignment['allow_late']): ?>
            <div style="background: #fee2e2; border: 1px solid #f87171; border-radius: 12px; padding: 2rem; text-align: center; margin-top: 2rem;">
                <i class="bx bx-x-circle" style="font-size: 3rem; color: #dc2626; margin-bottom: 1rem;"></i>
                <h3 style="color: #dc2626; margin-bottom: 0.5rem;">Submission Period Closed</h3>
                <p style="color: #991b1b;">This assignment no longer accepts submissions as the due date has passed.</p>
            </div>
        <?php else: ?>
            <form method="POST" enctype="multipart/form-data" class="submission-form" id="submissionForm">
                <h3 style="margin-bottom: 1.5rem; font-weight: 600; color: var(--text-primary);">
                    <i class="bx bx-edit"></i>
                    <?= $assignment['submission_id'] ? 'Update Your Submission' : 'Submit Your Assignment' ?>
                    <?php if ($assignment['is_overdue'] && $assignment['allow_late']): ?>
                        <span style="color: #f59e0b; font-size: 0.8rem; font-weight: normal;">
                            (Late submission - <?= $assignment['late_penalty'] > 0 ? $assignment['late_penalty'].'% penalty will apply' : 'no penalty' ?>)
                        </span>
                    <?php endif; ?>
                </h3>
                
                <!-- Display Success/Error Messages -->
                <?php if (isset($success_message)): ?>
                    <div class="alert alert-success">
                        <i class="bx bx-check-circle"></i>
                        <?= $success_message ?>
                    </div>
                <?php endif; ?>
                
                <?php if (isset($errors) && !empty($errors)): ?>
                    <div class="alert alert-error">
                        <i class="bx bx-error"></i>
                        <strong>Submission Error:</strong>
                        <ul style="margin: 0.5rem 0 0 1.5rem;">
                            <?php foreach ($errors as $error): ?>
                                <li><?= htmlspecialchars($error) ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                
                <!-- Text Entry Submission (if required) -->
                <?php if (in_array('text_entry', $submission_types)): ?>
                    <div class="form-group">
                        <label for="text_submission" class="form-label">
                            <i class="bx bx-text"></i>
                            Text Submission
                            <span style="color: #dc2626;">*</span>
                        </label>
                        <textarea 
                            id="text_submission" 
                            name="text_submission" 
                            class="form-textarea"
                            rows="8"
                            placeholder="Enter your assignment response here..."
                            required
                        ><?= htmlspecialchars($assignment['submission_text'] ?? '') ?></textarea>
                        <small style="color: var(--text-secondary);">
                            Provide a detailed written response to the assignment prompt.
                        </small>
                    </div>
                <?php endif; ?>
                
                <!-- Dynamic File Upload Sections Based on Submission Types -->
                <?php foreach ($submission_types as $type): ?>
                    <?php if ($type !== 'text_entry'): ?>
                        <div class="form-group">
                            <label class="form-label">
                                <?php 
                                    $type_labels = [
                                        'file_upload' => ['Generic File Upload', 'bx-upload'],
                                        'video' => ['Video Submission', 'bx-video'],
                                        'document' => ['Document Upload', 'bx-file-doc'],
                                        'image' => ['Image Upload', 'bx-image'],
                                        'excel' => ['Spreadsheet Upload', 'bx-spreadsheet'],
                                        'pdf' => ['PDF Document', 'bx-file-pdf'],
                                        'presentation' => ['Presentation Upload', 'bx-slideshow'],
                                        'code' => ['Code Files', 'bx-code-alt'],
                                        'multiple_files' => ['Multiple Files', 'bx-folder-plus']
                                    ];
                                    $label = $type_labels[$type] ?? ['File Upload', 'bx-file'];
                                ?>
                                <i class="bx <?= $label[1] ?>"></i>
                                <?= $label[0] ?>
                                <span style="color: #dc2626;">*</span>
                            </label>
                            
                            <div class="file-upload" style="position: relative;">
                                <input 
                                    type="file" 
                                    id="<?= $type ?>" 
                                    name="<?= $type ?>[]" 
                                    multiple
                                    accept="<?php 
                                        if (!empty($allowed_formats)) {
                                            echo '.' . implode(',.', $allowed_formats);
                                        } else {
                                            // Default based on type
                                            $defaults = [
                                                'video' => '.mp4,.avi,.mov,.wmv,.mkv',
                                                'image' => '.jpg,.jpeg,.png,.gif,.bmp,.webp',
                                                'document' => '.doc,.docx,.txt,.rtf',
                                                'pdf' => '.pdf',
                                                'excel' => '.xls,.xlsx,.csv',
                                                'presentation' => '.ppt,.pptx,.odp',
                                                'code' => '.js,.php,.py,.java,.cpp,.c,.html,.css,.sql'
                                            ];
                                            echo $defaults[$type] ?? '*';
                                        }
                                    ?>"
                                    class="file-input"
                                    onchange="handleFileSelect(this, '<?= $type ?>')"
                                >
                                <label for="<?= $type ?>" class="file-upload-label" id="label_<?= $type ?>">
                                    <i class="bx bx-cloud-upload"></i>
                                    <div>
                                        <span>Choose <?= strtolower($label[0]) ?> files</span>
                                        <small style="display: block; margin-top: 0.25rem; color: var(--text-secondary);">
                                            <?php if (!empty($allowed_formats)): ?>
                                                Allowed: <?= strtoupper(implode(', ', $allowed_formats)) ?>
                                            <?php endif; ?>
                                            <?php if ($assignment['max_file_size']): ?>
                                                • Max: <?= $assignment['max_file_size'] ?>MB per file
                                            <?php endif; ?>
                                            • Multiple files supported
                                        </small>
                                    </div>
                                </label>
                                <div id="preview_<?= $type ?>" class="file-preview" style="display: none; margin-top: 1rem;"></div>
                            </div>
                        </div>
                    <?php endif; ?>
                <?php endforeach; ?>
                
                <!-- Show Current Submissions if Any -->
                <?php if ($assignment['file_url']): ?>
                    <div style="background: #f0f9ff; border: 1px solid #0ea5e9; border-radius: 12px; padding: 1rem; margin: 1rem 0;">
                        <h5 style="color: #0369a1; margin-bottom: 0.75rem; font-weight: 600;">
                            <i class="bx bx-check-circle"></i> Current Submissions
                        </h5>
                        <?php 
                            $current_files = json_decode($assignment['file_url'], true);
                            if (is_array($current_files)):
                        ?>
                            <?php foreach ($current_files as $file): ?>
                                <div style="display: flex; align-items: center; gap: 0.5rem; margin-bottom: 0.5rem;">
                                    <i class="bx bx-file" style="color: #0369a1;"></i>
                                    <span><?= htmlspecialchars($file['original_name'] ?? $file['file_path'] ?? 'Unknown file') ?></span>
                                    <small style="color: var(--text-secondary);">(<?= ucfirst($file['type'] ?? 'file') ?>)</small>
                                </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <p style="color: var(--text-secondary); margin: 0;">Previous submission available</p>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
                
                <!-- Feedback Display -->
                <?php if ($assignment['feedback']): ?>
                    <div style="background: #fef3c7; border: 1px solid #f59e0b; border-radius: 12px; padding: 1rem; margin: 1rem 0;">
                        <h5 style="color: #92400e; margin-bottom: 0.75rem; font-weight: 600;">
                            <i class="bx bx-message-dots"></i> Instructor Feedback
                        </h5>
                        <div style="color: #78350f;"><?= nl2br(htmlspecialchars($assignment['feedback'])) ?></div>
                    </div>
                <?php endif; ?>
                
                <div style="margin-top: 2rem; padding-top: 1.5rem; border-top: 2px solid var(--border-color);">
                    <button type="submit" class="submit-btn" id="submitBtn">
                        <i class="bx bx-send"></i>
                        <?= $assignment['submission_id'] ? 'Update Submission' : 'Submit Assignment' ?>
                    </button>
                    
                    <p style="font-size: 0.85rem; color: var(--text-secondary); margin-top: 1rem; line-height: 1.5;">
                        <i class="bx bx-info-circle"></i>
                        By submitting this assignment, you confirm that the work is your own 
                        <?= $assignment['group_assignment'] ? '(or your team\'s)' : '' ?> 
                        and adheres to academic integrity policies.
                        <?php if ($assignment['contributes_to_final_grade']): ?>
                            <strong>This assignment counts toward your final grade and certification.</strong>
                        <?php endif; ?>
                    </p>
                </div>
            </form>
        <?php endif; ?>

    </div>

    <script>
        // Enhanced file upload handling for multiple submission types
        function handleFileSelect(input, type) {
            const label = document.getElementById(`label_${type}`);
            const preview = document.getElementById(`preview_${type}`);
            const files = input.files;
            
            if (files.length > 0) {
                // Update label to show selection
                const fileText = files.length === 1 ? files[0].name : `${files.length} files selected`;
                label.innerHTML = `
                    <i class="bx bx-check-circle" style="color: var(--success-color);"></i>
                    <div>
                        <span style="color: var(--success-color);">${fileText}</span>
                        <small style="display: block; margin-top: 0.25rem; color: var(--text-secondary);">
                            Click to change selection
                        </small>
                    </div>
                `;
                label.style.borderColor = 'var(--success-color)';
                label.style.background = '#f0fdf4';
                
                // Show file preview
                preview.style.display = 'block';
                preview.innerHTML = '';
                
                Array.from(files).forEach((file, index) => {
                    const fileItem = document.createElement('div');
                    fileItem.style.cssText = `
                        display: flex; align-items: center; gap: 0.75rem; padding: 0.75rem; 
                        background: white; border: 1px solid #e5e7eb; border-radius: 8px; margin-bottom: 0.5rem;
                    `;
                    
                    const fileSize = (file.size / 1024 / 1024).toFixed(2);
                    const maxSize = <?= $assignment['max_file_size'] ?? 50 ?>;
                    const isValidSize = file.size <= (maxSize * 1024 * 1024);
                    
                    fileItem.innerHTML = `
                        <i class="bx bx-file" style="color: ${isValidSize ? '#10b981' : '#dc2626'}; font-size: 1.2rem;"></i>
                        <div style="flex: 1;">
                            <div style="font-weight: 500; color: var(--text-primary);">${file.name}</div>
                            <div style="font-size: 0.8rem; color: var(--text-secondary);">
                                ${fileSize}MB ${!isValidSize ? `(Exceeds ${maxSize}MB limit)` : ''}
                            </div>
                        </div>
                        <div style="color: ${isValidSize ? '#10b981' : '#dc2626'};">
                            <i class="bx ${isValidSize ? 'bx-check-circle' : 'bx-error-circle'}"></i>
                        </div>
                    `;
                    
                    preview.appendChild(fileItem);
                });
            } else {
                // Reset label to default
                resetFileLabel(type);
                preview.style.display = 'none';
            }
            
            // Validate form
            validateForm();
        }
        
        function resetFileLabel(type) {
            const label = document.getElementById(`label_${type}`);
            const typeLabels = {
                'file_upload': 'Choose generic file upload files',
                'video': 'Choose video submission files',
                'document': 'Choose document upload files',
                'image': 'Choose image upload files',
                'excel': 'Choose spreadsheet upload files',
                'pdf': 'Choose PDF document files',
                'presentation': 'Choose presentation upload files',
                'code': 'Choose code files',
                'multiple_files': 'Choose multiple files'
            };
            
            const labelText = typeLabels[type] || 'Choose file upload files';
            
            label.innerHTML = `
                <i class="bx bx-cloud-upload"></i>
                <div>
                    <span>${labelText}</span>
                    <small style="display: block; margin-top: 0.25rem; color: var(--text-secondary);">
                        <?php if (!empty($allowed_formats)): ?>
                            Allowed: <?= strtoupper(implode(', ', $allowed_formats)) ?>
                        <?php endif; ?>
                        <?php if ($assignment['max_file_size']): ?>
                            • Max: <?= $assignment['max_file_size'] ?>MB per file
                        <?php endif; ?>
                        • Multiple files supported
                    </small>
                </div>
            `;
            label.style.borderColor = '';
            label.style.background = '';
        }
        
        function validateForm() {
            const submitBtn = document.getElementById('submitBtn');
            const submissionTypes = <?= json_encode($submission_types) ?>;
            let isValid = true;
            let errorMessage = '';
            
            // Check if all required submission types have files or content
            submissionTypes.forEach(type => {
                if (type === 'text_entry') {
                    const textArea = document.getElementById('text_submission');
                    if (textArea && textArea.value.trim() === '') {
                        isValid = false;
                        errorMessage = 'Text submission is required.';
                    }
                } else {
                    const fileInput = document.getElementById(type);
                    if (fileInput && fileInput.files.length === 0) {
                        isValid = false;
                        errorMessage = `${type.replace('_', ' ')} files are required.`;
                    }
                }
            });
            
            // Visual feedback for submit button
            if (isValid) {
                submitBtn.style.opacity = '1';
                submitBtn.style.cursor = 'pointer';
                submitBtn.disabled = false;
            } else {
                submitBtn.style.opacity = '0.6';
                submitBtn.style.cursor = 'not-allowed';
                submitBtn.disabled = true;
            }
        }
        
        // Form submission handling
        document.getElementById('submissionForm')?.addEventListener('submit', function(e) {
            const submitBtn = document.getElementById('submitBtn');
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i> Submitting...';
        });
        
        // Real-time validation for text area
        document.getElementById('text_submission')?.addEventListener('input', validateForm);
        
        // Initialize form validation
        document.addEventListener('DOMContentLoaded', function() {
            validateForm();
            
            // Add drag and drop functionality
            const fileUploadAreas = document.querySelectorAll('.file-upload');
            
            fileUploadAreas.forEach(area => {
                const input = area.querySelector('input[type="file"]');
                const label = area.querySelector('.file-upload-label');
                
                ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
                    area.addEventListener(eventName, preventDefaults, false);
                });
                
                ['dragenter', 'dragover'].forEach(eventName => {
                    area.addEventListener(eventName, highlight, false);
                });
                
                ['dragleave', 'drop'].forEach(eventName => {
                    area.addEventListener(eventName, unhighlight, false);
                });
                
                area.addEventListener('drop', function(e) {
                    const files = e.dataTransfer.files;
                    input.files = files;
                    handleFileSelect(input, input.id);
                }, false);
                
                function preventDefaults(e) {
                    e.preventDefault();
                    e.stopPropagation();
                }
                
                function highlight() {
                    label.style.borderColor = 'var(--panadite-primary)';
                    label.style.background = 'var(--panadite-light)';
                }
                
                function unhighlight() {
                    label.style.borderColor = '';
                    label.style.background = '';
                }
            });
        });
    </script>
</body>
</html>
