<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
requireRole('student');
require_once '../config/db_connect.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input) {
    echo json_encode(['success' => false, 'message' => 'Invalid JSON input']);
    exit;
}

$student_id = $_SESSION['user_id'];
$question = trim($input['question'] ?? '');
$type = $input['type'] ?? ''; // 'assignment' or 'quiz'

if (empty($question)) {
    echo json_encode(['success' => false, 'message' => 'Question is required']);
    exit;
}

if (!in_array($type, ['assignment', 'quiz'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid question type']);
    exit;
}

try {
    if ($type === 'assignment') {
        $assignment_id = intval($input['assignment_id'] ?? 0);
        
        if ($assignment_id <= 0) {
            echo json_encode(['success' => false, 'message' => 'Invalid assignment ID']);
            exit;
        }
        
        // Check if student is enrolled in the course for this assignment
        $check_query = "
            SELECT a.assignment_id, c.title as course_title
            FROM assignments a
            JOIN courses c ON a.course_id = c.course_id
            JOIN enrollments e ON c.course_id = e.course_id
            WHERE a.assignment_id = ? AND e.student_id = ?
        ";
        
        $stmt = $pdo->prepare($check_query);
        $stmt->execute([$assignment_id, $student_id]);
        $assignment = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$assignment) {
            echo json_encode(['success' => false, 'message' => 'Assignment not found or access denied']);
            exit;
        }
        
        // Insert assignment question
        $insert_query = "
            INSERT INTO assignment_questions (assignment_id, student_id, question, created_at)
            VALUES (?, ?, ?, NOW())
        ";
        
        $stmt = $pdo->prepare($insert_query);
        $stmt->execute([$assignment_id, $student_id, $question]);
        
    } else { // quiz
        $quiz_id = intval($input['quiz_id'] ?? 0);
        
        if ($quiz_id <= 0) {
            echo json_encode(['success' => false, 'message' => 'Invalid quiz ID']);
            exit;
        }
        
        // Check if student is enrolled in the course for this quiz
        $check_query = "
            SELECT q.quiz_id, c.title as course_title
            FROM quizzes q
            JOIN courses c ON q.course_id = c.course_id
            JOIN enrollments e ON c.course_id = e.course_id
            WHERE q.quiz_id = ? AND e.student_id = ?
        ";
        
        $stmt = $pdo->prepare($check_query);
        $stmt->execute([$quiz_id, $student_id]);
        $quiz = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$quiz) {
            echo json_encode(['success' => false, 'message' => 'Quiz not found or access denied']);
            exit;
        }
        
        // Insert quiz question
        $insert_query = "
            INSERT INTO quiz_questions (quiz_id, student_id, question, created_at)
            VALUES (?, ?, ?, NOW())
        ";
        
        $stmt = $pdo->prepare($insert_query);
        $stmt->execute([$quiz_id, $student_id, $question]);
    }
    
    echo json_encode(['success' => true, 'message' => 'Question submitted successfully']);
    
} catch (PDOException $e) {
    error_log("Submit question error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Database error occurred']);
}
?>
