<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';

// Require student role to access this page
requireRole('student');

// Include database connection
require_once '../config/db_connect.php';

// Get student ID
$student_id = $_SESSION['user_id'];

// Check if form was submitted
if ($_SERVER['REQUEST_METHOD'] != 'POST') {
    $_SESSION['error'] = "Invalid request method";
    header('Location: courses.php');
    exit();
}

// Get quiz ID from form
if (!isset($_POST['quiz_id']) || !is_numeric($_POST['quiz_id'])) {
    $_SESSION['error'] = "Invalid quiz ID";
    header('Location: courses.php');
    exit();
}

$quiz_id = mysqli_real_escape_string($conn, $_POST['quiz_id']);

// Get quiz details
$quizQuery = "
    SELECT q.*, c.course_id
    FROM quizzes q
    JOIN courses c ON q.course_id = c.course_id
    WHERE q.quiz_id = ?
";

$quizStmt = $conn->prepare($quizQuery);
$quizStmt->bind_param("i", $quiz_id);
$quizStmt->execute();
$quizResult = $quizStmt->get_result();

if ($quizResult->num_rows == 0) {
    $_SESSION['error'] = "Quiz not found";
    header('Location: courses.php');
    exit();
}

$quiz = $quizResult->fetch_assoc();
$course_id = $quiz['course_id'];

// Check if student is enrolled in this course
$enrollQuery = "SELECT * FROM enrollments WHERE student_id = ? AND course_id = ?";
$enrollStmt = $conn->prepare($enrollQuery);
$enrollStmt->bind_param("ii", $student_id, $course_id);
$enrollStmt->execute();
$enrollResult = $enrollStmt->get_result();

$is_enrolled = ($enrollResult->num_rows > 0);

if (!$is_enrolled) {
    $_SESSION['error'] = "You are not enrolled in this course";
    header('Location: courses.php');
    exit();
}

// Check if student has reached maximum attempts
$attemptQuery = "SELECT COUNT(*) as attempt_count FROM quiz_attempts WHERE student_id = ? AND quiz_id = ?";
$attemptStmt = $conn->prepare($attemptQuery);
$attemptStmt->bind_param("ii", $student_id, $quiz_id);
$attemptStmt->execute();
$attemptResult = $attemptStmt->get_result();
$attempt = $attemptResult->fetch_assoc();

$maxAttempts = isset($quiz['max_attempts']) ? $quiz['max_attempts'] : PHP_INT_MAX;
if ($attempt['attempt_count'] >= $maxAttempts) {
    $_SESSION['error'] = "You have reached the maximum number of attempts for this quiz";
    header("Location: course-quizzes.php?id={$course_id}");
    exit();
}

// Get questions for the quiz
$questionQuery = "
    SELECT q.*
    FROM quiz_questions q
    WHERE q.quiz_id = ?
";

$questionStmt = $conn->prepare($questionQuery);
$questionStmt->bind_param("i", $quiz_id);
$questionStmt->execute();
$questionResult = $questionStmt->get_result();

// Calculate total score
$totalPoints = 0;
$earnedPoints = 0;
$answersData = [];

// Start transaction
$conn->begin_transaction();

try {
    // Create a new attempt
    $attemptNumber = $attempt['attempt_count'] + 1;
    $insertAttemptQuery = "
        INSERT INTO quiz_attempts (quiz_id, student_id, attempt_number, attempt_date)
        VALUES (?, ?, ?, NOW())
    ";
    
    $insertAttemptStmt = $conn->prepare($insertAttemptQuery);
    $insertAttemptStmt->bind_param("iii", $quiz_id, $student_id, $attemptNumber);
    $insertAttemptStmt->execute();
    $attempt_id = $conn->insert_id;
    
    // Process each question
    while ($question = $questionResult->fetch_assoc()) {
        $question_id = $question['question_id'];
        $question_type = $question['question_type'] ?? 'multiple_choice';
        $points = $question['points'] ?? 1;
        $totalPoints += $points;
        
        $answerData = [
            'question_id' => $question_id,
            'attempt_id' => $attempt_id,
            'given_answer' => '',
            'correct_answer' => '',
            'is_correct' => 0,
            'points_earned' => 0
        ];
        
        // Get the correct answer(s) for this question
        $correctAnswerQuery = "
            SELECT o.*, o.option_text as answer_text
            FROM question_options o
            WHERE o.question_id = ? AND o.is_correct = 1
        ";
        
        $correctAnswerStmt = $conn->prepare($correctAnswerQuery);
        $correctAnswerStmt->bind_param("i", $question_id);
        $correctAnswerStmt->execute();
        $correctAnswerResult = $correctAnswerStmt->get_result();
        
        $correctAnswers = [];
        while ($correctOption = $correctAnswerResult->fetch_assoc()) {
            $correctAnswers[] = $correctOption['option_id'];
            $answerData['correct_answer'] .= ($answerData['correct_answer'] ? ', ' : '') . $correctOption['answer_text'];
        }
        
        // Process student's answer based on question type
        $isCorrect = false;
        
        if ($question_type == 'multiple_choice') {
            $studentAnswer = isset($_POST["question_{$question_id}"]) ? $_POST["question_{$question_id}"] : null;
            
            if ($studentAnswer !== null) {
                // Get the text of the student's answer
                $studentAnswerTextQuery = "SELECT option_text FROM question_options WHERE option_id = ?";
                $studentAnswerTextStmt = $conn->prepare($studentAnswerTextQuery);
                $studentAnswerTextStmt->bind_param("i", $studentAnswer);
                $studentAnswerTextStmt->execute();
                $studentAnswerTextResult = $studentAnswerTextStmt->get_result();
                
                if ($studentAnswerTextResult->num_rows > 0) {
                    $answerData['given_answer'] = $studentAnswerTextResult->fetch_assoc()['option_text'];
                }
                
                // Check if the answer is correct
                if (in_array($studentAnswer, $correctAnswers)) {
                    $isCorrect = true;
                    $answerData['is_correct'] = 1;
                    $answerData['points_earned'] = $points;
                    $earnedPoints += $points;
                }
            }
        } elseif ($question_type == 'multiple_answer') {
            $studentAnswers = isset($_POST["question_{$question_id}"]) ? $_POST["question_{$question_id}"] : [];
            
            if (!empty($studentAnswers) && is_array($studentAnswers)) {
                // Get the text of the student's answers
                $studentAnswerTexts = [];
                foreach ($studentAnswers as $optionId) {
                    $studentAnswerTextQuery = "SELECT option_text FROM question_options WHERE option_id = ?";
                    $studentAnswerTextStmt = $conn->prepare($studentAnswerTextQuery);
                    $studentAnswerTextStmt->bind_param("i", $optionId);
                    $studentAnswerTextStmt->execute();
                    $studentAnswerTextResult = $studentAnswerTextStmt->get_result();
                    
                    if ($studentAnswerTextResult->num_rows > 0) {
                        $studentAnswerTexts[] = $studentAnswerTextResult->fetch_assoc()['option_text'];
                    }
                }
                
                $answerData['given_answer'] = implode(', ', $studentAnswerTexts);
                
                // Check if all correct answers are selected and no incorrect ones
                $allCorrect = true;
                $pointsPerOption = $points / count($correctAnswers);
                $correctCount = 0;
                
                foreach ($studentAnswers as $answer) {
                    if (!in_array($answer, $correctAnswers)) {
                        $allCorrect = false;
                    } else {
                        $correctCount++;
                    }
                }
                
                if ($allCorrect && count($studentAnswers) == count($correctAnswers)) {
                    $isCorrect = true;
                    $answerData['is_correct'] = 1;
                    $answerData['points_earned'] = $points;
                    $earnedPoints += $points;
                } else if ($correctCount > 0) {
                    // Partial credit
                    $partialPoints = $pointsPerOption * $correctCount;
                    $answerData['points_earned'] = $partialPoints;
                    $earnedPoints += $partialPoints;
                }
            }
        } elseif ($question_type == 'true_false') {
            $studentAnswer = isset($_POST["question_{$question_id}"]) ? $_POST["question_{$question_id}"] : null;
            
            if ($studentAnswer !== null) {
                $answerData['given_answer'] = $studentAnswer;
                
                // Check if the answer is correct
                $correctAnswer = !empty($correctAnswers) ? 'true' : 'false';
                if ($studentAnswer == $correctAnswer) {
                    $isCorrect = true;
                    $answerData['is_correct'] = 1;
                    $answerData['points_earned'] = $points;
                    $earnedPoints += $points;
                }
            }
        } elseif ($question_type == 'short_answer') {
            $studentAnswer = isset($_POST["question_{$question_id}"]) ? trim($_POST["question_{$question_id}"]) : '';
            
            if ($studentAnswer !== '') {
                $answerData['given_answer'] = $studentAnswer;
                
                // For short answer, we'll need to check for keywords or exact matches
                // This is a simplified implementation
                $correctAnswerKeywords = explode(',', $answerData['correct_answer']);
                $keywordMatches = 0;
                
                foreach ($correctAnswerKeywords as $keyword) {
                    if (stripos($studentAnswer, trim($keyword)) !== false) {
                        $keywordMatches++;
                    }
                }
                
                if ($keywordMatches == count($correctAnswerKeywords)) {
                    $isCorrect = true;
                    $answerData['is_correct'] = 1;
                    $answerData['points_earned'] = $points;
                    $earnedPoints += $points;
                } else if ($keywordMatches > 0) {
                    // Partial credit
                    $partialPoints = $points * ($keywordMatches / count($correctAnswerKeywords));
                    $answerData['points_earned'] = $partialPoints;
                    $earnedPoints += $partialPoints;
                }
            }
        }
        
        // Insert answer record
        $insertAnswerQuery = "
            INSERT INTO quiz_answers 
            (attempt_id, question_id, given_answer, correct_answer, is_correct, points_earned)
            VALUES (?, ?, ?, ?, ?, ?)
        ";
        
        $insertAnswerStmt = $conn->prepare($insertAnswerQuery);
        $insertAnswerStmt->bind_param(
            "iissid",
            $attempt_id,
            $question_id,
            $answerData['given_answer'],
            $answerData['correct_answer'],
            $answerData['is_correct'],
            $answerData['points_earned']
        );
        $insertAnswerStmt->execute();
    }
    
    // Update attempt with score
    $updateAttemptQuery = "UPDATE quiz_attempts SET score = ? WHERE attempt_id = ?";
    $updateAttemptStmt = $conn->prepare($updateAttemptQuery);
    $updateAttemptStmt->bind_param("di", $earnedPoints, $attempt_id);
    $updateAttemptStmt->execute();
    
    // Commit transaction
    $conn->commit();
    
    // Redirect to results page
    $_SESSION['success'] = "Quiz submitted successfully!";
    header("Location: quiz_results.php?id={$quiz_id}");
    exit();
    
} catch (Exception $e) {
    // Roll back transaction on error
    $conn->rollback();
    
    $_SESSION['error'] = "Error submitting quiz: " . $e->getMessage();
    header("Location: course-quizzes.php?id={$course_id}");
    exit();
}
?>
