<?php
// IMPORTANT: Clean JSON response - no extra output allowed
ob_start(); // Start output buffering to catch any stray output
ini_set('display_errors', 0);
error_reporting(0);

// Set content type to JSON immediately  
header('Content-Type: application/json');
header('Cache-Control: no-cache, must-revalidate');
header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');

// Use the same session handling as other pages in the project
require_once '../includes/session_start.php';

// Include ID encryption functions
require_once '../includes/id_encryption.php';

// Database connection - try multiple possible paths
$db_connected = false;
$pdo = null;

// Try different database connection paths
$db_paths = [
    '../config/database.php',
    '../includes/db_connect.php', 
    '../config/db_connect.php',
    'includes/db_connect.php'
];

foreach ($db_paths as $path) {
    if (file_exists($path)) {
        try {
            require_once $path;
            if (function_exists('getDbConnection')) {
                $pdo = getDbConnection();
                $db_connected = true;
                break;
            }
        } catch (Exception $e) {
            continue;
        }
    }
}

// If no database connection found, create one inline
if (!$db_connected) {
    try {
        // LIVE SERVER DATABASE CONNECTION - CORRECT CREDENTIALS
        $host = 'localhost';
        $dbname = 'twpurnyc_panadite_DB'; // Correct database name from debug
        $username = 'twpurnyc_149586Alain'; // Correct username from debug
        $password = '149586Alain#'; // Correct password from debug
        
        $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        $db_connected = true;
        
        // Log successful connection for debugging
        error_log("📝 REVIEW SUBMIT: Database connected successfully to $dbname");
        
    } catch (Exception $e) {
        error_log("📝 REVIEW SUBMIT: Database connection failed - " . $e->getMessage());
        send_json_response(['success' => false, 'message' => 'Database connection failed']);
        exit;
    }
}

// Debug logging function
function debug_log($message, $data = null) {
    error_log("📝 REVIEW SUBMIT: " . $message . ($data ? " | Data: " . json_encode($data) : ""));
}

debug_log("Review submission script started");

try {
    // Check if user is logged in
    if (!isset($_SESSION['user_id'])) {
        debug_log("User not logged in");
        send_json_response(['success' => false, 'message' => 'Please log in to submit a review']);
        exit;
    }

    // Check if request is POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        debug_log("Invalid request method", $_SERVER['REQUEST_METHOD']);
        send_json_response(['success' => false, 'message' => 'Invalid request method']);
        exit;
    }

    // Get form data
    $user_id = $_SESSION['user_id'];
    $course_id = $_POST['course_id'] ?? '';
    $rating = $_POST['rating'] ?? '';
    $review_title = trim($_POST['review_title'] ?? '');
    $review_text = trim($_POST['review_text'] ?? '');
    $user_token = $_POST['user_token'] ?? '';

    debug_log("Form data received", [
        'user_id' => $user_id,
        'course_id' => $course_id,
        'rating' => $rating,
        'title_length' => strlen($review_title),
        'text_length' => strlen($review_text)
    ]);

    // Validate required fields
    if (empty($course_id) || empty($rating) || empty($review_title) || empty($review_text)) {
        debug_log("Missing required fields");
        send_json_response(['success' => false, 'message' => 'No data received']);
        exit;
    }

    // Validate rating
    $rating = intval($rating);
    if ($rating < 1 || $rating > 5) {
        debug_log("Invalid rating value", $rating);
        send_json_response(['success' => false, 'message' => 'Rating must be between 1 and 5']);
        exit;
    }

    // Validate review title
    if (strlen($review_title) < 3 || strlen($review_title) > 255) {
        debug_log("Invalid review title length");
        send_json_response(['success' => false, 'message' => 'Review title must be 3-255 characters']);
        exit;
    }

    // Validate review text
    if (strlen($review_text) < 10) {
        debug_log("Review text too short");
        send_json_response(['success' => false, 'message' => 'Review text must be 10-1000 characters']);
        exit;
    }

    // Validate user token (basic security check)
    $expected_token = md5($user_id . 'panadite_review_token');
    if ($user_token !== $expected_token) {
        debug_log("Invalid user token");
        send_json_response(['success' => false, 'message' => 'Security token validation failed']);
        exit;
    }

    // Decrypt course ID using the project's proper encryption system
    $decrypted_course_id = decryptId($course_id);
    
    debug_log("Course ID decryption", [
        'encrypted' => $course_id,
        'decrypted' => $decrypted_course_id,
        'is_numeric' => is_numeric($decrypted_course_id)
    ]);
    
    if (!$decrypted_course_id || !is_numeric($decrypted_course_id)) {
        debug_log("Invalid course ID after decryption", [
            'original' => $course_id,
            'decrypted' => $decrypted_course_id
        ]);
        send_json_response(['success' => false, 'message' => 'Invalid course ID']);
        exit;
    }

    debug_log("Course ID decrypted", $decrypted_course_id);

    // Use the already connected database
    if (!$pdo) {
        send_json_response(['success' => false, 'message' => 'Database connection not available']);
        exit;
    }
    
    // Check if user is enrolled in the course (try both user_id and student_id)
    $enrollment_queries = [
        "SELECT enrollment_id FROM enrollments WHERE student_id = ? AND course_id = ? AND status = 'active'",
        "SELECT enrollment_id FROM enrollments WHERE user_id = ? AND course_id = ? AND status = 'active'",
        "SELECT id FROM enrollments WHERE student_id = ? AND course_id = ? AND status = 'active'", 
        "SELECT id FROM enrollments WHERE user_id = ? AND course_id = ? AND status = 'active'",
        "SELECT * FROM enrollments WHERE student_id = ? AND course_id = ? LIMIT 1",
        "SELECT * FROM enrollments WHERE user_id = ? AND course_id = ? LIMIT 1"
    ];
    
    $enrollment = null;
    foreach ($enrollment_queries as $query) {
        try {
            $stmt = $pdo->prepare($query);
            $stmt->execute([$user_id, $decrypted_course_id]);
            $enrollment = $stmt->fetch();
            if ($enrollment) {
                debug_log("Found enrollment with query: " . $query);
                break;
            }
        } catch (Exception $e) {
            debug_log("Enrollment query failed: " . $query . " - " . $e->getMessage());
            continue;
        }
    }

    if (!$enrollment) {
        debug_log("User not enrolled in course after trying all queries");
        send_json_response(['success' => false, 'message' => 'You must be enrolled in this course to leave a review']);
        exit;
    }

    // Check if user already reviewed this course (use correct column name: student_id)  
    $review_queries = [
        "SELECT review_id FROM course_reviews WHERE student_id = ? AND course_id = ?",
        "SELECT review_id FROM course_reviews WHERE user_id = ? AND course_id = ?",
        "SELECT * FROM course_reviews WHERE student_id = ? AND course_id = ? LIMIT 1",
        "SELECT * FROM course_reviews WHERE user_id = ? AND course_id = ? LIMIT 1"
    ];
    
    $existing_review = null;
    foreach ($review_queries as $query) {
        try {
            $stmt = $pdo->prepare($query);
            $stmt->execute([$user_id, $decrypted_course_id]);
            $existing_review = $stmt->fetch();
            if ($existing_review) {
                debug_log("Found existing review with query: " . $query);
                break;
            }
        } catch (Exception $e) {
            debug_log("Review check query failed: " . $query . " - " . $e->getMessage());
            continue;
        }
    }

    if ($existing_review) {
        debug_log("User already reviewed this course");
        send_json_response(['success' => false, 'message' => 'You have already submitted a review for this course. You can edit your existing review']);
        exit;
    }

    // Create course_reviews table if it doesn't exist (flexible schema)
    $create_table_sql = "CREATE TABLE IF NOT EXISTS course_reviews (
        review_id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        course_id INT NOT NULL,
        rating INT NOT NULL CHECK (rating >= 1 AND rating <= 5),
        review_title VARCHAR(255) NOT NULL,
        review_text TEXT NOT NULL,
        status ENUM('pending', 'approved', 'rejected') DEFAULT 'approved',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_course_reviews_course (course_id),
        INDEX idx_course_reviews_user (user_id),
        INDEX idx_course_reviews_status (status),
        UNIQUE KEY unique_user_course_review (user_id, course_id)
    )";

    try {
        $pdo->exec($create_table_sql);
        debug_log("Course reviews table ensured successfully");
    } catch (Exception $e) {
        debug_log("Table creation failed (table may already exist): " . $e->getMessage());
    }

    // Insert the review (use correct column names: student_id and is_approved)
    $insert_queries = [
        "INSERT INTO course_reviews (student_id, course_id, rating, review_title, review_text, is_approved, created_at, updated_at) VALUES (?, ?, ?, ?, ?, 1, NOW(), NOW())",
        "INSERT INTO course_reviews (student_id, course_id, rating, review_title, review_text, is_approved, created_at) VALUES (?, ?, ?, ?, ?, 1, NOW())",
        "INSERT INTO course_reviews (student_id, course_id, rating, review_title, review_text, is_approved) VALUES (?, ?, ?, ?, ?, 1)",
        "INSERT INTO course_reviews (student_id, course_id, rating, review_title, review_text) VALUES (?, ?, ?, ?, ?)"
    ];
    
    $result = false;
    $review_id = null;
    
    foreach ($insert_queries as $query) {
        try {
            $stmt = $pdo->prepare($query);
            $result = $stmt->execute([$user_id, $decrypted_course_id, $rating, $review_title, $review_text]);
            
            if ($result) {
                $review_id = $pdo->lastInsertId();
                debug_log("Review inserted successfully with query: " . $query, ['review_id' => $review_id]);
                break;
            }
        } catch (Exception $e) {
            debug_log("Insert query failed: " . $query . " - " . $e->getMessage());
            continue;
        }
    }

    if ($result && $review_id) {
        debug_log("Review saved successfully", ['review_id' => $review_id]);

        // Update course average rating (use correct column name: is_approved)
        $stmt = $pdo->prepare("
            SELECT AVG(rating) as avg_rating, COUNT(*) as review_count 
            FROM course_reviews 
            WHERE course_id = ? AND is_approved = 1
        ");
        $stmt->execute([$decrypted_course_id]);
        $rating_data = $stmt->fetch();

        if ($rating_data) {
            $stmt = $pdo->prepare("UPDATE courses SET average_rating = ?, review_count = ? WHERE id = ?");
            $stmt->execute([round($rating_data['avg_rating'], 1), $rating_data['review_count'], $decrypted_course_id]);
            debug_log("Course rating updated", $rating_data);
        }

        send_json_response([
            'success' => true,
            'message' => 'Review submitted successfully!',
            'review_id' => $review_id
        ]);
    } else {
        debug_log("CRITICAL ERROR: All insert queries failed", [
            'result' => $result,
            'review_id' => $review_id,
            'user_id' => $user_id,
            'decrypted_course_id' => $decrypted_course_id,
            'rating' => $rating,
            'review_title' => $review_title,
            'review_text' => substr($review_text, 0, 100) . '...',
            'all_insert_queries_tried' => count($insert_queries)
        ]);
        
        // Try to get more specific error information
        $error_info = $pdo->errorInfo();
        debug_log("PDO Error Info", $error_info);
        
        send_json_response([
            'success' => false,
            'message' => 'Failed to save review. Please try again.',
            'debug' => 'All insert queries failed. Check debug logs for details.'
        ]);
    }

} catch (Exception $e) {
    debug_log("Exception occurred", $e->getMessage());
    error_log("📝 REVIEW SUBMIT ERROR: " . $e->getMessage() . " | File: " . $e->getFile() . " | Line: " . $e->getLine());
    
    // Clean any previous output before sending JSON
    ob_clean();
    
    echo json_encode([
        'success' => false, 
        'message' => 'An error occurred while submitting your review. Please try again.',
        'debug' => $e->getMessage() // Remove this in production
    ]);
    
    // Ensure clean output
    ob_end_flush();
    exit;
}

// Clean any stray output and ensure proper JSON response
function send_json_response($data) {
    ob_clean(); // Clear any previous output
    echo json_encode($data);
    ob_end_flush();
    exit;
}
?>
