<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
requireRole('student');
require_once '../config/db_connect.php';
require_once '../includes/id_encryption.php';

$student_id = $_SESSION['user_id'];

// Handle both encrypted and raw quiz IDs
$quiz_id = 0;
$raw_id = $_GET['id'] ?? '';

if (!empty($raw_id)) {
    // Try to decrypt first (for encrypted IDs from course detail)
    try {
        if (class_exists('IdEncryption') && method_exists('IdEncryption', 'decrypt')) {
            $decrypted_id = IdEncryption::decrypt($raw_id);
            if (is_numeric($decrypted_id) && $decrypted_id > 0) {
                $quiz_id = intval($decrypted_id);
            }
        }
    } catch (Exception $e) {
        // Decryption failed, try raw integer
        $quiz_id = 0;
    }
    
    // If decryption failed, try as raw integer (for backward compatibility)
    if ($quiz_id <= 0 && is_numeric($raw_id)) {
        $quiz_id = intval($raw_id);
    }
}

if ($quiz_id <= 0) {
    header('Location: assignments.php');
    exit;
}

// Get quiz details with video information using MySQLi
$quiz_query = "
    SELECT q.*, c.title as course_title, u.first_name, u.last_name,
           v.title as video_title, v.thumbnail_path, v.duration
    FROM quizzes q 
    JOIN courses c ON q.course_id = c.course_id 
    JOIN users u ON q.teacher_id = u.user_id 
    LEFT JOIN videos v ON q.video_id = v.video_id
    WHERE q.quiz_id = ? AND q.is_published = 1
";

$quiz_stmt = $conn->prepare($quiz_query);

// If enhanced query fails, fallback to original query
if (!$quiz_stmt) {
    // Fallback to original query without video information
    $quiz_query = "
        SELECT q.*, c.title as course_title, u.first_name, u.last_name 
        FROM quizzes q 
        JOIN courses c ON q.course_id = c.course_id 
        JOIN users u ON q.teacher_id = u.user_id 
        WHERE q.quiz_id = ? AND q.is_published = 1
    ";
    $quiz_stmt = $conn->prepare($quiz_query);
}

if ($quiz_stmt) {
    $quiz_stmt->bind_param("i", $quiz_id);
    $quiz_stmt->execute();
    $quiz_result = $quiz_stmt->get_result();
    $quiz = $quiz_result->fetch_assoc();
} else {
    // If both queries fail, redirect
    header('Location: assignments.php');
    exit;
}

if (!$quiz) {
    header('Location: assignments.php');
    exit;
}

// Check if student is enrolled in the course
$enrollment_stmt = $conn->prepare("SELECT enrollment_id FROM enrollments WHERE student_id = ? AND course_id = ?");
$enrollment_stmt->bind_param("ii", $student_id, $quiz['course_id']);
$enrollment_stmt->execute();
$enrollment_result = $enrollment_stmt->get_result();
$enrollment = $enrollment_result->fetch_assoc();

if (!$enrollment) {
    header('Location: assignments.php');
    exit;
}

// Get quiz questions and options
$questions_stmt = $pdo->prepare("
    SELECT q.*, qo.option_id, qo.option_text, qo.is_correct
    FROM quiz_questions q
    LEFT JOIN quiz_question_options qo ON q.question_id = qo.question_id
    WHERE q.quiz_id = ?
    ORDER BY q.question_id, qo.option_id
");
$questions_stmt->execute([$quiz_id]);
$question_results = $questions_stmt->fetchAll(PDO::FETCH_ASSOC);

// Organize questions and options
$questions = [];
foreach ($question_results as $row) {
    $question_id = $row['question_id'];
    if (!isset($questions[$question_id])) {
        $questions[$question_id] = [
            'question_id' => $row['question_id'],
            'question_text' => $row['question_text'],
            'question_type' => $row['question_type'],
            'points' => $row['points'],
            'options' => []
        ];
    }
    
    if ($row['option_id']) {
        $questions[$question_id]['options'][] = [
            'option_id' => $row['option_id'],
            'option_text' => $row['option_text'],
            'is_correct' => $row['is_correct']
        ];
    }
}

// Check for existing attempt (but allow retries with retry=1 parameter)
$retry_mode = isset($_GET['retry']) && $_GET['retry'] == '1';
$existing_attempt = null;

if (!$retry_mode) {
    $attempt_stmt = $pdo->prepare("
        SELECT * FROM quiz_attempts 
        WHERE quiz_id = ? AND student_id = ? 
        ORDER BY start_time DESC LIMIT 1
    ");
    $attempt_stmt->execute([$quiz_id, $student_id]);
    $existing_attempt = $attempt_stmt->fetch(PDO::FETCH_ASSOC);
}

// If quiz is completed, fetch student's responses
$student_responses = [];
if ($existing_attempt && $existing_attempt['end_time']) {
    $responses_stmt = $pdo->prepare("
        SELECT qr.*, qo.option_text, qo.is_correct
        FROM quiz_responses qr
        LEFT JOIN quiz_question_options qo ON qr.answer_id = qo.option_id
        WHERE qr.attempt_id = ?
    ");
    $responses_stmt->execute([$existing_attempt['attempt_id']]);
    $responses_result = $responses_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($responses_result as $response) {
        $student_responses[$response['question_id']] = $response;
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && (isset($_POST['submit_quiz']) || isset($_POST['time_expired']))) {
    $time_expired = isset($_POST['time_expired']) && $_POST['time_expired'] == '1';
    try {
        $pdo->beginTransaction();
        
        // Create quiz attempt
        $attempt_stmt = $pdo->prepare("
            INSERT INTO quiz_attempts (quiz_id, student_id, start_time, end_time, score, pass_status, created_at)
            VALUES (?, ?, NOW(), NOW(), 0, 0, NOW())
        ");
        $attempt_stmt->execute([$quiz_id, $student_id]);
        $attempt_id = $pdo->lastInsertId();
        
        $total_points = 0;
        $earned_points = 0;
        
        // Process answers
        foreach ($questions as $question) {
            $question_id = $question['question_id'];
            $total_points += $question['points'];
            
            $answer = $_POST['question_' . $question_id] ?? '';
            
            // For time expired submissions, treat empty answers as 0 points
            if ($time_expired && empty($answer)) {
                continue; // Skip processing, will get 0 points
            }
            $is_correct = 0;
            $points_earned = 0;
            
            if ($question['question_type'] === 'multiple_choice' || $question['question_type'] === 'true_false') {
                $selected_option_id = intval($answer);
                
                // Check if answer is correct
                foreach ($question['options'] as $option) {
                    if ($option['option_id'] == $selected_option_id && $option['is_correct']) {
                        $is_correct = 1;
                        $points_earned = $question['points'];
                        $earned_points += $points_earned;
                        break;
                    }
                }
                
                // Save response
                $response_stmt = $pdo->prepare("
                    INSERT INTO quiz_responses (attempt_id, question_id, answer_id, is_correct, points_earned, created_at)
                    VALUES (?, ?, ?, ?, ?, NOW())
                ");
                $response_stmt->execute([$attempt_id, $question_id, $selected_option_id, $is_correct, $points_earned]);
                
            } elseif ($question['question_type'] === 'short_answer') {
                // For short answer, save text response (will need manual or AI grading)
                $response_stmt = $pdo->prepare("
                    INSERT INTO quiz_responses (attempt_id, question_id, text_response, is_correct, points_earned, created_at)
                    VALUES (?, ?, ?, 0, 0, NOW())
                ");
                $response_stmt->execute([$attempt_id, $question_id, trim($answer)]);
            }
        }
        
        // Calculate final score
        $score_percentage = $total_points > 0 ? round(($earned_points / $total_points) * 100, 2) : 0;
        $pass_status = $score_percentage >= $quiz['passing_score'] ? 1 : 0;
        
        // Update attempt with final score
        $update_attempt = $pdo->prepare("
            UPDATE quiz_attempts 
            SET score = ?, pass_status = ? 
            WHERE attempt_id = ?
        ");
        $update_attempt->execute([$score_percentage, $pass_status, $attempt_id]);
        
        $pdo->commit();
        
        // Redirect to results
        header("Location: quiz_results.php?attempt_id=" . $attempt_id);
        exit;
        
    } catch (Exception $e) {
        $pdo->rollBack();
        $error = "Error submitting quiz: " . $e->getMessage();
    }
}

$page_title = "Take Quiz: " . $quiz['title'];

// Set current page for navigation
$currentPage = 'assignments';

// Include modern header with sidebar and navbar
include_once __DIR__ . '/components/header.php';
?>

<!-- Take Quiz Content -->
<div class="take-quiz-content">
    <style>
        :root {
            --panadite-primary: #2563eb;
            --panadite-secondary: #1d4ed8;
            --panadite-light: #dbeafe;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --error-color: #ef4444;
            --text-primary: #1f2937;
            --text-secondary: #6b7280;
            --border-color: #e5e7eb;
            --bg-light: #f8fafc;
        }

        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, var(--bg-light) 0%, #e2e8f0 100%);
            color: var(--text-primary);
            min-height: 100vh;
        }

        /* Simple Quiz Header */
        .simple-quiz-header {
            padding: 2rem 0;
            margin-bottom: 2rem;
            background: white;
            border-bottom: 1px solid var(--border-color);
        }

        .quiz-title {
            font-size: 2rem;
            font-weight: 700;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
        }

        .instructor-info {
            color: var(--text-secondary);
            font-size: 1rem;
            margin-bottom: 0;
        }

        .header-background {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grid" width="10" height="10" patternUnits="userSpaceOnUse"><path d="M 10 0 L 0 0 0 10" fill="none" stroke="rgba(255,255,255,0.05)" stroke-width="0.5"/></pattern></defs><rect width="100" height="100" fill="url(%23grid)"/></svg>');
            opacity: 0.3;
        }

        /* Video Card in Header */
        .header-video-card {
            background: rgba(255, 255, 255, 0.15);
            backdrop-filter: blur(10px);
            border-radius: 16px;
            overflow: hidden;
            border: 1px solid rgba(255, 255, 255, 0.2);
            transition: all 0.3s ease;
        }

        .header-video-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
        }

        .header-video-thumbnail {
            position: relative;
            width: 100%;
            aspect-ratio: 16/9;
            overflow: hidden;
        }

        .header-thumbnail-img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: transform 0.3s ease;
        }

        .header-video-placeholder {
            display: flex;
            align-items: center;
            justify-content: center;
            height: 100%;
            background: rgba(255, 255, 255, 0.1);
            color: rgba(255, 255, 255, 0.7);
            font-size: 2rem;
        }

        .header-video-overlay {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(37, 99, 235, 0.9);
            color: white;
            border-radius: 50%;
            width: 50px;
            height: 50px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .header-video-thumbnail:hover .header-video-overlay {
            opacity: 1;
        }

        .header-video-thumbnail:hover .header-thumbnail-img {
            transform: scale(1.05);
        }

        .header-duration-badge {
            position: absolute;
            bottom: 8px;
            right: 8px;
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 4px 8px;
            border-radius: 6px;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .header-video-info {
            padding: 1rem;
        }

        .video-label {
            font-size: 0.75rem;
            color: rgba(255, 255, 255, 0.7);
            text-transform: uppercase;
            font-weight: 600;
            letter-spacing: 0.5px;
            margin-bottom: 0.25rem;
        }

        .video-title {
            font-size: 0.9rem;
            font-weight: 600;
            color: white;
        }

        /* Quiz Header Content */
        .quiz-header-content {
            position: relative;
            z-index: 2;
        }

        .quiz-type-badge {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            background: rgba(16, 185, 129, 0.2);
            border: 1px solid rgba(16, 185, 129, 0.3);
            color: #10b981;
            padding: 0.5rem 1rem;
            border-radius: 25px;
            font-size: 0.85rem;
            font-weight: 600;
            margin-bottom: 1rem;
        }

        .quiz-main-title {
            font-size: 2.25rem;
            font-weight: 700;
            margin-bottom: 1rem;
            color: white;
            text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }

        .quiz-meta-info {
            display: flex;
            align-items: center;
            gap: 1rem;
            margin-bottom: 1.5rem;
            flex-wrap: wrap;
        }

        .meta-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            color: rgba(255, 255, 255, 0.9);
            font-size: 0.95rem;
        }

        .meta-item i {
            color: rgba(255, 255, 255, 0.7);
        }

        .meta-separator {
            color: rgba(255, 255, 255, 0.5);
            font-weight: 300;
        }

        .quiz-stats-row {
            display: flex;
            gap: 1.5rem;
            flex-wrap: wrap;
        }

        .stat-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .stat-icon {
            background: rgba(255, 255, 255, 0.15);
            border-radius: 10px;
            width: 44px;
            height: 44px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.25rem;
        }

        .stat-content {
            line-height: 1.2;
        }

        .stat-value {
            font-size: 1.1rem;
            font-weight: 700;
            color: white;
        }

        .stat-label {
            font-size: 0.75rem;
            color: rgba(255, 255, 255, 0.7);
            text-transform: uppercase;
            font-weight: 500;
            letter-spacing: 0.5px;
        }

        /* Grade Status Card */
        .grade-status-card {
            position: relative;
            z-index: 2;
        }

        .grade-indicator {
            background: rgba(255, 255, 255, 0.15);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.2);
            border-radius: 12px;
            padding: 1rem;
            transition: all 0.3s ease;
        }

        .grade-indicator:hover {
            background: rgba(255, 255, 255, 0.25);
            transform: scale(1.05);
        }

        .grade-icon {
            font-size: 1.5rem;
            margin-bottom: 0.5rem;
            color: #fbbf24;
        }

        .grade-title {
            font-size: 0.85rem;
            font-weight: 600;
            color: white;
            margin-bottom: 0.25rem;
        }

        .grade-subtitle {
            font-size: 0.7rem;
            color: rgba(255, 255, 255, 0.7);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        /* Header Decorations */
        .header-decoration {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            pointer-events: none;
            overflow: hidden;
        }

        .decoration-circle {
            position: absolute;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.05);
        }

        .circle-1 {
            width: 200px;
            height: 200px;
            top: -50px;
            right: -50px;
        }

        .circle-2 {
            width: 150px;
            height: 150px;
            bottom: -30px;
            left: -30px;
        }

        .circle-3 {
            width: 80px;
            height: 80px;
            top: 50%;
            right: 20%;
        }

        /* Responsive Design for Header */
        @media (max-width: 768px) {
            .enhanced-quiz-header {
                padding: 2rem 0;
            }
            
            .quiz-main-title {
                font-size: 1.75rem;
            }
            
            .quiz-stats-row {
                gap: 1rem;
            }
            
            .stat-item {
                flex-direction: column;
                text-align: center;
                gap: 0.5rem;
            }
            
            .quiz-meta-info {
                flex-direction: column;
                align-items: flex-start;
                gap: 0.5rem;
            }
        }

        .grade-indicator {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.75rem 1.25rem;
            border-radius: 25px;
            font-size: 0.9rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            white-space: nowrap;
            cursor: help;
            transition: all 0.3s ease;
            margin-top: 0.5rem;
        }

        .grade-counts {
            background: linear-gradient(135deg, #fef3e2 0%, #fde68a 100%);
            color: #d97706;
            border: 2px solid #f59e0b;
        }

        .grade-counts:hover {
            background: linear-gradient(135deg, #fde68a 0%, #f59e0b 100%);
            color: white;
            transform: scale(1.05);
        }

        .grade-practice {
            background: linear-gradient(135deg, #e0f2fe 0%, #b3e5fc 100%);
            color: #0277bd;
            border: 2px solid #03a9f4;
        }

        .grade-practice:hover {
            background: linear-gradient(135deg, #b3e5fc 0%, #03a9f4 100%);
            color: white;
            transform: scale(1.05);
        }

        .quiz-info {
            background: white;
            border-radius: 15px;
            padding: 1.5rem;
            margin-bottom: 2rem;
            box-shadow: 0 4px 15px rgba(0,0,0,0.05);
        }

        .question-card {
            background: white;
            border-radius: 15px;
            padding: 2rem;
            margin-bottom: 1.5rem;
            box-shadow: 0 4px 15px rgba(0,0,0,0.05);
            border-left: 4px solid var(--panadite-primary);
        }

        .question-number {
            background: var(--panadite-primary);
            color: white;
            width: 35px;
            height: 35px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            margin-bottom: 1rem;
        }

        .question-text {
            font-size: 1.1rem;
            font-weight: 500;
            margin-bottom: 1.5rem;
            line-height: 1.6;
        }

        .option-item {
            background: #f8fafc;
            border: 2px solid var(--border-color);
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 0.75rem;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .option-item:hover {
            border-color: var(--panadite-primary);
            background: var(--panadite-light);
        }

        .option-item.selected {
            border-color: var(--panadite-primary);
            background: var(--panadite-light);
        }

        .timer-widget {
            position: fixed;
            top: 20px;
            right: 20px;
            background: white;
            padding: 1rem;
            border-radius: 10px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
            z-index: 1000;
        }

        .submit-section {
            background: white;
            border-radius: 15px;
            padding: 2rem;
            text-align: center;
            box-shadow: 0 4px 15px rgba(0,0,0,0.05);
        }

        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn-success {
            background: var(--success-color);
            color: white;
        }

        .btn-success:hover {
            background: #059669;
        }

        .form-control {
            border: 2px solid var(--border-color);
            border-radius: 8px;
            padding: 0.75rem;
            transition: all 0.3s ease;
        }

        .form-control:focus {
            border-color: var(--panadite-primary);
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        /* Enhanced Quiz Information Styles */
        .quiz-info-enhanced {
            background: white;
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            border: 1px solid var(--border-color);
            margin-bottom: 2rem;
        }

        /* Video Info Card */
        .video-info-card {
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 2px 12px rgba(0, 0, 0, 0.1);
            border: 1px solid var(--border-color);
            transition: transform 0.2s ease, box-shadow 0.2s ease;
        }

        .video-info-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
        }

        .video-thumbnail-container {
            position: relative;
            width: 100%;
            aspect-ratio: 16/9;
            overflow: hidden;
            background: var(--bg-light);
        }

        .video-thumbnail {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: transform 0.3s ease;
        }

        .video-placeholder {
            display: flex;
            align-items: center;
            justify-content: center;
            height: 100%;
            background: linear-gradient(135deg, #f1f5f9 0%, #e2e8f0 100%);
            color: var(--text-secondary);
        }

        .video-duration-overlay {
            position: absolute;
            bottom: 8px;
            right: 8px;
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 2px 6px;
            border-radius: 4px;
            font-size: 0.75rem;
            font-weight: 500;
        }

        .video-overlay {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(37, 99, 235, 0.9);
            color: white;
            border-radius: 50%;
            width: 48px;
            height: 48px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .video-thumbnail-container:hover .video-overlay {
            opacity: 1;
        }

        .video-thumbnail-container:hover .video-thumbnail {
            transform: scale(1.05);
        }

        .video-info-content {
            padding: 1rem;
        }

        .video-title {
            color: var(--text-primary);
            font-weight: 600;
            font-size: 0.9rem;
        }

        /* Quiz Tips Section */
        .quiz-tips-section {
            background: linear-gradient(135deg, #f0fdf4 0%, #dcfce7 100%);
            border: 1px solid #bbf7d0;
            border-radius: 12px;
            padding: 1.25rem;
        }

        .tips-header h6 {
            font-weight: 600;
            display: flex;
            align-items: center;
        }

        .tips-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 0.75rem;
        }

        .tip-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.85rem;
            color: var(--text-primary);
        }

        .tip-item i {
            font-size: 1rem;
        }

        /* Quiz Details Cards */
        .quiz-details-grid .detail-card {
            background: white;
            border: 1px solid var(--border-color);
            border-radius: 10px;
            padding: 1rem;
            text-align: center;
            transition: all 0.2s ease;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
        }

        .quiz-details-grid .detail-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            border-color: var(--panadite-primary);
        }

        .detail-icon {
            color: var(--panadite-primary);
            font-size: 1.5rem;
            margin-bottom: 0.5rem;
        }

        .detail-value {
            font-size: 1.25rem;
            font-weight: 700;
            color: var(--text-primary);
            margin-bottom: 0.25rem;
        }

        .detail-label {
            font-size: 0.8rem;
            color: var(--text-secondary);
            font-weight: 500;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        /* Quiz Completion Status */
        .quiz-completion-status {
            margin-top: 1.5rem;
            padding-top: 1.5rem;
            border-top: 1px solid var(--border-color);
        }

        .score-summary-card {
            background: linear-gradient(135deg, #fff7ed 0%, #fed7aa 100%);
            border: 1px solid #fdba74;
            border-radius: 12px;
            padding: 1.25rem;
            text-align: center;
        }

        .score-header {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            margin-bottom: 0.75rem;
            color: var(--text-secondary);
            font-size: 0.9rem;
            font-weight: 500;
        }

        .score-value {
            font-size: 2rem;
            font-weight: 700;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
        }

        .score-status {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 0.25rem;
        }

        /* Quiz Title Section inside card */
        .quiz-title-section {
            border-bottom: 1px solid var(--border-color);
            padding-bottom: 1rem;
        }

        .quiz-main-title {
            font-size: 1.75rem;
            font-weight: 700;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
        }

        .quiz-course-info {
            color: var(--text-secondary);
            font-size: 0.95rem;
            margin-bottom: 0;
        }

        /* Enhanced Instructions Header */
        .instructions-header h4 {
            color: var(--text-primary);
            font-weight: 600;
            display: flex;
            align-items: center;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .quiz-info-enhanced {
                padding: 1.25rem;
            }
            
            .tips-grid {
                grid-template-columns: 1fr;
                gap: 0.5rem;
            }
            
            .score-value {
                font-size: 1.5rem;
            }
            
            .video-overlay {
                width: 40px;
                height: 40px;
                font-size: 1.25rem;
            }
        }

        /* Quiz Review Styles */
        .correct-answer {
            border-left-color: var(--success-color) !important;
            background: rgba(16, 185, 129, 0.03);
        }

        .incorrect-answer {
            border-left-color: var(--error-color) !important;
            background: rgba(239, 68, 68, 0.03);
        }

        .correct-option {
            background: rgba(16, 185, 129, 0.1) !important;
            border-color: var(--success-color) !important;
            color: var(--success-color);
        }

        .incorrect-option {
            background: rgba(239, 68, 68, 0.1) !important;
            border-color: var(--error-color) !important;
            color: var(--error-color);
        }

        .option-item.selected {
            border-width: 3px;
            font-weight: 500;
        }

        .answer-status {
            display: flex;
            gap: 0.5rem;
            align-items: center;
        }

        .answer-review {
            background: #f8fafc;
            border-radius: 8px;
            padding: 1rem;
            border: 2px solid var(--border-color);
        }

        .question-card.correct-answer .question-number {
            background: var(--success-color);
        }

        .question-card.incorrect-answer .question-number {
            background: var(--error-color);
        }
    </style>
</head>
<body>


    <div class="container">
        <!-- Timer Widget -->
        <?php if ($quiz['time_limit'] > 0 && !($existing_attempt && $existing_attempt['end_time'])): ?>
        <div class="timer-widget">
            <div class="d-flex align-items-center gap-2">
                <i class="bx bx-time-five text-warning"></i>
                <span class="fw-bold">Time Remaining:</span>
                <span id="timer" class="text-primary fw-bold"></span>
            </div>
        </div>
        <?php endif; ?>

        <!-- Enhanced Quiz Information -->
        <div class="quiz-info-enhanced">
            <div class="row g-4 mb-4">
                <!-- Video Information Column -->
                <?php if (!empty($quiz['video_title'])): ?>
                <div class="col-lg-4">
                    <div class="video-info-card">
                        <div class="video-thumbnail-container">
                            <?php if (!empty($quiz['thumbnail_path'])): ?>
                                <img src="<?= htmlspecialchars($quiz['thumbnail_path']) ?>" 
                                     alt="<?= htmlspecialchars($quiz['video_title']) ?>"
                                     class="video-thumbnail">
                                <div class="video-duration-overlay">
                                    <?= htmlspecialchars($quiz['duration']) ?>
                                </div>
                            <?php else: ?>
                                <div class="video-placeholder">
                                    <i class="bx bx-video fs-1"></i>
                                </div>
                            <?php endif; ?>
                            <div class="video-overlay">
                                <i class="bx bx-play-circle"></i>
                            </div>
                        </div>
                        <div class="video-info-content">
                            <h6 class="video-title mb-1"><?= htmlspecialchars($quiz['video_title']) ?></h6>
                            <small class="text-muted">Related Video Lesson</small>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Quiz Instructions Column -->
                <div class="<?= !empty($quiz['video_title']) ? 'col-lg-8' : 'col-12' ?>">
                    <!-- Quiz Title and Instructor Info with Back Button -->
                    <div class="quiz-title-section mb-3">
                        <div class="d-flex justify-content-between align-items-start mb-2">
                            <div>
                                <h1 class="quiz-main-title mb-1"><?= htmlspecialchars($quiz['title']) ?></h1>
                                <p class="quiz-course-info mb-0">
                                    <?= htmlspecialchars($quiz['course_title']) ?> • 
                                    Instructor: <?= htmlspecialchars($quiz['first_name'] . ' ' . $quiz['last_name']) ?>
                                </p>
                            </div>
                            <?php
                            // Encrypt course ID for the back button
                            $encrypted_course_id = '';
                            try {
                                if (class_exists('IdEncryption') && method_exists('IdEncryption', 'encrypt')) {
                                    $encrypted_course_id = IdEncryption::encrypt($quiz['course_id']);
                                    if (empty($encrypted_course_id) || $encrypted_course_id == $quiz['course_id']) {
                                        $encrypted_course_id = $quiz['course_id'];
                                    }
                                } else {
                                    $encrypted_course_id = $quiz['course_id'];
                                }
                            } catch (Exception $e) {
                                $encrypted_course_id = $quiz['course_id'];
                            }
                            ?>
                            <a href="course-detail.php?id=<?= htmlspecialchars($encrypted_course_id) ?>" class="btn btn-outline-secondary">
                                <i class="bx bx-arrow-back me-1"></i> Back to Course
                            </a>
                        </div>
                    </div>
                    
                    <div class="instructions-header mb-3">
                        <h4 class="mb-2">
                            <i class="bx bx-info-circle text-primary me-2"></i>
                            Quiz Instructions
                        </h4>
                        <?php if (!empty($quiz['description'])): ?>
                            <p class="text-secondary mb-0"><?= htmlspecialchars($quiz['description']) ?></p>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Quiz Success Tips -->
                    <div class="quiz-tips-section mb-4">
                        <div class="tips-header mb-3">
                            <h6 class="text-success mb-0">
                                <i class="bx bx-lightbulb me-1"></i>
                                Tips for Success
                            </h6>
                        </div>
                        <div class="tips-grid">
                            <div class="tip-item">
                                <i class="bx bx-time text-warning"></i>
                                <span>Manage your time wisely</span>
                            </div>
                            <div class="tip-item">
                                <i class="bx bx-book-reader text-info"></i>
                                <span>Read questions carefully</span>
                            </div>
                            <div class="tip-item">
                                <i class="bx bx-check-double text-success"></i>
                                <span>Review before submitting</span>
                            </div>
                            <div class="tip-item">
                                <i class="bx bx-target-lock text-primary"></i>
                                <span>Stay focused and confident</span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Quiz Details Cards -->
                    <div class="quiz-details-grid">
                        <div class="row g-3">
                            <div class="col-md-4">
                                <div class="detail-card">
                                    <div class="detail-icon">
                                        <i class="bx bx-list-ol"></i>
                                    </div>
                                    <div class="detail-content">
                                        <div class="detail-value"><?= count($questions) ?></div>
                                        <div class="detail-label">Questions</div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="detail-card">
                                    <div class="detail-icon">
                                        <i class="bx bx-time"></i>
                                    </div>
                                    <div class="detail-content">
                                        <div class="detail-value"><?= $quiz['time_limit'] > 0 ? $quiz['time_limit'] . 'min' : 'No limit' ?></div>
                                        <div class="detail-label">Time Limit</div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="detail-card">
                                    <div class="detail-icon">
                                        <i class="bx bx-target-lock"></i>
                                    </div>
                                    <div class="detail-content">
                                        <div class="detail-value"><?= $quiz['passing_score'] ?>%</div>
                                        <div class="detail-label">Passing Score</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Quiz Status for Completed Attempts -->
            <?php if ($existing_attempt && $existing_attempt['end_time']): ?>
            <div class="quiz-completion-status">
                <div class="row g-3">
                    <div class="col-md-8">
                        <div class="alert alert-info border-0">
                            <div class="d-flex align-items-center justify-content-between">
                                <div class="d-flex align-items-center">
                                    <i class="bx bx-info-circle fs-4 me-3"></i>
                                    <div>
                                        <h6 class="mb-1">Quiz Already Completed</h6>
                                        <p class="mb-0">You can review your answers below or <a href="quiz_results.php?attempt_id=<?= $existing_attempt['attempt_id'] ?>" class="fw-bold">view detailed results</a>.</p>
                                    </div>
                                </div>
                                <div>
                                    <a href="?id=<?= htmlspecialchars($_GET['id']) ?>&retry=1" class="btn btn-primary">
                                        <i class="bx bx-refresh me-2"></i>Retry Quiz
                                    </a>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="score-summary-card">
                            <div class="score-header">
                                <i class="bx bx-trophy text-warning"></i>
                                <span>Your Score</span>
                            </div>
                            <div class="score-value"><?= number_format($existing_attempt['score'], 1) ?>%</div>
                            <div class="score-status">
                                <?php if ($existing_attempt['pass_status']): ?>
                                    <span class="badge bg-success">PASSED</span>
                                <?php else: ?>
                                    <span class="badge bg-danger">FAILED</span>
                                <?php endif; ?>
                                <small class="text-muted d-block mt-1">
                                    <?= date('M d, Y g:i A', strtotime($existing_attempt['start_time'])) ?>
                                </small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>

        <!-- Completed Quiz Review -->
        <?php if ($existing_attempt && $existing_attempt['end_time']): ?>
            <div class="mb-4">
                <h4 class="mb-3">
                    <i class="bx bx-list-check"></i>
                    Quiz Review - Your Answers
                </h4>
                
                <?php $question_num = 1; ?>
                <?php foreach ($questions as $question): ?>
                    <?php 
                    $student_response = $student_responses[$question['question_id']] ?? null;
                    $is_correct = $student_response ? $student_response['is_correct'] : 0;
                    ?>
                    
                    <div class="question-card <?= $is_correct ? 'correct-answer' : 'incorrect-answer' ?>">
                        <div class="d-flex justify-content-between align-items-start mb-3">
                            <div class="question-number <?= $is_correct ? 'bg-success' : 'bg-danger' ?>">
                                <?= $question_num ?>
                            </div>
                            <div class="answer-status">
                                <?php if ($is_correct): ?>
                                    <span class="badge bg-success">
                                        <i class="bx bx-check"></i> Correct
                                    </span>
                                <?php else: ?>
                                    <span class="badge bg-danger">
                                        <i class="bx bx-x"></i> Incorrect
                                    </span>
                                <?php endif; ?>
                                <span class="badge bg-primary ms-1">
                                    <?= $student_response && $is_correct ? $question['points'] : 0 ?>/<?= $question['points'] ?> points
                                </span>
                            </div>
                        </div>
                        
                        <div class="question-text mb-3">
                            <?= nl2br(htmlspecialchars($question['question_text'])) ?>
                        </div>

                        <?php if ($question['question_type'] === 'multiple_choice' || $question['question_type'] === 'true_false'): ?>
                            <?php foreach ($question['options'] as $option): ?>
                                <?php
                                $is_student_answer = $student_response && $student_response['answer_id'] == $option['option_id'];
                                $is_correct_option = $option['is_correct'];
                                $option_class = '';
                                $icon = '';
                                
                                if ($is_correct_option) {
                                    $option_class = 'correct-option';
                                    $icon = '<i class="bx bx-check-circle text-success"></i>';
                                } elseif ($is_student_answer && !$is_correct_option) {
                                    $option_class = 'incorrect-option';
                                    $icon = '<i class="bx bx-x-circle text-danger"></i>';
                                }
                                ?>
                                
                                <div class="option-item <?= $option_class ?> <?= $is_student_answer ? 'selected' : '' ?>" style="pointer-events: none;">
                                    <div class="d-flex justify-content-between align-items-center">
                                        <span>
                                            <?= htmlspecialchars($option['option_text']) ?>
                                            <?php if ($is_student_answer): ?>
                                                <span class="badge bg-secondary ms-2">Your Answer</span>
                                            <?php endif; ?>
                                        </span>
                                        <?= $icon ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>

                        <?php elseif ($question['question_type'] === 'short_answer'): ?>
                            <div class="answer-review">
                                <strong>Your Answer:</strong>
                                <div class="form-control bg-light" style="min-height: 100px;">
                                    <?= htmlspecialchars($student_response['text_response'] ?? 'No answer provided') ?>
                                </div>
                                <small class="text-muted mt-2 d-block">
                                    Note: Short answer questions require manual grading by the instructor.
                                </small>
                            </div>
                        <?php endif; ?>
                    </div>
                    <?php $question_num++; ?>
                <?php endforeach; ?>
            </div>
        
        <!-- Quiz Form -->
        <?php else: ?>
        <form method="POST" id="quizForm">
            <?php $question_num = 1; ?>
            <?php foreach ($questions as $question): ?>
                <div class="question-card">
                    <div class="question-number"><?= $question_num ?></div>
                    <div class="question-text">
                        <?= nl2br(htmlspecialchars($question['question_text'])) ?>
                        <span class="badge bg-primary ms-2"><?= $question['points'] ?> point<?= $question['points'] > 1 ? 's' : '' ?></span>
                    </div>

                    <?php if ($question['question_type'] === 'multiple_choice'): ?>
                        <?php foreach ($question['options'] as $option): ?>
                            <label class="option-item">
                                <input type="radio" name="question_<?= $question['question_id'] ?>" 
                                       value="<?= $option['option_id'] ?>" required>
                                <span><?= htmlspecialchars($option['option_text']) ?></span>
                            </label>
                        <?php endforeach; ?>

                    <?php elseif ($question['question_type'] === 'true_false'): ?>
                        <?php foreach ($question['options'] as $option): ?>
                            <label class="option-item">
                                <input type="radio" name="question_<?= $question['question_id'] ?>" 
                                       value="<?= $option['option_id'] ?>" required>
                                <span><?= htmlspecialchars($option['option_text']) ?></span>
                            </label>
                        <?php endforeach; ?>

                    <?php elseif ($question['question_type'] === 'short_answer'): ?>
                        <textarea name="question_<?= $question['question_id'] ?>" 
                                  class="form-control" rows="4" required
                                  placeholder="Type your answer here..."></textarea>
                    <?php endif; ?>
                </div>
                <?php $question_num++; ?>
            <?php endforeach; ?>

            <div class="submit-section">
                <h4 class="mb-3">Ready to Submit?</h4>
                <p class="text-secondary mb-4">
                    Please review your answers before submitting. Once submitted, you cannot change your responses.
                </p>
                <button type="submit" name="submit_quiz" class="btn btn-success btn-lg">
                    <i class="bx bx-send"></i>
                    Submit Quiz
                </button>
            </div>
        </form>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Handle option selection styling
        document.querySelectorAll('input[type="radio"]').forEach(radio => {
            radio.addEventListener('change', function() {
                // Remove selected class from all options in this question
                const questionName = this.name;
                document.querySelectorAll(`input[name="${questionName}"]`).forEach(input => {
                    input.closest('.option-item').classList.remove('selected');
                });
                
                // Add selected class to chosen option
                this.closest('.option-item').classList.add('selected');
            });
        });

        // Timer functionality
        <?php if ($quiz['time_limit'] > 0 && !($existing_attempt && $existing_attempt['end_time'])): ?>
        let timeLeft = <?= $quiz['time_limit'] * 60 ?>; // Convert minutes to seconds
        
        function updateTimer() {
            const minutes = Math.floor(timeLeft / 60);
            const seconds = timeLeft % 60;
            
            const timerElement = document.getElementById('timer');
            if (timerElement) {
                timerElement.textContent = 
                    `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
            }
            
            if (timeLeft <= 0) {
                // Time expired - auto submit with bypass validation
                autoSubmitQuiz();
                return;
            }
            
            timeLeft--;
        }
        
        // Update timer every second
        setInterval(updateTimer, 1000);
        updateTimer(); // Initial call
        <?php endif; ?>

        // Auto-submit function for time expired
        function autoSubmitQuiz() {
            // Create form data with time_expired flag
            const formData = new FormData();
            formData.append('time_expired', '1');
            
            // Add any answered questions
            const form = document.getElementById('quizForm');
            const inputs = form.querySelectorAll('input[type="radio"]:checked, textarea');
            inputs.forEach(input => {
                if (input.value.trim() !== '') {
                    formData.append(input.name, input.value);
                }
            });
            
            // Submit via fetch to bypass validation
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            }).then(response => {
                if (response.redirected) {
                    window.location.href = response.url;
                } else {
                    // Fallback redirect to results
                    window.location.href = 'assignments.php';
                }
            }).catch(error => {
                console.error('Auto-submit failed:', error);
                alert('Time expired! Quiz will be submitted automatically.');
                window.location.href = 'assignments.php';
            });
        }

        // Form submission confirmation (only for manual submission)
        document.getElementById('quizForm').addEventListener('submit', function(e) {
            if (e.submitter && e.submitter.name === 'submit_quiz') {
                if (!confirm('Are you sure you want to submit your quiz? You cannot change your answers after submission.')) {
                    e.preventDefault();
                }
            }
        });
    </script>
</div>

<!-- Close home-content div -->
        </div>
    </section>
</body>
</html>
