<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include session and encryption
require_once '../includes/session_start.php';
require_once '../includes/id_encryption.php';

// Set JSON header
header('Content-Type: application/json');

// Debug logging function
function debug_log($message, $data = null) {
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[$timestamp] UPDATE REVIEW: $message";
    if ($data) {
        $log_entry .= " - Data: " . json_encode($data);
    }
    error_log($log_entry);
}

debug_log("Update review script started");

try {
    // Check if request is POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        debug_log("Invalid request method: " . $_SERVER['REQUEST_METHOD']);
        echo json_encode(['success' => false, 'message' => 'Invalid request method']);
        exit;
    }

    // Get JSON input
    $json_input = file_get_contents('php://input');
    $data = json_decode($json_input, true);

    debug_log("Received data", $data);

    if (!$data) {
        debug_log("No JSON data received");
        echo json_encode(['success' => false, 'message' => 'No data received']);
        exit;
    }

    // Extract and validate input
    $review_id = isset($data['review_id']) ? (int)$data['review_id'] : 0;
    $rating = isset($data['rating']) ? (int)$data['rating'] : 0;
    $review_title = isset($data['title']) ? trim($data['title']) : '';
    $review_text = isset($data['text']) ? trim($data['text']) : '';

    debug_log("Parsed input", [
        'review_id' => $review_id,
        'rating' => $rating,
        'title_length' => strlen($review_title),
        'text_length' => strlen($review_text)
    ]);

    // Validation
    if (!$review_id || $review_id <= 0) {
        debug_log("Invalid review ID: " . $review_id);
        echo json_encode(['success' => false, 'message' => 'Invalid review ID']);
        exit;
    }

    if (!$rating || $rating < 1 || $rating > 5) {
        debug_log("Invalid rating: " . $rating);
        echo json_encode(['success' => false, 'message' => 'Rating must be between 1 and 5']);
        exit;
    }

    if (strlen($review_title) < 3 || strlen($review_title) > 255) {
        debug_log("Invalid title length: " . strlen($review_title));
        echo json_encode(['success' => false, 'message' => 'Review title must be 3-255 characters']);
        exit;
    }

    if (strlen($review_text) < 10 || strlen($review_text) > 1000) {
        debug_log("Invalid text length: " . strlen($review_text));
        echo json_encode(['success' => false, 'message' => 'Review text must be 10-1000 characters']);
        exit;
    }

    // Check if user is logged in
    if (!isset($_SESSION['user_id'])) {
        debug_log("User not logged in");
        echo json_encode(['success' => false, 'message' => 'Please log in to update your review']);
        exit;
    }

    $user_id = $_SESSION['user_id'];
    debug_log("User ID: " . $user_id);

    // Database connection - try multiple methods
    $pdo = null;

    // Method 1: Try db_connect.php
    try {
        require_once '../includes/db_connect.php';
        if (isset($conn) && $conn instanceof mysqli) {
            // Convert mysqli to PDO
            $host = 'localhost';
            $dbname = 'twpurnyc_panadite_DB';
            $username = 'twpurnyc_149586Alain';
            $password = '149586Alain#';
            $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            debug_log("Database connected via db_connect.php (converted from mysqli)");
        }
    } catch (Exception $e) {
        debug_log("db_connect.php failed: " . $e->getMessage());
    }

    // Method 2: Direct PDO connection
    if (!$pdo) {
        try {
            $host = 'localhost';
            $dbname = 'twpurnyc_panadite_DB';
            $username = 'twpurnyc_149586Alain';
            $password = '149586Alain#';
            $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            debug_log("Database connected via direct PDO");
        } catch (Exception $e) {
            debug_log("Direct PDO connection failed: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Database connection failed']);
            exit;
        }
    }

    // Check if review exists and belongs to current user
    $check_queries = [
        "SELECT review_id, student_id, course_id FROM course_reviews WHERE review_id = ? AND student_id = ?",
        "SELECT review_id, user_id as student_id, course_id FROM course_reviews WHERE review_id = ? AND user_id = ?",
        "SELECT * FROM course_reviews WHERE review_id = ? AND student_id = ?",
        "SELECT * FROM course_reviews WHERE review_id = ? AND user_id = ?"
    ];
    
    $existing_review = null;
    foreach ($check_queries as $query) {
        try {
            $stmt = $pdo->prepare($query);
            $stmt->execute([$review_id, $user_id]);
            $existing_review = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($existing_review) {
                debug_log("Found existing review with query: " . $query, $existing_review);
                break;
            }
        } catch (Exception $e) {
            debug_log("Review check query failed: " . $query . " - " . $e->getMessage());
            continue;
        }
    }

    if (!$existing_review) {
        debug_log("Review not found or access denied", ['review_id' => $review_id, 'user_id' => $user_id]);
        echo json_encode(['success' => false, 'message' => 'Review not found or you do not have permission to edit it']);
        exit;
    }

    // Update the review - try different column combinations
    $update_queries = [
        "UPDATE course_reviews SET rating = ?, review_title = ?, review_text = ?, updated_at = NOW() WHERE review_id = ? AND student_id = ?",
        "UPDATE course_reviews SET rating = ?, review_title = ?, review_text = ?, updated_at = NOW() WHERE review_id = ? AND user_id = ?",
        "UPDATE course_reviews SET rating = ?, review_title = ?, review_text = ? WHERE review_id = ? AND student_id = ?",
        "UPDATE course_reviews SET rating = ?, review_title = ?, review_text = ? WHERE review_id = ? AND user_id = ?"
    ];
    
    $update_success = false;
    foreach ($update_queries as $query) {
        try {
            $stmt = $pdo->prepare($query);
            $result = $stmt->execute([$rating, $review_title, $review_text, $review_id, $user_id]);
            
            if ($result && $stmt->rowCount() > 0) {
                debug_log("Review updated successfully with query: " . $query);
                $update_success = true;
                break;
            }
        } catch (Exception $e) {
            debug_log("Update query failed: " . $query . " - " . $e->getMessage());
            continue;
        }
    }

    if (!$update_success) {
        debug_log("All update queries failed");
        echo json_encode(['success' => false, 'message' => 'Failed to update review. Please try again.']);
        exit;
    }

    // Update course average rating
    $course_id = $existing_review['course_id'];
    try {
        $stmt = $pdo->prepare("
            SELECT AVG(rating) as avg_rating, COUNT(*) as review_count 
            FROM course_reviews 
            WHERE course_id = ? AND is_approved = 1
        ");
        $stmt->execute([$course_id]);
        $rating_data = $stmt->fetch();

        if ($rating_data) {
            $stmt = $pdo->prepare("UPDATE courses SET average_rating = ?, review_count = ? WHERE id = ?");
            $stmt->execute([round($rating_data['avg_rating'], 1), $rating_data['review_count'], $course_id]);
            debug_log("Course rating updated", $rating_data);
        }
    } catch (Exception $e) {
        debug_log("Course rating update failed: " . $e->getMessage());
        // Don't fail the whole operation for this
    }

    // Success response
    debug_log("Review update completed successfully");
    echo json_encode([
        'success' => true,
        'message' => 'Review updated successfully!',
        'review_id' => $review_id
    ]);

} catch (Exception $e) {
    debug_log("Unexpected error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'An unexpected error occurred. Please try again.',
        'debug' => $e->getMessage()
    ]);
}
?>
