<?php
/**
 * URL Video Progress Tracking Demo
 * Shows how URL videos integrate with existing tracking system
 */

// Initialize session if not already started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

require_once '../config/db_connect.php';

// Debug session info
error_log("🔍 URL Video Demo Session Debug: " . json_encode([
    'user_id' => $_SESSION['user_id'] ?? 'NOT_SET',
    'user_role' => $_SESSION['user_role'] ?? 'NOT_SET',
    'username' => $_SESSION['username'] ?? 'NOT_SET',
    'session_status' => session_status(),
    'session_id' => session_id()
]));

// Fallback demo login only if no session exists at all
if (!isset($_SESSION['user_id']) && isset($_GET['demo_login'])) {
    $_SESSION['user_id'] = 1;
    $_SESSION['user_role'] = 'student';
    $_SESSION['username'] = 'demo_student';
    error_log("🎭 Using demo login fallback");
}

// Sample video data for demonstration
$demo_videos = [
    [
        'video_id' => 1,
        'course_id' => 13,
        'title' => 'What is Bootstrap? Beginner CSS Framework',
        'description' => 'Learn the basics of Bootstrap CSS framework in this comprehensive tutorial.',
        'video_url' => 'https://www.youtube.com/watch?v=vpAJ0s5S2t0',
        'platform' => 'youtube',
        'duration' => '19:45',
        'embed_code' => '<iframe width="100%" height="315" src="https://www.youtube.com/embed/vpAJ0s5S2t0" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" allowfullscreen></iframe>',
        'views' => 1250,
        'created_at' => '2024-01-15 10:30:00'
    ],
    [
        'video_id' => 2,
        'course_id' => 13,
        'title' => 'TikTok Web Development Tips',
        'description' => 'Quick web development tips in bite-sized format.',
        'video_url' => 'https://www.tiktok.com/@webdevtips/video/7234567890123456789',
        'platform' => 'tiktok',
        'duration' => '0:45',
        'embed_code' => '<blockquote class="tiktok-embed" cite="https://www.tiktok.com/@webdevtips/video/7234567890123456789" data-video-id="7234567890123456789" style="max-width: 605px;min-width: 325px;" > <section> <a target="_blank" title="@webdevtips" href="https://www.tiktok.com/@webdevtips">@webdevtips</a> </section> </blockquote> <script async src="https://www.tiktok.com/embed.js"></script>',
        'views' => 850,
        'created_at' => '2024-01-20 14:15:00'
    ],
    [
        'video_id' => 3,
        'course_id' => 13,
        'title' => 'Instagram CSS Tutorial',
        'description' => 'Learn CSS through Instagram posts and stories.',
        'video_url' => 'https://www.instagram.com/p/ABC123DEF456/',
        'platform' => 'instagram',
        'duration' => '2:30',
        'embed_code' => '<blockquote class="instagram-media" data-instgrm-captioned data-instgrm-permalink="https://www.instagram.com/p/ABC123DEF456/" data-instgrm-version="14"></blockquote><script async src="//www.instagram.com/embed.js"></script>',
        'views' => 650,
        'created_at' => '2024-01-22 16:45:00'
    ]
];

$selected_video = $demo_videos[0]; // Default to first video
if (isset($_GET['video']) && isset($demo_videos[$_GET['video']])) {
    $selected_video = $demo_videos[$_GET['video']];
}

$video = $selected_video; // Set for component
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>URL Video Progress Tracking Demo</title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    
    <style>
        .demo-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem 0;
        }
        
        .demo-section {
            margin: 2rem 0;
        }
        
        .video-selector {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 1rem;
            margin-bottom: 2rem;
        }
        
        .video-option {
            display: flex;
            align-items: center;
            padding: 0.75rem;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            margin-bottom: 0.5rem;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            color: inherit;
        }
        
        .video-option:hover {
            background: white;
            border-color: #007bff;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
            color: inherit;
        }
        
        .video-option.active {
            background: #e3f2fd;
            border-color: #2196f3;
        }
        
        .platform-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 1rem;
        }
        
        .youtube-icon { background: #ff0000; color: white; }
        .tiktok-icon { background: #000000; color: white; }
        .instagram-icon { background: linear-gradient(45deg, #f09433 0%,#e6683c 25%,#dc2743 50%,#cc2366 75%,#bc1888 100%); color: white; }
        
        .integration-info {
            background: #f8f9fa;
            border-left: 4px solid #28a745;
            padding: 1rem;
            margin: 2rem 0;
        }
        
        .tracking-explanation {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem;
            border-radius: 15px;
            margin: 2rem 0;
        }
        
        .feature-list {
            list-style: none;
        }
        
        .feature-list li {
            padding: 0.5rem 0;
            border-bottom: 1px solid rgba(255,255,255,0.2);
        }
        
        .feature-list li:last-child {
            border-bottom: none;
        }
    </style>
</head>
<body>

    <!-- Demo Header -->
    <div class="demo-header">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h1><i class="fas fa-video me-3"></i>URL Video Progress Tracking</h1>
                    <p class="mb-0 lead">Enhanced tracking system for embedded videos from YouTube, TikTok, Instagram, and more</p>
                </div>
                <div class="col-md-4 text-md-end">
                    <div class="demo-status">
                        <?php if (isset($_SESSION['user_id'])): ?>
                            <div class="text-end">
                                <span class="badge bg-success fs-6 mb-2">
                                    <i class="fas fa-user me-1"></i>Logged in as <?= htmlspecialchars($_SESSION['user_role'] ?? 'User') ?>
                                </span>
                                <div class="small text-light">
                                    <i class="fas fa-id-badge me-1"></i>User ID: <?= htmlspecialchars($_SESSION['user_id'] ?? 'N/A') ?>
                                    <?php if (isset($_SESSION['username'])): ?>
                                        <br><i class="fas fa-user-tag me-1"></i><?= htmlspecialchars($_SESSION['username']) ?>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php else: ?>
                            <div class="text-end">
                                <div class="alert alert-warning mb-2 py-2">
                                    <small><i class="fas fa-exclamation-triangle me-1"></i>No session detected</small>
                                </div>
                                <a href="?demo_login=1" class="btn btn-outline-light">
                                    <i class="fas fa-sign-in-alt me-1"></i>Demo Login
                                </a>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="container mt-4">
        
        <!-- Video Selector -->
        <div class="video-selector">
            <h5 class="mb-3">
                <i class="fas fa-list me-2 text-primary"></i>Select a Demo Video
            </h5>
            
            <div class="row">
                <?php foreach ($demo_videos as $index => $demo_video): ?>
                    <div class="col-md-4">
                        <a href="?video=<?= $index ?>&demo_login=1" 
                           class="video-option <?= $demo_video['video_id'] === $selected_video['video_id'] ? 'active' : '' ?>">
                            
                            <div class="platform-icon <?= strtolower($demo_video['platform']) ?>-icon">
                                <?php
                                $platform_icons = [
                                    'youtube' => 'fab fa-youtube',
                                    'tiktok' => 'fab fa-tiktok', 
                                    'instagram' => 'fab fa-instagram'
                                ];
                                ?>
                                <i class="<?= $platform_icons[$demo_video['platform']] ?? 'fas fa-globe' ?>"></i>
                            </div>
                            
                            <div>
                                <strong><?= htmlspecialchars($demo_video['title']) ?></strong><br>
                                <small class="text-muted">
                                    <i class="fas fa-clock me-1"></i><?= $demo_video['duration'] ?>
                                    • <i class="fas fa-eye me-1"></i><?= number_format($demo_video['views']) ?> views
                                </small>
                            </div>
                        </a>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>

        <div class="row">
            <div class="col-lg-8">
                
                <!-- Video Player Component -->
                <div class="demo-section">
                    <?php include '../components/url-video-player.php'; ?>
                </div>
                
            </div>
            
            <div class="col-lg-4">
                
                <!-- How It Works -->
                <div class="tracking-explanation">
                    <h5><i class="fas fa-cogs me-2"></i>How It Works</h5>
                    
                    <ul class="feature-list">
                        <li><i class="fas fa-check-circle me-2"></i>Multi-tier tracking system</li>
                        <li><i class="fas fa-check-circle me-2"></i>Platform API integration (YouTube, Vimeo)</li>
                        <li><i class="fas fa-check-circle me-2"></i>User interaction monitoring</li>
                        <li><i class="fas fa-check-circle me-2"></i>Manual progress controls</li>
                        <li><i class="fas fa-check-circle me-2"></i>Automatic completion detection</li>
                        <li><i class="fas fa-check-circle me-2"></i>Course progression unlock</li>
                    </ul>
                    
                    <div class="mt-3 p-3 bg-white bg-opacity-20 rounded">
                        <small>
                            <strong>🎯 Result:</strong> Students can progress through courses with URL videos 
                            just like file uploads, ensuring proper learning sequence and completion tracking.
                        </small>
                    </div>
                </div>
                
                <!-- Integration Info -->
                <div class="card">
                    <div class="card-header">
                        <h6 class="mb-0">
                            <i class="fas fa-database me-2 text-success"></i>Database Integration
                        </h6>
                    </div>
                    <div class="card-body">
                        <div class="integration-info">
                            <h6 class="text-success">✅ Uses Existing System</h6>
                            <ul class="mb-0 small">
                                <li><strong>Table:</strong> <code>video_progress</code></li>
                                <li><strong>Endpoint:</strong> <code>update_video_progress.php</code></li>
                                <li><strong>Queries:</strong> Same as file uploads</li>
                                <li><strong>Progress:</strong> Percentage + completion</li>
                            </ul>
                        </div>
                        
                        <h6 class="mt-3">🔧 Implementation</h6>
                        <div class="small">
                            <ol class="mb-0">
                                <li>Include <code>embedded-video-tracker.js</code></li>
                                <li>Use <code>url-video-player.php</code> component</li>
                                <li>Students control their progress</li>
                                <li>System tracks engagement automatically</li>
                            </ol>
                        </div>
                    </div>
                </div>
                
                <!-- Test Progress -->
                <?php if (isset($_SESSION['user_id'])): ?>
                <div class="card mt-3">
                    <div class="card-header">
                        <h6 class="mb-0">
                            <i class="fas fa-flask me-2 text-warning"></i>Test Progress System
                        </h6>
                    </div>
                    <div class="card-body">
                        <div class="d-grid gap-2">
                            <button class="btn btn-outline-primary btn-sm" onclick="simulateProgress(25)">
                                Simulate 25% Progress
                            </button>
                            <button class="btn btn-outline-success btn-sm" onclick="simulateProgress(80)">
                                Simulate 80% Progress
                            </button>
                            <button class="btn btn-outline-warning btn-sm" onclick="simulateProgress(100)">
                                Simulate Completion
                            </button>
                            <button class="btn btn-outline-secondary btn-sm" onclick="resetProgress()">
                                Reset Progress
                            </button>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
                
            </div>
        </div>
        
        <!-- Implementation Guide -->
        <div class="demo-section">
            <div class="card border-primary">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0">
                        <i class="fas fa-rocket me-2"></i>Implementation Guide
                    </h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <h6>📁 Files Created:</h6>
                            <ul class="small">
                                <li><code>js/embedded-video-tracker.js</code></li>
                                <li><code>components/url-video-player.php</code></li>
                                <li><code>student/url-video-demo.php</code></li>
                            </ul>
                            
                            <h6 class="mt-3">🔌 Integration Steps:</h6>
                            <ol class="small">
                                <li>Include the JavaScript tracker</li>
                                <li>Use the PHP component for URL videos</li>
                                <li>Existing database and endpoints work unchanged</li>
                                <li>Students get manual controls + auto-tracking</li>
                            </ol>
                        </div>
                        <div class="col-md-6">
                            <h6>🎯 Features:</h6>
                            <ul class="small">
                                <li>✅ Works with all video platforms</li>
                                <li>✅ Uses existing database schema</li>
                                <li>✅ Manual progress controls</li>
                                <li>✅ Automatic engagement tracking</li>
                                <li>✅ Course progression unlock</li>
                                <li>✅ Mobile-responsive UI</li>
                            </ul>
                            
                            <h6 class="mt-3">🎛️ Student Controls:</h6>
                            <ul class="small">
                                <li><strong>Start Watching:</strong> Begin tracking</li>
                                <li><strong>Mark Complete:</strong> Manual completion</li>
                                <li><strong>Reset:</strong> Start over</li>
                                <li><strong>Auto-tracking:</strong> Background monitoring</li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
    // Test functions for demo
    function simulateProgress(progress) {
        <?php if (isset($_SESSION['user_id'])): ?>
        const videoId = <?= json_encode($selected_video['video_id']) ?>;
        const courseId = <?= json_encode($selected_video['course_id']) ?>;
        
        const formData = new FormData();
        formData.append('video_id', videoId);
        formData.append('course_id', courseId);
        formData.append('progress_percentage', progress);
        formData.append('current_time', Math.round(progress * 2)); // Simulate time
        formData.append('completed', progress >= 100 ? 1 : 0);
        
        fetch('ajax/update_video_progress.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                console.log('✅ Progress simulation successful:', data);
                
                // Update UI
                const progressBar = document.querySelector('.video-progress');
                const progressText = document.querySelector('.progress-text strong');
                
                if (progressBar && progressText) {
                    progressBar.style.width = `${progress}%`;
                    progressText.textContent = `${progress}%`;
                    
                    if (progress >= 100) {
                        progressBar.classList.remove('bg-primary');
                        progressBar.classList.add('bg-success');
                    }
                }
                
                // Show success message
                showToast(`Progress updated to ${progress}%`, 'success');
                
            } else {
                console.error('❌ Progress simulation failed:', data);
                showToast('Progress update failed', 'error');
            }
        })
        .catch(error => {
            console.error('❌ Progress simulation error:', error);
            showToast('Progress update error', 'error');
        });
        <?php else: ?>
        showToast('Please login first', 'warning');
        <?php endif; ?>
    }
    
    function resetProgress() {
        simulateProgress(0);
    }
    
    function showToast(message, type = 'info') {
        // Simple toast notification
        const toast = document.createElement('div');
        toast.className = `alert alert-${type === 'success' ? 'success' : type === 'error' ? 'danger' : 'info'} position-fixed`;
        toast.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 250px;';
        toast.textContent = message;
        
        document.body.appendChild(toast);
        
        setTimeout(() => {
            toast.remove();
        }, 3000);
    }
    
    // Demo info
    console.log('🎬 URL Video Progress Tracking Demo');
    console.log('📊 This demo shows how URL videos integrate with your existing tracking system');
    console.log('🔧 All database queries and endpoints remain unchanged');
    </script>

</body>
</html>
