<?php
// Use session_start.php for consistent session handling
require_once '../includes/session_start.php';
require_once '../auth/functions.php';

// Require student role authentication - no exceptions
requireRole('student');

// Check if video ID is provided
if (!isset($_GET['id'])) {
    header("Location: videos.php");
    exit();
}

$videoId = intval($_GET['id']);
$video = null;
$error = null;

// Include database connection
require_once '../config/db_connect.php';

// Fetch video details
try {
    $userId = $_SESSION['user_id'] ?? 0;
    
    if (!$userId) {
        $error = "You must be logged in to view videos.";
    } else {
        // Check if the user has purchased this video
        $stmt = $conn->prepare("
            SELECT COUNT(*) as has_access 
            FROM video_purchases 
            WHERE user_id = ? AND video_id = ? AND payment_status = 'completed'
        ");
        $stmt->bind_param("ii", $userId, $videoId);
        $stmt->execute();
        $result = $stmt->get_result();
        $accessRow = $result->fetch_assoc();
        
        if (!$accessRow || $accessRow['has_access'] == 0) {
            $error = "You do not have access to this video. Please purchase it first.";
        } else {
            // Get video details
            $stmt = $conn->prepare("
                SELECT 
                    v.video_id as id,
                    v.title,
                    v.description,
                    v.video_url,
                    v.duration,
                    CONCAT(FLOOR(LENGTH(v.video_url)/1024/1024), ' MB') as file_size,
                    v.created_at,
                    v.is_free as is_downloadable,
                    v.thumbnail,
                    v.category_name,
                    u.first_name as teacher_first_name,
                    u.last_name as teacher_last_name,
                    IFNULL(vp.last_position, 0) as last_position,
                    IFNULL(vp.completed, 0) as completed
                FROM videos v
                JOIN users u ON v.teacher_id = u.user_id
                LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = ?
                WHERE v.video_id = ? AND v.is_published = 1
            ");
            $stmt->bind_param("ii", $userId, $videoId);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                $video = $result->fetch_assoc();
            } else {
                $error = "Video not found or is not published.";
            }
        }
    }
} catch (Exception $e) {
    $error = "Error fetching video: " . $e->getMessage();
}

// No sample data - we only use real data from the database
if (!$video && !$error) {
    $error = "The requested video could not be found.";
}

// Update video progress on page load
if ($video && isset($_SESSION['user_id'])) {
    $userId = $_SESSION['user_id'];
    $videoId = $video['id'];
    $lastPosition = $video['last_position'];
    
    try {
        // Check if a progress record exists
        $stmt = $conn->prepare("
            SELECT id FROM video_progress 
            WHERE student_id = ? AND video_id = ?
        ");
        $stmt->bind_param("ii", $userId, $videoId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            // Update last_watched timestamp
            $stmt = $conn->prepare("
                UPDATE video_progress 
                SET last_watched = NOW() 
                WHERE student_id = ? AND video_id = ?
            ");
            $stmt->bind_param("ii", $userId, $videoId);
            $stmt->execute();
        } else {
            // Insert new progress record
            $stmt = $conn->prepare("
                INSERT INTO video_progress (student_id, video_id, last_position, completed, last_watched)
                VALUES (?, ?, 0, 0, NOW())
            ");
            $stmt->bind_param("ii", $userId, $videoId);
            $stmt->execute();
        }
    } catch (Exception $e) {
        // Silently handle this error to not disrupt user experience
    }
}

// Get related videos
$relatedVideos = [];
try {
    if ($video) {
        // Get videos from the same category
        $stmt = $conn->prepare("
            SELECT 
                v.video_id as id,
                v.title,
                v.thumbnail,
                v.duration,
                v.category_name,
                v.is_free,
                CASE WHEN vp.video_id IS NOT NULL THEN 1 ELSE 0 END as purchased
            FROM videos v
            LEFT JOIN video_purchases vp ON v.video_id = vp.video_id AND vp.user_id = ? AND vp.payment_status = 'completed'
            WHERE v.category_name = ? AND v.video_id != ? AND v.is_published = 1
            LIMIT 5
        ");
        $stmt->bind_param("isi", $userId, $video['category_name'], $video['id']);
        $stmt->execute();
        $relatedVideos = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }
} catch (Exception $e) {
    // Silently handle this error
}

// No sample related videos - only use real data
// We don't add sample related videos - only use real data from the database

// Include custom page-specific CSS
$customCss = 'video-player';

// Include header
include_once 'components/header.php';
?>

<?php if ($error): ?>
<!-- Error state -->
<div class="page-header">
    <div class="page-title">
        <h1>Video Not Available</h1>
    </div>
    <div class="page-actions">
        <a href="videos.php" class="btn btn-primary">
            <i class="bx bx-arrow-back"></i> Back to Videos
        </a>
    </div>
</div>

<div class="card">
    <div class="card-body text-center p-5">
        <div class="empty-state">
            <div class="empty-state-icon">
                <i class="bx bx-error-circle"></i>
            </div>
            <h4>Video Access Error</h4>
            <p><?= $error ?></p>
            <a href="videos.php" class="btn btn-primary mt-3">Back to Videos</a>
        </div>
    </div>
</div>

<?php else: ?>
<!-- Page Header -->
<div class="page-header">
    <div class="page-title">
        <h1><?= $video['title'] ?></h1>
        <p>
            <span class="badge bg-primary-light text-primary"><?= $video['category_name'] ?></span>
            <span class="text-muted ms-2">Instructor: <?= $video['teacher_first_name'] . ' ' . $video['teacher_last_name'] ?></span>
        </p>
    </div>
    <div class="page-actions">
        <a href="videos.php" class="btn btn-light">
            <i class="bx bx-arrow-back"></i> Back to Videos
        </a>
        <?php if ($video['is_downloadable']): ?>
        <a href="download-video.php?id=<?= $video['id'] ?>" class="btn btn-primary ms-2">
            <i class="bx bx-download"></i> Download Video
        </a>
        <?php endif; ?>
    </div>
</div>

<div class="row">
    <div class="col-lg-8">
        <!-- Video Player -->
        <div class="card mb-4">
            <div class="card-body p-0">
                <div class="video-player-wrapper">
                    <!-- HTML5 Video Player -->
                    <video id="videoPlayer" class="video-js vjs-big-play-centered vjs-fluid" controls preload="auto" poster="<?= $video['thumbnail'] ?>" data-setup='{}'>
                        <source src="<?= $video['video_url'] ?>" type="video/mp4">
                        <p class="vjs-no-js">
                            To view this video please enable JavaScript, and consider upgrading to a
                            web browser that <a href="https://videojs.com/html5-video-support/" target="_blank">supports HTML5 video</a>
                        </p>
                    </video>
                </div>
            </div>
        </div>

        <!-- Video Info -->
        <div class="card mb-4">
            <div class="card-body">
                <h4 class="mb-3">About this video</h4>
                <p class="video-description"><?= $video['description'] ?></p>
                
                <div class="video-meta mt-4">
                    <div class="row g-3">
                        <div class="col-sm-6 col-md-4">
                            <div class="meta-item">
                                <div class="meta-icon">
                                    <i class="bx bx-time"></i>
                                </div>
                                <div class="meta-info">
                                    <h6>Duration</h6>
                                    <p><?= $video['duration'] ?></p>
                                </div>
                            </div>
                        </div>
                        <div class="col-sm-6 col-md-4">
                            <div class="meta-item">
                                <div class="meta-icon">
                                    <i class="bx bx-file"></i>
                                </div>
                                <div class="meta-info">
                                    <h6>File Size</h6>
                                    <p><?= $video['file_size'] ?></p>
                                </div>
                            </div>
                        </div>
                        <div class="col-sm-6 col-md-4">
                            <div class="meta-item">
                                <div class="meta-icon">
                                    <i class="bx bx-calendar"></i>
                                </div>
                                <div class="meta-info">
                                    <h6>Added On</h6>
                                    <p><?= date('M j, Y', strtotime($video['created_at'])) ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Notes Section -->
        <div class="card">
            <div class="card-header">
                <h5 class="card-title">Video Notes</h5>
                <p class="card-subtitle">Take notes while watching the video</p>
            </div>
            <div class="card-body">
                <form id="notesForm">
                    <div class="mb-3">
                        <textarea class="form-control" id="videoNotes" rows="5" placeholder="Write your notes here..."></textarea>
                    </div>
                    <div class="d-flex justify-content-end">
                        <button type="button" class="btn btn-light me-2" id="clearNotes">Clear</button>
                        <button type="button" class="btn btn-primary" id="saveNotes">Save Notes</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-lg-4">
        <!-- Related Videos -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="card-title">Related Videos</h5>
            </div>
            <div class="card-body p-0">
                <div class="related-videos-list">
                    <?php foreach ($relatedVideos as $relatedVideo): ?>
                    <a href="video-player.php?id=<?= $relatedVideo['id'] ?>" class="related-video-item">
                        <div class="related-video-thumbnail">
                            <?php if (file_exists($relatedVideo['thumbnail'])): ?>
                            <img src="<?= $relatedVideo['thumbnail'] ?>" alt="<?= $relatedVideo['title'] ?>">
                            <?php else: ?>
                            <div class="placeholder-thumbnail small">
                                <i class="bx bx-video"></i>
                            </div>
                            <?php endif; ?>
                            <div class="related-video-duration"><?= $relatedVideo['duration'] ?></div>
                            <?php if ($relatedVideo['completed']): ?>
                            <div class="video-completed small">
                                <i class="bx bx-check-circle"></i>
                            </div>
                            <?php endif; ?>
                        </div>
                        <div class="related-video-info">
                            <h6 class="related-video-title"><?= $relatedVideo['title'] ?></h6>
                            <div class="related-video-status">
                                <?php if ($relatedVideo['completed']): ?>
                                <span class="badge bg-success-light text-success">Completed</span>
                                <?php elseif ($relatedVideo['started']): ?>
                                <span class="badge bg-warning-light text-warning">In progress</span>
                                <?php else: ?>
                                <span class="badge bg-gray-light text-gray">Not started</span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </a>
                    <?php endforeach; ?>
                    
                    <?php if (count($relatedVideos) === 0): ?>
                    <div class="text-center p-4">
                        <p class="text-muted">No related videos found.</p>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <!-- Video Resources -->
        <div class="card">
            <div class="card-header">
                <h5 class="card-title">Resources</h5>
            </div>
            <div class="card-body">
                <ul class="resource-list">
                    <li class="resource-item">
                        <div class="resource-icon pdf">
                            <i class="bx bxs-file-pdf"></i>
                        </div>
                        <div class="resource-info">
                            <h6>Lecture Notes</h6>
                            <span>PDF, 2.4 MB</span>
                        </div>
                        <a href="#" class="resource-download">
                            <i class="bx bx-download"></i>
                        </a>
                    </li>
                    <li class="resource-item">
                        <div class="resource-icon code">
                            <i class="bx bx-code-alt"></i>
                        </div>
                        <div class="resource-info">
                            <h6>Example Code</h6>
                            <span>ZIP, 1.8 MB</span>
                        </div>
                        <a href="#" class="resource-download">
                            <i class="bx bx-download"></i>
                        </a>
                    </li>
                    <li class="resource-item">
                        <div class="resource-icon slide">
                            <i class="bx bx-slideshow"></i>
                        </div>
                        <div class="resource-info">
                            <h6>Presentation Slides</h6>
                            <span>PDF, 5.2 MB</span>
                        </div>
                        <a href="#" class="resource-download">
                            <i class="bx bx-download"></i>
                        </a>
                    </li>
                </ul>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Include VideoJS from CDN -->
<link href="https://vjs.zencdn.net/7.20.3/video-js.css" rel="stylesheet" />
<script src="https://vjs.zencdn.net/7.20.3/video.min.js"></script>

<!-- Include VideoAPIIntegration for automatic progress tracking -->
<script src="../js/video-api-integration.js"></script>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize video player
    const player = videojs('videoPlayer', {
        controlBar: {
            pictureInPictureToggle: true,
            playbackRateMenuButton: true
        }
    });
    
    // Set initial position if available
    <?php if ($video && $video['last_position'] > 0): ?>
    player.ready(function() {
        // Ask user if they want to resume
        if (confirm('Would you like to resume from where you left off?')) {
            player.currentTime(<?= $video['last_position'] ?>);
        }
    });
    <?php endif; ?>
    
    // Update video progress
    const updateProgressInterval = 10; // Update every 10 seconds
    let lastUpdateTime = 0;
    
    player.on('timeupdate', function() {
        const currentTime = Math.floor(player.currentTime());
        
        // Only update every updateProgressInterval seconds
        if (currentTime - lastUpdateTime >= updateProgressInterval) {
            updateVideoProgress(currentTime);
            lastUpdateTime = currentTime;
        }
    });
    
    // Mark video as completed when it ends
    player.on('ended', function() {
        updateVideoProgress(Math.floor(player.duration()), true);
    });
    
    function updateVideoProgress(position, completed = false) {
        const videoId = <?= $video ? $video['id'] : 0 ?>;
        if (!videoId) return;
        
        // Send AJAX request to update progress
        fetch('ajax/update-video-progress.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `video_id=${videoId}&position=${position}&completed=${completed ? 1 : 0}`
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && completed) {
                // Show completion notification
                showNotification('Video completed! Great job!', 'success');
            }
        })
        .catch(error => {
            console.error('Error updating video progress:', error);
        });
    }
    
    // Handle notes functionality
    const notesTextarea = document.getElementById('videoNotes');
    const saveNotesBtn = document.getElementById('saveNotes');
    const clearNotesBtn = document.getElementById('clearNotes');
    const videoId = <?= $video ? $video['id'] : 0 ?>;
    
    // Load saved notes
    if (videoId) {
        loadSavedNotes();
    }
    
    // Save notes
    if (saveNotesBtn) {
        saveNotesBtn.addEventListener('click', function() {
            const notes = notesTextarea.value;
            
            // Save to localStorage for demo purposes
            // In a real app, would save to database
            localStorage.setItem(`video_notes_${videoId}`, notes);
            
            showNotification('Notes saved successfully!', 'success');
        });
    }
    
    // Clear notes
    if (clearNotesBtn) {
        clearNotesBtn.addEventListener('click', function() {
            if (confirm('Are you sure you want to clear all notes?')) {
                notesTextarea.value = '';
                localStorage.removeItem(`video_notes_${videoId}`);
                showNotification('Notes cleared!', 'info');
            }
        });
    }
    
    function loadSavedNotes() {
        const savedNotes = localStorage.getItem(`video_notes_${videoId}`);
        if (savedNotes && notesTextarea) {
            notesTextarea.value = savedNotes;
        }
    }
});
</script>

<?php include_once 'components/footer.php'; ?>
