/**
 * YouTube iframe API Integration for Automatic Progress Tracking
 * Integrates with existing ajax/update_video_progress.php endpoint
 */

// YouTube API variables
let youtubePlayer = null;
let youtubePlayerReady = false;
let youtubeProgressInterval = null;

// Create visible debug panel
function createYouTubeDebugPanel() {
    // Remove existing debug panel if it exists
    const existingPanel = document.getElementById('youtube-debug-panel');
    if (existingPanel) {
        existingPanel.remove();
    }
    
    const debugPanel = document.createElement('div');
    debugPanel.id = 'youtube-debug-panel';
    debugPanel.innerHTML = 
        '<div style="position: fixed; top: 100px; right: 10px; background: #1f2937; color: white; padding: 15px; border-radius: 8px; font-family: monospace; font-size: 12px; z-index: 9999; max-width: 300px; box-shadow: 0 4px 6px rgba(0,0,0,0.3);">' +
            '<div style="font-weight: bold; color: #fbbf24; margin-bottom: 10px;">🎬 YouTube Debug Panel</div>' +
            '<div id="yt-debug-api-status">API Status: <span style="color: #f87171;">Not Loaded</span></div>' +
            '<div id="yt-debug-player-status">Player Status: <span style="color: #f87171;">Not Created</span></div>' +
            '<div id="yt-debug-events-status">Events Status: <span style="color: #f87171;">No Events</span></div>' +
            '<div id="yt-debug-progress-status">Progress Status: <span style="color: #f87171;">Not Tracking</span></div>' +
            '<div id="yt-debug-ajax-status">AJAX Status: <span style="color: #f87171;">No Calls</span></div>' +
            '<div id="yt-debug-current-progress" style="margin-top: 10px; padding-top: 10px; border-top: 1px solid #374151;">Current Progress: <span style="color: #60a5fa;">0%</span></div>' +
            '<button onclick="document.getElementById(\'youtube-debug-panel\').remove()" style="position: absolute; top: 5px; right: 5px; background: #ef4444; border: none; color: white; width: 20px; height: 20px; border-radius: 50%; cursor: pointer;">×</button>' +
        '</div>';
    document.body.appendChild(debugPanel);
    
    return debugPanel;
}

// Update debug panel
function updateDebugStatus(type, message, color = '#10b981') {
    const element = document.getElementById(`yt-debug-${type}-status`);
    if (element) {
        element.innerHTML = `${type.charAt(0).toUpperCase() + type.slice(1)} Status: <span style="color: ${color};">${message}</span>`;
    }
}

// Initialize YouTube API for iframe videos
function initializeYouTubeProgressTracking(videoUrl, videoId, lessonId, courseId) {
    console.log('🎬 YOUTUBE: Initializing automatic progress tracking...');
    console.log('🎬 VIDEO DATA:', { videoUrl, videoId, lessonId, courseId });
    
    // Create visible debug panel
    createYouTubeDebugPanel();
    updateDebugStatus('api', 'Initializing...', '#fbbf24');
    updateDebugStatus('player', 'Waiting for API...', '#fbbf24');
    
    if (!videoUrl || !videoUrl.includes('youtube.com/embed')) {
        console.log('🎬 NOT A YOUTUBE VIDEO: Skipping YouTube API');
        updateDebugStatus('api', 'Not YouTube Video', '#f87171');
        return false;
    }
    
    // Load YouTube iframe API if not already loaded
    if (!window.YT) {
        console.log('📥 LOADING: YouTube iframe API script...');
        updateDebugStatus('api', 'Loading Script...', '#fbbf24');
        const tag = document.createElement('script');
        tag.src = 'https://www.youtube.com/iframe_api';
        tag.onload = function() {
            console.log('✅ LOADED: YouTube API script');
            updateDebugStatus('api', 'Script Loaded ✅', '#10b981');
        };
        tag.onerror = function() {
            console.error('❌ FAILED: YouTube API script load failed');
            updateDebugStatus('api', 'Script Failed ❌', '#f87171');
        };
        const firstScriptTag = document.getElementsByTagName('script')[0];
        firstScriptTag.parentNode.insertBefore(tag, firstScriptTag);
    }
    
    // Set up API ready callback
    window.onYouTubeIframeAPIReady = function() {
        console.log('🎬 YOUTUBE API: Ready - creating player...');
        updateDebugStatus('api', 'API Ready ✅', '#10b981');
        updateDebugStatus('player', 'Creating Player...', '#fbbf24');
        createYouTubePlayer(videoId, lessonId, courseId);
    };
    
    // If API already loaded, initialize immediately
    if (window.YT && window.YT.Player) {
        console.log('🎬 YOUTUBE API: Already loaded - creating player...');
        updateDebugStatus('api', 'Already Loaded ✅', '#10b981');
        updateDebugStatus('player', 'Creating Player...', '#fbbf24');
        createYouTubePlayer(videoId, lessonId, courseId);
    }
    
    return true;
}

function createYouTubePlayer(videoId, lessonId, courseId) {
    try {
        // Get the existing iframe and extract video ID from its src
        const existingIframe = document.getElementById('lessonVideo');
        if (!existingIframe || !existingIframe.src) {
            console.error('❌ YOUTUBE PLAYER: No iframe found or no src');
            return;
        }
        
        // Extract YouTube video ID from the embed URL
        const srcUrl = existingIframe.src;
        const match = srcUrl.match(/\/embed\/([^?&]+)/);
        if (!match) {
            console.error('❌ YOUTUBE PLAYER: Could not extract video ID from', srcUrl);
            return;
        }
        const youtubeVideoId = match[1];
        console.log('🎬 EXTRACTED YouTube Video ID:', youtubeVideoId);
        
        // Replace the iframe with a div, then let YouTube API create new iframe
        const parentContainer = existingIframe.parentNode;
        const playerDiv = document.createElement('div');
        playerDiv.id = 'lessonVideo';
        playerDiv.style.width = '100%';
        playerDiv.style.height = '100%';
        
        // Replace the iframe with the div
        parentContainer.replaceChild(playerDiv, existingIframe);
        
        // Now create the YouTube player
        console.log('🎬 CREATING PLAYER with config:', {
            videoId: youtubeVideoId,
            origin: window.location.origin,
            divId: 'lessonVideo'
        });
        
        youtubePlayer = new YT.Player('lessonVideo', {
            height: '100%',
            width: '100%',
            videoId: youtubeVideoId,
            playerVars: {
                'autoplay': 0,
                'controls': 1,
                'rel': 0,
                'showinfo': 0,
                'enablejsapi': 1,
                'origin': window.location.origin
            },
            events: {
                'onReady': function(event) {
                    console.log('🎬 YOUTUBE EVENT: onReady fired!');
                    onYouTubePlayerReady(event, videoId, lessonId, courseId);
                },
                'onStateChange': function(event) {
                    console.log('🎬 YOUTUBE EVENT: onStateChange fired!', event.data);
                    onYouTubePlayerStateChange(event, videoId, lessonId, courseId);
                },
                'onError': function(event) {
                    console.log('🎬 YOUTUBE EVENT: onError fired!', event.data);
                    onYouTubePlayerError(event);
                }
            }
        });
        
        // Additional debug: Check if player object was created
        setTimeout(() => {
            console.log('🎬 PLAYER STATUS CHECK:', {
                playerExists: !!youtubePlayer,
                playerState: youtubePlayer ? youtubePlayer.getPlayerState() : 'N/A',
                duration: youtubePlayer ? youtubePlayer.getDuration() : 'N/A'
            });
        }, 2000);
        console.log('✅ YOUTUBE PLAYER: Creation initiated with video ID:', youtubeVideoId);
        updateDebugStatus('player', 'Player Created ✅', '#10b981');
        updateDebugStatus('events', 'Waiting for Ready...', '#fbbf24');
    } catch (error) {
        console.error('❌ YOUTUBE PLAYER: Creation failed', error);
        updateDebugStatus('player', 'Creation Failed ❌', '#f87171');
    }
}

function onYouTubePlayerReady(event, videoId, lessonId, courseId) {
    console.log('🎬 YOUTUBE PLAYER: Ready for tracking!');
    updateDebugStatus('events', 'Player Ready ✅', '#10b981');
    updateDebugStatus('progress', 'Starting Tracking...', '#fbbf24');
    youtubePlayerReady = true;
    
    // Start periodic progress tracking
    startYouTubeProgressInterval(videoId, lessonId, courseId);
}

function onYouTubePlayerStateChange(event, videoId, lessonId, courseId) {
    const states = {
        '-1': 'unstarted',
        '0': 'ended', 
        '1': 'playing',
        '2': 'paused',
        '3': 'buffering',
        '5': 'video cued'
    };
    
    const stateName = states[event.data] || 'unknown';
    console.log('🎬 YOUTUBE STATE:', stateName);
    updateDebugStatus('events', `State: ${stateName}`, '#10b981');
    
    if (event.data === 1) { // playing
        console.log('🎬 YOUTUBE: Video started playing - starting progress tracking');
        startYouTubeProgressInterval(videoId, lessonId, courseId);
    } else if (event.data === 2 || event.data === 0) { // paused or ended
        console.log('🎬 YOUTUBE: Video paused/ended - stopping progress tracking');
        stopYouTubeProgressInterval();
        
        if (event.data === 0) { // ended
            console.log('🎬 YOUTUBE: Video completed! Sending 100% progress');
            const duration = youtubePlayer.getDuration();
            sendYouTubeProgressToServer(100, duration, videoId, lessonId, courseId);
        }
    }
}

function onYouTubePlayerError(event) {
    console.error('🎬 YOUTUBE ERROR:', event.data);
    const errorCodes = {
        2: 'Invalid parameter',
        5: 'HTML5 player error', 
        100: 'Video not found',
        101: 'Video not allowed in embedded players',
        150: 'Video not allowed in embedded players'
    };
    console.error('🎬 ERROR DETAILS:', errorCodes[event.data] || 'Unknown error');
}

function startYouTubeProgressInterval(videoId, lessonId, courseId) {
    if (!youtubePlayerReady || !youtubePlayer) {
        updateDebugStatus('progress', 'Player Not Ready ❌', '#f87171');
        return;
    }
    
    // Clear any existing interval
    stopYouTubeProgressInterval();
    updateDebugStatus('progress', 'Tracking Active ✅', '#10b981');
    
    youtubeProgressInterval = setInterval(function() {
        try {
            const currentTime = youtubePlayer.getCurrentTime();
            const duration = youtubePlayer.getDuration();
            
            if (duration > 0) {
                const progressPercentage = Math.round((currentTime / duration) * 100);
                
                console.log('🎬 YOUTUBE PROGRESS:', {
                    currentTime: Math.round(currentTime),
                    duration: Math.round(duration),
                    progress: progressPercentage + '%'
                });
                
                // Update visible progress in debug panel
                const progressElement = document.getElementById('yt-debug-current-progress');
                if (progressElement) {
                    progressElement.innerHTML = `Current Progress: <span style="color: #60a5fa;">${progressPercentage}% (${Math.round(currentTime)}s / ${Math.round(duration)}s)</span>`;
                }
                
                // Send progress to server using existing endpoint
                sendYouTubeProgressToServer(progressPercentage, currentTime, videoId, lessonId, courseId);
                
                // Update UI if functions exist
                if (typeof updateLessonProgress === 'function') {
                    updateLessonProgress(progressPercentage);
                }
            }
        } catch (error) {
            console.error('🎬 YOUTUBE PROGRESS ERROR:', error);
            updateDebugStatus('progress', 'Error: ' + error.message, '#f87171');
        }
    }, 3000); // Update every 3 seconds
}

function stopYouTubeProgressInterval() {
    if (youtubeProgressInterval) {
        clearInterval(youtubeProgressInterval);
        youtubeProgressInterval = null;
    }
}

function sendYouTubeProgressToServer(progressPercentage, currentTime, videoId, lessonId, courseId) {
    console.log('📤 SENDING YOUTUBE PROGRESS:', {
        videoId: videoId, 
        lessonId: lessonId, 
        courseId: courseId, 
        progress: progressPercentage + '%',
        time: Math.round(currentTime) + 's'
    });
    
    updateDebugStatus('ajax', 'Sending...', '#fbbf24');
    
    // Use direct AJAX call to working endpoint (bypass potentially broken existing function)
    $.ajax({
        url: 'ajax/update_video_progress.php',
        method: 'POST',
        dataType: 'json', // Expect JSON response
        data: {
            video_id: videoId,
            lesson_id: lessonId,
            course_id: courseId,
            progress_percentage: Math.round(progressPercentage),
            current_time: Math.round(currentTime),
            completed: progressPercentage >= 90 ? 1 : 0
        },
        success: function(response) {
            console.log('✅ YOUTUBE PROGRESS SAVED:', response);
            updateDebugStatus('ajax', 'Success ✅', '#10b981');
            
            // Update course progress if response contains it
            if (response && response.course_progress !== undefined) {
                updateCourseProgressDisplay(response.course_progress);
            }
            
            // Update main course progress banner if available
            if (typeof updateMainCourseProgress === 'function') {
                updateMainCourseProgress();
            }
        },
        error: function(xhr, status, error) {
            console.error('❌ YOUTUBE PROGRESS SAVE FAILED:', {
                status: status,
                error: error,
                responseText: xhr.responseText,
                statusCode: xhr.status
            });
            
            updateDebugStatus('ajax', 'Failed: ' + status, '#f87171');
            
            // If it's a JSON parsing error, log the raw response
            if (error.includes('JSON') || status === 'parsererror') {
                console.error('❌ RAW RESPONSE:', xhr.responseText);
                console.error('❌ This suggests the PHP endpoint returned invalid JSON');
            }
        });
}

function updateCourseProgressDisplay(courseProgress) {
    const progressElement = document.getElementById('main-course-progress-percentage');
    const progressBar = document.getElementById('main-course-progress-bar');
    
    if (progressElement && progressBar) {
        progressElement.textContent = Math.round(courseProgress) + '%';
        progressBar.style.width = Math.round(courseProgress) + '%';
        console.log('✅ COURSE PROGRESS UPDATED:', Math.round(courseProgress) + '%');
    }
}

// Export functions for global use
window.initializeYouTubeProgressTracking = initializeYouTubeProgressTracking;
