<?php
/**
 * Database Migration: Add contributes_to_final_grade column to quizzes table
 * This allows teachers to control which quizzes count toward students' final grades
 */

require_once '../config/db_connect.php';

echo "<h1>🎯 Quiz Final Grade Column Migration</h1>";

try {
    // Check if column already exists
    echo "<h3>📋 Step 1: Checking current table structure</h3>";
    
    $check_query = "SHOW COLUMNS FROM quizzes LIKE 'contributes_to_final_grade'";
    $result = $pdo->query($check_query);
    
    if ($result->rowCount() > 0) {
        echo "<p>✅ <strong>Column already exists!</strong> No migration needed.</p>";
    } else {
        echo "<p>📝 Column doesn't exist. Proceeding with migration...</p>";
        
        // Add the new column
        echo "<h3>🔧 Step 2: Adding contributes_to_final_grade column</h3>";
        
        $alter_query = "
            ALTER TABLE quizzes 
            ADD COLUMN contributes_to_final_grade TINYINT(1) DEFAULT 1 COMMENT 'Whether this quiz contributes to final grade calculation'
        ";
        
        $pdo->exec($alter_query);
        echo "<p>✅ <strong>Successfully added contributes_to_final_grade column!</strong></p>";
        
        // Verify the addition
        echo "<h3>✅ Step 3: Verifying column addition</h3>";
        $verify_query = "SHOW COLUMNS FROM quizzes LIKE 'contributes_to_final_grade'";
        $verify_result = $pdo->query($verify_query);
        
        if ($verify_result->rowCount() > 0) {
            echo "<p>✅ <strong>Verification successful!</strong> Column added correctly.</p>";
            
            // Show column details
            $column_info = $verify_result->fetch(PDO::FETCH_ASSOC);
            echo "<div style='background: #f0f8ff; padding: 15px; border-radius: 8px; margin: 10px 0;'>";
            echo "<strong>Column Details:</strong><br>";
            echo "• <strong>Field:</strong> " . $column_info['Field'] . "<br>";
            echo "• <strong>Type:</strong> " . $column_info['Type'] . "<br>";
            echo "• <strong>Default:</strong> " . ($column_info['Default'] ?? 'NULL') . "<br>";
            echo "• <strong>Comment:</strong> " . ($column_info['Comment'] ?? 'None') . "<br>";
            echo "</div>";
        } else {
            echo "<p>❌ <strong>Verification failed!</strong> Column may not have been added correctly.</p>";
        }
    }
    
    // Show current table structure
    echo "<h3>📊 Step 4: Current quizzes table structure</h3>";
    $structure_query = "SHOW COLUMNS FROM quizzes";
    $structure_result = $pdo->query($structure_query);
    
    echo "<table border='1' cellpadding='8' cellspacing='0' style='border-collapse: collapse; width: 100%; margin: 10px 0;'>";
    echo "<tr style='background-color: #f2f2f2;'>";
    echo "<th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th>";
    echo "</tr>";
    
    while ($column = $structure_result->fetch(PDO::FETCH_ASSOC)) {
        echo "<tr>";
        echo "<td><strong>" . htmlspecialchars($column['Field']) . "</strong></td>";
        echo "<td>" . htmlspecialchars($column['Type']) . "</td>";
        echo "<td>" . htmlspecialchars($column['Null']) . "</td>";
        echo "<td>" . htmlspecialchars($column['Key']) . "</td>";
        echo "<td>" . htmlspecialchars($column['Default'] ?? 'NULL') . "</td>";
        echo "<td>" . htmlspecialchars($column['Extra']) . "</td>";
        echo "</tr>";
    }
    echo "</table>";
    
    // Test data update (set some quizzes to not contribute to final grade)
    echo "<h3>🧪 Step 5: Sample data update</h3>";
    
    $count_query = "SELECT COUNT(*) as total FROM quizzes";
    $count_result = $pdo->query($count_query)->fetch(PDO::FETCH_ASSOC);
    $total_quizzes = $count_result['total'];
    
    echo "<p>📊 <strong>Total quizzes in database:</strong> $total_quizzes</p>";
    
    if ($total_quizzes > 0) {
        // Update some sample quizzes for demonstration
        $sample_update = "UPDATE quizzes SET contributes_to_final_grade = 1 WHERE quiz_id > 0";
        $pdo->exec($sample_update);
        echo "<p>✅ <strong>All existing quizzes set to contribute to final grade by default.</strong></p>";
        
        // Show quiz breakdown
        $breakdown_query = "
            SELECT 
                contributes_to_final_grade,
                COUNT(*) as count,
                ROUND((COUNT(*) * 100.0 / $total_quizzes), 1) as percentage
            FROM quizzes 
            GROUP BY contributes_to_final_grade
        ";
        
        $breakdown_result = $pdo->query($breakdown_query);
        
        echo "<h4>📈 Quiz Final Grade Contribution Breakdown:</h4>";
        echo "<table border='1' cellpadding='8' cellspacing='0' style='border-collapse: collapse; margin: 10px 0;'>";
        echo "<tr style='background-color: #f2f2f2;'>";
        echo "<th>Contributes to Final Grade</th><th>Count</th><th>Percentage</th>";
        echo "</tr>";
        
        while ($row = $breakdown_result->fetch(PDO::FETCH_ASSOC)) {
            $contributes = $row['contributes_to_final_grade'] ? 'Yes' : 'No';
            $color = $row['contributes_to_final_grade'] ? '#d1fae5' : '#fee2e2';
            echo "<tr style='background-color: $color;'>";
            echo "<td><strong>$contributes</strong></td>";
            echo "<td>" . $row['count'] . "</td>";
            echo "<td>" . $row['percentage'] . "%</td>";
            echo "</tr>";
        }
        echo "</table>";
    }
    
    echo "<h3>🎉 Migration Summary</h3>";
    echo "<div style='background: #d1fae5; border: 1px solid #059669; padding: 15px; border-radius: 8px; margin: 15px 0;'>";
    echo "<h4 style='color: #059669; margin-top: 0;'>✅ Migration Completed Successfully!</h4>";
    echo "<p><strong>What was done:</strong></p>";
    echo "<ul>";
    echo "<li>✅ Added <code>contributes_to_final_grade</code> column to quizzes table</li>";
    echo "<li>✅ Set default value to 1 (contributes by default)</li>";
    echo "<li>✅ All existing quizzes set to contribute to final grade</li>";
    echo "<li>✅ Column verified and structure confirmed</li>";
    echo "</ul>";
    echo "<p><strong>Next step:</strong> Update the quiz creation form to include the final grade option.</p>";
    echo "</div>";
    
} catch (PDOException $e) {
    echo "<div style='background: #fee2e2; border: 1px solid #dc2626; padding: 15px; border-radius: 8px; margin: 15px 0;'>";
    echo "<h4 style='color: #dc2626; margin-top: 0;'>❌ Database Error</h4>";
    echo "<p><strong>Error:</strong> " . htmlspecialchars($e->getMessage()) . "</p>";
    echo "<p><strong>Code:</strong> " . $e->getCode() . "</p>";
    echo "</div>";
} catch (Exception $e) {
    echo "<div style='background: #fef3c7; border: 1px solid #d97706; padding: 15px; border-radius: 8px; margin: 15px 0;'>";
    echo "<h4 style='color: #d97706; margin-top: 0;'>⚠️ General Error</h4>";
    echo "<p><strong>Error:</strong> " . htmlspecialchars($e->getMessage()) . "</p>";
    echo "</div>";
}

echo "<hr>";
echo "<p><strong>Instructions:</strong></p>";
echo "<ol>";
echo "<li>✅ Run this script first to add the database column</li>";
echo "<li>🔧 Update the quiz creation form to include the final grade checkbox</li>";
echo "<li>📝 Modify the quiz creation logic to handle the new field</li>";
echo "<li>🧪 Test creating new quizzes with/without final grade contribution</li>";
echo "</ol>";
?>

<style>
body { font-family: Arial, sans-serif; margin: 20px; line-height: 1.6; }
h1, h3 { color: #333; }
table { width: 100%; }
th { background-color: #f2f2f2; font-weight: bold; }
td, th { padding: 8px; text-align: left; border: 1px solid #ddd; }
code { background-color: #f4f4f4; padding: 2px 5px; border-radius: 3px; }
</style>
